#define _XOPEN_SOURCE 500

#include "global.h"
#include "struct.h"

#include "ch_time_mod.h"
#include "log.h"

#include <stdio.h>
#include <stdlib.h>

#include <dirent.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <glib.h>

// 関数プロトタイプ
void create_symbolink(struct cp_target *cp_tmp);

/*******************************************************************************
シンボリックリンクを作成
*******************************************************************************/
void create_symbolink(struct cp_target *cp_tmp)
{
	int readlink_len;
	char link_buf[PATH_LEN];
	errno = 0;

	if((readlink_len = readlink(cp_tmp->from, link_buf, PATH_LEN)) == -1)
	{
		log_errors(__FILE__, __LINE__, errno, cp_tmp->from);
		fprintf(stderr, "シンボリックリンク読み込みエラーです\n");
		return;
	}
	else
	{
		/*
		readlinkは終端にヌル文字を付加しない。
		という仕様の所為でlink_bufの文字列がおかしくなるバグに悩まされた。
		ローカル変数はスコープから抜けると解放されるが、
		即座に解放されるわけではなく（パフォーマンスが落ちるし）、
		領域を使い回すこともしばしばある。
		この関数が呼び出されるとlink_bufにシンボリックリンクの内容がコピーされるが、
		関数から抜けてもlink_bufの内容は保持されたままになる。
		次回呼び出されたときにlink_bufの内容が上書きされ、
		それが前回よりも短い文字列だった場合、ヌル文字がないので、
		文字列が混ざり合っておかしな事になっていた。
		*/
		link_buf[readlink_len] = '\0';

		if(g_file_test(cp_tmp->to, G_FILE_TEST_EXISTS) == TRUE)
			unlink(cp_tmp->to);

		if(symlink(link_buf, cp_tmp->to) == -1)
		{
			log_errors(__FILE__, __LINE__, errno, cp_tmp->to);
			fprintf(stderr, "シンボリックリンク作成エラーです\n");
		}
		else
		{
			struct stat stat_buf;
			if(lstat(cp_tmp->from, &stat_buf) == -1)
			{
				log_errors(__FILE__, __LINE__, errno, cp_tmp->from);
			}
			else
			{
				ch_time_mod(&stat_buf, cp_tmp->to, SYMBOLIC);
			}
		}
	}
}
