{*********************************************************

 SlavaNap source code.

 Copyright 2001,2002 by CyberAlien@users.sourceforge.net
 Released under GNU General Public License

 Latest version is available at
 http://slavanap2.sourceforge.net

**********************************************************

 Unit: share

 Class for list of shared files

*********************************************************}
unit share;

interface

uses
 Windows, Classes, SysUtils, stypes, constants, slavastrings, class_cmdlist;

{$I defines.pas}

const
 SHARED_AUDIO = 0;
 SHARED_VIDEO = 1;
 SHARED_TEXT  = 2;
 SHARED_IMAGE = 3;
 SHARED_APP   = 4;
 SHARED_CD    = 5;
  // MP3 - sorted by bitrate - this way search is several times faster is bitrate is specified.
 SHARED_320   = 6;
 SHARED_256   = 7;
 SHARED_224   = 8;
 SHARED_192   = 9;
 SHARED_160   = 10;
 SHARED_128   = 11;
 SHARED_112   = 12;
 SHARED_64    = 13;
 SHARED_OTHER = 14; // MP3 with other bitrate
  // mime types
 TYPE_MP3     = 0;
 TYPE_AUDIO   = 1;
 TYPE_VIDEO   = 2;
 TYPE_IMAGE   = 3;
 TYPE_APP     = 4;
 TYPE_CD      = 5;
 TYPE_TEXT    = 6;
 // totals
 SHARED_MAX   = 14;
 SHARED_OTHER_MAX = 5;
 SHARED_MP3_MIN = 6;
 TYPE_MAX     = 6;
 TYPE_INVALID = 255;
 SHARED_INVALID = 255;

type
  TShare = packed record
    short        : String; // short file name (see ExtractMPName function)
    name         : String; // full file name
    index        : Word; // index in folders list
    size         : LongWord; // size in bytes
    bitrate      : Word; // bitrate (MP3 only)
    frequency    : Word; // frequency (MP3 only)
    time         : Word; // time (MP3 only, in seconds)
    mime         : Byte; // mime type
    id           : Byte; // SHARED_xxx
  end;
  PShare = ^TShare;
  TShareList = class(TList)
    dirs         : TNapCmdList;
    function  Add(Value: TShare):PShare;
    procedure Delete(index: Integer; delete_index: Boolean = true);
    function  FindFile(filename: String): Integer; overload;
    function  FindFile(index: Integer; shortname: String): Integer; overload;
    function  FindFile(folder: String; shortname: String): Integer; overload;
    function  FindRec(filename: String): PShare;
    procedure Clear; override;
    constructor Create;
    destructor Destroy; override;
    function  Id(index: Integer): Integer;
    procedure DecreaseIndex(n: Integer);
    procedure IncreaseIndex(n: Integer);
    function  AddFolder(folder: String): Integer;
    function  GetFileName(index: Integer): String;
    function  GetFolder(index: Integer): String;
  end;

function  GetType(ext: String): Integer;
function  ExtractMPName(src: String): String;
function  StrToType(str:String): Integer;
procedure SearchInclude(var data: TSearchStruct; str: String; include: Boolean=true);
procedure SplitFileName(filename: String; var folder: String; var name: String);
function  CreateShareList: TShareList;
procedure FreeShareList(list: TShareList);
{$IFNDEF DISABLE_MEMORY_MANAGER}
procedure ExpireShareLists;
{$ENDIF}

var
 count_sharelist,
 count_sharelist_max: Integer;
 count_sharelist_items,
 count_sharelist_items_max: Integer;

implementation

uses
 vars, thread, memory_manager;

{$IFNDEF DISABLE_MEMORY_MANAGER}
var
 pool_lists: TList;
 pool_items: TList;
{$ENDIF}

{* * * * * internal functions * * * * *}

function  CreateShareList: TShareList;
begin
{$IFNDEF DISABLE_MEMORY_MANAGER}
 if pool_lists.count>0 then
 begin
  Result:=pool_lists.Items[pool_lists.count-1];
  pool_lists.Delete(pool_lists.count-1);
 end
 else
{$ENDIF}
  Result:=TShareList.Create;
end;

procedure FreeShareList(list: TShareList);
begin
 list.Clear;
{$IFNDEF DISABLE_MEMORY_MANAGER}
 if not running then
{$ENDIF}
  list.Free
{$IFNDEF DISABLE_MEMORY_MANAGER}
 else
  pool_lists.Add(list);
{$ENDIF}
end;

function  CreateShareItem: PShare;
var
 p: PShare;
begin
{$IFNDEF DISABLE_MEMORY_MANAGER}
 if pool_items.count>0 then
 begin
  Result:=pool_items.Items[pool_items.count-1];
  pool_items.Delete(pool_items.count-1);
 end
 else
 begin
{$ENDIF}
  p:=AllocMem(sizeof(TShare));
  Pointer(p^.short):=nil;
  Pointer(p^.name):=nil;
  inc(count_sharelist_items);
  if count_sharelist_items>count_sharelist_items_max then count_sharelist_items_max:=count_sharelist_items;
  Result:=p;
{$IFNDEF DISABLE_MEMORY_MANAGER}
 end;
{$ENDIF}
end;

procedure ClearShareItem(item: PShare);
begin
 SetLength(item^.short,0);
 SetLength(item^.name,0);
end;

procedure FreeShareItem(item: PShare);
begin
 ClearShareItem(item);
 Finalize(item^);
 FreeMem(item,sizeof(TShare));
 dec(count_sharelist_items);
end;

procedure DeleteShareItem(item: PShare);
begin
{$IFNDEF DISABLE_MEMORY_MANAGER}
 if not running then
{$ENDIF}
  FreeShareItem(item)
{$IFNDEF DISABLE_MEMORY_MANAGER}
 else
 begin
  ClearShareItem(item);
  pool_items.Add(item);
 end; 
{$ENDIF}
end;

{$IFNDEF DISABLE_MEMORY_MANAGER}
procedure ExpireShareLists;
var
 l: TShareList;
begin
 if pool_lists.count>50 then
 while (pool_lists.count * 3) > count_sharelist do
 try
   l:=pool_lists.Items[pool_lists.Count-1];
   pool_lists.Delete(pool_lists.count-1);
   l.Free;
  except
   exit;
 end;
 if pool_items.count>1000 then
 while (pool_items.count * 3) > count_sharelist_items do
 try
   FreeShareItem(pool_items.Items[pool_items.count-1]);
   pool_items.Delete(pool_items.count-1);
  except
   exit;
 end;
end;
{$ENDIF}

{$IFNDEF DISABLE_MEMORY_MANAGER}
procedure ClearShareLists;
var
 i: Integer;
 l: TShareList;
begin
 for i:=0 to pool_lists.count-1 do
 try
   l:=pool_lists.Items[i];
   l.Free;
  except
 end;
 pool_lists.Clear;
 for i:=0 to pool_items.count-1 do
 try
  FreeShareItem(pool_items.Items[i]);
  except
 end;
 pool_items.Clear; 
end;
{$ENDIF}

{* * * * * TShareList * * * * *}

constructor TShareList.Create;
begin
 inherited Create;
 inc(count_sharelist);
 if count_sharelist>count_sharelist_max then count_sharelist_max:=count_sharelist;
 dirs:=CreateCmdList;
end;

destructor TShareList.Destroy;
begin
 Clear;
 dec(count_sharelist);
 if dirs<>nil then FreeCmdList(dirs);
 dirs:=nil;
 inherited Destroy;
end;

function  TShareList.Add(Value: TShare):PShare;
var
 data:PShare;
begin
 data:=CreateShareItem;
 with data^ do
 begin
  short:=Value.short;
  name:=Value.name;
  size:=Value.size;
  bitrate:=Value.bitrate;
  frequency:=Value.frequency;
  time:=Value.time;
  mime:=Value.mime;
  id:=Value.id;
  index:=Value.index;
 end;
 inherited Add(data);
 Result:=data;
 IncreaseIndex(value.index);
end;

procedure TShareList.Delete(index: Integer; delete_index: Boolean = true);
begin
 if (Index<0) or (Index>=count) then exit;
 if delete_index then DecreaseIndex(PShare(Items[index])^.index);
 if Items[Index]<>nil then
  DeleteShareItem(Items[Index]);
 Inherited Delete(index);
end;

function  TShareList.Id(index: Integer): Integer;
begin
 if (Index<0) or (Index>=count) then
 begin
  Result:=SHARED_INVALID;
  exit;
 end;
 Result:=PShare(Items[index]).id;
end;

procedure TShareList.Clear;
begin
 while count>0 do
  Delete(count-1,false);
 if dirs<>nil then
  dirs.Clear;
 inherited Clear;
end;

function  TShareList.FindFile(filename: String): Integer;
var
 i,num:Integer;
 name, folder: String;
begin
 Result:=-1;
 SplitFileName(filename,folder,name);
 num:=dirs.FindByCmd(folder);
 if num=-1 then exit;
 for i:=count-1 downto 0 do
  if PShare(Items[i])^.index=num then
   if PShare(Items[i])^.name=name then
   begin
    Result:=i;
    exit;
   end;
{$IFNDEF DISABLE_CPU_THROTTLE}
 if (count>50) then CheckSync;
{$ENDIF}
end;

function  TShareList.FindFile(index: Integer; shortname: String): Integer;
var
 i:Integer;
begin
 Result:=-1;
 if index=-1 then exit;
 for i:=count-1 downto 0 do
  if PShare(Items[i])^.index=index then
   if PShare(Items[i])^.name=shortname then
   begin
    Result:=i;
    exit;
   end;
{$IFNDEF DISABLE_CPU_THROTTLE}
 if (count>50) then CheckSync;
{$ENDIF}
end;

function  TShareList.FindFile(folder: String; shortname: String): Integer;
var
 i, index: Integer;
begin
 Result:=-1;
 index:=dirs.FindByCmd(folder);
 if index=-1 then exit;
 for i:=count-1 downto 0 do
  if PShare(Items[i])^.index=index then
   if PShare(Items[i])^.name=shortname then
   begin
    Result:=i;
    exit;
   end;
{$IFNDEF DISABLE_CPU_THROTTLE}
 if (count>50) then CheckSync;
{$ENDIF}
end;

function  TShareList.FindRec(filename: String): PShare;
var
 i,num:Integer;
 name, folder: String;
begin
 Result:=nil;
 SplitFileName(filename,folder,name);
 num:=dirs.FindByCmd(folder);
 if num=-1 then exit;
 for i:=count-1 downto 0 do
  if PShare(Items[i])^.index=num then
   if PShare(Items[i])^.name=name then
   begin
    Result:=Items[i];
    exit;
   end;
{$IFNDEF DISABLE_CPU_THROTTLE}
 if (count>50) then CheckSync;
{$ENDIF}
end;

function TShareList.AddFolder(folder: String): Integer;
begin
 Result:=dirs.FindByCmd(folder);
 if Result=-1 then Result:=dirs.AddCmd(0,folder);
end;

procedure TShareList.IncreaseIndex(n: Integer);
begin
 if n=-1 then exit;
 inc(PNapCmd(dirs.Items[n])^.id); 
end;

procedure TShareList.DecreaseIndex(n: Integer);
var
 i: Integer;
 rec: PShare;
begin
 if n=-1 then exit;
 if dirs=nil then exit;
 i:=dirs.Id(n);
 PNapCmd(dirs.Items[n])^.id:=i-1;
 if i<2 then
 begin
  dirs.Delete(n);
  for i:=0 to count-1 do
  begin
    rec:=Items[i];
    if rec^.index>n then
     dec(rec^.index);
  end;
 end;
end;

function  TShareList.GetFileName(index: Integer): String;
var
 rec: PShare;
begin
 rec:=Items[index];
 if rec^.index=-1 then Result:=rec^.name
 else Result:=GetFolder(rec^.index)+rec^.name;
end;

function  TShareList.GetFolder(index: Integer): String;
begin
  if index=-1 then Result:=''
  else Result:=dirs.Str(index);
end;

procedure SplitFileName(filename: String; var folder: String; var name: String);
var
 i,p: Integer;
begin
 for i:=Length(filename)-3 downto 1 do
 if ((filename[i]='\') and (ByteType(filename,i-1)=mbSingleByte)) or (filename[i]='/') then
 begin
   folder:=Copy(filename,1,i);
   name:=Copy(filename,i+1,Length(filename)-i);
   exit;
 end;
 folder:='';
 name:=filename;
end;

function GetType(ext: String): Integer;
begin
  Result:=TYPE_MAX+1;
  ext:=lowercase(ext);
  if Length(ext)<2 then exit;
  if ext[1]='.' then ext:=Copy(ext,2,Length(ext));
  if Length(ext)<2 then exit;
  if StrHash_FindString(ext_mp3_list,ext,false) then
  begin
    Result:=TYPE_MP3;
    exit;
  end;
  if StrHash_FindString(ext_audio_list,ext,false) then
  begin
    Result:=TYPE_AUDIO;
    exit;
  end;
  if StrHash_FindString(ext_video_list,ext,false) then
  begin
    Result:=TYPE_VIDEO;
    exit;
  end;
  if StrHash_FindString(ext_app_list,ext,false) then
  begin
    Result:=TYPE_APP;
    exit;
  end;
  if StrHash_FindString(ext_image_list,ext,false) then
  begin
    Result:=TYPE_IMAGE;
    exit;
  end;
  if StrHash_FindString(ext_cd_list,ext,false) then
  begin
    Result:=TYPE_CD;
    exit;
  end;
  if StrHash_FindString(ext_text_list,ext,false) then
  begin
    Result:=TYPE_TEXT;
    exit;
  end;
end;

function ExtractMPName(src: String): String;
// makes filename short - to speed up search process
var
 str: String;
 hash: TStringHash;
 h1: PStringHash;
 i,j,k: Integer;
 b: Boolean;
 p: PStringHashItem;
 function count(str: String):Integer;
 var
  i,j:Integer;
 begin
   j:=0;
   for i:=1 to Length(str) do
    if (str[i]='\') and (ByteType(str,i-1)=mbSingleByte) then
    begin
     str[i]:=' ';
     inc(j);
    end;
   Result:=j;
 end;
begin
 str:=AnsiLowerCase(src);
 for i:=1 to Length(str) do
  if str[i]='/' then str[i]:='\';
 if str[1]='\' then
  str:=copy(str,2,length(str));
  // removing extension
// j:=Length(ExtractFileExt(str));
// str:=copy(str,1,Length(str)-j);
 // counting "\";
 k:=count(str);
 if k>folder_depth then
 begin
  j:=0;
  if k>folder_depth+1 then
  for i:=1 to Length(str) do
   if (str[i]='\') and (ByteType(str,i-1)=mbSingleByte) then
   begin
    str[i]:='_';
    inc(j);
    if j=(k-(folder_depth+1)) then break;
   end;
  j:=AnsiPos('\',str)+1;
  k:=Length(str)-j;
  str:=Copy(str,j,k+1);
 end;
 StrHash_Reset(hash);
 SplitString(StripString(str),hash);
 str:='';
 if share_matchedfile_only then
 begin
  b:=false;
  for i:=0 to db_blocks.Count-1 do
  begin
{$IFNDEF DISABLE_CPU_THROTTLE}
   if ((i mod 10)=0) then CheckSync;
{$ENDIF}
   h1:=db_blocks.Items[i];
   p:=h1^.first;
   while (p<>nil) do
   begin
    if StrHash_FindString(hash,p^.data,false)then begin b:=true; break; end;
    p:=p^.next;
   end;
   if b then break;
{$IFNDEF DISABLE_CPU_THROTTLE}
   if ((i mod 10)=0) then CheckSync;
{$ENDIF}
  end;
  if not b then
  begin
   StrHash_Clear(hash);
   Result:='_';
   exit;
  end;
 end
 else
 begin
  for i:=0 to db_blocks.Count-1 do
  begin
{$IFNDEF DISABLE_CPU_THROTTLE}
   if ((i mod 10)=0) then CheckSync;
{$ENDIF}
   h1:=db_blocks.Items[i];
   b:=true;
   p:=h1^.first;
   while (p<>nil) and b do
   begin
    if not StrHash_FindString(hash,p^.data,false)then b:=false;
    p:=p^.next;
   end;
   if b then
   begin
     StrHash_Clear(hash);
     Result:='_';
     exit;
   end;
{$IFNDEF DISABLE_CPU_THROTTLE}
   if ((i mod 10)=0) then CheckSync;
{$ENDIF}
  end;
 end;
 str:=JoinString(hash);
 StrHash_Clear(hash);
 str:=Trim(str);
{ if Length(str)>maxshareindex then
  if (ByteType(str,Length(str)-maxshareindex)=mbTrailByte) then
   str:=Copy(str,Length(str)-maxshareindex+2,maxshareindex)
  else
   str:=Copy(str,Length(str)-maxshareindex+1,maxshareindex);}
 if Length(str)>0 then Result:=' '+str+' '
 else Result:=str;
end;

function  StrToType(str:String): Integer;
begin
 str:=lowercase(str);
 Result:=TYPE_INVALID;
 if pos('mp3',str)<>0 then Result:=TYPE_MP3;
 if pos('audio',str)<>0 then Result:=TYPE_AUDIO;
 if pos('video',str)<>0 then Result:=TYPE_VIDEO;
 if pos('text',str)<>0 then Result:=TYPE_TEXT;
 if pos('image',str)<>0 then Result:=TYPE_IMAGE;
 if pos('app',str)<>0 then Result:=TYPE_APP;
 if pos('cd',str)<>0 then Result:=TYPE_CD;
end;

procedure SearchInclude(var data: TSearchStruct; str: String; include: Boolean=true);
var
 lst: TStringList;
 str1: String;
 num,i,j: Integer;
 b: Boolean;
begin
 lst:=CreateStringList;
 try
   str1:=StripString(AnsiLowerCase(str),true);
   SplitString(str1,lst);
   if lst.count>0 then
   for i:=lst.Count-1 downto 0 do
   begin
     str1:=Trim(lst.Strings[i]);
     if Length(str1)=0 then
      lst.Delete(i)
     else if str1[1]='-' then
      if Length(str1)=1 then
       lst.Delete(i);
   end;
   if lst.Count=0 then
   begin
     FreeStringList(lst);
     exit;
   end;
   for i:=0 to lst.Count-1 do
   begin
     b:=include;
     str1:=lst.Strings[i];
     if str1[1]='-' then
     begin
        str1:=Copy(str1,2,64);
        b:=false;
     end;
     if Length(str1)=0 then continue;
     if b then
     begin
       num:=-1;
       for j:=max_search_array-1 downto 0 do
       begin
          if data.include[j]=str1 then b:=false;
          if data.include[j]='' then num:=j;
       end;
       if num=-1 then b:=false;
       if b then data.include[num]:=str1;
     end
     else
     begin
       num:=-1;
       for j:=max_search_array-1 downto 0 do
       begin
          if data.exclude[j]=str1 then b:=true;
          if data.exclude[j]='' then num:=j;
       end;
       if num=-1 then b:=true;
       if not b then data.exclude[num]:=str1;
     end
   end;
  except
 end;
 FreeStringList(lst);
end;

initialization
begin
 count_sharelist:=0;
 count_sharelist_max:=0;
 count_sharelist_items:=0;
 count_sharelist_items_max:=0;
{$IFNDEF DISABLE_MEMORY_MANAGER}
 pool_lists:=TList.Create;
 pool_items:=TList.Create;
{$ENDIF}
end;

{$IFNDEF DISABLE_MEMORY_MANAGER}
finalization
begin
 ClearShareLists;
 pool_lists.Free;
 pool_items.Free;
end;
{$ENDIF}
end.
