%module skf

%{
/* -------------------------------------------- */
/* SWIG interface C definitions			*/
/* -------------------------------------------- */
/* ********************************************************************
**
** skf_convert.i: skf extension interfaces
**
** Copyright (c) 2005-2015 Seiji Kaneko. All rights reserved.
** Everyone is permitted to use this program in source and binary
** form, with or without modification if and only if the following
** conditions are met:
** 1. Redistributions of source code must retain the above copyright
**   notice, copyright notice written in source code, additional list
**   of conditions and the following disclaimer.
** 2. Redistributions in machine readable form must reproduce the 
**   above copyright notice and the following disclaimer in the
**   documentation or other material provided with the distribution.
** 3. Neither the name of the copyright holders nor the names of its 
**   contributors may be used to endorse or promote products derived
**   from this software without specific prior written permission.
**********************************************************************
** Disclaimer: This software is provided and distributed AS IS, 
**	without any implicit or explicit warranties, and not
**	guaranteed to be error-free. In no event shall the author be
**	liable for any direct, indirect or incidental damages,
**	including, but not limited to, loss of data, use or profits
**	responsibility for any direct or indirect damages or results
**	arising by using whole or a part of this software.
**********************************************************************
** RUBY EXTENSION
** Usage: Skf.guess(option_string, string_to_convert) -- returns states
**	  Skf.convert(option_string, string_to_convert) -- conversion
**	  Skf.init(option_string, string_to_convert) -- initialize
**	  Skf.destruct(option_string, string_to_convert) -- free buffers
**
** skf options except -b,-u works as shown in documents. Environment
** variables skfenv and SKFENV is ignored regardless uid.
** conversion refers previous state hold inside skf. To suppress this
** behavior, use Skf.init explicitly.
******************************************************************* */
/* $Id: skf_convert.i,v 1.22 2022/11/12 16:58:38 seiji Exp seiji $ */

/* invocation for some ghosts */
#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
#include <stdlib.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <unistd.h>
#include <errno.h>

#ifdef HAVE_STRING_H
#include <string.h>
#else
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif
#endif

/* --- perl swig --- */
#ifdef SWIGPERL
#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"
#endif

/* --- ruby swig --- */
#ifdef SWIGRUBY

#if defined(SKF_RUBY20) && !defined(SKF_RUBY2)
#define SKF_RUBY2
#endif

/* 1.9-handling: HAS_RUBY_RUBY_H may be broken on Mac OS X */
#if	defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3)
#include "ruby/ruby.h"
#include "ruby/encoding.h"
#else
#include "ruby.h"
#endif
#endif	/* SWIGRUBY */

/* --- php swig --- */
#ifdef SWIGPHP
#include "php.h"
#endif

/* --- python swig --- */
#ifdef SWIGPYTHON
#include "Python.h"
#define LWLINTSTRING 
const char *skf_notstring = "skf: not string\n";
#endif

#if defined(SKF_PYTHON3) && defined(SWIGPYTHON)
#define SWIG_FILE_WITH_INIT
#include "unicodeobject.h"
#include "bytearrayobject.h"
#define	Py_SKFSTR	PyObject
#ifdef USE_PEP393		/* after python 3.3 */
#define skf_PyUnicode_GetSize(x) PyUnicode_GET_LENGTH(x) 
#define skf_PyUnicode_AsUnicode(x) PyUnicode_AS_UNICODE(x)	
#else	/* NOT PEP393 */
#define skf_PyUnicode_GetSize	PyUnicode_GETSIZE
#define skf_PyUnicode_AsUnicode	PyUnicode_AsUnicode
#endif
#else	/* PYTHON2.x */
#define	Py_SKFSTR	char
#endif

#ifndef PyUnicode_4BYTE_KIND	/* care for !PEP393 */
#define PyUnicode_4BYTE_KIND 4
#define PyUnicode_2BYTE_KIND 2
#define PyUnicode_1BYTE_KIND 1
#endif

/* --- others --- */

/* --- skf -------- */
#include "skf.h"
#include "skf_fileio.h"
#include "convert.h"
#include "oconv.h"

#include "skf_convert.h"

/* Note:
   maximum output size is ((input_string_length * 5) + 1) for skf (non-unic*de).
   maximum output size is ((input_string_length * 4) + 6) for skf (unic*de).
   maximum output size is ((input_string_length * 10) + 6) for skf (unic*de w. decomp).
   See skf_fileio.h and .c
*/
#define INCSIZE		256
#define GUESSSIZE	128
#define OPTSTRLEN	1024

#define NO_INIT		0
#define FORCE_INIT	1
#define GUESS_INIT	2

#define SKFNOERR	0
#define SKFINTERR	2

#ifdef __cplusplus
}
#endif

int	swig_state = 0;
int	skf_swig_result = 0;

int	errorcode = 0;		/* return error result condition   */
int	in_saved_codeset;

#if	defined(SKF_PYTHON3) && defined(SWIGPYTHON)
/* p_out_binary: 1(TRUE) - binary 0 - string (unicode) mode	   */
int	p_out_binary = 0;	/* currently python3 only variable */
#endif
struct Skf_localestring	lwlstr_b;
struct Skf_localestring	lwlopt_b;

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)
int	ruby_out_locale_index = 0;
int	ruby_out_ascii_index = 0;
#endif

/* --------------------------------------------------------------- */
/* input/output string conversion				   */
/* --------------------------------------------------------------- */
#if !defined(SKF_PYTHON3) || !defined(SWIGPYTHON)
static struct Skf_localestring	*istrdef;
#endif

/* option string */
char	*optionStr;
struct Skf_localestring *optstr;

#if	!defined(SKF_RUBY19) && !defined(SKF_RUBY2) && (!defined(SKF_PYTHON3) || !defined(SWIGPYTHON))
static int optstr_len = 0;
#endif

static int iencode = -1;

#if	defined(SWIGPYTHON) && defined(SKF_PYTHON3)
void setsstrdummies(sstrdef)
struct Skf_localestring	*sstrdef;
{
    sstrdef->length = 0;	/* return dummies  */
    sstrdef->sstr = NULL;
    sstrdef->lwl_codeset = codeset_utf16be;
};
#endif

/* cygwin64 requires a dummy main function */
#if	defined(__CYGWIN__)
int	main() 
{
}
#endif

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2) || (defined(SKF_PYTHON3) && defined(SWIGPYTHON))
#define SKFSTRUCT_RETURN
#endif

jmp_buf		skf_errbuf;

/* --------------------------------------------------------------- */
/* misc utilities						   */
/* --------------------------------------------------------------- */
void skf_exit(int eval)
{
    errorcode = skf_swig_result;
#if defined(SWIGRUBY)
    if (eval != EXIT_SUCCESS) rb_raise(rb_eSignal,"skf detected fatal error");
#endif
#if defined(SWIGPERL)
    if (eval != EXIT_SUCCESS) croak("skf detected fatal error");
#endif
    longjmp(skf_errbuf,SKFINTERR);
}

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2)|| defined(SKF_RUBY3)
int get_rstr_enc(cname)
char *cname;
{
    return(skf_search_chname(cname));
}
#endif

/* --------------------------------------------------------------- */
/* we need function to calculate string length			   */
/* --------------------------------------------------------------- */
#if	defined(SWIGRUBY) && (defined(SKF_RUBY19) || defined(SKF_RUBY2)) || defined(SKF_RUBY3)
/* ruby 1.9 or later */
size_t  skf_swig_strlen(str,maxlen)
@SKFSTRINGS@ *str;
int maxlen;
{
    int len;
    
    len = str->length;
    if ((len < 0) || (len > maxlen)) len = maxlen;

    return(len);
}
#elif defined(SWIGPYTHON) && defined(SKF_PYTHON3)
/* python 3.1 or later */
size_t  skf_swig_strlen(str,maxlen)
@SKFSTRINGS@ *str;
int maxlen;
{
    size_t len;

    if (PyUnicode_Check(str)) {		/* if SANE object	   */
	len = skf_PyUnicode_GetSize(str);
	if ((len < 0) || (len > maxlen)) len = maxlen;
#ifdef	SKFDEBUG	
	if (is_v_debug) fprintf(stderr,"get unicodestr(len: %ld)",len);
#endif
	return len;
    } else if (PyByteArray_Check(str)) { /* if SANE object	   */
	len = PyByteArray_Size(str);
	if ((len < 0) || (len > maxlen)) len = maxlen;
#ifdef	SKFDEBUG	
	if (is_v_debug) fprintf(stderr,"get bytestr(len: %ld)",len);
#endif
	return len;
    } else {
    	return (1);
    };

}
#else
size_t  skf_swig_strlen(str,maxlen)
@SKFSTRINGS@ *str;
int maxlen;
{
    int i;
    size_t len;

    for (i=0,len=0;((i<maxlen) && (*str!='\0')); i++,str++,len++);

    return(len);
}
#endif

/* --------------------------------------------------------------- */
/* output control for LWL interface				   */
/* --------------------------------------------------------------- */
/* output string */
static struct Skf_localestring	*ostrdef;

int	skf_olimit = 0;
unsigned char	*skfobuf = NULL;

/* initialize */
void	skf_ioinit(skfoFILE *fout,int mode)
{

    skf_swig_result = 0;
    errorcode = 0;

#ifdef SKFDEBUG
	if (is_v_debug) fprintf(stderr,"-- ioinit --");
#endif
/* --- output-side buffer prepare -------------------------------- */
    if (ostrdef == NULL) {
    	ostrdef = (struct Skf_localestring *)
			malloc(sizeof(struct Skf_localestring));
	if (ostrdef == NULL) {
	    skferr(SKF_OBUFERR,0,skf_olimit);
	    	/* should be break at this point */
	};
    } else;

    if (skfobuf == NULL) {
#ifdef SKFDEBUG
	if (is_v_debug) fprintf(stderr,"buffer allocation\n");
#endif
	skf_olimit = SKF_STRBUFLEN;
    	skfobuf = (unsigned char *)malloc(skf_olimit * sizeof(unsigned char));
	if (skfobuf == NULL) {
	    skferr(SKF_OBUFERR,0,skf_olimit);
	};
    };
    ostrdef->sstr = skfobuf;
    ostrdef->length = 0;
    ostrdef->codeset = out_codeset;

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3)
    if ((mode == GUESS_INIT) || is_o_encode) {
    		/* return ascii for guess and ACE/MIMEs		   */
	ostrdef->lwl_codeset =
		rb_enc_find_index("US_ASCII");
    } else if (mode == FORCE_INIT) {
	ostrdef->lwl_codeset =
		rb_enc_find_index(i_codeset[out_codeset].cname);
    } else;	/* continuing.. do not initialize		   */
#else
    ostrdef->lwl_codeset = -1;
#endif
    if (o_add_bom) show_endian_out();
    if (add_annon) print_announce(out_codeset);
    show_lang_tag();

    return;
}

/* dummy initialize */
void	skf_dmyinit()
{

    skf_swig_result = 0;
    errorcode = 0;

#ifdef SKFDEBUG
	if (is_v_debug) fprintf(stderr,"-- dmyinit --");
#endif
/* --- output-side buffer prepare -------------------------------- */
    if (ostrdef == NULL) {
    	ostrdef = (struct Skf_localestring *)
			malloc(sizeof(struct Skf_localestring));
	if (ostrdef == NULL) {
	    skferr(SKF_OBUFERR,0,skf_olimit);
	    	/* should be break at this point */
	};
    } else;

    if (skfobuf == NULL) {
#ifdef SKFDEBUG
	if (is_v_debug) fprintf(stderr,"buffer allocation\n");
#endif
	skf_olimit = SKF_STRBUFLEN;
    	skfobuf = (unsigned char *)malloc(4 * sizeof(unsigned char));
	if (skfobuf == NULL) {
	    skferr(SKF_OBUFERR,0,skf_olimit);
	};
    };
    skfobuf[0] = ' ';
    skfobuf[1] = '\0';
    ostrdef->sstr = skfobuf;
    ostrdef->length = 1;
    ostrdef->codeset = out_codeset;

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3)
    ostrdef->lwl_codeset =
		rb_enc_find_index("US_ASCII");
#else
    ostrdef->lwl_codeset = -1;
#endif

    return;
}

/* --------------------------------------------------------------- */
/* putchar for LWL interface					   */
/* --------------------------------------------------------------- */
int	lwl_putchar(c)
int	c;
 {
    unsigned char	*newbuf;

    if (ostrdef->length >= skf_olimit) {
#ifdef SKFDEBUG
	if (is_v_debug) fprintf(stderr,"buffer re-allocation\n");
#endif
	skf_olimit += BUFINCSIZE;
	newbuf = realloc(skfobuf, (skf_olimit * sizeof(unsigned char)));
	if (newbuf == NULL) {
	    skferr(SKF_OBUFREERR,0,skf_olimit);
	};
	skfobuf = newbuf;
	ostrdef->sstr = newbuf;
    };
    skfobuf[ostrdef->length] = c;
    ostrdef->length = ostrdef->length + 1;

    return(0);
}

#ifdef LWLINTSTRING
#if defined(SWIGPYTHON) && defined(SKF_PYTHON3)
/* --------------------------------------------------------------- */
/* this routine handles only optstr in a python extension	   */
/* --------------------------------------------------------------- */
char	*skfstrstrconv(PyObject *robj,size_t len)
{
    char	*dstr;
    Py_UNICODE	*srcstr;
    char	*bsrcstr;
    Py_ssize_t	slen;
    int		i;

#ifdef SKFDEBUG
	if (is_v_debug) fprintf(stderr," --strstrconv --");
#endif
#ifdef Py_UNICODE_WIDE
    dstr = calloc((len+2)*4,sizeof(char));
#else
    dstr = calloc((len+2)*2,sizeof(char));
#endif
    if (dstr == NULL) {
    	skferr(SKF_MALLOCERR,24,0);
    } else;
    srcstr = NULL; bsrcstr = NULL;

    if (PyUnicode_Check(robj)) {	/* if SANE object	   */
        if ((slen = skf_PyUnicode_GetSize(robj)) <= 0) {
		/* may be just nullstring, not error.		   */
	    return(NULL);
	} else;
	if ((srcstr = skf_PyUnicode_AsUnicode(robj)) == NULL) {
		/* LIMITED_API: which should I use?		   */
	    skferr(SKF_IBUFERR,0,0);
	} else;
	if (slen > len) slen = len;
		/* do not process too long string.		   */
	for (i=0;i<slen;i++) {
	    if (srcstr[i] >= 0x7f) break;
	    dstr[i] = srcstr[i];
	};
    } else if (PyByteArray_Check(robj)) { /* if SANE object	   */
        if ((slen = PyByteArray_Size(robj)) <= 0) {
	    return(NULL);
	} else;
	if ((bsrcstr = PyByteArray_AsString(robj)) == NULL) {
	    skferr(SKF_IBUFERR,0,0);
	} else;
	if (slen > len) slen = len;
	for (i=0;i<(slen);i++) {
	    dstr[i] = bsrcstr[i];
	};

        dstr[slen] = '\0';

    };
    return(dstr);
}
#else
/* --------------------------------------------------------------- */
char	*skfstrstrconv(@SKFSTRINGS@ *robj,int len)
{
    char	*dstr;
    int		i;

    dstr = calloc(len+1,sizeof(char));
    if (dstr == NULL) {
    	skferr(SKF_MALLOCERR,24,1);
    } else;
    for (i=0;i<len;i++) dstr[i] = (char) robj[i];
    robj[len] = '\0';

    return(dstr);
}
#endif
#endif

#if	defined(SWIGRUBY) && (defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3))
/* --------------------------------------------------------------- */
struct Skf_localestring *skf_rbstring2skfstring(VALUE rstr)
{
    struct Skf_localestring	*sstrdef;


    if ((sstrdef = calloc(1,sizeof(struct Skf_localestring))) == NULL) {
    	skferr(SKF_MALLOCERR,24,2);
    } else {
	sstrdef->sstr = (unsigned char *)RSTRING_PTR(rstr);
	sstrdef->length = RSTRING_LEN(rstr);
	sstrdef->codeset = 
	    skf_search_cname_w_alias((char *)rb_enc_name(rb_enc_get(rstr)));
	sstrdef->lwl_codeset = -1;	/* do not assume input locale */
    };

    istrdef = sstrdef;
    return sstrdef;
}

#elif	defined(SWIGPYTHON) && defined(SKF_PYTHON3)
	/* Note: Python string comes with UCS2(BMP only) or UCS4.
	   The width depends on compile option, which is described below. */
	/* The note above is obsolete since PEP_393. See specifications. */
/* --------------------------------------------------------------- */
/* Note: md == 1 --> input string, md == 0 --> options		   */
/*  return character string -					   */
/*	option string: unsigned char				   */
/*	target string: unsigned char * 4 (big endian UTF32 packed) */
/* --------------------------------------------------------------- */
struct Skf_localestring *skf_pystring2skfstring(PyObject *robj,int mo)
{
    struct Skf_localestring	*sstrdef;
    Py_ssize_t len;
#ifdef USE_PEP393
    void	*instr = NULL;
#else
    Py_UNICODE	*instr = NULL;
#endif
    char	*instr_c = NULL;
    unsigned char	*istr;
    int		i;
#if USE_PEP393
    int		kind = 2;
    int		off;
#endif
    unsigned int re;
    char	sy;
    skf_ucode	ch,c2;

    if ((sstrdef = calloc(1,sizeof(struct Skf_localestring))) == NULL) {
    	skferr(SKF_MALLOCERR,24,3);
	skf_exit(-1);
    } else {
	sstrdef->lwl_codeset = -1;	/* assumed input codeset   */
	sstrdef->codeset = -1;	/* does it necessary? 		   */
/* case: if a string given is an Unicode string */
    	if (PyUnicode_Check(robj)) {	/* if SANE object	   */
#if USE_PEP393

	    if (PyUnicode_READY(robj) != 0) {
		setsstrdummies(sstrdef);	/* return dummy    */
		return(sstrdef);
	    } else;
	    kind = PyUnicode_KIND(robj);
	    if (((len = PyUnicode_GET_LENGTH(robj)) <= 0) ||
	        ((instr = PyUnicode_DATA(robj)) == NULL)) {
		skf_openerr("",7);
		setsstrdummies(sstrdef);
		return(sstrdef);
	    } else;
#else	/* !USE_PEP393 */
	    if (((len = skf_PyUnicode_GetSize(robj)) <= 0) ||
		((instr = skf_PyUnicode_AsUnicode(robj)) == NULL)) {
		skf_openerr("",7);
		setsstrdummies(sstrdef);
		return(sstrdef);
	    } else;
#endif
	    if (mo == 0) {	/* options should come here */
		/* pseude code. packed 8-bit asciis.		   */
		optstr = sstrdef;
		if ((optionStr = malloc(OPTSTRLEN * sizeof(char)))
		    	== NULL) {
		    skferr(SKF_MALLOCERR,24,4);
		    skf_exit(-1);
		} else;
		for (i=0;(i<len) && (i < OPTSTRLEN);i++) {
#ifdef USE_PEP393
		    sy = PyUnicode_READ(kind,instr,i);
#else	/* !USE_PEP393 */
		    sy = (char) (*(instr + i) & 0x0ffu);
#endif
		    optionStr[i] = sy;
		};

		optionStr[i] = '\0';
		optstr->length = len;
		optstr->sstr = (unsigned char *)optionStr;
		return(optstr);
#ifdef USE_PEP393	/* !PEP393 never comes here */
	    } else if (kind == PyUnicode_1BYTE_KIND) {
		/* this case assumes to be latin-1 */
		if ((istr = calloc((len) + 4 + 4,
				   sizeof(unsigned char))) == NULL) {
		    PyErr_NoMemory();
		    return(sstrdef);	  /*NOTREACHED*/
		} else;
		for (i=0;i<len;i++) {
		    ch = PyUnicode_READ(kind,instr,i);
		    *(istr + i) = ch;
		};
		*(istr+len) = '\0';
		sstrdef->length = len;
		sstrdef->sstr = istr;
		sstrdef->lwl_codeset = codeset_8859_1;
#endif	/* USE_PEP393 */
	    } else {	/* UTF16/UTF32 target strings.	*/
		if ((istr = calloc((len)*4 + 4 + 4,
				   sizeof(unsigned char))) == NULL) {
		    PyErr_NoMemory();
		    return(sstrdef);	  /*NOTREACHED*/
		} else;
		off = 0;
		for (i=0;(i+off)<len;i++) {
#ifdef USE_PEP393
		    *(istr + i*4 + 0) = 0;	/* should be 0 */
		    ch = PyUnicode_READ(kind,instr,i+off);
		    if ((ch >= 0xd800) && (ch < 0xdc00)) {
			off++;
			if ((i+off)>= len) {
			    in_undefined(ch,SKF_UNEXPEOF);
			    *(istr + i*4 + 1) = 0;
			    *(istr + i*4 + 2) = 0x30;
			    *(istr + i*4 + 3) = 0x13;
			    continue;
			} else;
			c2 = PyUnicode_READ(kind,instr,(i+off));
			if ((c2 < 0xdc00) || (c2 >= 0xe000)) {
			    in_undefined(ch,SKF_DECODERR);
			    *(istr + i*4 + 1) = 0;
			    *(istr + i*4 + 2) = 0x30;
			    *(istr + i*4 + 3) = 0x13;
			    continue;
			} else;
			ch = decode_surrogate(ch,c2);
			*(istr + i*4 + 1) = (unsigned char)(ch >> 16);
			*(istr + i*4 + 2) =
				(unsigned char)((ch & 0xff00) >> 8);
			*(istr + i*4 + 3) = (unsigned char)(ch & 0xffU);
		    } else {	/* normal UTF-32 characters	   */
			if (kind == PyUnicode_4BYTE_KIND) {
			    *(istr + i*4 + 1) =
				    (unsigned char)((ch >> 16) & 0xffU);
			} else;
			*(istr + i*4 + 2) =
				    (unsigned char)((ch >> 8) & 0xffU);
			*(istr + i*4 + 3) = (unsigned char)(ch & 0xffU);
		    };
#else /* !USE_PEP393 */
		    ch = *(instr + i);
#ifdef Py_UNICODE_WIDE		/* UCS4 is used			   */
		    *(istr + i*4 + 0) =
			(unsigned char) (((*(instr + i)) >> 24) & 0x0ffU);
		    *(istr + i*4 + 1) =
			(unsigned char) (((*(instr + i)) >> 16) & 0x0ffU);
#else
		    *(istr + i*4 + 0) = 0;
		    *(istr + i*4 + 1) = 0;
#endif
		    *(istr + i*4 + 2) =
			(unsigned char) (((*(instr + i)) >> 8) & 0x0ffU);
		    *(istr + i*4 + 3) =
			(unsigned char) (((*(instr + i)) >> 0) & 0x0ffU);
#endif /* !USE_PEP393 */
		};
		*(istr+(len*4)) = '\0';
		sstrdef->length = len * 4;
		sstrdef->sstr = istr;
		sstrdef->lwl_codeset = codeset_utf32be;
	    };
/* case: if a string given is a bytes (not byte array) */
    	} else if (PyBytes_Check(robj)) {	/* if SANE object	   */
	    if ((len = PyBytes_Size(robj)) <= 0) {
		setsstrdummies(sstrdef);	/* return dummy    */
		return(sstrdef);
	    } else;
	    if ((instr_c = PyBytes_AsString(robj)) == NULL) {
		skf_openerr("",8);
		setsstrdummies(sstrdef);
		return(sstrdef);
	    } else;
	    /* we have 8-bit asciis.			   */
	    if ((istr = calloc((len + 2),sizeof(unsigned char)))
			== NULL) {
		PyErr_NoMemory();
		return(NULL);		/*NOTREACHED*/
	    } else;
    
	    for (i=0;i<len;i++) {
		re = (int)(*(instr_c+i) & 0xffu);
		*(istr + i) = (unsigned char) (*(instr_c+ i) & 0x0ffu);
	    };
	    *(istr + len) = '\0';
	    sstrdef->length = len;
		    /* Note: codeset is not yet determined */
/* case: if a string given is a byte array */
    	} else if (PyByteArray_Check(robj)) {	/* if SANE object	   */
	    if ((len = PyByteArray_Size(robj)) <= 0) {
		setsstrdummies(sstrdef);	/* return dummy    */
		return(sstrdef);
	    } else;
	    if ((instr_c = PyByteArray_AsString(robj)) == NULL) {
		skferr(SKF_IBUFERR,0,0);
	    } else;
	    /* we have 8-bit asciis.			   */
	    if ((istr = calloc((len + 2),sizeof(unsigned char)))
			== NULL) {
		PyErr_NoMemory();
		return(NULL);		/*NOTREACHED*/
	    } else;
    
	    for (i=0;i<len;i++) {
		re = (int)(*(instr_c+i) & 0xffu);
		*(istr + i) = (unsigned char) (*(instr_c+ i) & 0x0ffu);
	    };
	    *(istr + len) = '\0';
	    sstrdef->length = len;
		    /* Note: codeset is not yet determined */
	} else {	/* we got neither string nor byte array		  */
		/* we got a byte array */
	    PyErr_SetString(PyExc_RuntimeError,skf_notstring);
	};
    };
    sstrdef->sstr = istr;

    return sstrdef;
}

#ifdef USE_PEP393
PyObject *skf_skfstring2pystring(struct Skf_localestring *robj,
	int out_codeset, int p_out_binary)
{
    long	olen,oolen;
    int	i,j;
    unsigned char *oostr;
    PyObject *presult;
    int	kind;
    Py_UCS4	*uary4 = NULL;
    Py_UCS2	*uary2 = NULL;
    Py_UCS1	*uary = NULL;
    skf_ucode	ch,ci;

    oolen = (olen = robj->length);
#ifdef SKFDEBUG
    if (is_vv_debug) {
	fprintf(stderr,"\noutput string conversion - pep393(%c,%d) ",
		((p_out_binary) ? 'B' : 'S'),(int)olen);
	fflush(stderr);
    };
#endif
    oostr = robj->sstr;
    if (olen < 0) olen = 8;

    /* Note: if memory is so short that even 8 byte can't be allocated, */
    /*  skf may crash internally. */
    /* oostr may be leaked. TODO; check. */
    
    if (p_out_binary == 0) {
	if (is_o_encode_asciiish(o_encode)) {
	    /* any encoding is ascii, if MIMEish encode is enabled */
	    kind = PyUnicode_1BYTE_KIND;
	} else if (is_ucs_utf32(conv_cap)) kind = PyUnicode_4BYTE_KIND;
	else if (is_ucs_utf16(conv_cap)) {
	    kind = PyUnicode_2BYTE_KIND;
	    for (i=0;i<oolen;i=i+2) {
		if (out_endian(conv_cap)) ch = oostr[i];
		else ch = oostr[i+1];
		if (((oostr[i+1] >= 0xd8) && (oostr[i+1] < 0xe0) 
		      && !out_endian(conv_cap)) ||
		    ((oostr[i] >= 0xd8) && (oostr[i] < 0xe0)
		      && out_endian(conv_cap))) {
		    kind = PyUnicode_4BYTE_KIND;
		    break;
		} else;
	    };
	} else kind = PyUnicode_1BYTE_KIND;
	/* allocate buffer */
	if (kind == PyUnicode_4BYTE_KIND) {
	    if ((uary4 = calloc(olen + 8,sizeof(Py_UCS4))) == NULL) {
		skferr(SKF_MALLOCERR,24,4); skf_exit(-1);
	    } else;
	} else if (kind == PyUnicode_2BYTE_KIND) {
	    if ((uary2 = calloc(olen + 8,sizeof(Py_UCS2))) == NULL) {
		skferr(SKF_MALLOCERR,24,4); skf_exit(-1);
	    } else;
	} else {
	    if ((uary = calloc(olen + 8, sizeof(Py_UCS1))) == NULL) {
		skferr(SKF_MALLOCERR,24,4); skf_exit(-1);
	    } else;
	};
	if (kind == PyUnicode_4BYTE_KIND) {
	    j = 0;
	    if (is_ucs_utf16(conv_cap)) {
		for (i=0;i<oolen;i+=2) {
		    if (out_endian(conv_cap)) {
			ch = (oostr[i] << 8) + oostr[i+1];
		    } else 
			ch = (oostr[i+1] << 8) + oostr[i];
		    if ((ch >= 0xd800) && (ch < 0xdc00)) {
			i += 2;
			if (out_endian(conv_cap)) {
			    ci = (oostr[i] << 8) + oostr[i+1];
			} else 
			    ci = (oostr[i+1] << 8) + oostr[i];
			ch = decode_surrogate(ch,ci);
		    } else
		    uary4[j++] = ch;
		};
	    } else {
		for (i=0;i<oolen;i+=4) {
		    if (out_endian(conv_cap)) {
			uary4[j++] = (oostr[i] << 24) +
				(oostr[i+1] << 16) +
				(oostr[i+2] << 8) + oostr[i+3];
		    } else {
			uary4[j++] = (oostr[i+3] << 24) +
				(oostr[i+2] << 16) +
				(oostr[i+1] << 8) + oostr[i];
		    };
		};
	    };
	    uary4[j] = 0;
	} else if ((kind == PyUnicode_4BYTE_KIND) &&
			!out_endian(conv_cap)) {
	    j = 0;
	    for (i=0;i<oolen;i+=4) {	/* 4 is to skip BOM	*/
		uary4[j++] = (oostr[i+3] << 24) + (oostr[i+2] << 16) +
			    (oostr[i+1] << 8) + oostr[i];
	    };
	    uary4[j] = 0;
	} else if ((kind == PyUnicode_2BYTE_KIND) &&
			out_endian(conv_cap)) {
	    j = 0;
	    for (i=0;i<oolen;i+=2) {	/* 2 is to skip BOM	*/
		uary2[j++] = (oostr[i] << 8) + oostr[i+1];
	    };
	    uary2[j] = 0;
	} else if ((kind == PyUnicode_2BYTE_KIND) &&
			!out_endian(conv_cap)) {
	    j = 0;
	    for (i=0;i<oolen;i+=2) {
		uary2[j++] = (oostr[i+1] << 8) + oostr[i];
	    };
	    uary2[j] = 0;
	} else {
	    /* last one byte is NULL */
	    j = 0;
	    for (i=0;i<oolen;i++) {
		uary[j++] = oostr[i];
	    };
	    uary[j] = 0;
	};
	if (kind == PyUnicode_4BYTE_KIND) {
	    presult = PyUnicode_FromKindAndData(kind,uary4,j-1);
	} else if (kind == PyUnicode_2BYTE_KIND) {
	    presult = PyUnicode_FromKindAndData(kind,uary2,j-1);
	} else {
	    presult = PyUnicode_FromKindAndData(kind,uary,j-1);
	};
    } else {
	/* last one byte is NULL */
	presult = PyByteArray_FromStringAndSize((const char *)oostr,
	    			(Py_ssize_t)(oolen-1));
    };

    return(presult);
}

#else	/* !USE_PEP393 */
PyObject *skf_skfstring2pystring(struct Skf_localestring *robj,
	int out_codeset, int p_out_binary)
{
    long	olen,oolen;
    int	i,j;
    unsigned char *oostr;
    PyObject *presult;
    int	kind;
    Py_UNICODE	*uary;

#ifdef SKFDEBUG
    if (debug_opt) {
	fprintf(stderr," output string conversion - oldstyle\n");
	fflush(stderr);
    };
#endif
    oolen = (olen = robj->length);
    oostr = robj->sstr;
    if (olen < 0) olen = 8;

    if ((uary = calloc(olen + 8,sizeof(Py_UNICODE))) == NULL) {
	skferr(SKF_MALLOCERR,24,4);
    } else;

    /* Note: if memory is so short that even 8 byte can't be allocated, */
    /*  skf may crash internally. */
    /* oostr may be leaked. TODO; check. */
    
    if (p_out_binary == 0) {
	kind = sizeof(Py_UNICODE);
	if ((uary = calloc(olen + 8,kind)) == NULL) {
	    skferr(SKF_MALLOCERR,24,4);
	} else;

	if ((kind == PyUnicode_4BYTE_KIND) && out_endian(conv_cap)) {
	    j = 0;
	    for (i=0;i<oolen;i+=4) {
		uary[j++] = (oostr[i] << 24) + (oostr[i+1] << 16) +
			    (oostr[i+2] << 8) + oostr[i+3];
	    };
	    uary[j] = 0;
	} else if ((kind == PyUnicode_4BYTE_KIND) &&
			!out_endian(conv_cap)) {
	    j = 0;
	    for (i=0;i<oolen;i+=4) {	/* 4 is to skip BOM	*/
		uary[j++] = (oostr[i+3] << 24) + (oostr[i+2] << 16) +
			    (oostr[i+1] << 8) + oostr[i];
	    };
	    uary[j] = 0;
	} else if ((kind == PyUnicode_2BYTE_KIND) &&
			out_endian(conv_cap)) {
	    j = 0;
	    for (i=0;i<oolen;i+=2) {	/* 2 is to skip BOM	*/
		uary[j++] = (oostr[i] << 8) + oostr[i+1];
	    };
	    uary[j] = 0;
	} else if ((kind == PyUnicode_2BYTE_KIND) &&
			!out_endian(conv_cap)) {
	    j = 0;
	    for (i=0;i<oolen;i+=2) {
		uary[j++] = (oostr[i+1] << 8) + oostr[i];
	    };
	    uary[j] = 0;
	} else {
	    /* last one byte is NULL */
	    j = 0;
	    for (i=0;i<oolen;i++) {
		uary[j++] = oostr[i];
	    };
	    uary[j] = 0;
	};
#ifdef USE_PEP393
	presult = PyUnicode_FromKindAndData(kind,uary,j+1);
#else
	presult = PyUnicode_FromUnicode(uary,j+1);
#endif
    } else {
	fprintf(stderr,"#POB#");
	/* last one byte is NULL */
	presult = PyByteArray_FromStringAndSize((const char *)oostr,
	    			(Py_ssize_t)(oolen-1));
    };
#if 0
    free (uary);
#endif

    return(presult);
}
#endif	/* !USE_PEP393 */

int get_py_out_binary(unsigned long ccap, int pb, unsigned long popt,
	unsigned long oe)
{
    if (pb == 1) return(1);
    if (is_o_encode_asciiish(oe)) {
	return(0);
    } else if (!is_ucs_utf32(ccap) && !is_ucs_utf16(ccap) &&
	     !is_input_inquiry(popt) && !is_ucs_utf7m(ccap)) {
	return(1);
    } else return(0);
}

#else	/* PYTHON2 or other lwl*/
/* --------------------------------------------------------------- */
struct Skf_localestring *skf_lwlstring2skfstring(char *rstr)
{
    struct Skf_localestring	*sstrdef;
    int i;
    char *pstr = rstr;
    unsigned char	 *istr;
    int	 buflen = LWL_MAXLEN;

    if (istrdef == NULL) {
	if ((sstrdef = calloc(1,sizeof(struct Skf_localestring))) == NULL) {
	    skferr(SKF_MALLOCERR,24,5);
	} else;
    } else {
    	sstrdef = istrdef;
    };
    if ((sstrdef->sstr) == NULL) {
	if ((istr = calloc(LWL_MAXLEN,sizeof(unsigned char))) == NULL) {
	/* Note: Maybe I should free istr at this point, but do nothing. */
	    skferr(SKF_MALLOCERR,24,6);
	} else;
    } else {
    	istr = sstrdef->sstr;
    };

    for (i=0;((i<LWL_MAXLEN) && (*pstr!='\0')); i++,pstr++) {
	istr[i] = (*pstr) & 0xff;
	if (i >= (buflen - 2)) {
	    istr = realloc(istr,(size_t)(sizeof(int) *(buflen * 2)));
	    if (istr == NULL) {
		skferr(SKF_MALLOCERR,24,7);
	    } else;
	    break;
	} else;
    };

    istr[i] = sEOF;
    sstrdef->sstr = istr;
    sstrdef->length = i;
    sstrdef->codeset = -1;
    sstrdef->lwl_codeset = -1;	/* do not assume input locale */

    istrdef = sstrdef;
    return sstrdef;
}

#endif
/* ---------------------------------------------------------------
    common output routine.
    output buffer should be prepared before calling.
*/
static void r_skf_convert(struct Skf_localestring *lstr, long ibuflen,
				int mode,int ienc)
{
    int sy;
    int errc = 0;

    errorcode = 0;	/* forget previous errors.		   */
/* --- codeset specific initialization is done in convert -------- */
/* --- preparing output buffers ---------------------------------- */
    skf_ioinit((skfoFILE *) 0,mode);

/* --- preconversion output prepare ------------------------------ */
    if (o_add_bom) show_endian_out();
    if (add_annon) print_announce(out_codeset);

/* --- compatibility various hooks ------------------------------- */
#ifdef SKF196COMPAT
    if (encode_enbl) set_skf196mime;
#endif
    if (is_nkf_compat) {
    	mime_fold_llimit -= 1;
	if (!encode_enbl) set_mimeb_encode;
    };

    reset_kanji_shift;
/* --- codeset specific initialization, include table setting ---- */
    skf_output_table_set();

/* --- reflect in_param into conv_cap -------------------------- */
    if (is_set_ad_ann && !is_ucs_ufam(conv_cap)) set_add_annon;
    else if (is_set_no_ann && !is_ucs_ufam(conv_cap)) res_add_annon;
    else;

    if (is_set_p_iso8859 && !is_ucs_ufam(conv_cap)) set_use_iso8859;
    else if (is_set_p_niso8859 && !is_ucs_ufam(conv_cap)) res_use_iso8859;
    else;

    if (is_par_hk_enbl && !is_ucs_ufam(conv_cap)) {
	set_put_hk_enbl(get_par_hkenbl);
    };

    if (is_ucs_utf16(conv_cap)) {
    if (is_set_out_le && is_ucs_utf16(conv_cap)) set_o_ltl_endian;
    else if (is_set_out_be && is_ucs_utf16(conv_cap)) set_o_big_endian;
    else;
    } else;

    if (is_set_ucs_bom && is_ucs_ufam(conv_cap)) set_o_add_bom;
    else if (is_set_ucs_nobom && is_ucs_ufam(conv_cap)) res_o_add_bom;
    else ;

    if (is_set_mscpt) set_use_ms_compat;
    else if (is_set_no_mscpt) res_use_ms_compat;
    else;

    if (is_set_pux212) set_use_x0212;
    else if (is_set_pnux212) res_use_x0212;
    else;

/* --- output initialize ----------------------------------------- */
    if (oconv_init() < 0) {	/* various initialize		   */
	skferr(SKF_MALLOCERR,(long)24,(long)16);
    } else;

#if defined(SWIGPYTHON) && defined(SKF_PYTHON3)
    p_out_binary = get_py_out_binary(conv_cap,p_out_binary,
    			preconv_opt, o_encode);
#endif
    init_all_stats();

/* pass parameter to ioinit input side	-------------------------- */
    stdibuf = lstr->sstr;
    buf_p = lstr->length;

#ifdef SKFDEBUG
    if (is_vv_debug) {
	fprintf(stderr,"#buf_p:%ld#",buf_p);
    } else;
#endif

    if (mode == FORCE_INIT) {
	if (o_add_bom) show_endian_out();
	show_lang_tag();

/* --- preconversion output prepare ------------------------------ */
	if (add_annon) print_announce(out_codeset);

/* --- fold value fix -------------------------------------------- */
#ifdef FOLD_SUPPORT
	fold_value_setup();
#endif
    };
/* --- debug analyze ---- */
#ifdef SKFDEBUG
    debug_analyze();
#endif
/* --- open provided strings ------------------------------------- */
/* Note: this point messed up int and long. Be warned.		   */

    sy = ibuflen;
    if (sy > ibuflen) {
	errorcode = 1;
	skferr(SKF_DEBUGERR_2,(long)0,(long)0);
    };

    skf_fpntr = 0;
    Qflush();

    in_codeset = ienc;

/* --- conversion loop ------------------------------------------- */
    if ((errc = setjmp(skf_errbuf)) == 0) {
	sy = skf_in_converter((FILE *)0);
    } else {
#if defined(SKF_PYTHON3) && defined(SWIGPYTHON)
	if (errc == 1) PyErr_NoMemory();
	else;
#else
	;
#endif
    };

    in_saved_codeset = in_codeset;

    if (is_jis(conv_cap)) JIS_finish_procedure();
    else if (is_euc(conv_cap)) EUC_finish_procedure();
    else if (is_msfam(conv_cap)) SJIS_finish_procedure();
    else if (is_ucs_utf7m(conv_cap)) utf7_finish_procedure();
    else if (is_ucs_utf8(conv_cap)) utf8_finish_procedure();
    else if (is_ucs_utf16(conv_cap)) ucod_finish_procedure();
    else if (out_bg(conv_cap)) BG_finish_procedure();
    else if (is_brgt(conv_cap)) BRGT_finish_procedure();
    else if (is_keis(conv_cap)) KEIS_finish_procedure();
    else if (is_big5fam(conv_cap)) BG_finish_procedure();
    else if (is_nnisofam(conv_cap)) BG_finish_procedure();
    else if (is_gbkrfam(conv_cap)) GBKR_finish_procedure();
    else if (is_ucs_punyuri(conv_cap)) puny_finish_procedure();
    else;
#ifdef SKFDEBUG
    if (is_v_debug) fprintf(stderr,"\n[EOF]\n");
#endif

#if defined(FOLD_SUPPORT) 
/* folding at the end of file					   */
    if (is_nkf_compat && (fold_count != 0) && fold_fclap && !notrunc_le) {
#ifdef SKFDEBUG
	if (is_vv_debug) { fprintf(stderr,"#LEADD#");
	} else;
#endif
    	SKFCRLF();
    } else;
#endif

/* if any characters hang in encoder, push out these stuffs. 	   */
/* and also put eol if base64 encode.				   */
    oconv(sFLSH);  /* remains should be cleared	   */

    if (o_encode) encoder_tail();

    if (skf_swig_result == 0) skf_swig_result = errc;

    return;
}

/*
    routines visible from (i.e. provided to) scripting languages
    convert, quickconvert, guess
*/
@SKFOSTRINGS@ *convert(@SKFCSTRINGS@ *optstr, @SKFSTRINGS@ *cstr)
{
    long	ibuflen;
    int		result;
    struct	Skf_localestring *lwlstr = &lwlstr_b;
#if defined(SWIGPYTHON) && defined(SKF_PYTHON3)
    struct	Skf_localestring *lwlopt = &lwlopt_b;
#endif
#if defined(FUTURESUPPORT)
    int		errc = 0;
#endif
    int actual_out_code = -1;

    in_saved_codeset = -1;

    skf_script_init(); swig_state = 1;

#if	defined(SWIGRUBY) && (defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3))
    lwlstr = cstr;
    ibuflen = get_rstr_len(cstr);
#elif	defined(SWIGPYTHON) 
    lwlstr = skf_pystring2skfstring(cstr,1);
    ibuflen = lwlstr->length;
#else
    lwlstr = skf_lwlstring2skfstring(cstr);
    ibuflen = lwlstr->length;
#endif

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3)
    if (optstr->sstr != NULL) {
	result = skf_script_param_parse((char *)(optstr->sstr),optstr->length);
    } else result = 0;
#elif defined(SWIGPYTHON) 
    lwlopt = skf_pystring2skfstring(optstr,0);
    if ((lwlopt != NULL) && (lwlopt->sstr != NULL)) {
	result = skf_script_param_parse((char *)lwlopt->sstr,lwlopt->length);
	free(lwlopt->sstr);
#ifdef USE_PEP393
	free(lwlopt);
#endif
    } else result = 0;
#else
    if (optstr != NULL) result = skf_script_param_parse((char *)optstr,optstr_len);
    else result = 0;
#endif
    if (lwlstr->lwl_codeset != -1) {
	in_codeset = lwlstr->lwl_codeset;
    } else;
/* --- determine output codeset ---------------------------------- */
#if defined(SWIGPYTHON) && defined(SKF_PYTHON3)
    if (out_code < 0) out_code = Py_Default_ocode;
#endif

    if (out_code == codeset_binary) {
    	out_code = DEFAULT_O;
	in_codeset = codeset_transp;
    } else if ((out_code == codeset_locale) ||
    		((input_inquiry) && (out_code < 0))){
#ifndef SWIG_EXT
    	locale_codeset = get_output_locale();
	if (locale_codeset < 0) {
	    out_undefined(sOCD,SKF_NOOUT);
	    out_code = DEFAULT_O;
	} else out_code = locale_codeset;
#else
    	out_code = DEFAULT_O;
#endif
#ifdef DYNAMIC_LOADING
    /* Note: B-Right/V is not ISO-2022 type codeset, but it needs  */
    /*   iso-2022-kr code for hangul processing. This treatment	   */
    /*   would be a sheer hack, but proper handling needs a table  */
    /*   duplication for just to go straight without exceptions.   */
    } else if (out_code == codeset_brgt) {
    	actual_out_code = out_code;
	out_code = codeset_euckr;
	skf_charset_parser(out_code); 
	out_code = actual_out_code;
#endif
    } else;

    skf_charset_parser(out_code); 
    if (result < 0) {
#if	defined(SKFSTRUCT_RETURN)
	skf_dmyinit();	/* generate dmy ostrdef			   */
#endif
    } else {

	iencode = in_codeset;

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3)
	ruby_out_locale_index = rb_enc_find_index(i_codeset[out_codeset].cname);
	if (cstr->codeset != codeset_binary) iencode = cstr->codeset;
#endif
    /* --- conversion call ------------------------------------------- */
	skf_in_text_type = 0;
	r_skf_convert(lwlstr,ibuflen,FORCE_INIT,iencode);
	lwl_putchar(0x00);	/* add terminater			   */
	errorcode = skf_swig_result;

#ifdef DEBUG
	if (is_vv_debug) {
#if	defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3)
	    fprintf(stderr,"\nswig_result-len:%d,ptr:%lx\n",ostrdef->length,ostrdef->sstr);
#else
	    fprintf(stderr,"\nswig_result-len:%d,ptr:%lx\n",ostrdef->length,cstr);
#endif
	};

#endif
    };

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3)
    if (cstr != NULL) free(cstr);
    return (ostrdef);
#elif defined(SWIGPYTHON) && defined(SKF_PYTHON3)
    if (lwlstr->sstr != NULL) free(lwlstr->sstr);
    if (lwlstr != NULL) free(lwlstr);
    ostrdef->sstr = (unsigned char *)skfobuf;
    ostrdef->length = ostrdef->length;
    return(skf_skfstring2pystring(ostrdef,out_codeset,p_out_binary));
#else
    if (lwlstr->sstr != NULL) free(lwlstr->sstr);
    if (lwlstr != NULL) free(lwlstr);
    return ((char *)skfobuf);
#endif
}

@SKFOSTRINGS@ *quickconvert(@SKFCSTRINGS@ *optstr, @SKFSTRINGS@ *cstr)
{
    long	ibuflen;
    int		result = 0;
    struct	Skf_localestring *lwlstr = &lwlstr_b;
#if defined(SWIGPYTHON) && defined(SKF_PYTHON3)
    struct	Skf_localestring *lwlopt = &lwlopt_b;
#endif

    if (swig_state == 0) {
#ifdef SKFDEBUG
	if (is_vv_debug) fprintf(stderr,"\nextension initialize\n");
#endif
	skf_script_init(); swig_state = 1;
    };
    debug_opt = 0;

#if	defined(SWIGRUBY) && (defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3))
    lwlstr = cstr;
    ibuflen = get_rstr_len(cstr);
#elif	defined(SWIGPYTHON) && defined(SKF_PYTHON3)
    lwlstr = skf_pystring2skfstring(cstr,1);
    ibuflen = lwlstr->length;
#else
    lwlstr = skf_lwlstring2skfstring(cstr);
    ibuflen = lwlstr->length;
#endif

    lwlstr->codeset = in_saved_codeset;

#if	defined(SWIGRUBY) && (defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3))
    if (optstr->sstr != NULL) {
	result = skf_script_param_parse((char *)(optstr->sstr),optstr->length);
    } else result = 0;
#elif defined(SWIGPYTHON) && defined(SKF_PYTHON3)
    lwlopt = skf_pystring2skfstring(optstr,0);
    if ((lwlopt != NULL) && (lwlopt->sstr != NULL)) {
	result = skf_script_param_parse((char *)lwlopt->sstr,lwlopt->length);
	free(lwlopt->sstr);
#ifdef USE_PEP393
	free(lwlopt);
#endif
    } else result = 0;
#else
    if (optstr != NULL) result = skf_script_param_parse((char *)optstr,optstr_len);
    else result = 0;
#endif

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3)
    if (optstr->sstr != NULL) {
	result = skf_script_param_parse((char *)(optstr->sstr),optstr->length);
    } else result = 0;
#elif defined(SWIGPYTHON) && defined(SKF_PYTHON3)
    lwlopt = skf_pystring2skfstring(optstr,0);
    if ((lwlopt != NULL) && (lwlopt->sstr != NULL)) {
	result = skf_script_param_parse((char *)lwlopt->sstr,lwlopt->length);
	free(lwlopt->sstr);
    } else result = 0;
#else
    if (optstr != NULL) result = skf_script_param_parse((char *)optstr,optstr_len);
    else result = 0;
#endif

    if (result < 0) {
#if	defined(SKFSTRUCT_RETURN)
	skf_dmyinit();	/* generate dmy ostrdef			   */
#endif
    } else {

	iencode = in_codeset;
	if (in_codeset < 0) iencode = in_saved_codeset;

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3)
	ruby_out_locale_index = rb_enc_find_index(i_codeset[out_codeset].cname);
	if (iencode < 0) iencode = cstr->codeset;
#endif
    /* --- conversion call ------------------------------------------- */
	r_skf_convert(lwlstr,ibuflen,FORCE_INIT,iencode);
	lwl_putchar(0x00);	/* add terminater			   */
	errorcode = skf_swig_result;

#ifdef DEBUG
	if (is_vv_debug) {
#if	defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3)
	    fprintf(stderr,"\nswig_result-len:%d,ptr:%lx\n",ostrdef->length,ostrdef->sstr);
#else
	    fprintf(stderr,"\nswig_result-len:%d,ptr:%lx\n",ostrdef->length,cstr);
#endif
	};

#endif
    };

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3)
    if (cstr != NULL) free(cstr);
    return (ostrdef);
#elif defined(SWIGPYTHON) && defined(SKF_PYTHON3)
    if (lwlstr->sstr != NULL) free(lwlstr->sstr);
    if (lwlstr != NULL) free(lwlstr);
    ostrdef->sstr = (unsigned char *)skfobuf;
    ostrdef->length = ostrdef->length;
    return(skf_skfstring2pystring(ostrdef,out_codeset,p_out_binary));
#else
    if (lwlstr->sstr != NULL) free(lwlstr->sstr);
    if (lwlstr != NULL) free(lwlstr);
    return ((char *)skfobuf);
#endif
}

@SKFOSTRINGS@ *guess(@SKFCSTRINGS@ *optstr, @SKFSTRINGS@ *cstr)
{
    long	ibuflen;
    int		result;
    struct	Skf_localestring *lwlstr = &lwlstr_b;
#if defined(SWIGPYTHON) && defined(SKF_PYTHON3)
    struct	Skf_localestring *lwlopt = &lwlopt_b;
#endif
    skf_script_init();		/* guess does not use old value	   */
    in_saved_codeset = -1;

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3)
    lwlstr = cstr;
    ibuflen = get_rstr_len(cstr);
#elif	defined(SWIGPYTHON) && defined(SKF_PYTHON3)
    lwlstr = skf_pystring2skfstring(cstr,1);
    ibuflen = lwlstr->length;
#else
    lwlstr = skf_lwlstring2skfstring(cstr);
    ibuflen = lwlstr->length;
#endif

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3)
    if (optstr->sstr != NULL) {
	result = skf_script_param_parse((char *)(optstr->sstr),optstr->length);
    } else result = 0;
#elif defined(SWIGPYTHON) && defined(SKF_PYTHON3)
    lwlopt = skf_pystring2skfstring(optstr,0);
    if ((lwlopt != NULL) && (lwlopt->sstr != NULL)) {
	result = skf_script_param_parse((char *)lwlopt->sstr,lwlopt->length);
	free(lwlopt->sstr);
#ifdef USE_PEP393
	free(lwlopt);
#endif
    } else result = 0;
#else
    if (optstr != NULL) result = skf_script_param_parse((char *)optstr,optstr_len);
    else result = 0;
#endif

    if (result < 0) {
#if	defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3)
	skf_dmyinit();	/* generate dmy ostrdef			   */
	return (ostrdef);
#elif defined(SWIGPYTHON) && defined(SKF_PYTHON3)
	skf_dmyinit();	/* generate dmy ostrdef			   */
	return(skf_skfstring2pystring(ostrdef,out_codeset,p_out_binary));
#else
	return ((char *)skfobuf);
#endif
    };
    set_input_inquiry;
    iencode = in_codeset;

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3)
    ruby_out_ascii_index = rb_enc_find_index("US_ASCII");
    iencode = cstr->codeset;
#endif
    if (lwlstr->lwl_codeset != -1) {
	in_codeset = lwlstr->lwl_codeset;
    } else;
/* --- conversion call ------------------------------------------- */
    skf_in_text_type = 0;
    r_skf_convert(lwlstr,ibuflen,FORCE_INIT,iencode);
    lwl_putchar(0x00);	/* add terminater			   */
    errorcode = skf_swig_result;

#ifdef DEBUG
    if (is_vv_debug) {
#if	defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3)
    	fprintf(stderr,"\nswig_result-len:%d,ptr:%lx\n",ostrdef->length,ostrdef->sstr);
#else
	fprintf(stderr,"\nswig_result-len:%d,ptr:%lx\n",ostrdef->length,cstr);
#endif
    };

#endif

#if	defined(SKF_RUBY19) || defined(SKF_RUBY2) || defined(SKF_RUBY3)
    if (cstr != NULL) free(cstr);
    return (ostrdef);
#elif defined(SWIGPYTHON) && defined(SKF_PYTHON3)
    if (lwlstr->sstr != NULL) free(lwlstr->sstr);
    if (lwlstr != NULL) free(lwlstr);
#ifdef USE_PEP393
    return (PyUnicode_FromKindAndData(PyUnicode_2BYTE_KIND,ostrdef->sstr,
    	ostrdef->length));
#else
    ostrdef->sstr = (unsigned char *)skfobuf;
    return(skf_skfstring2pystring(ostrdef,out_codeset,p_out_binary));
#endif
#else
    if (lwlstr->sstr != NULL) free(lwlstr->sstr);
    if (lwlstr != NULL) free(lwlstr);
    return ((char *)skfobuf);
#endif
}

void destruct()
{
    skf_script_init();
    if (skfobuf != NULL) free(skfobuf);
    skfobuf = NULL;
}

char *inputcode()
{
    return(i_codeset[in_codeset].cname);
}

%}

%import "skf_convert.h"
%newobject convert;
%newobject quickconvert;
%newobject guess;
%newobject inputcode;
%immutable cstr;
%immutable optstr;

#if defined(SWIGRUBY)
%typemap (in) (SKFSTRINGS *cstr) {
	$1 = skf_rbstring2skfstring($input);
}

%typemap (in) (SKFSTRINGS *optstr) {
	$1 = skf_rbstring2skfstring($input);
}

%typemap (out) (SKFSTRINGS *) {
    {
	VALUE res;
	int	i;
	char	*resstr;
	unsigned char	*oostr;
	long	olen,oolen;

	oolen = (olen = $1->length);
	if (olen < 0) olen = 8;

	/* olen includes terminate \0. ruby does not expect this value, so */
	/* we have to fix up the return value.				   */
	olen--;

	res = rb_str_new(0, olen + 5);
	rb_str_set_len(res,olen);
	resstr = RSTRING_PTR(res);
	oostr = $1->sstr;
	if (o_encode) {
	    rb_enc_associate(res, rb_usascii_encoding());
	} else {
	    rb_enc_associate(res,
	      rb_enc_from_index(rb_enc_find_index(i_codeset[out_codeset].cname)));
	};
	for (i=0;i<$1->length;i++) {
	    if (oolen < 0) {
	    	*(resstr++) = ' ';
	    } else {
		*(resstr++) = *(oostr++);
	    };
	};
	$result = res;
    };
}
#endif

/* module definitions */
/* %apply @SKFSTRINGS@ *INPUT { @SKFSTRINGS@ *cstr }; */
@SKFOSTRINGS@ *convert(@SKFCSTRINGS@ *optstr, @SKFSTRINGS@ *cstr);
@SKFOSTRINGS@ *quickconvert(@SKFCSTRINGS@ *optstr, @SKFSTRINGS@ *cstr);
@SKFOSTRINGS@ *guess(@SKFCSTRINGS@ *optstr, @SKFSTRINGS@ *cstr);
@SKFOSTRINGS@ *inputcode();

/* %rename(init) skf_script_init; */
void skf_script_init();
void destruct();

int  in_codeset;
int  out_codeset;
int  errorcode;
