require 'test/unit/testcase'
require 'amrita/template'
require 'amrita/testsupport'

class TestTemplate < Test::Unit::TestCase
  include Amrita

  def test_template_text_direct1
    html = '<p id="a">sample_text</p>'
    t = TemplateText.new(html)
    result = ""
    t.expand(result, { :a=>"aaa" })
    assert_equal('<p>aaa</p>', result)
  end

  def test_template_text_direct2
    html = '<ul><li id="list">list</li>'
    t = TemplateText.new(html)
    result = ""
    data = { :list => [1, 2, 3] }
    t.expand(result, data)
    assert_equal('<ul><li>1</li><li>2</li><li>3</li></ul>', result)
  end

  def test_template_compile1
    html = '<p id="a">sample_text</p>'
    t = TemplateText.new(html)
    t.use_compiler = true
    result = ""
    #t.set_hint(HashData[:a=>ScalarData])
    t.expand(result, { :a=>"aaa" })
    #puts t.src
    assert_equal('<p>aaa</p>', result)
  end

  def test_template_file1
    tempfile = "/tmp/amritatest#{$$}"
    File.open(tempfile, "w") { |f| f.print <<-END }
    <p id="a">sample_text</p>
    END

    t = TemplateFile.new(tempfile)  
    assert_equal(true, t.need_update?)
    result = ""
    t.expand(result, { :a=>"aaa" })
    sleep 1.1 # for test of need_update?
    assert_equal("    <p>aaa</p>\n", result)
    result = ""
    t.expand(result, { :a=>"111" })
    assert_equal(false, t.need_update?)
    assert_equal("    <p>111</p>\n", result)
    File.open(tempfile, "w") { |f| f.print <<-END }
    <p id="a">sample_text</p>xxx
    END
    result = ""
    assert_equal(true, t.need_update?)
    t.expand(result, { :a=>"aaa" })
    assert_equal("    <p>aaa</p>xxx\n", result)
    assert_equal(false, t.need_update?)
  ensure
    #File.open(tempfile) { |f| puts f.read }
    File::unlink(tempfile)
  end

  def test_template_file2
    tempfile = "/tmp/amritatest#{$$}"
    File.open(tempfile, "w") { |f| f.print <<-END }
    <p id="a">sample_text</p>
    END

    t = TemplateFile.new(tempfile)  
    t.xml = true
    result = ""
    t.expand(result, { :a=>"aaa" })
    assert_equal("<p>aaa</p>", result)
  ensure
    File::unlink(tempfile)
  end

  def test_keep_id
    html = '<p id="a">sample_text</p>'
    t = TemplateText.new(html)
    t.delete_id = false

    result = ""
    t.expand(result, { :a=>"aaa" })
    assert_equal('<p id="a">aaa</p>', result)

  end

  def test_escaped_id
    html = '<p id="a" __id="realid">sample_text</p>'
    t = TemplateText.new(html)
    t.escaped_id = :__id

    result = ""
    t.expand(result, { :a=>"aaa" })
    assert_equal_node('<p id="realid">aaa</p>', result)

  end

  def test_amrita_id
    html = '<p amrita_id="a" id="realid">sample_text</p>'
    t = TemplateText.new(html)
    t.amrita_id = :amrita_id

    result = ""
    t.expand(result, { :a=>"aaa" })
    assert_equal_node('<p id="realid">aaa</p>', result)

  end

  def test_amrita_id2
    html = '<p amrita_id="a" id="realid">sample_text</p><a href="xxx">yyy</a>'
    t = TemplateText.new(html)
    t.amrita_id = :amrita_id

    result = ""
    t.expand(result, { :a=>"aaa" })
    assert_equal_node('<p id="realid">aaa</p><a href="xxx">yyy</a>', result)

  end

  def test_sanitize
    html = '<p id="a">sample_text</p>'
    t = TemplateText.new(html)
    result = ""
    t.expand(result, { :a=>"<xxx>&<yyy>" })
    assert_equal('<p>&lt;xxx&gt;&amp;&lt;yyy&gt;</p>', result)

    result = ""
    t.expand(result, { :a=>Amrita::SanitizedString["<xxx>&<yyy>" ]  })
    #puts t.src
    assert_equal('<p><xxx>&<yyy></p>', result)

  end

  def test_sanitize2
    html = '<p id="a">sample_text</p>'
    t = TemplateText.new(html)
    result = ""
    t.expand(result, { :a=>"<xxx>&<yyy>" })
    assert_equal('<p>&lt;xxx&gt;&amp;&lt;yyy&gt;</p>', result)

    return 
    t = TemplateText.new(html)
    result = ""
    t.prettyprint = true
    t.expand(result, { :a=>"<xxx>&<yyy>" })
    assert_equal("\n<p>&lt;xxx&gt;&amp;&lt;yyy&gt;</p>\n", result)

    t = TemplateText.new(html)
    result = ""
    t.pre_format = true
    t.expand(result, { :a=>"<xxx>&<yyy>" })
    assert_equal('<p>&lt;xxx&gt;&amp;&lt;yyy&gt;</p>', result)
  end


  def test_prettyprint
    return 
    html = '<body><p id="a">sample_text</p></body>'
    t = TemplateText.new(html)
    t.prettyprint = true
    result = ""
    t.expand(result, { :a=>"aaa" })
    assert_equal("\n<body>\n  <p>aaa</p>\n</body>\n", result)

    def t.setup_taginfo
      ret = HtmlTagInfo.new
      ret[:body].pptype = 3
      ret[:p].pptype = 3
      ret
    end
    result = ""
    t.expand(result, { :a=>"aaa" })
    assert_equal("\n<body><p>aaa</p></body>\n", result)
  end

  def test_sanitizedstring
    html = '<p id="a">sample_text</p>'
    t = TemplateText.new(html)
    result = ""
    t.expand(result, { :a=>"<xxx>" })
    assert_equal('<p>&lt;xxx&gt;</p>', result)
    result = ""
    t.expand(result, { :a=>SanitizedString["<xxx>"] })
    assert_equal('<p><xxx></p>', result)

    html = '<p xxx="@aaa">sample_text</p>'
    t = TemplateText.new(html)
    result = ""
    t.expand(result, { :aaa=>"<xxx>"} )
    assert_equal('<p xxx="&lt;xxx&gt;">sample_text</p>', result)
    result = ""
    t.expand(result, { :aaa=>SanitizedString['"<xxx>"']} )
    assert_equal('<p xxx=""<xxx>"">sample_text</p>', result)
  end

  def test_url_sanitizing
    html = '<a href="@a">sample_text</a>'
    xss = 'xxx">yyy</a><script>XSS</script><a href="zzz"'
    t = TemplateText.new(html)
    result = ""
    t.expand(result, { :a=> xss})
    assert_equal('<a href="">sample_text</a>', result)
    result = ""
    t.expand(result, { :a=> SanitizedString[xss]})
    assert_equal('<a href="xxx">yyy</a><script>XSS</script><a href="zzz"">sample_text</a>', result)
  end

  def test_url_sanitizing2
    html = '<xmlelement aaa="@a">sample_text</xmlelement>'
    xss = 'javascritpt:XSS'
    t = TemplateText.new(html)
    result = ""
    t.expand(result, { :a=> xss})
    assert_equal('<xmlelement aaa="javascritpt:XSS">sample_text</xmlelement>', result)

    t = TemplateText.new(html)
    result = ""
    t.tag_info = TagInfo.new
    t.tag_info[:xmlelement].set_url_attr(:aaa)
    t.expand(result, { :a=> xss})
    assert_equal('<xmlelement aaa="">sample_text</xmlelement>', result)

    result = ""
    t.expand(result, { :a=> SanitizedString[xss]})
    assert_equal('<xmlelement aaa="javascritpt:XSS">sample_text</xmlelement>', result)

  end
  
  def test_xhtml
    xhtml1 = <<-END
<?XML version="1.0"?>
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN"
  "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">
<!-- comment1 -->
<html xmlns="http://www.w3.org/1999/xhtml">
<head>
    <title>xhtml sample</title>
</head>
<body>
  <h1 id="title">title</h1>
  <p id="body">body text</p>
  <hr />
</body>
</html>
<!-- comment2 -->
END

    t =  TemplateText.new xhtml1
    def t.setup_template
      super
      #p @template
    end

    result = ""
    t.expand(result, { :title=>'aaa', :body=>'bbb' })
    assert_equal_node(<<END, result)
<?XML version="1.0"?>
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN"
  "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">
<!-- comment1 -->
<html xmlns="http://www.w3.org/1999/xhtml">
<head>
    <title>xhtml sample</title>
</head>
<body>
  <h1>aaa</h1>
  <p>bbb</p>
  <hr>
</body>
</html>
<!-- comment2 -->
END
  end

end


#--- main program ----
if __FILE__ == $0
  require 'test/unit/ui/console/testrunner'

  if ARGV.size == 0
    Test::Unit::UI::Console::TestRunner.run(TestTemplate, Test::Unit::UI::Console::TestRunner::VERBOSE)
    require 'amrita/accel'
    Test::Unit::UI::Console::TestRunner.run(TestTemplate)
  else
    #require 'amrita/accel'
    ARGV.each do |method|
      Test::Unit::UI::Console::TestRunner.run(TestTemplate.new(method))
    end
  end
end

