
require 'amrita/node'

class Proc
  alias :execute :call 
end

module Amrita
  module Amulet
  end

  module ByteCodeCommon
    module Instruction
      include Amrita

      attr_reader :seq

      def instruction_name
        self.class.to_s.gsub(/.*::/, "")
      end

      def prepare(vm, opt, iset)
        super
        set_seqno(iset)
        if opt[:debug]
          define_debug_execute(vm)
        else
          if self.respond_to?(:execute_func)
            class << self
              alias :execute :execute_func 
            end
          else
            define_execute(vm)
          end
        end
      end

      def set_seqno(iset)
        @seq = iset.new_seqno
      end

      def define_execute(vm)
        @src = <<-END
        def execute(register, stack, out, vm)
          #{ruby_code(vm)}
          register  
        rescue StandardError
          puts "an exeption happend here"
          puts @src
          puts self.class
          raise
        end
        END
        instance_eval @src
      rescue ScriptError
        puts @src
        raise
      end

      def has_cond_instruction?
        false
      end

      def ruby_code(vm, iset=nil)
        raise "subclass responsibility #{self.class}"
      end

      # optimize self and return true if changed
      def optimize
        false
      end

      # return an instruction if optimized
      # return array[self, other] unless optimized
      def optimize_with(other_instruction)
        case other_instruction
        when NullInstruction
          self
        when Sequence
          ByteCode::Sequence[*([self] + other_instruction.instructions) ]
        else
          [ self, other_instruction]
        end
      end

      def inspect
        if param
          %Q[#{instruction_name}[ #{param.inspect} ] ] 
        else
          %Q[#{instruction_name}[] ] 
        end
      end

      def dump(out, level=0)
        out << sprintf("%03d:",@seq) if @seq
        out << ("  " * level) << instruction_name << " "
        out << param.inspect if param
        out << "\n"
      end

      def param
        nil
      end

      def inspect_var(v)
        s = v.inspect
        if s.size > 100
          s[0..100] + "..."
        else
          s
        end
      end

      def define_debug_execute(vm)
        body = nil
        if self.respond_to?(:execute_func)
          body = "register = execute_func(register, stack, out, vm)"
        else
          body = ruby_code(vm)
        end
        @src = <<-END
        def execute(register, stack, out, vm)
          vm.debug_out << "#{@seq}:#{instruction_name}  \n"
          vm.debug_out.puts "register=\#{inspect_var(register)} stack=\#{inspect_var(stack)}"
          pre_value = pre_check(register, stack, out, vm)
          ### body code start ###
          #{body}
          ### body code end   ###
          vm.debug_out.puts "register=\#{inspect_var(register)} stack=\#{inspect_var(stack)}"
          post_check(register, stack, out, vm, pre_value)
          register  
        rescue Exception
          vm.debug_out << "an exeption happend here\n"
          vm.debug_out.puts @src
          raise
        end
        END
        instance_eval @src
      rescue ScriptError
        vm.debug_out << "exeption"
        dump(vm.debug_out)
        vm.debug_out.puts @src
        raise
      end

      def pre_check(register, stack, out, vm)
        true
      end

      def post_check(register, stack, out, vm, pre_value)
      end

      def assert(x)
        unless x
          puts "error happend in "
          dump(STDOUT)
          raise 
        end
      end

      def assert_equal(a, b)
        assert(a == b)
      rescue RuntimeError
        puts "expected #{a.inspect} but found #{b.inspect}"
        raise
      end

      def assert_type(typ, x)
        assert(x.kind_of?(typ))
      rescue RuntimeError
        puts "expected #{typ} but found #{x.class}"
        raise
      end
    end

    module InstructionMarshal
      def _dump(level)
        ret = Marshal::dump(param) 
        ret
      end
      
      def InstructionMarshal::append_features(subclass)
        super
        def subclass::_load(s)
          param = Marshal::load(s)
          ret = new(*param)
          ret
        end
      end
    end

    module NullInstruction
      include ByteCodeCommon::Instruction

      def ruby_code(vm, iset=nil)
        "# do nothing"
      end

      def optimize_with(other_instruction)
        other_instruction
      end

      def pre_check(register, stack, out, ep)
        [register, stack.dup, out.dup]
      end

      def post_check(register, stack, out, np, pre_value)
        assert_equal(register, pre_value[0])
        assert_equal(stack, pre_value[1])
        assert_equal(out, pre_value[2]) unless out.kind_of?(IO)
      end
    end

    module GetDataByKey 
      include Instruction

      def ruby_code(vm, iset=nil)
        <<-END
        stack.push register
        register = register[#{key.inspect}]
        END
      end

      def param
        [key]
      end

      def pre_check(register, stack, out, ep)
        raise "can't handle #{register.inspect}" unless register.kind_of?(AmritaDictionary) or register.respond_to?(:[])
        register.dup
      end

      def post_check(register, stack, out, np, pre_value)
        assert_equal(register, pre_value[key])
        assert_equal(stack[-1], pre_value) unless pre_value.kind_of?(Struct)
      end
    end

    module ExtractData
      include Instruction

      def ruby_code(vm, iset=nil)
        hash_code = keys.collect do |k, v|
          case v
          when Symbol
            "#{k.intern.to_ruby} => stack[-1][:#{v}]"
          when Array
            vexp = v.collect do |x|
              case x
              when String
                x.inspect
              when Symbol
                %Q["#\{stack[-1][:#{x}]}"]
              else
                raise "can't happen"
              end
            end.join(" + ")
            %Q[#{k.intern.to_ruby} => #{vexp}]
          else
            raise "can't happen"
          end
        end.join(",\n")
        ret = <<-END
        stack.push register
        register = {
          #{hash_code}
        }
        END
        ret
      end

      def param
        keys
      end

      def pre_check(register, stack, out, ep)
        raise unless register.kind_of?(Hash)
        register.dup
      end

      def post_check(register, stack, out, np, pre_value)
        keys.each do |k|
          assert_equal(register[k], pre_value[k])
        end
        assert_equal(stack[-1], pre_value)
      end
    end

    module PopData
      include Instruction

      def ruby_code(vm, iset=nil)
        %Q[ register = stack.pop ]
      end

      def pre_check(register, stack, out, ep)
        assert(stack.size > 0)
        [register, stack.dup]
      end

      def post_check(register, stack, out, np, pre_value)
        assert_equal(register, pre_value[1][-1])
        assert_equal(stack.size , pre_value[1].size - 1)
      end
    end

    module SwapData
      include Instruction

      def ruby_code(vm, iset=nil)
        %Q[ register, stack[-1] = stack[-1], register ]
      end

      def pre_check(register, stack, out, ep)
        assert(stack.size > 0)
        [register, stack.dup]
      end

      def post_check(register, stack, out, np, pre_value)
        assert_equal(register, pre_value[1][-1])
        assert_equal(stack[-1], pre_value[0])
      end
    end


    module PrintBase
      include Instruction
      attr_reader :text

      def ruby_code(vm, iset=nil)
        <<-END
        out << #{self.text.to_s.inspect}
        register
        END
      end

      def param
        [text]
      end

      def optimize_with(other_instruction)
        if self.text and self.text.size > 0
          case other_instruction
          when PrintBase
            ByteCode::PrintStaticText[text.to_s + other_instruction.text.to_s]
          else
            super
          end
        else
          other_instruction
        end
      end

      def pre_check(register, stack, out, ep)
        out.dup
      end

      def post_check(register, stack, out, np, pre_value)
        assert_equal(out, pre_value + text.to_s) if pre_value.kind_of?(String)
      end
    end
    
    module PrintStaticText
      include PrintBase
    end

    module PrintStaticNode 
      include PrintBase
      def prepare(vm, opt, iset)
        self.text = vm.ep.format_node(node)
        super
      end

      def param
        [node]
      end
    end
    
    module PushElement
      include Instruction
      def ruby_code(vm, iset=nil)
        <<-END
        stack.push register
        register = #{element.to_ruby}
        END
      end

      def param
        [element]
      end
    end

    module PrintRegister
      include Instruction

      def ruby_code(vm, iset=nil)
        <<-END
        case register
        when Node
          out << vm.ep.format_node(register).to_s 
        else
          out << register.to_s.amrita_sanitize 
        end
        END
      end
    end

    module MergeElement
      include Instruction

      def ruby_code(vm, iset=nil)
        iset = vm.iset unless iset
        c1 = iset.new_constant(element)
        <<-END
        stack.push register
        register = #{c1}
        stack[-1].each do |key, value|
          register[key] = value
        end
        stack.pop
        END
      end

      def pre_check(register, stack, out, ep)
        assert_type(AmritaDictionary, register)
        [register, stack.dup]
      end

      def post_check(register, stack, out, np, pre_value)
        assert_type(Element, register)
        assert_equal(element.tag, register.tag)
        pre_value[0].each do |k, v|
          #assert_equal(v, register[k])
        end
      end

      def param
        [element]
      end
    end

    module GetDataFromAttrArray 

      include Instruction

      def ruby_code(vm, iset=nil)
        <<-END
          stack.push register.body
          register = register.to_hash
        END
      end

      def pre_check(register, stack, out, ep)
        assert_type(AttrArray, register)
        [register, stack.dup]
      end

      def post_check(register, stack, out, np, pre_value)
        assert_type(Hash, register)
        assert_equal(pre_value[0].body, stack[-1])
        register.each do |k, v|
          assert_equal(v, pre_value[0].to_hash[k])
        end
      end
    end

    module GetDataFromAmulet
      include Instruction

      def ruby_code(vm, iset=nil)
        <<-END
          stack.push register
          stack.push register[:amulet_data]
          register = register[:amulet_iset]
          register.prepare(vm, vm.iset.opt)
        END
      end
    end

    module SimpleSpanSupport
      include Amrita
      Methods = []
      def SimpleSpanSupport::add_method(cls, mid)
        cls = cls.class unless cls.kind_of?(Module)
        Methods << [cls, mid]
      end

      def do_trueclass(register, stack, out, vm, text)
        out << text
      end
      add_method true, :do_trueclass

      def do_nilclass(register, stack, out, vm, text)
      end
      add_method nil, :do_nilclass
      add_method false, :do_nilclass

      def do_string(register, stack, out, vm, text)
        out << register.amrita_sanitize
      end
      add_method String, :do_string
      
      def do_integer(register, stack, out, vm, text)
        out << register.to_s.amrita_sanitize
      end
      add_method Integer, :do_integer

      def do_node(register, stack, out, vm, text)
        out << vm.ep.format_node(register)
      end
      add_method Node, :do_node

      def do_attrarray(register, stack, out, vm, text)
        out << "<span "
        register.each do |a|
          out << %Q[#{a.key}="#{a.value}"]
        end
        if register.body != Null
          out << ">" << register.body << "</span>"
        else
          out << ">" << text << "</span>"
        end
      end
      add_method AttrArray, :do_attrarray

      def do_enumerable(register, stack, out, vm, text)
        register.each do |r|
          simplespan_execute_func(r, stack, out, vm, text)
        end
      end
      add_method Enumerable, :do_enumerable

      
      def do_proc(register, stack, out, vm, text)
        elem = e(:span) { text }
        ret =  register.call(elem, stack, out, vm)
        simplespan_execute_func(ret, stack, out, vm, text)
      end
      add_method Proc, :do_proc

      def do_amulet(register, stack, out, vm, text)
        stack.push register
        iset= register[:amulet_iset]
        iset.prepare(vm, vm.iset.opt)
        iset.execute(register[:amulet_data], stack, out, vm)
        register = stack.pop
      end
      add_method Amrita::Amulet, :do_amulet

      def simplespan_execute_func(register, stack, out, vm, text)
        rcls = register.class
        @cache ||= Hash.new
        m = @cache[rcls]
        unless m
          cls, mid = *Methods.find do |cls, meth|
            register.kind_of?(cls)
          end
          if mid
            m = method(mid)
          else
            register = register.to_s
            m = method(:do_string)
          end
          @cache[rcls] = m
        end
        m.call(register, stack, out, vm, text)
      end
    end

    module SimpleSpan
      include Instruction
      include SimpleSpanSupport

      def param
        [tag, text]
      end

      def ruby_code(vm, iset=nil)
        t = case text
            when String; text.inspect ;
            when NullNode; '""' ;
            else text.to_s.inspect ;
            end
        <<-END
        stack.push(register)
        register = register[:#{tag}]
        text = #{t}
        simplespan_execute_func(register, stack, out, vm, text)
        register = stack.pop
        END
      end
    end

    module SimpleSpanMarshal
      def _dump(level)
        Marshal::dump([tag, text])
      end

      def SimpleSpanMarshal::append_features(mod)
        super
        def mod::_load(s)
          tag, text= *Marshal::load(s)
          Amrita::ByteCode::SimpleSpan::new(tag, text)
        end
      end
    end

    module Sequence
      include Instruction

      def <<(instruction)
        self.instructions << instruction
      end

      def prepare(vm, opt, iset)
        super

        each do |i|
          i.prepare(vm, opt, iset)
        end
        
#         unless has_cond_instruction? or Object::const_defined? :AmritaAccelerator
#           src = <<-END
#           def self::execute(register, stack, out, vm)
#             #{ruby_code(vm, iset)}
#             register  
#           rescue StandardError
#             puts "an exeption happend here"
#             puts src
#             puts self.class
#             raise
#           end
#           END
#           eval src
#           return
#         end
          
        if opt[:optimize]
          if optimize 
            each do |i|
              i.prepare(vm, opt, iset)
            end
          end
        end
      end

      def execute_func(register, stack, out, vm)
        each do |i|
          register = i.execute(register, stack, out, vm)
        end
        register
      end

      def has_cond_instruction?
        instructions.each do |i|
          return true if i.has_cond_instruction?
        end
        false
      end

      def ruby_code(vm, iset=nil)
        ret = self.instructions.collect do |i|
          i.ruby_code(vm, iset) 
        end.join("\n")
        ret
      end

      def optimize
        if instructions.size < 2
          false
        else
          result = []
          a = b = x = nil
          changed = false
          each do |b|
            if a
              changed ||= a.optimize
              changed ||= b.optimize
              x = a.optimize_with(b)
              case x
              when Array
                result << x[0]
                a = x[1]
              when Instruction
                a = x
                changed = true
              else
                raise "bad optimize"
              end
            else
              a = b
            end
          end
          case x
          when Array
            result << x[1]
          when Instruction
            result << x
            changed = true
          else
            raise "bad optimize"
          end
          replace_instructions(result)
          if changed
            optimize 
            true
          else
            false
          end
        end
      end

      def optimize_with(other_instruction)
        case other_instruction
        when Sequence
          i = instructions || []
          i += other_instruction.instructions
          replace_instructions(i)
          self
        else
          super
        end
      end

      def inspect
        instruction_name + "[\n" +
          instructions.collect do |i|
            i.inspect
          end.join(",\n") + "\n]"
      end

      def dump(out, level=0)
        super
        each do |i|
          i.dump(out, level + 1)
        end
      end
    end

    module SequenceMarshal
      def _dump(level)
        Marshal::dump(instructions)
      end

      def SequenceMarshal::append_features(mod)
        super
        def mod::_load(s)
          a = Marshal::load(s)
          ret = Amrita::ByteCode::Sequence::new
          a.each do |i|
            ret << i
          end
          ret
        end
      end
    end

    module PrintDynamicElement
      include Instruction

      def prepare(vm, opt, iset)
        super
        body.prepare(vm, opt, iset)
      end

      def execute_func(register, stack, out, vm)
        out << vm.ep.format_start_tag(register)
        register, stack[-1] = stack[-1], register 
        register = body.execute(register, stack, out, vm)
        out << vm.ep.format_end_tag(stack.pop)
        register
      end

      def ruby_code(vm, iset=nil)
        <<-END
        out << vm.ep.format_start_tag(register) 
        register, stack[-1] = stack[-1], register 
        #{body.ruby_code(vm, iset)}
        out << vm.ep.format_end_tag(stack.pop) 
        register
        END
      end

      def dump(out, level=0)
        super
        body.dump(out, level + 1)
      end

      def _dump(s)
        Marshal::dump(body)
      end

      def PrintDynamicElement::append_features(mod)
        super
        def mod::_load(s)
          body = Marshal::load(s)
          new(body)
        end
      end
    end

    module PrintDynamicElementMarshal
      def _dump(level)
        Marshal::dump(body)
      end

      def PrintDynamicElementMarshal::append_features(mod)
        super
        def mod::_load(s)
          body = Marshal::load(s)
          new(body)
        end
      end
    end

    module ExecuteRegister
      include Instruction

      def ruby_code(vm, iset=nil)
        %Q[register = register.execute(stack.pop, stack, out, vm)]
      end
    end

    module Loop
      include Instruction

      def prepare(vm, opt, iset)
        super
        body.prepare(vm, opt, iset)
      end

      def execute_func(register, stack, out, vm)
        register.each do |x|
          register = body.execute(x, stack, out, vm)
        end
        register
      end

      def has_cond_instruction?
        body.has_cond_instruction?
      end

      def ruby_code(vm, iset=nil)
        <<-END
          register.each do |register|
          #{body.ruby_code(vm, iset)}
          end
        END
      end

      def optimize
        ret = body.optimize
        super or ret
      end

      def dump(out, level=0)
        super
        if body
          body.dump(out, level + 1)
        else
          out << " has no body!!!"
        end
      end
    end

    module LoopMarshal
      def _dump(level)
        Marshal::dump(body)
      end

      def LoopMarshal::append_features(mod)
        super
        def mod::_load(s)
          body = Marshal::load(s)
          new(body)
        end
      end
    end

    module SelectByType
      include Instruction

      def []=(*a)
        types = a[0..-2]
        instruction = a[-1]
        conds << [types, instruction]
      end

      def prepare(vm, opt, iset)
        super
        conds.each do |types, instruction|
          instruction.prepare(vm, opt, iset)
        end
        else_.prepare(vm, opt, iset) if else_
        @cache = {}
      end


      def execute_func(register, stack, out, vm)
        c = @cache[register.class]
        return c.execute(register, stack, out, vm) if c

        conds.each do |types, instruction|
          if types.find { |typ| register.kind_of?(typ) }
            @cache[register.class] = instruction
            return instruction.execute(register, stack, out, vm)
          end
        end
        if else_
          @cache[register.class] = else_
          else_.execute(register, stack, out, vm)
        else
          raise "type mismatch"
        end
      end

      def has_cond_instruction?
        true
      end

      def ruby_code(vm, iset=nil)
        when_ = conds.collect do |types, i|
          types = types.collect { |t| t.to_s }
          %Q[ when #{types.join(',')}
              #{i.ruby_code(vm, iset)}
            ]
          end.join("\n")
        else_code = ""
        else_code = <<-END if else_
        else
          #{else_.ruby_code(vm, iset)}
        END
        <<-END
        case register
          #{when_}
          #{else_code}
        end
      END
      end

      def optimize
        ret = false
        conds.each do |types, i|
          ret ||= i.optimize
        end

        if else_
          ret ||= else_.optimize
        end
        super or ret
      end

      def dump(out, level=0)
        super
        conds.each do |types, i|
          out << "    " << ("  " * (level+1)) << %Q[#{types.join(', ')}] << "\n"
          i.dump(out, level + 2)
        end

        if else_
          out << "    " << ("  " * (level+1)) << "ELSE \n"
          else_.dump(out, level + 2)
        end
      end

      def inspect
        %Q[SelectByType[#{conds.inspect}, #{else_.inspect}]]
      end
    end

    module SelectByTypeMarshal
      def _dump(level)
        Marshal::dump([conds, else_])
      end

      def SelectByTypeMarshal::append_features(mod)
        super
        def mod::_load(s)
          conds, else_ = *Marshal::load(s)
          ret = Amrita::ByteCode::SelectByType::new
          conds.each do |types, instructions|
            ret[*types] = instructions
          end
          ret.else_ = else_
          ret
        end
      end
    end
  end

  module ByteCode
    MOD_PREFIX = 'Amrita::ByteCode::'

    class Instruction
      include Amrita

      def Instruction.inherited(subclass)
        super
        def subclass.[] (*a)
          new(*a)
        end
      end

      def initialize
      end

      def prepare(vm, opt, iset)
      end
    end


    class NullInstruction < Instruction
      include ByteCodeCommon::NullInstruction
      include ByteCodeCommon::InstructionMarshal
    end

    class GetDataByKey < Instruction
      include ByteCodeCommon::GetDataByKey
      include ByteCodeCommon::InstructionMarshal
      attr_reader :key
      def initialize(key)
        super()
        @key = key.intern
      end
    end

    class ExtractData < Instruction
      include ByteCodeCommon::ExtractData
      include ByteCodeCommon::InstructionMarshal
      attr_reader :keys
      def initialize(keys)
        super()
        @keys = keys
      end
    end

    class PopData < Instruction
      include ByteCodeCommon::PopData
      include ByteCodeCommon::InstructionMarshal
    end

    class SwapData < Instruction
      include ByteCodeCommon::SwapData
      include ByteCodeCommon::InstructionMarshal
    end


    class PrintStaticNode < Instruction
      include ByteCodeCommon::PrintStaticNode
      include ByteCodeCommon::InstructionMarshal
      attr_accessor :text
      attr_reader :node
      def initialize(node=Null)
        super()
        @node = node
      end
    end

    class PrintStaticText < Instruction
      include ByteCodeCommon::PrintStaticText
      include ByteCodeCommon::InstructionMarshal
      attr_accessor :text
      def initialize(text="")
        super()
        @text = text
      end
    end

    class PushElement < Instruction
      include ByteCodeCommon::PushElement
      include ByteCodeCommon::InstructionMarshal
      attr_reader :element
      
      def initialize(elem)
        @element = elem
      end
    end

    class PrintRegister < Instruction
      include ByteCodeCommon::PrintRegister
      include ByteCodeCommon::InstructionMarshal
    end

    class MergeElement < Instruction
      include ByteCodeCommon::MergeElement
      include ByteCodeCommon::InstructionMarshal
      attr_reader :element
      def initialize(element)
        @element = element.clone { nil }
      end
    end

    class GetDataFromAttrArray < Instruction
      include ByteCodeCommon::GetDataFromAttrArray
      include ByteCodeCommon::InstructionMarshal
    end

    class GetDataFromAmulet < Instruction
      include ByteCodeCommon::GetDataFromAmulet
      include ByteCodeCommon::InstructionMarshal
    end

    class SimpleSpan < Instruction
      include ByteCodeCommon::SimpleSpan
      include ByteCodeCommon::SimpleSpanMarshal
      attr_reader :tag, :text

      def initialize(tag, text)
        @tag, @text = tag, text
      end
    end

    class Sequence < Instruction
      include ByteCodeCommon::Sequence
      include ByteCodeCommon::SequenceMarshal
      attr_reader :instructions

      def initialize(*instructions)
        super()
        @instructions = instructions
      end

      def each(&block)
        @instructions.each(&block)
      end

      def replace_instructions(instructions)
        raise "invalid parameter #{instructions.class}" unless instructions.kind_of?(Array)
        instructions.each do |i|
          raise "invalid parameter #{i.inspect}" unless i.kind_of?(Instruction)
        end
        @instructions = instructions
      end
    end

    class PrintDynamicElement < Instruction
      include ByteCodeCommon::PrintDynamicElement
      include ByteCodeCommon::PrintDynamicElementMarshal
      attr_reader :body
      def initialize(body)
        super()
        @body = body
      end
    end

    class ExecuteRegister < Instruction
      include ByteCodeCommon::ExecuteRegister
      include ByteCodeCommon::InstructionMarshal
    end

    class Loop < Instruction
      include ByteCodeCommon::Loop
      include ByteCodeCommon::LoopMarshal
      attr_reader :body
      def initialize(body)
        super()
        @body = body
      end
    end

    class SelectByType < Instruction
      include ByteCodeCommon::SelectByType
      include ByteCodeCommon::SelectByTypeMarshal

      attr_reader :conds
      attr_accessor :else_
      def initialize(conds=[], else_=nil)
        super()
        @conds = conds
        @else_ = else_
      end
    end
  end


  class InstructionSet
    attr_reader :opt

    def initialize(entry_point_or_src, constants_=nil)
      @seq = @cseq = @vseq = @subseq = 0
      case entry_point_or_src
      when ByteCodeCommon::Instruction
        @entry_point = entry_point_or_src
        @vm = @opt = @src = nil
        @module = nil
      when String # restoreing used by Marshal::load
        init_from_src(entry_point_or_src, constants_)
      else
         raise "invalid parameter #{entry_point_or_src} for InstructionSet#new" 
       end
    end

    def prepare(vm, opt)
      return if @module # dumped after compiled and loaded again
      @opt = opt
      @module = nil
      @seq = @cseq = @vseq = @subseq = 0
      @constants = []
      @vars = []   
      @entry_point.prepare(vm, opt, self)
      if opt[:use_compiler]
        compile(vm, opt)
      end
    end

                  

    def new_seqno
      @seq += 1
      @seq
    end

    def new_constant(cvalue)
      return cvalue.to_ruby unless @opt[:use_compiler]
      @cseq += 1
      cname = sprintf("C%03d", @cseq)
      @constants << [cname, cvalue]
      @module.const_set(cname.intern, cvalue)  if @module
      cname
    end

    def new_variable
      @vseq += 1
      vname = sprintf("@_tmp_%03d", @vseq)
      vname
    end

    def ruby_sub
      @subseq += 1
      subname = sprintf("sub%03d", @subseq)
      @module.module_eval <<-END
         def #{subname}(register, stack, out, vm)
           #{yield(subname)}
         end
         module_function :#{subname}
      END
      @module.method(subname.intern)
    end

    def compile(vm, opt)
      rubycode = @entry_point.ruby_code(vm, self)

      @module = Module.new

      #cdefs = @constants.collect do |cname, cdef|
      #  "#{cname} = #{cdef}"
      #end.join("\n")
      @constants.each do |cname, cvalue|
         @module.const_set(cname.intern, cvalue)
      end

      vinit = (1...@vseq).collect do |i|
        sprintf("@_tmp_%3d = nil",i)
      end.join
      @src = <<-END
      extend Amrita
      extend ByteCodeCommon::SimpleSpanSupport
      def self::execute(register, stack, out, vm)
        #{vinit}
        #{rubycode}
        register  
      rescue StandardError
        puts "an exeption happend here"
        puts @src
        puts self.class
        raise
      end
      END
      # puts @src
      @module.module_eval(@src)
    rescue ScriptError
      puts @src
      raise
    end

    def execute(register, stack, out, vm)
      if @module
        @module::execute(register, stack, out, vm)
      else
        @entry_point.execute(register, stack, out, vm)
      end
    end

    def dump(out)
      if @entry_point
        @entry_point.dump(out)
      else
        out << "Translated Bytecode Without original\n"
      end
    end

    def set_opt(opt)
      @opt = opt
    end

    def _dump(level)
      if @src and not @opt[:partial_compile]
        Marshal::dump([:src, @src, @constants, @opt])
      else
        Marshal::dump([:bytecode, @entry_point, @opt])
      end
    end
    
    def InstructionSet::_load(stream)
      ret = nil
      param = Marshal::load(stream)
      case param[0]
      when :bytecode
        ret = InstructionSet.new(param[1])
        ret.set_opt(param[2])
      when :src
        ret = InstructionSet.new(param[1], param[2]) # init_from_src
        ret.set_opt(param[3])
      end
      ret
    end

    def init_from_src(src, constants_)
      @src = src
      @constants = constants_
      @module = Module.new
      @constants.each do |cname, cvalue|
         @module.const_set(cname.intern, cvalue)
      end
      @module.module_eval(@src)
    end
  end

  class VirtualMachine
    include Amrita

    class VMAsserttionFail < RuntimeError
      attr_reader :instruction
      def initialize(instruction)
        @instruction = instruction
      end
    end

    class NullOut
      def <<(x)
        self
      end

      def puts(x)
      end
    end

    attr_accessor :register, :out, :debug_out, :use_compiler, :optimize_bytecode, :lazy_evaluation, :partial_compile
    attr_reader :iset, :stack, :ep, :debug, :opt

    def initialize(ep=DefaultElementProcessor)
      @iset = nil
      @register = nil
      @stack = []
      @out = ""
      @ep = ep
      @use_compiler = self.debug = @optimize_bytecode = @lazy_evaluation = false
      @partial_compile = true
    end

    def debug=(f)
      @debug = f
      if @debug
        @debug_out = STDOUT
      else
        @debug_out = NullOut.new
      end
      f
    end

    def load_bytecode(bytecode)
      @opt = { 
        :use_compiler=>use_compiler ,
        :debug=>debug,
        :optimize=>optimize_bytecode,
        :lazy_evaluation=>lazy_evaluation,
        :partial_compile=>partial_compile
      }
      if bytecode.kind_of?(InstructionSet)
        @iset = bytecode
      else
        @iset = InstructionSet.new(bytecode)
      end
      @iset.prepare(self, @opt)
    end

    def dump_bytecode
      @iset.dump(@debug_out)
    end

    def go
      @iset.execute(@register, @stack, @out, self)
    rescue VMAsserttionFail
      print "vm failed at "
      $!.instruction.dump(STDOUT)
      #puts #stack=#{@stack.inspect}"
      #puts #out=#{@out}"
      raise
    rescue RuntimeError, NameError
      #puts "stack=#{@stack.inspect}"
      #puts "out=#{@out}"
      raise
    end
  end

end

