#----------------------------------------------------------------------------
#   node.rb
#----------------------------------------------------------------------------
#  Copyright (C) 2003 August Nowake(野分) nowake@fiercewinds.net
#  This program is free software; you can redistribute it and/or modify it 
#  under the terms of the GNU General Public License version 2 as published 
#  by the Free Software Foundation.
#  This program is distributed in the hope that it will be useful, but 
#     WITHOUT ANY WARRANTY;
#  without even the implied warranty of MERCHANTABILITY or FITNESS FOR
#  A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#  You should have received a copy of the GNU General Public License along 
#  with this program; if not, write to the Free Software Foundation, Inc., 
#  59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#----------------------------------------------------------------------------
#  このプログラムはフリーソフトウェアです。あなたはこれを、フリーソフトウェア
#  財団によって発行された GNU一般公衆利用許諾契約書(バージョン2)の定める条件
#  の下で再頒布または改変することができます。
#  このプログラムは有用であることを願って頒布されますが、
#    *全くの無保証*
#  です。商業可能性の保証や特定の目的への適合性は、言外に示されたものも含め
#  全く存在しません。詳しくはGNU 一般公衆利用許諾契約書をご覧ください。
#  あなたはこのプログラムと共に、GNU 一般公衆利用許諾契約書の複製物を一部
#  受け取ったはずです。もし受け取っていなければ、フリーソフトウェア財団まで
#  請求してください。
#  (宛先は the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
#  Boston, MA 02111-1307 USA)
#----------------------------------------------------------------------------

require 'cgi'
require 'fileutils'
require 'date'
require 'digest/md5'
require 'algorithm/diff'

module Node
  class NodeInitializeError < StandardError; end
  class NodeContainerError < StandardError; end
  module Target end
  module Comeback end
    
  class Reader
    attr_reader( :name, :escaped_name, :container_path )
    def initialize( node_name, variables )
      @variables = variables
      @name = node_name
      @name = @name != "" ? @name : variables.default_name
      @escaped_name = CGI.escape( @name )
      raise( NodeInitializeError, "Can't use sign #{@name}") if
          not (@name =~ /^[\w\/\!\.]+$/)
      if @name.include?( '!' )
        @file_data = ContainerFile.new( self, variables )
      else
        @file_data = NodeFile.new( self, variables )
        name_path = @file_data
        @backup_file_data = BackupFile.new( self, variables )
        @cache_file_data = CacheFile.new( self, variables )
      end
    end
    def html_data
      if recency_node? or not @variables.use_cache
        @variables.formatter.on_show(
            escapeHTML_content, 0..(escapeHTML_content.size-1), self )
      elsif @cache_file_data.exist?
        @cache_file_data.content
      else
        c = @variables.formatter.on_show(
            escapeHTML_content, 0..(escapeHTML_content.size-1), self )
        @cache_file_data.content = c if @variables.use_cache
        c
      end
    end
    def data; @file_data.data end
    def container?; @file_data.container? end
    def file_exist?; @file_data.exist? end
    def recency_node?; name == @variables.recency_name end
    def content; @file_data.content end
    def escapeHTML_content; @file_data.escapeHTML_content end
    def clear_cache; @cache_file_data.clear_cache end
    def base_name; @file_data.base_name end
    def container_items; @file_data.container_items end
    def container_file_data( file_name )
      File.open( @container_path + file_name, "r+b" ) do | f | f.read end
    end
    def backup_datetime
      @backup_file_data.datetime
    end
    def backup_datetime=( datetime )
      @backup_file_data.datetime = datetime
    end
    def backup_modification
      @backup_file_data.modification
    end
  end
  
  class Accessor < Reader
    def initialize( node_name, variables )
      super
      @can_modify = true
      @tdiff = nil
    end
    def save
      return if not @can_modify
      if not container?
        CacheFile.new( self, @variables ).clear_cache
        BackupFile.new( self, @variables, DateTime.now ) << @tdiff.diff
        update_recency
      end
      @file_data.content = @content
    end
    def content=( val ) #old_content, new_content
      return if not @can_modify
      if container?
        @content = val
      else
        val[0].gsub!( /(?:\r\n)|\r|\n/ , "\n" )
        val[1].gsub!( /(?:\r\n)|\r|\n/ , "\n" )
        @tdiff = SikiTool::TriDiff.new( content, val[0], val[1] )
        @content = @tdiff.integrate
      end
    end
  private
    def update_recency
      return if @variables.recency_name == CGI.unescape( self.name )
      recency_node = Node::Accessor.new(@variables.recency_name, @variables)
      t = @variables.formatter[Formatter::Link.base_name].format( self.name )
      p = @variables.formatter[Formatter::UnorderedList.base_name]
      s = recency_node.content
      s = p.delete( s, "#{Regexp.escape(t)}.*" )
      s = p.push( s, "#{t} #{DateTime.now.to_s}" )
      s = p.top20( s )
      recency_node.content = recency_node.content,  s
      recency_node.save
    end
  end

  class DataFile
    attr_reader( :name )
    def initialize( node, variables )
      @node = node
      @variables = variables
    end
    def data
      return "" if not exist?
      File.open( @name, "r+b" ) do | f | f.read end
    end
    def exist?; File.exist?( @name ) end
  protected
    def create_name_path( node_name )
      name_path = "";
      node_name.split( /[\/]/ ).each do | n |
        name_path << "_#{Digest::MD5.hexdigest(n)[0..1]}/#{CGI.escape(n)}/"
      end
      name_path.chop!.untaint
    end
  end

  class NodeFile < DataFile
    def initialize( node, variables )
      super
      @can_modify = true
      @media_type = %s{text/html}
      name_path = create_name_path( node.name )
      @container_path = "#{@variables.container_root}#{name_path}".untaint
      @name = "#{@variables.node_root}#{name_path}.txt".untaint
    end
    def container?; false end
    def content
      m = /(?:^$\n)|(?:\n\Z)/.match( data )
      m ? m.post_match.to_s : ""
    end
    def escapeHTML_content; CGI.escapeHTML( content ) end
    def content=( val )
      return if not @can_modify
      FileUtils.mkdir_p( File.dirname( @name ) )
      File.open( @name, "w+b" ) do | file |
        #Headerの挿入
        file << "\n" #区切り文字
        #本文の挿入
        file <<
            @variables.formatter.on_save( val, 0..(val.size-1), self )
      end
    end
    def container_items
      ary = Dir.entries( @container_path )
      ary[2..ary.size]
    end
  end

  class CacheFile < DataFile
    def initialize( node, variables )
      super
      @name = "#{variables.cache_root}#{create_name_path(node.name)}.html"
      @name.untaint
    end
    def exist?; File.exist?( @name ) end
    def content; data end
    def content=( val )
      FileUtils.mkdir_p( File.dirname( @name ) )
      File.open( @name, "w+b" ) do | file | file << val end
    end
    def clear_cache
      FileUtils.rm( @name ) if File.exist?( @name )
    end
  end
  
  class ContainerFile < DataFile
    attr_reader( :base_name )
    def initialize( node, variables )
      super
      @media_type = %s{application/octet-stream}
      m = node.name.split('!')
      m[1] = CGI.escape( m[1] )
      @name = "#{@variables.container_root}#{create_name_path(m[0])}/#{m[1]}"
      @name.untaint
      @base_name = m[1]
    end
    def container?; true end
    def content; data end
    def content=( val )
      FileUtils.mkdir_p( File.dirname( @name ) )
      File.open( @name, "w+b" ) do | file | file << val end
    end
  end

  class BackupFile < DataFile
    attr_reader( :datetime )
    def initialize( node, variables, backup_datetime = nil )
      super( node, variables )
      @name_path = create_name_path( node.name )
      self.datetime = backup_datetime ? backup_datetime : DateTime.now
    end
    def datetime=( val )
      @datetime = DateTime.parse( val.to_s )
      @name =
          "#{@variables.backup_root}#{@datetime.to_s[0..6]}/#{@name_path}.txt"
    end
    def modification
      m = /(?:^$\n)|(?:\n\Z)/.match( data )
      return [[], []] if not m
      diff_set = create_diff_set( m.pre_match )
      [patch( m.post_match, diff_set ), diff_set]
    end
    def <<( diff )
      return if diff == "[]"
      s = ""
      if File.exist?( @name )
        File.open( @name, "r+b" ) do | file | s << file.read end
      else
        FileUtils.mkdir_p( File.dirname( @name ) )
        s = "Backup-Date: #{@now.to_s}\n"
        s << Node::Reader.new( @node.name, @variables ).data
      end
      File.open( @name, "w+b" ) do | file |
        file << "Diff: #{DateTime.now.to_s} #{diff.inspect}\n" +  s
      end
    end
  private
    def create_diff_set( header )
      diff = []
      header.each_line do | line |
        md = /^Diff\: ([^\s]+) (.*)/.match( line )
        next if not md
        array = eval( md[2].untaint ) if md[2]  =~ /^\[.*\]$/
        array.each do | i | i[2] = i[2].split(//) end
        diff << [DateTime.parse( md[1] ), array]
      end
      diff.reverse!
    end
    def patch( text, diff_set )
      c = text.split(//); last = nil
      diff_set.each do | i |
        if @datetime <= i[0] or i == diff_set[-1]
          last = i
          break
        end
        c = c.patch( i[1] )
      end
      @datetime = last[0]
      SikiTool::TriDiff.patch( c.join, last[1] )
    end
  end
end
