
require 'singleton'
require 'amrita/node.rb'
require 'amrita/vm.rb'
require 'amrita/parser.rb'
require 'amrita/compiler.rb'

module Amrita

  module Amulet
    include ExpandByMember
    attr_reader :amulet_iset

    def init_amulet(seed)
      @amulet_iset = seed.iset
    end

    def amulet_data
      self
    end
  end

  class DataAmulet
    include Amulet

    attr_reader :amulet_data
    def initialize(data)
      @amulet_data = data
    end
  end

  class AmuletSeed
    attr_reader :template, :element, :iset, :amulet_class
    def initialize(template, element, spec)
      @element = element
      compiler = spec[:compiler] || template.compiler
      pragma = spec[:pragma] || compiler.default_pragma.clone_without(Amulet)
      @iset = compiler.compile_element(element, pragma)
      @amulet_class = spec[:amulet_class] || DataAmulet
    end

    def create_amulet(*args, &block)
      ret = @amulet_class.new(*args, &block)
      ret.init_amulet(self)
      ret
    end

    alias :[] :create_amulet
  end

  class Template
    include Amrita

    # delete id attribute of output if set. Default is true
    attr_accessor :delete_id
    # delete id attribute of output even if the elements are copied. Default is true
    attr_accessor :delete_id_on_copy

    # The name of attribute that turns into _id_.
    # You will need to set this if you use _id_ attribute for DOM/CSS/etc...
    # For expample, if this was set to "__id", 
    # you can use _id_ for amrita and __id for DOM/CSS/etc....
    attr_accessor :escaped_id
    
    # The name of attribute that will be used for template expandion by amrita.
    # You will need to set this if you use _id_ attribute fom DOM.
    # For expample, if this was set to "amrita_id", 
    # you can use amrita_id for amrita and id for DOM.
    attr_accessor :amrita_id

    # If set, use REXML-based parser instead of Amrita's own html-parser
    attr_accessor :xml

    # If set, the output is an xhtml document.
    attr_accessor :asxml

    # If Set, use compiler. 
    attr_accessor :use_compiler

    attr_accessor :use_accelerater, :lazy_evaluation, :partial_compile, :tag_info, :use_simplespan, :optimize_bytecode
    attr_accessor :amulet_attr, :amulet_seeds

    # debug compiler
    attr_accessor :debug_compiler

    # The source code that generated by template compiler
    attr_reader :src

    attr_reader :template, :iset, :ep, :compiler, :vm

    def initialize
      @template = @iset = @ep = @compiler = @vm = nil
      @xml = @asxml = false
      @delete_id = @delete_id_on_copy = true
      @use_simplespan = true
      @escaped_id = @amrita_id = nil
      @debug_compiler = false
      @tag_info = DefaultHtmlTagInfo
      @amulet_seeds = {}
      if Amrita::accelerator_loaded?
        accel_mode
      else
        pureruby_mode
      end
    end

    def accel_mode
      @use_compiler =  false
      @partial_compile = true
      @lazy_evaluation = true
      @optimize_bytecode = false
    end

    def pureruby_mode
      @use_compiler =  true
      @partial_compile = true
      @lazy_evaluation = true
      @optimize_bytecode = false
    end

    def compiler_mode
      @use_compiler =  true
      @partial_compile = false
      @lazy_evaluation = false
      @optimize_bytecode = true
    end

    # 
    # 1. load template if it was changed
    #
    # 2. compile template if +use_compiler+ was set.
    #
    # 3. expand template with +model+
    #
    # 4. print template to +stream+
    #
    def expand(stream, model)
      setup
      do_expand(stream, model)
    end

    def setup
      setup_ep unless @ep
      setup_template if need_update?
      setup_vm
    end

    def do_expand(stream, model)
      @vm.out = stream
      @vm.register = model
      @vm.go
      @vm.out
    end

    def setup_ep
      @ep = ElementProcessor.new(@tag_info)
      @ep.amrita_id = amrita_id
      @ep.escaped_id = escaped_id
      @ep.delete_id = delete_id
      @ep.delete_id_on_copy = delete_id_on_copy
    end
    
    def get_parser_class
      if @xml
        require 'amrita/xml'
        Amrita::XMLParser
      else
        Amrita::HtmlParser
      end
    end

    def setup_compiler
      setup_ep unless @ep
      @compiler = Compiler.new(ep)
      @compiler.use_simplespan = use_simplespan
    end

    def setup_template
      @template = load_template
      setup_compiler unless @compiler
      @iset = compiler.compile(@template)
    end

    def setup_vm
      unless @vm
        @vm = VirtualMachine.new(@ep)
        @vm.use_compiler = use_compiler
        @vm.debug = debug_compiler
        @vm.lazy_evaluation = lazy_evaluation
        @vm.partial_compile = partial_compile
        @vm.optimize_bytecode = optimize_bytecode
      end
      @vm.load_bytecode(@iset)
      @vm.dump_bytecode if @debug_compiler
    end

    def source_mtime
      nil
    end


    def need_update?
      not @template 
    end

    def define_amulet_all
      @template = load_template
      ids = @template.children.collect do |e|
        e.amrita_id.intern if e.amrita_id
      end.compact
      define_amulet(*ids)
    end

    def define_amulet(*args)
      @amulet_attr = nil unless defined? @amulet_attr
      h = nil
      if args[-1].kind_of?(Hash)
        h = args.pop
      else
        h = {}
      end

      args.each do |x|
        case x
        when Hash
          x.each do |k, v|
            h[k] = v
          end
        when String, Symbol
          h[x] = {}
        else
          raise "illeagal param for declar_amulet"
        end
      end
      
      h.each do |k, v|
        case v
        when Class
          h[k] = { :amulet_class=>v }
        when Hash
        else
          raise "illeagal parameter #{v.inspect}"
        end
      end
      setup_amulet_seed(h)
    end

    def extract_amulet(x, h)
      case x
      when Element
        aid = x.amrita_id
        aid = x[amulet_attr.intern] if amulet_attr
        if aid
          aid = aid.intern
          spec = h[aid]
          if spec
            h.delete(aid)
            x.hide_amrita_id!
            seedklass = spec[:seedclass] || AmuletSeed
            @amulet_seeds[aid] = seedklass.new(self, x, spec)
            spec[:placeholder] or e(:span, :id=>aid)
          else
            x.apply_to_children do |xx|
              extract_amulet(xx, h)
            end
          end
        else
          x.apply_to_children do |xx|
            extract_amulet(xx, h)
          end
        end
      else
        x.apply_to_children do |xx|
          extract_amulet(xx, h)
        end
      end
    end

    def setup_amulet_seed(h)
      setup_compiler
      @template = load_template
      # wrap with dummy span to make enable extracting top node
      t = e(:span) { @template }.apply_to_children do |x|
        extract_amulet(x, h)
      end
      @template = t.body
      raise "unknown id #{h.to_a[0][0]}" if h.size > 9
      @iset = compiler.compile(@template)
    end

    def create_amulet(aid, *args, &block)
      seed = @amulet_seeds[aid]
      raise "Amulet #{aid} was  not declared" unless seed
      seed.create_amulet(*args, &block)
    end

    def [](*args)
      aid = args.shift
      if args.size == 0
        @amulet_seeds[aid]
      else
        @amulet_seeds[aid][*args]
      end
    end

    def get_opts_for_save
      {
        :xml => @xml,
        :asxml => @asxml,
        :delete_id => @delete_id ,
        :delete_id_on_copy => @delete_id_on_copy,
        :escaped_id => @escaped_id,
        :amrita_id => @amrita_id,
        :use_compiler => @use_compiler,
        :lazy_evaluation => @lazy_evaluation,
        :optimize_bytecode => @optimize_bytecode,
        :use_simplespan => @use_simplespan,
        :debug_compiler => @debug_compiler
      }
    end

    def set_opts_from(opts)
      @xml = opts[:xml]
      @asxml = opts[:asxml]
      @delete_id = opts[:delete_id]
      @delete_id_on_copy = opts[:delete_id_on_copy]
      @escaped_id = opts[:escaped_id]
      @amrita_id = opts[:amrita_id]
      @use_compiler = opts[:use_compiler]
      @lazy_evaluation = opts[:lazy_evaluation]
      @optimize_bytecode = opts[:optimize_bytecode]
      @use_simplespan = opts[:use_simplespan]
      @debug_compiler = opts[:debug_compiler]
    end

    def get_ivars_for_dump
      [@template, @iset, @amulet_seeds, get_opts_for_save]
    end

    def set_ivars_from(a)
      @template, @iset, @amulet_seeds, opts = *a
      set_opts_from(opts)
    end

    def _dump(level)
      Marshal::dump(get_ivars_for_dump, level)
    end
  end

  class TemplateText < Template
    def initialize(template_text, fname="", lno=0)
      super()
      @template_text, @fname, @lno = template_text, fname, lno
      @template = nil
    end

    def load_template
      setup_ep unless @ep
      @template = get_parser_class.parse_text(@template_text, @fname, @lno, tag_info, ep)
    end

    def need_update?
      @template == nil
    end

    def path
      nil
    end

    def TemplateText::_load(s)
      ret = TemplateText.new ''
      ret.set_ivars_from(Marshal::load(s))
      ret
    end
  end

  class TemplateFile < Template
    attr_reader :path
    attr_accessor :auto_save

    def initialize(path)
      super()
      @path = path
      @auto_save = false
      @lastread = nil
    end

    # template will be loaded again if modified.
    def need_update?
      return true unless @lastread
      @lastread < File::stat(@path).mtime
    end

    def load_template
      setup_ep unless @ep
      ret = get_parser_class.parse_file(@path, tag_info, ep)
      @lastread = Time.now
      ret
    end

    def expand(stream, data)
      super
      TemplateManager.instance.save_template(self) if @auto_save
    end

    def get_ivars_for_dump
      super + [ @path, @lastread ]
    end

    def set_ivars_from(a)
      @lastread = a.pop
      @path = a.pop
      super(a)
    end

    def TemplateFile::_load(s)
      ret = TemplateFile.new('')
      ret.set_ivars_from(Marshal::load(s))
      ret
    end
  end


  class CCompiledAmuletSeed < AmuletSeed
    include Amrita
    attr_accessor :iset

    def initialize(template, spec)
      @amulet_class = spec[:amulet_class] || DataAmulet
      @element = instance_eval(spec[:element_code])
    end
  end

  module ByteCode
    class RubyMethodInstruction < Instruction
      include ByteCodeCommon::Instruction
      def initialize(mod, mname)
        @method = mod.method(mname)
      end

      def prepare(vm , opt, iset=nil)
      end

      def execute(reg, stack, out, vm)
        @method.call(reg, stack, out, vm)
      end
    end
  end

  class CCompiledTemplate < Template

    def use_compiler
      false
    end

    def setup_template
      instruction = ByteCode::RubyMethodInstruction.new(self, :execute)
      @iset = InstructionSet.new instruction
      @amulet_seeds = {}
      amulet_specs.each do |k, v|
        methodname = "execute_amulet_" + k.to_s
        instruction = ByteCode::RubyMethodInstruction[self, methodname.intern]
        iset = InstructionSet.new instruction
        seed = CCompiledAmuletSeed.new(self, v)
        seed.iset = iset
        @amulet_seeds[k] = seed
      end
    end

    def amulet_specs
      {}
    end
  end

  class CacheBase
    Item = Struct.new(:path, :mtime, :template)

    def get_template(path)
      source_mtime = File::stat(path).mtime
      item = get_item(path) 
      if item and valid_item?(item, source_mtime)
        item.template
      else
        nil
      end
    end

    def valid_item?(item, source_mtime)
      item.mtime && source_mtime && item.mtime >= source_mtime
    end

    def get_item(filename, key)
      raise 'subclass resposibility'
    end

    def save_template(path, template)
      item = Item.new
      item.path = path
      item.template = template
      save_item(item)
    end

    def save_item(item)
      raise 'subclass resposibility'
    end
  end


  class MemoryCache < CacheBase
    def initialize
      @hash = {}
    end

    def clear
      @hash = {}
    end

    def get_item(path)
      ret = @hash[path]
      ret
    end

    def save_item(item)
      item.mtime = Time.new
      @hash[item.path] = item
    end
  end

  class FileCache < CacheBase

    def initialize(dir)
      @dir = dir
    end

    def get_item(path)
      path = make_cache_path(path)
      File::open(path) do |f|
        item = Item.new
        item.path = path
        item.mtime = f.mtime
        item.template = Marshal::load(f)
        return item
      end
    rescue Errno::ENOENT, Errno::EACCES
      nil
    end

    def save_item(item)
      path = make_cache_path(item.path)
      File::open(path, "w") do |f|
        Marshal::dump(item.template, f)
      end
    end

    private

    def make_cache_path(path)
      base = path.to_s + '.amc'
      base.gsub!("/", "_")
      File::join(@dir, base)
    end
  end


  class TemplateManager
    include Singleton
    def TemplateManager::[](path)
      TemplateManager.instance.get_template(path)
    end

    def TemplateManager::set_cache_dir(path)
      TemplateManager.instance.set_cache_dir(path)
    end

    # <em>CAUTION: be careful to prevent users to edit the cache file.</em>
    # It's *YOUR* resposibility to protect the cache files from
    # crackers. Don't use <tt>TemplateFileWithCache::set_cache_dir</tt> if
    # you don't understand this.
    def set_cache_dir(path)
      if path
        @file_cache = FileCache.new(path)
      else
        @file_cache = nil
      end
    end

    def initialize
      @mem_cache = MemoryCache.new
      set_cache_dir(ENV["AmritaCacheDir"].untaint)  # be careful whether this directory is safe
    end

    def get_template(path)
      ret = nil
      if TemplateManager::const_defined? :Templates # if output of amcc was loaded
        ret = Templates[path]
      end

      ret = @mem_cache.get_template(path) unless ret
      ret = @file_cache.get_template(path) if @file_cache and not ret
      unless ret
        ret = TemplateFile.new(path)   
        ret.auto_save = true
      end
      ret
    end

    def clear_mem_cache
      @mem_cache.clear
    end

    def save_template(template)
      if template.path
        @mem_cache.save_template(template.path, template)
        @file_cache.save_template(template.path, template) if @file_cache
      end
    end
  end
end
