/*SIE under the MIT Lisence
 */
/* Copyright 2016 dhrname and other contributors
 * http://sie.osdn.jp/
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 * OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
 * WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/*$frame オブジェクト
 * 全体のフレームの管理を行う
 */
base("$frame").mix ( {
  /*フレームレート。１ミリ秒何フレームか。計算を省略するためミリ秒使用*/
  fpms: 0.024,

  /*タイムラインのリスト (時間区間の設定ができる）*/
  timelines: [],

  /*開始フレーム数の候補。アニメーションの開始条件となる
   * 単位はフレーム数であって、秒数ではない
   * なお、初期値については、開始フレームが負の値を考慮しているため*/
  begin: -Number.MAX_VALUE,

  /*活動継続時間 (Active Duration)のフレーム数。アニメーションの継続条件となる
   * 単位はフレーム数であって、秒数ではない*/
  activeTime: Number.MAX_VALUE,
  
  /*現在のフレーム数*/
  currentFrame: 0,
  
  /*アニメーションを開始させるメソッド*/
  startAnimation: function() {
    /*$getDocument.step関数は最後に書く*/
    base("$getDocument").step();
  },
  
  /*アニメーションが停止した状態かどうか、停止しているならばtrue*/
  isPaused: false,

  /*アニメーションを停止させるメソッド*/
  pauseAnimation: function() {
    this.isPaused = true;
  },
  
  /*後述のinitializeメソッドで使うオブジェクトリスト*/
  objList: [],
  
  /*オブジェクトの初期化処理*/
  initialize: function() {
    var list = this.objList;
    for (var i=0;i<list.length;++i) {
      list[i].initialize();
    }
  },

  /*setFrame メソッド
   * フレーム数を数値num まで進めるか、戻す*/
  setFrame: function( /*number*/ num) {
    this.currentFrame = num;
    var timelines = this.timelines;
    for (var i=0;i<timelines.length;++i) {
      if (timelines[i] === this) {
        /*自分自身が含まれていると、永久に再帰を繰り返して、「スタック領域が不足」してしまう*/
        continue;
      }
      timelines[i].setFrame(num);
    }
  },
  
  /*addLineメソッド
   * タイムラインを追加したあと、trueを返す
   * ただし、引数objのobj.beginとobj.activeTimeが定まっていない場合はfalseを返す*/
  addLine: function( /*$frame*/ obj ) {
    if(!obj || (!obj.begin && (obj.begin !== 0))
    || (!obj.activeTime && (obj.activeTime !== 0)) ) {
      /*どちらのプロパティも未確認の場合、タイムラインは追加されない*/
      return false;
    }
    var timelines = this.timelines;
    if ( timelines.indexOf(obj) >= 0 ) {
      this.removeLine(obj);
    }
    timelines.push(obj);
    timelines = void 0;
    return true;
  },
  
  /*removeLine メソッド
   * 指定されたタイムラインのオブジェクトを、リストから削除する*/
  removeLine: function( /*$frame*/ timeline ) {
    var list = this.timelines,
        j = list.indexOf(timeline);
    if (j > -1) {
      list.splice(j, 1);      //Arrayのspliceを利用して、リストからtimelineを排除
    }
    list = j = void 0;
  }
} ).mix( function($frame) {
  $frame.up("$list").mix( {
    /*終了時刻（単位フレーム数）のキャッシュとして使う*/
    end: 0,
    
    /*開始時刻から終了時刻までのフレーム数
     * これがactiveTimeより短ければ、活動継続時間とみなす*/
    beginEnd: Number.MAX_VALUE,
    
    /*開始時刻（単位フレーム数）リスト (後述のupdateStateメソッドで使う)*/
    beginList: { 
      next: null,
      value: Number.MAX_VALUE
    },
    
    /*終了時刻（単位フレーム数）リスト (後述のupdateStateメソッドで使う)*/
    endList: {
       next: null,
       value: Number.MAX_VALUE
    },
    
    addBeginList: function (num) {
      return ( this.beginList = {
                  value: num,
                  next: this.beginList
                } );
    },
    
    addEndList: function (num) {
      return ( this.endList = {
                  value: num,
                  next: this.endList
                } );
    },
    
    /*引数に渡された時刻リストの中から、現在フレームf以下の、最大値を求めるメソッド
     * Number.MIN_VALUEの値を返したときはリストの中にf以下の値がないことを示す*/
    getMaxList: function (f, list) {
      var maxTime = -Number.MAX_VALUE; /*時刻の最大値*/
      while( list ) {
        var v = list.value;
        /*f以下の開始リスト値のうち、最大値をmaxTimeに代入*/
        if ( (v <= f)
             && (maxTime <= v) ) {
          maxTime = v;
        }
        list = list.next;
      }
      return maxTime;
    },
      
    /*現在の要素の状態を数値で示す（マジックナンバーは後述の大文字プロパティを使う）*/
    state: 0,
    
    /*アニメーション待機状態を示す定数*/
    WAITING: 0,
    
    /*アニメーション開始状態を示す定数*/
    BEGINNING: 1,
    
    /*アニメーション再生中の状態を示す定数*/
    PLAYING: 2,
    
    /*アニメーション終了状態を示す定数*/
    ENDING: 3,
    
    /*終了処理をした後の待機状態を示す定数*/
    POSTWAITING: 4,
    
    /*初期化用メソッド*/
    init: function() {
      this.state = this.WAITING;
      this.begin = 0;
      return this;
    },
    
    /*引数で指定されたフレーム数に応じて、stateプロパティを更新するメソッド*/
    updateState: function( /*number*/ f) {
      if (f === void 0) {
        /*引数fが指定されないときには状態を更新しない*/
        return this;
      }
      var state = this.state,
          wait = /*this.WAITING*/ 0,
          begin = /*this.BEGINNING*/ 1,
          play = /*this.PLAYING*/ 2,
          end = /*this.ENDING*/ 3,
          post = /*this.POSTWAITING*/ 4;
      /*beginListプロパティと、endListプロパティの中で、
       * 現在フレーム数 f より大きい数値は、更新できる条件と無関係なので、除外しておく
       * だから、f以下の値の中から、最大値を探して、
       * それをbeginプロパティの値cacheBeginと比較する*/
      var startTime = this.getMaxList(f, this.beginList);
      if (startTime === -Number.MAX_VALUE) {
        (state > post) && (this.state = begin);
        return this;
      }
      var endTime = this.getMaxList(f, this.endList),
          isWait = (state === wait),
          cacheBegin = this.begin;
      if ( !startTime && isWait) {
          /*開始時刻が0ならば、アニメーションを開始
           * ただし、アニメが終了した後のPOSTWAITING状態の時には作動させない*/
          this.begin = 0;
          this.state = begin;
      } else if ( isWait || (state === post) ) {
        if (startTime > cacheBegin) {
          this.state = begin;
          /*beginプロパティに開始時刻をキャッシュ用に保存*/
          this.begin = startTime;
        }
      } else if (state === begin) {
        if (endTime >= cacheBegin) {
          /*終了時刻にもう到達したときは、直接BEGINNING状態からENDING状態へ移行*/
          this.state = end;
          /*endTimeを終了時刻とみなす*/
          (endTime > 0) && (this.end = endTime);
          /*activeTimeプロパティは、begin属性とend属性が反映されていないため、
           * beginEndプロパティに別に設定しておく*/
          this.beginEnd = 0;
        } else {
          this.state = play;
        }
      } else if (state === play) {
        /*activeTimeプロパティを比較して、変数endTimeを書き換える*/
        var act = cacheBegin + this.activeTime;
        endTime = (endTime > act) ? act
                                  : endTime;
        if ( (f >= act) || (endTime >= cacheBegin) || (startTime > cacheBegin) ) {
          /*終了時刻に到達したか、再び開始イベントが発火されたとき*/
          this.state = end;
          if (endTime > 0) {
            /*endTimeを終了時刻とみなす*/
            this.end = endTime;
            /*activeTimeプロパティは、begin属性とend属性が反映されていないため、
             * beginEndプロパティに別に設定しておく*/
            this.beginEnd = endTime - cacheBegin;
          }
        }
      } else if (state === end) {
        if (startTime > cacheBegin) {
          /*再生中に開始イベントが発火されて、終了状態となったとき*/
          this.state = begin;
          this.begin = startTime;
        } else {
          this.state = post;
        }
      } else {
        this.state = begin;
      }
      cacheBegin = startTime = endTime = isWait = state = void 0;
      return this;
    },
    
    /*addEventメソッドで使われるイベントリスト（開始時に登録されたリスナー関数が呼び出される）*/
    _beginListenerList: [],
    
    /*addEventメソッドで使われるイベントリスト（終了時に登録されたリスナー関数が呼び出される）*/
    _endListenerList: [],
    
    /*addEventメソッドで使われるイベントリスト（再生時に登録されたリスナー関数が呼び出される）*/
    _playListenerList: [],
    
    /*開始と再生と終了時に発火されるイベントリスナーを登録するメソッド*/
    addEvent: function ( /*string*/ eventName, /*fnction*/ listener) {
      var evtName = "_" +eventName+ "ListenerList";
      /*プロトタイプ継承していた場合は新しく配列を作成*/
      if (!this.hasOwnProperty(evtName)) {
        this[evtName] = [];
      }
      this[evtName].push(listener);
    },
    
    /*入力されたフレーム数fの場面に切り替えるメソッド*/
    setFrame: function( /*number*/ f) {
      this.currentFrame = f;
      var state = this.updateState(f).state;
      /*アニメーション開始と再生と、終了状態のときに、beginとplayとendイベントを呼び出しておいて、
       * 次の状態(再生状態)に遷移する*/
      if (state === /*this.PLAYING*/ 2) {
        var list = this._playListenerList;
        for (var i=0;i<list.length;++i) {
          list[i](this);
        }
      } else if (state === /*this.BEGINNING*/ 1) {
        list = this._beginListenerList;
        for (var i=0;i<list.length;++i) {
          list[i](this);
        }
        /*開始時刻と終了時刻が一致した場合はstateはENDING状態
         * それ以外はPLAYING状態*/
        state = this.updateState(f).state;
      }
      if (state === /*this.ENDING*/ 3) {
        list = this._endListenerList;
        for (var i=0;i<list.length;++i) {
          list[i](this);
        }
        if (this.updateState(f).state === /*this.BEGINNING*/ 1) {
          /*再生中にbeginイベントが呼び出された場合*/
          this.updateState(f);
        }
      }
      state = list = void 0;
    }
  } ).mix( function() {
    /*後述の$beginや$endで使うメソッド*/
    this.addList = this.addBeginList;
  } );
   
  /*$begin オブジェクト
   * 開始のタイミングを計算する*/
  $frame.up("$begin").mix( {

    /*開始時刻やタイミングが書かれた文字列*/
    string: "",

    /*イベントやindefinteで未解決かどうか*/
    isResolved: false,
    
    /*イベントが適用される要素*/
    eventTarget: document.documentElement,
    
    /*現在のフレーム数を改めて初期化*/
    currentFrame: 0,
    
    /*イベント同期で使う時間差のフレーム数*/
    eventOffset: 0,
    
    /*repeat(1)など文字列内に書かれたリピート回数*/
    repeat: 0,
    
    /*accessKey(a)の"a"などキーイベントの対象となる文字列*/
    accessKey: "",

    /*trim メソッド
     * 文字列中の空白を除去*/
    trim: function(str) {
      /*strがString型以外のときは必ずエラーを出す*/
      return str.replace(/[\s\n]+/g, "");
    },

    /*offset メソッド
     * 引数に渡された文字列から、ミリ秒単位に変換した時間を、解析して返す*/
    offset: function(str) {
      str = str || "0";
      var plusminus = str.charAt(0),
          /*parseFloatのエイリアス*/
          _float = parseFloat,
          s = _float( str.match(/[\d.]+ms$/) || "0") + sec() + min() + h();
      if (plusminus === "-") {
        s *= -1;
      }
      plusminus = _float = sec = min = h = void 0;
      return s;
      
      /*00:00:0と00:0と、0sなどの文字列をミリ秒へ変換*/
      function sec() {
        return str2num( 1000, /[\d.]+s$/, /[\d.]+$/ );
      };
      function min() {
        return str2num( 60000, /[\d.]+min$/, /\d\d:[^:]+$/ );
      };
      function h() {
        return str2num( 3600000, /\d+:\d\d:/, /[\d.]+h$/ );
      };
      function str2num(s, /*RegExp*/ a, /*RegExp*/ b) {
        return s*( _float(str.match(a) || "0") || _float(str.match(b) || "0") );
      };
    },

    /*event メソッド
     * 引数の文字列から、idとイベントに相当する文字列のプロパティを持ったオブジェクトを返す
     * idがない場合や、イベントがない場合は空文字列を該当のプロパティに入れる*/
    event: function(str) {
      str = str || "";
      if (/[\+\-]/.test(str)) {
        /*数値がある場合は切り取っておく*/
        str = str.slice(0, str.search(/[\+\-]/));
      }
      if (str.indexOf(".") > -1) {
        /*ドットが見つかった場合、IDとイベントに分けておく*/
        var ide = str.split(".");
        /* エラーが起きて、idが空文字列ならば、evtも空文字列。逆も然り*/
        return {
          id: (ide[1] && ide[0]),
          event: (ide[0] && ide[1])
        };
      } else {
        return {
          id: "",
          event: str
        };
      }
    }, 
    
    /*_parse メソッド
     * 引数の文字列を解析して、フレーム数を算出し、結果を$frame.beginプロパティに出力
     * また、イベントリスナーに登録をしておく*/
    _parse: function (str) {
      var plusminus = str.search(/[\+\-]/),
          event = null,
          ele,
          /*endListのvalueプロパティには、活動継続フレーム数と開始フレーム数を足したものが入る*/
          endList = this.$list.addEndList(Number.MAX_VALUE);
      if (str === "indefinite") {
        this.begin = Number.MAX_VALUE;
      } else if (plusminus > 0) {
        /*Event-Value +/- Clock-Value の場合*/
        this.begin = this.offset( str.slice(plusminus) );
        event = this.event(str);
      } else if ( /[^\+\-\d]/.test(str.charAt(0)) ) {
        /*Event-Valuen　のみの場合*/
        event = this.event(str);
      } else {
        /*+/- Clock-Value のみの場合*/
        this.begin = this.offset( str );
        /*イベントもindefiniteもないので、解決済みと考える*/
        this.isResolved = true;
      }
      /*もしもあれば、リピートの回数を求める*/
      this.repeat = /repeat\((\d+)\)/.test(str) ? +RegExp.$1 : 0;
      /*もしもあれば、押されるはずのキーを求める*/
      this.accessKey = /accessKey\(([^\)]+?)\)/.test(str) ? RegExp.$1 : "";
      this.begin = Math.floor( this.begin * this.fpms);
      if (str === "indefinite") {
        /*begin属性の値がindefiniteの場合は、何もしない。
         * 開始時刻はbeginElementメソッドに依存*/
      } else if (event) {
        ele = event.id ? this.eventTarget.ownerDocument.getElementById(event.id)
                        : this.eventTarget;
        /*イベントの時間差を設定しておく
         * eventOffsetとobjListの変数はクロージャとしてlistener関数で使われる*/
        var eventOffset = this.begin,
            /*objListのvalueプロパティはあとで書き換えられる（イベントの場合のみ）*/
            objList = this.$list.addList(Number.MAX_VALUE),
        /*イベントのリスナーとして使う*/
            listener = function(evt) {
              objList.value = this.begin = eventOffset + base("$frame").currentFrame;
              this.isResolved = true;
            };
        this.eventOffset = eventOffset;
        if (this.repeat > 0) {
          ele && ele.addEventListener("repeatEvent", (function(evt) {
            if (evt.detail === this.repeat) {
              listener.call(this, evt);
            } }).bind(this), true);
        } else if (this.accessKey) {
          var doc = document || ele.ownerDocument;
          doc.documentElement.addEventListener("keydown", (function(evt) {
            if (evt.key === this.accessKey) {
                listener.call(this, evt);
              } }).bind(this), false);
        } else {
          var evtName = /^(?:begin|end|repeat)$/.test(event.event) ? event.event + "Event"
                          : event.event;
          ele && ele.addEventListener(evtName, listener.bind(this), false);
        }
      } else {
        /*開始リストに登録しておく（$endの場合は終了リストに登録）*/
        this.$list.addList(this.begin);
      }
      s = event = str = plusminus = ele = void 0;
    },
    
    /*stringプロパティを解析して、
     * 開始フレーム数の算出や、イベントリスナーの登録をするメソッド*/
    parse: function() {
      /*初期値を設定*/
      this.begin = 0;
      this.isResolved = false;
      var str = this.trim(this.string);
      if (str.indexOf(";") > -1){
        /*;で区切られたリストを一つずつ解析*/
        var list = str.split(";");
        for (var i=0;i<list.length;++i) {
          this._parse(list[i]);
        }
      } else {
        this._parse(str);
      }
      s = str = void 0;
      return this;
    },
    
    /*$listと$activateオブジェクトを更新して、活動継続時間を求めるメソッド*/
    updateList: function() {
      this.$list = this.$list.up();
      /*beginとend属性を考慮に入れないで、活動継続時間を求める*/
      var s = ( this.$activate = this.$activate.up() );
      /*$endオブジェクトに付属している$listプロパティを更新したものと一致させておく*/
      s.end__ && (s.end__.$list = this.$list);
      this.activeTime = this.$list.activeTime = s.call() || Number.MAX_VALUE;
      this.simpleDuration = s.simpleDur;
      return this;
    }
    
  /*$activate オブジェクト
   * 活動継続時間などを計算するための計算実体
   * $begin オブジェクトからの継承*/
  } ).up("$activate").mix( {
        
    /*単純継続時間のパースされる前の文字列*/
    dur: "indefinite",
    
    /*リピート回数*/
    repeatCount: null,
    
    /*繰り返し時間*/
    repeatDur: null,
    
    /*最小値に制限される
     * 最小値 <= 活動継続時間 とならなければならない*/
     min: "0",
     
    /*最大値に制限される
     * 活動継続時間 <= 最大値 とならなければならない*/
     max: "indefinite"
  } ).of( {

    /*活動をストップさせるためのオブジェクト*/
    end: $frame.$begin.up("$end"),

    /*単純継続時間 (単位はフレーム数)*/
    simpleDur: function() {
      return ( (this.dur === "indefinite") || !this.dur ) ?
                null
              : Math.floor(this.offset(this.dur) * this.fpms) ;
    },
    
    /*解決した(計算する)ときの時間*/
    resolvedTime: function() {
      return Date.now();
    },
    
    /*関数型の呼び出しメソッド
     * base.jsのofメソッドを活用して活動継続時間を算出
     * ただし、begin属性とend属性については、別途、$frame.$listで定める
     * 計算方法はSMILアニメーション 3.3.4節を参照
     * http://www.w3.org/TR/smil-animation/#ComputingActiveDur
     */
    call: function() {
      var ind = "indefinite",
          dur = this.simpleDur,
          isIndefRepeatCount = (this.repeatCount === ind),
          isIndefRepeatDur = (this.repeatDur === ind),
          isDur = dur || (dur === 0),
          isRepeatCount = this.repeatCount || (this.repeatCount === 0),
          isRepeatDur = this.repeatDur || (this.repeatDur === 0),
          actList = [],
          min = Math.floor(this.offset(this.min) * this.fpms),
          max = (this.max === ind) ? null : Math.floor(this.offset(this.max) * this.fpms),
          s;
      if (indef()) {
        /*注意点として、活動継続時間が不定かつ、min属性とmax属性が指定されたときの記述が仕様にないため、
         * W3Cのテストスイート(animate-elem-66-t.svg)に従い、max属性の値を返す*/
        if (max) {
          return max;
        }
        return null;
      }
      if (isDur && this.repeatCount && !isIndefRepeatCount) {
        actList.push( dur * this.repeatCount );
      }
      if (isRepeatDur && !isIndefRepeatDur) {
        actList.push( Math.floor( this.offset(this.repeatDur) * this.fpms) );
      }
      if (isDur && !isRepeatCount && !isRepeatDur) {
        /*repeatCountやrepeatDur属性が指定されていない場合*/
        actList.push( dur );
      }

      /*長くなるため、インライン関数を活用
       * indef関数はbeginとend属性を考慮せずに、
       * 活動継続時間が不定かどうか、もし、不定なら真を返す*/
      function indef() {
        return !!( (!isDur && !isRepeatDur)
                   || (isIndefRepeatCount && !isRepeatDur)
                   || (isIndefRepeatDur && !isRepeatCount)
                   || (isIndefRepeatCount && isIndefRepeatDur)
                 );
      };
      
      ind = dur = isIndefRepeatCount = isIndefRepeatDurindef = isDur = isEnd = isRepeatDur = isRepeatCount = indef = void 0;

      if (actList.length === 1) {
        s = actList[0];
      } else if(actList.length > 1) {
        /*属性が競合するときは、最小値をとること (SMILアニメーション 3.3.4節)*/
        s = Math.min.apply(Math, actList);
      } else {
        return null;
      }
      if ( max && (min > max) ) {
        return s;
      }
      min && (min > s) && (s = min);
      max && (max < s) && (s = max);
      return s;
    }
  } );
  $frame.$begin.$end.of( {
    call: function() {
      if (!this.string) {
        return null;
      }
      /*addListメソッドには、addBeginList関数が入っているはずなので、それを一時的に変更する*/
      this.$list.addList = this.$list.addEndList;
      this.parse(this.string);
      this.$list.addList = this.$list.addBeginList;
      return this.isResolved ? this.begin
                             : "indefinite";
    }
  } ).mix( {
    $list: $frame.$begin.$list.up()
  } );
} );
/*$from オブジェクト
 * 呈示値 (presentation value)の計算をする。値そのものを返すための計算実体*/
base("$from").of( {
  
  /*呈示値の数値の部分だけを抜き出した配列を返す*/
  numList: function() {
    var s  = this.string.match(/[\-\+]?[\d\.]+(?:[eE][\-\+]?[\d\.]+)?/g)
             || [];
    if (s) {
      /*mapメソッドで代用してもよい*/
      for (var i=0;i<s.length;++i) {
        s[i] = parseFloat(s[i]);
      }
    }
    return s;
  },
  
  /*呈示値の文字部分だけを抜き出した配列を返す*/
  strList: function() {
    /*replaceメソッドで1E-10などの対策*/
    return this.string.replace(/\d[eE][\-\+\d]/g, "")
                      .match(/[^\d\-\+\.]+/g);
  },
  
  from: base("$from").up().mix( {
          from: null
        } ),
  
  /*$toオブジェクトにこのオブジェクトを適用させる関数*/
  call: function() {
    /*advanceメソッドで使うために、stringを保持*/
    this.numList.string = this.string;
    if (this.numList.length
          && (this.additive[0] === 0)
          && (this.accumulate[0] === 0)
          ) {
      /*配列の項目がundefinedだと困るので、配列を初期化する*/
      var additive = [],
          accumulate = [];
      for (var i=0;i<this.numList.length;++i) {
        /*0で配列を初期化しておく*/
        additive[i] = accumulate[i] = 0;
      }
      this.additive = additive;
      this.accumulate = accumulate;
    }
    /*文字部分の配置パターンは4通りあるので、ここでstrListを処理
     * (1) a 0 の場合
     * (2) 0 a
     * (3) a 0 a (ノーマルパターン)
     * (4) 0 a 0
     * これらのパターンのうち、(1)(2)(4)を(3)のパターンに統一したのが以下の処理*/
    /*文字列が1aのように、数値で始まるかどうか。始まったら真*/
    if (!this.string || !this.numList.length || !this.strList) {
      return this.numList;
    }
    var isNormal = (this.numList.length < this.strList.length);
    if (/^[\-\+]?[\d\.]/.test(this.string) && !isNormal) {
      /*文字列が1aのように、数値で始まる場合*/
      this.strList.unshift("");
    }
    if (/\d$/.test(this.string) && !isNormal) {
      /*文字列がa1のように、数値で終わる場合*/
      this.strList.push("");
    }
    return this.numList;
  }
    
} )
 .mix( {
   /*呈示値が書かれた文字列*/
   string: "",
   
   /*advanceメソッドの返り値で使われる小数点以下の桁数*/
   degit: 0,
   
   /*additve属性やaccumulate属性が設定された、累積アニメーションか、加法アニメーションで使われる*/
   additive: [0],
   accumulate: [0],
   
   /*advance メソッド
    * アニメーションの進行具合を示す進捗率 t (0 <= t <= 1)をもとに、現在の呈示値を算出するためのもの
    * callメソッドが前もって呼び出されていることが前提となる*/
    advance: function(t) {
      if ( (t < 0) || (1 < t)) {
        throw new Error("An Invalid Number Error");
      }
      if (!this.string) {
        return "";
      } else if (!this.from.length) {
        /*discreteのために、this.stringに数値が入っていない場合の対処*/
        if (t === 1) {
          return this.string.trim();
        }
        return this.from.string.trim();
      }
      var str = "",
          numList = this.numList,
          strList = this.strList,
          fromNumList = this.from,
          deg = this.degit,
          additive = this.additive,
          accumulate = this.accumulate;
      for (var i=0,nuli=numList.length;i<nuli;++i) {
        /*原点Oを(0,0,...0)とおく
         *$fromと$toを、原点Oからの二つのベクトル (n次空間のベクトル)、ベクトルOFとベクトルOTと考える
         *$fromと$toの二つの端の点FとTを結ぶ線分を、t : 1-t で内分する点をPとおく
         * このときのベクトルOPを求めたのが以下の式*/
        str += ( t * numList[i] + (1 - t) * fromNumList[i] + additive[i] + accumulate[i]).toFixed(deg);
        strList && ( str += strList[i+1] );
      }
      /*文字列はcallメソッドにより、a0aのパターンになっているので、aの部分を追加*/
      str = (strList ? strList[0] : "") + str;
      numList = strList = fromNumList = i = nuli = deg = additive = accumulate = void 0;
      return str.trim();
    },
    
    /*distanceメソッド
     * fromベクトルから自分自身のベクトルへの距離 (ノルム)の数値を返す。callメソッドを使うので注意すること*/
     distance: function(from) {
       if (!from) {
          return 0;
       }
       var toList = this.call(),
           fromList = from.call ? from.call() : from,
           s = 0;
       if (!toList || !fromList) {
         return 0;
       }
       for (var i=0, tli=toList.length; i<tli; ++i) {
         s += (toList[i] - fromList[i])*(toList[i] - fromList[i]);
       }
       return Math.sqrt(s);
     },
     
     /*setAdditive メソッド
      * additve属性がsumのときに使われるメソッド
      * 引数は親要素の、現在の属性値*/
      setAdditive: function(str) {
        if (!str) {
          return 0;
        }
        var from = this.$from.up();
        from.string = str;
        return ( this.additive = from.call() );
      },
     
     /*setAccumulate メソッド
      * accumulate属性がsumのときに使われるメソッド
      * 引数は現在のリピート回数*/
      setAccumulate: function(num) {
        if (!num || isNaN(num)) {
          return 0;
        }
        return ( this.accumulate = this.numList.map( function(d) {
          return d * num;
        } ) );
      }
  } )
  /*fromプロパティの初期化*/
 .up("$to").from = null;
 
 /*計算モードを定めるための計算実体
  *補間の細かい制御などを行う*/
base("$calcMode").mix({
   /*計算モード　(calcMode属性の値)*/
   mode: "linear",

   /*keyTimesの区間
    * たとえば、"0, 0.5, 0.7, 1"の場合、時間の区間はそれぞれ、0.5 (=0.5-0)  0.2 (=0.7-0.5)  0.3 (=1-0.7)である
    * このうち、どれか一つが値として入力される*/
   keyTime: 1,
   
   /*keySpline属性の値を設定*/
   keySplines: null,
      
   /*全体の行列ノルム（距離）*/
   norm: 1,

   /*無効だった場合の呈示値*/
   string: "",

   /*callメソッドで使う関数
    * 進捗率を時間の圧縮率( = keyTimeプロパティ)で割ることで、現在、どこまで進んでいるのかを求めることができる*/
   _f: function (t) {
         /*tは進捗率*/
         var tkey = this.keyTime;
         if ( (tkey === 0) && t) {
           t = 0; 
         } else if (!tkey || !isFinite(tkey) ) {
           return this.string;
         } else {
           t = t / tkey;
           t = (t > 1) ? Math.floor(t) : t;
         }
         tkey = void 0;
         return isNaN(t) ? this.string
                         : this.to__.advance(t);
     },
     
     /*discreteモードのときには、上記の_f関数の代わりに、以下の関数を用いる*/
     funcForDiscrete: function (t) {
        if (isNaN(t)) {
          return this.string;
        } else if (t === 1) {
          return this.to__.string;
        } else {
          return this.to__.advance(0);
        }
      }
 }).of( {
   
   /*$fromオブジェクトを使っているが、toオブジェクトを指している*/
   to: base("$from").$to,
  
   /*与えられたアニメーションの進捗率を使った時間の圧縮率を計算して呈示値を返すための関数を作る*/
   call: function() {
     var f = this._f.bind(this);
     if (this.mode === "linear") {
       return f;
     } else if (this.mode === "paced") {
       /*keyTimes属性は無視され、ベクトルの距離の割合から計算される*/
       this.keyTime = this.to__.distance(this.to__.from__) / this.norm;
       return f;
     } else if (this.mode === "spline") {
       var tk = this.keySplines,
           /*必ず関数を返すようにするため、円周率を返す関数tfを返して、nullの代わりとする*/
           tf = function(x) {
                 return Math.PI;
           };
      tf.isNotAnimate = true;
      if (!tk) {
         return tf;
       }
      for (var i=0,tki = NaN;i<tk.length;++i) {
       tki = tk[i];
       if (isNaN(tki)) {
         return tf;
       }
       if ( (tki < 0) || (1 < tki)) {
         return tf;
       }
     }
     var x2 = tk[0],
         y2 = tk[1],
         x3 = tk[2],
         y3 = tk[3],
         x4 = 1,
         y4 = 1,
         Ax = x4-3*(x3-x2),
         Bx = 3*(x3-2*x2),
         Cx = 3*x2,
         Ay = y4-3*(y3-y2),
         By = 3*(y3-2*y2),
         Cy = 3*y2,
         _newton = Math.qubicnewton; //高速化のためのエイリアス
     if ( ( (x2 === 0) || (x2 === 1) )
          && (y2 === 0)
          && ( (x3 === 1) || (x3 === 0) )
          && (y3 === 1) ) {
            /*linearモードと同じ効果 (収束ではない可能性を考慮)*/
            return f;
     }
     var tkey = this.keyTime;
     if (tkey || isFinite(tkey) ) {
       /*keyTimeから時間の収縮率を3次ベジェ曲線に適用しておく*/
       Ax *= tkey;
       Bx *= tkey;
       Cx *= tkey;
       Ay *= tkey;
       By *= tkey;
       Cy *= tkey;
     }
     tkey = tk = x2 = y2 = x3 = y3 = x4 = y4 = void 0;
     return function (x) {
        /*3次ベジェ曲線は媒介曲線
         *x = (x4-3*(x3-x2)-x1)*t*t*t + 3*(x3-2*x2+x1)*t*t + 3*(x2-x1)*t + x1
         *y = (y4-3*(y3-y2)-y1)*t*t*t + 3*(y3-2*y2+y1)*t*t + 3*(y2-y1)*t + y1
         * ただし、0 <= t <= 1
         * スプラインモードの場合、x1 = y1 = 0, x4 = y4 = 1
         * ベジェ曲線のxの式が三次方程式であるため、その解 t から、ベジェ曲線の y を求める
         * なお、ニュートン法の初期値はxとする
         * なぜなら、xの式をみると、xが増加傾向となるスプラインモードでは、係数が負となる可能性が低いため*/
        var t = _newton(Ax, Bx, Cx, -x, x);
        return f(Ay*t*t*t + By*t*t + Cy*t);
      };
    } else if (this.mode === "discrete") {
      return this.funcForDiscrete.bind(this);
    }
  }
} );


/*ニュートン法により、三次方程式 a0x^3 + a1x^2 + a2x + a3 の解を求める
 * 引数bは初期値*/
Math.qubicnewton = function(a0, a1, a2, a3, b) {
  var eps = 1e-15,                          //収束誤差
      fb = a0 *b*b*b + a1 *b*b + a2*b + a3; //方程式の結果
  if (fb === 0) {
    return b;
  }
  /*限界の収束回数は100回*/
  for (var i=0;i<100;i=(i+1)|0) {
    /*数値nは与えられた三次方程式を微分したもの*/
    var n = 3* a0 *b*b + 2 * a1 *b + a2;
    if (!n || ( (fb < eps) && (fb > -eps) )) {
      fb = eps = void 0;
      return b;
    } else {
      /*以下は収束の漸化式*/
      b =  b - fb / n;
      fb = a0 *b*b*b + a1 *b*b + a2*b + a3;
    }
  }
  return b; //収束しなかった結果
};

/*$attribute オブジェクト
 * アニメーションの時間調整と、呈示値の調整を一つのオブジェクトにまとめて行うことで、
 * アニメーションサンドイッチの実装をする
 * $calcModeオブジェクトから継承*/
base("$calcMode").up("$attribute").mix( {
  
  /*アニメーションの対象となる要素。たとえば、animate要素の親要素*/
  element: null,
  
  /*$fromオブジェクトを作るためのひな形となるオブジェクト*/
  $from: base("$from").up(),
  
  /*attributeName属性の値*/
  attrName: "",
  
  /*属性の名前空間*/
  attrNameSpace: null,
  
    /*指定された属性の規定値*/
  defaultValue: "",
  
  /*もともと属性がターゲットの要素につけられていたかどうか*/
  isDefault: false,
  
  /*attributeType属性がCSSだったときや、autoで属性名がCSSプロパティ名だったときには、true*/
  isCSS: false,
  
  /*計算モードdicreteやsplineなど*/
  mode: "linear",
  
  /*指定した要素の属性値を取得するメソッド*/
  getAttr: function(/*string*/ name, def) {
    var nameSpace = null;
    if (name.indexOf("xlink:") > -1) {
      nameSpace = "http://www.w3.org/1999/xlink";
    }
    var s = this._ele.getAttributeNS(nameSpace, name);
    if (this.element) {
        var view = this.element.ownerDocument.defaultView;
      if (s === "inherit") {
        return view.getComputedStyle(this.element.parentNode, "").getPropertyValue(this.attrName);
      } else if (s === "currentColor") {
        return view.getComputedStyle(this._ele, "").getPropertyValue("color");
      }
    }
    if ( s && (this._ele.getAttributeNS(null, "attributeName") === "font-size") 
          && /\d\s*$/.test(s) ) {
        s += "px";
        s = s.replace(/;/g, "px;");
    }
    /*DOM Level2やIE11では、getAttributeNSメソッドは空文字列を返す。他のブラウザではnullを返すことが多い
     * 
     * >the empty string if that attribute does not have a specified or default value
     * http://www.w3.org/TR/DOM-Level-2-Core/core.html#ID-ElGetAttrNS*/
    return (s || def);
  },
  
  /*アニメーション関連要素を収納*/
  _ele: document.documentElement,
  
  /*指定された要素にvalues属性が付いているかどうかのチェックできるメソッド*/
  hasAttrValues: function () {
    var ele = this._ele;
    if (!ele) {
      return false;
    } else {
      return ( ele.hasAttribute("from") || ele.hasAttribute("to")
         || ele.hasAttribute("by") || ele.hasAttribute("values") );
    }
  },
  
  /*後述のsetAttributeメソッドで使うキャッシュ
   * これを使うことで、二度同じ値を設定せずにすむので、高速化できる*/
  __cacheAttr: "",
  
  /*アニメーションするために、対象の属性やプロパティを変化させるメソッド*/
  setAttribute: function (value) {
    var attrName = this.attrName;
    if (!attrName) {
      return;
    }
    if (this.__cacheAttr === value) {
      /*同じ値であれば、再設定しない*/
      return;
    } else {
      this.__cacheAttr = value;
    }
    var ele = this.element;
    if (this.isCSS) {
      /*スタイルシートのプロパティを上書きしておく*/
      ele.style.setProperty(attrName, value, "");
    } else {
      ele.setAttributeNS(this.attrNameSpace, attrName, value);
    }
    value = attrName = ele = void 0;
  },
  
  /*setAttributeメソッドとは逆の効果で、無効化させるメソッド*/
  removeAttribute: function () {
    var attrName = this.attrName;
    if (!attrName) {
      return;
    }
    var ele = this.element;
    if (this.isDefault) {
      this.setAttribute(this.defaultValue);
    } else {
      /*初期段階でターゲットの要素に属性が指定されていない場合は、
       * 現在の属性値を削除するだけでよい*/
      ele.removeAttributeNS(this.attrNameSpace, attrName);
      /*スタイルシートのプロパティも削除しておく。removePropertyでないことに注意*/
      this.isCSS && ele.style.setProperty(attrName, this.defaultValue, "");
    }
    this.__cacheAttr = "";
    value = attrName = ele = void 0;
  },
  
  /*アニメーションの対象となる要素を値として返すメソッド
   * もっぱら、pushメソッドで使われる*/
  initTargetElement: function() {
    var ele = this._ele;
    var s = ele.parentNode || null;
    var id = ele.getAttribute("xlink:href");
    /*getAttributeNSメソッドでうまくいかなかったため、NSなしで代用*/
    if (id) {
      return ele.ownerDocument.getElementById(id.slice(1));
    }
    if ( id = ele.getAttributeNS(null, "targetElement") ) {
      return ele.ownerDocument.getElementById(id);
    }
    return s;
  },
  
  /*repeatイベントの発火時刻リスト
   * setSmilEventメソッドを見よ*/
  _repeatList: [],
  
  /*リピート回数を示すプロパティ
   * setSmilEventメソッドを見よ*/
  _repeatCount: 0,
  
  /*SMILイベント関連を発火させるためのメソッド
   * もっぱら、$attributeオブジェクトのpush メソッドで使われる*/
   setSmilEvent: function($list) {
    $list.addEvent("begin", function($list) {
        var target = this._ele,
            detail = 0;
        /*IE11のために、MouseEventsでSMILEventsの代用をする*/
        var evt = target.ownerDocument.createEvent("MouseEvents");
        evt.initMouseEvent("beginEvent" ,true, true, window, detail, 0, 0, 0, 0, false, false, false, false, 0, target);
        target.dispatchEvent(evt);
        /*repeatイベントのために、_repeatListプロパティを初期化する*/
        var list = this._repeatList = [],
            active = $list.activeTime,
            begin = $list.begin,
            simpleDuration = this.timeline.simpleDuration;
            if (simpleDuration && (simpleDuration !== active)
                  && (active !== Number.MAX_VALUE) ) {
              /*活動継続時間と単純持続時間が異なるとき、repeatイベントを設定*/
              for (var m= simpleDuration, n=1;m < active ; m+=simpleDuration) {
                list.push( {
                  frame: begin + m,
                  /*リピートの回数 (n >= 1)*/
                  count: n
                } );
                ++n;
              }
            }
    }.bind(this) );
    
    $list.addEvent("play", function($list) {
      var target = this._ele,
          detail = 0,
          frame = $list.currentFrame,
          list = this._repeatList;
      if (!list.length) {
        return;
      }
      for (var i=0;i<list.length;++i) {
        if ( (this._repaetCount >= i+1)
            || (list[i].frame >= frame) ) {
          this._repeatCount = detail;
          break;
        } 
        detail = list[i].count;
        var evt = target.ownerDocument.createEvent("MouseEvents");
        evt.initMouseEvent("repeatEvent" ,true, true, window, detail, 0, 0, 0, 0, false, false, false, false, 0, target);
        target.dispatchEvent(evt);
      }
    }.bind(this) );
    
    $list.addEvent("end", function() {
        var target = this._ele,
            detail = 0;
        var evt = target.ownerDocument.createEvent("MouseEvents");
        evt.initMouseEvent("endEvent" ,true, true, window, detail, 0, 0, 0, 0, false, false, false, false, 0, target);
        target.dispatchEvent(evt);
    }.bind(this) );
   },

  /*引数で指定した要素 ele の属性を解析して、フレームに追加する*/
  push: function(/*Element Node*/ ele) {
    if (!ele || !ele.hasAttribute) {
      return null;
    }
    /*キャッシュを初期化しておく*/
    this.__cacheAttr = "";
    

   /*getAttrメソッドとhasAttrValuesメソッドで必要*/
    this._ele = ele;
    
    /*initTargetElementメソッドを使って、elementプロパティの初期化*/
    this.element = this.initTargetElement();
    
    if (!this.hasAttrValues()) {
      /*from属性、to、by、values属性が指定されていない場合、アニメーションの効果が出ないように調整する
       *SMILアニメーションの仕様を参照
       *
       *>if none of the from, to, by or values attributes are specified, the animation will have no effect
       *「3.2.2. Animation function values」より引用
       *http://www.w3.org/TR/2001/REC-smil-animation-20010904/#AnimFuncValues
      */
      return null;
    }
 
    /*属性値の設定*/
    this.attrName = this.getAttr("attributeName", "");
    var attrName = this.attrName;
    
    /*属性タイプの設定*/
    var attrType = this.getAttr("attributeType", "auto");
    var computedStyle = this.element && this.element.ownerDocument.defaultView.getComputedStyle(this.element, "");
    if ( (attrType === "CSS")
         || ( (attrType === "auto") && this.element 
             && computedStyle.getPropertyValue(attrName)
             && !/^(width|height|transform)$/.test(attrName)
            )
        ) {
        this.isCSS = true;
    }
    
    /*xlink関連の属性のときは名前空間を変更しておく*/
    if (attrName.indexOf("xlink") > -1) {
      this.attrNameSpace = "http://www.w3.org/1999/xlink";
    }
    /*thiseleはターゲットとなる要素（たとえば、親要素）*/
    var thisele = this.element;
    if (thisele) {
      /*規定値を設定しておく。属性を初期化するときに使われる*/
      this._ele = thisele;
      this.isDefault = thisele.hasAttributeNS(this.attrNameSpace, attrName);
      this.defaultValue = this.getAttr(attrName,
       computedStyle.getPropertyValue(attrName) );
      this._ele = ele; /*_eleプロパティを元に戻しておく*/
    }
    /*eleの属性の値を、それぞれオブジェクトに割り当て*/
    var $frame = base("$frame"),
        begin = $frame.$begin,
        frame = begin.up().mix( {
                  /*targetプロパティはbeginEventなどの発火で使う*/
                  target: ele,
                  eventTarget: (this.element || begin.eventTarget),
                  string: this.getAttr("begin", "0"),
                  $activate: begin.$activate.up().of( {
                    end: begin.$end.up().mix( {
                          eventTarget: (this.element || begin.eventTarget),
                          string: this.getAttr("end", null)
                        } )
                  } ).mix( {
                    dur: this.getAttr("dur", null),
                    repeatCount: this.getAttr("repeatCount", null),
                    repeatDur: this.getAttr("repeatDur", null),
                    min: this.getAttr("min", "0"),
                    max: this.getAttr("max", "indefinite")
                  } )
                } ).updateList().parse();
    $frame.addLine(frame.$list.init());
    
    /*beginElementメソッドを追加*/
    var objList = frame.$list.addList(Number.MAX_VALUE),
        /*endListのvalueプロパティには、活動継続フレーム数と開始フレーム数を足したものが入る*/
        endList = frame.$list.addEndList(Number.MAX_VALUE);
    ele.beginElement = (frame.string !== "indefinite") ? function(){}
                        : function() {
                            objList.value = frame.begin = base("$frame").currentFrame;
                            frame.isResolved = true;
                            var evt = this.ownerDocument.createEvent("MouseEvents");
                            evt.initMouseEvent("beginEvent" ,true, true, window, 0, 0, 0, 0, 0, false, false, false, false, 0, this);
                            this.dispatchEvent(evt);
                          };
    /*endElementメソッドを追加*/
    var endFrame = frame.$activate.end__ || {};
    ele.endElement = (endFrame.string !== "indefinite") ? function(){}
                        : function() {
                            if (frame.isResolved) {
                              endFrame.isResolved = true;
                              endList.value  = base("$frame").currentFrame;
                              var evt = this.ownerDocument.createEvent("MouseEvents");
                              evt.initMouseEvent("endEvent" ,true, true, window, 0, 0, 0, 0, 0, false, false, false, false, 0, this);
                              this.dispatchEvent(evt);
                            }
                          };
    /*setFrameメソッドを使ったときの、再帰スタックの使いすぎを防ぐため*/
    frame.timelines = [];
    this.setSmilEvent(frame.$list);
    begin = ele = void 0;
    return frame;
  },
  
  /*setValuesメソッド
   * values属性やfrom属性やto属性を処理するためのメソッド
   * valuesは配列、それ以外の引数は文字列
   * 返り値は、values属性は配列、それ以外の属性のときは、
   * 自分自身となる$attributeオブジェクトのコピーを返す*/
   setValues: function(values, from, to, by) {
     var $from = this.$from,
         s = [this.up().of( {
               to: $from.up().of( {
                 from: $from.up()
               } )
             } )],
         sto = s[0].to__;
     values = values && values.split(";");
     /*from属性はオプションなので、条件には付け加えない*/
     if (values && values.length) {
       /*values属性が指定された場合、他の属性は無視される
        * W3C仕様　SMIL アニメーション 3.2.2. アニメーション関数の値*/
        s = [];
       for (var i=1;i<values.length;++i) {
         s.push( this.up().of( {
               to: $from.up().of( {
                 from: $from.up()
               } )
             } ) );
         sto = s[s.length-1].to__;
         sto.string = values[i];
         sto.from__.string = values[i-1];
       }
     } else if (to) {
       sto.string = to;
       sto.from__.string = from || "0";
     } else if (by) {
       sto.string = by;
       sto.from__.string = from || "0";
       var toNumList = sto.call(),
           fromNumList = sto.from;
       for (var i=0;i<toNumList.length;++i) {
         /*初期値と差分を足していく*/
         toNumList[i] += fromNumList[i];
       }
     } else {
       return null;
     }
     $from = sto = toNumList = fromNumList = void 0;
     return s;
   },
   
  /*isKeyErrorメソッド
   * 後述のsetKeyメソッドで使われる。keyTimes属性のエラーをチェックするメソッド
   * 属性値にエラーがあれば、trueを返し、なければ、falseを返す*/
  isKeyError: function(/*number*/ keyLength, /*number*/toLength) {
    return !!(keyLength && (keyLength !== (toLength+1)) );
  },
   
   /*setKeyメソッド
    * 引数の要素のkeyTimes属性やkeySplines属性を処理するためのメソッド
    * 必要な他の属性処理はsetValuesメソッドに任せている*/
   setKey: function(ele) {
     this._ele = ele;
     var to = this.setValues(this.getAttr("values", null),
          this.getAttr("from", null),
          this.getAttr("to", null),
          this.getAttr("by", null) ),
         toLength = to ? to.length : 0,
         keyTimes = this.getAttr("keyTimes", null),
         keySplines = this.getAttr("keySplines", null),
         keys,
         splines = keySplines && keySplines.split(";"),
         isDiscrete = (this.mode === "discrete"),
         toiKeySplines;
    if (!isDiscrete && keyTimes && to) {
      keys = this.$from.numList__.call( {
        string: keyTimes
      } );
      /*keysTime属性の最初は0でなければならない。そうでなければエラー（SVG 1.1 2ndの仕様を参照)*/
      if (keys.length && (keys[0] !== 0)) {
        return null;
      }
      /*toオブジェクトはtoとfromで一組となっているのでlengthが加算される*/
      if (this.isKeyError(keys.length, toLength)) {
        /*keyTimes属性とvalues属性のリストの個数が合致しない場合、アニメーションの効果がない
         * 仕様を参照 SMIL Animation 3.2.3. Animation function calculation modes
         * http://www.w3.org/TR/smil-animation/#AnimFuncCalcMode*/
         
        /*ただし、animateMotion要素においては、keyPoints属性が
         * values属性のリストよりも優先されるため、
         * keyPoints属性があるときは、アニメーションの効果がある
         * 
         * >Regarding determining the points which correspond to the ‘keyTimes’ attributes, the ‘keyPoints’ attribute overrides ‘path’, which overrides ‘values’
         * 
         * http://www.w3.org/TR/SVG11/animate.html#AnimateMotionElement
         */
        return null;
      }
      for (var i=0;i<toLength;++i) {
        to[i].keyTime = keys[i+1] - keys[i];
        if (splines) {
          toiKeySplines = this.$from.numList__.call( {
            string: splines[i]
          } );
          /*空配列を返すため、nullに変えておく*/
          to[i].keySplines = toiKeySplines.length ? toiKeySplines : null;
        }
      }
    } else if (!isDiscrete && to) {
      var per = 1 / toLength;
      for (var i=0;i<toLength;++i) {
        to[i].keyTime = per;
        if (splines) {
          toiKeySplines = this.$from.numList__.call( {
            string: splines[i]
          } );
          to[i].keySplines = toiKeySplines.length ? toiKeySplines : null;
        }
      }
    } else if (to) {
        /*discreteモードの処理*/
      if (keyTimes) {
        keys = this.$from.numList__.call( {
          string: keyTimes
        } );
      /*keysTime属性の最初は0でなければならない。そうでなければエラー（SVG 1.1 2ndの仕様を参照)*/
        if (keys.length && (keys[0] !== 0)) {
          return null;
        }
        if (this.isKeyError(keys.length, toLength)) {
          return null;
        }
        for (var i=0;i<toLength;++i) {
          to[i].keyTime = keys[i+1] - keys[i];
        }
      } else {
        var per = 1 / (toLength+1);
        for (var i=0;i<toLength;++i) {
          to[i].keyTime = per;
        }
      }
      /*toオブジェクトが足らないので、一つ追加しておく*/
      to.push( to[toLength-1].up().mix( function(){
          if (!keys) {
            return;
          }
          this.keyTime = 1-keys[keys.length-1];
        } ).of( {
            call: function() {
              return function (t) {
                 return isNaN(t) ? this.string
                                 : this.to__.advance(1);
              }.bind(this);
            }
          } 
        ) );
    }
    if (this.mode === "paced") {
      /*ベクトル全体の距離を算出*/
      var norm = 0;
      to.forEach( function(x) {
        norm += x.to__.distance(x.to__.from__);
      } );
      to.forEach( function(x) {
         x.norm = norm;
      } );
    }
    ele = keyTimes = keys = per = splines = void 0;
    return to;
   }
} ).up("$setElement").mix( {
  /*to属性の値、文字列*/
  to: "",
  
  
  /*後述のinitializeメソッドで使う要素リスト
   * getElementsByTagNameNSメソッドの返り値をArray化したことを想定*/
  elementList: [],
  
  /*何番目からelementListを処理しているかの数値*/
  numberOfElemList: 0,
  
  /*initialize メソッド
   * 要素リストを初期化させる
   * 初期化処理を分散させるために使う*/
  initialize: function() {
    var eles = this.elementList;
    if (!eles || !eles.length) {
      return;
    }
    var length = this.numberOfElemList+50;
    for (var i=length-50; i<length; ++i) {
      if (eles.length <= i) {
        this.elementList = null;
        return;
      }
      this.up().init(eles[i]);
    }
    this.numberOfElemList += 50;
    eles = length = void 0;
  },
   
  /*initメソッドで使われるアニメーション関数*/
  _setFrame: function ($list) {
    this.setAttribute(this.to);
  },
  
  /*開始を設定されたタイムライン ($beginオブジェクト)*/
  timeline: base("$frame").$begin,
  
  /*アニメが終了した際の後処理*/
  _setEndFrame: function ($list) {
    /*removeの場合、アニメーションを凍結せずに、もとに戻す*/
    if (this.fill === "remove") {
      this.removeAttribute();
    }
  },
  
  /*アニメーションの呈示値を呼び出す関数*/
  tocall: function() {},
  
  init: function(ele) {
    var line = this.push(ele);
    if (ele && ele.getAttributeNS) {
      this._ele = ele;
      this.to = this.getAttr("to", "");
      this.fill = this.getAttr("fill", "remove");
    }
    var thisele = this.element;
    if (line && thisele) {
      this.timeline = line;
      /*$begin.$listのイベントに属性処理を追加*/
      line.$list.addEvent("begin", this._setFrame.bind(this));
      line.$list.addEvent("play", this._setFrame.bind(this));
      line.$list.addEvent("end", this._setEndFrame.bind(this));
      /*アニメーションが再起動する可能性もあるため、$listのstateプロパティはここで初期化*/
      line.$list.state = line.$list.WAITING;
    }
    line = thisele = void 0;
  }
}).up("$animateElement").mix( {
  /*アニメ関数の配列*/
  funcs: [],

  /*進捗率advanceから、呈示値を求める*/
  tocall: function(advance) {
    var tf = this.funcs;
    if (this.mode !== "discrete") {
      for (var i=0;i<tf.length;++i) {
        var tfi = tf[i];
        /*keyTime（keyTimes属性で指定されたような値）で実行するかどうかを判別*/
        if (tfi.endKeyTime >= advance) {
          return tfi(advance - tfi.startKeyTime);
        }
      }
    } else {
      var result = "";
      for (var i=0;i<tf.length;++i) {
        var tfi = tf[i];
        if (advance >= tfi.startKeyTime) {
          result = tfi(advance);
        }
      }
      advance = tf = tfi = void 0;
      return result;
    }
    tf = i = tfi = void 0;
    return "";
  },
  
  _setFrame: function($list) {
    var currentFrame = $list.currentFrame;
    /*durationは単純継続時間
     *advanceは継続時間内での、進捗率
     * 　仕様を参照　http://www.w3.org/TR/smil-animation/#AnimFuncValues
     *進捗率advanceは、durationと進捗フレーム数とを割った余り(REMAINDER)で算出する
     * 仕様を参照　SMIL Animation 3.6.2　Interval timing
     * http://www.w3.org/TR/2001/REC-smil-animation-20010904/#IntervalTiming*/
    var line = this.timeline,
        duration = line.simpleDuration,
        /*単純継続時間が不定の場合、補間はせずに初期値が採用されるため、advanceは0となる
         * 仕様を参照　SMIL Animation 3.2.2. Animation function values のInterpolation and indefinite simple durations
         * http://www.w3.org/TR/2001/REC-smil-animation-20010904/#AnimFuncValues*/
        advance = duration ? ( (currentFrame - $list.begin) % duration ) / duration
                    : 0;
    this.setAttribute(this.tocall(advance));
    line = duration = advance = void 0;
  },
  
  /*凍結処理をするために、進捗率を最後まで進めて調整するメソッド
   * 返り値は調整された進捗率
   * もっぱら、_setEndFrameメソッドで使われる*/
  getAdvanceEnd: function($list) {
    var line = this.timeline,
        duration = line.simpleDuration;
    if (duration) {
      var time = (line.activeTime > $list.beginEnd) ? $list.beginEnd
                  : line.activeTime;
      var advance = ( time % duration ) / duration;
      /*例外が発生するため、進捗率が1を超えないように処理*/
      advance = (advance > 1) ? 1 : advance;
       /*活動継続時間と単純継続時間が一致すると、余りは0となるため以下の処理*/
      advance = advance || 1;
    } else {
      advance = 0;
    }
    return advance;
  },
  
  /*_setEndFrameメソッドは、終了処理と凍結作業をする*/
  _setEndFrame: function($list) {
    /*上書きされたメソッドを呼び出してアニメーションの凍結作業をする*/
    if (this.fill === "freeze") {
      this.setAttribute(this.tocall( this.getAdvanceEnd($list) ));
      line = duration = advance = void 0;
    } else {
      this.removeAttribute();
    }
  },
  
  /*getAttrメソッドをオーバライド*/
  getAttr: function (name, def) {
    var s= this.$attribute.getAttr.apply(this, arguments);
    if ((name === "from") && !s && this.defaultValue) {
      /*from属性がない場合、対象要素の既定値を返す*/
      return this.defaultValue;
    }
    return s;
  },
  
  _keywords: {
    aliceblue:    [240,248,255],
    antiquewhite: [250,235,215],
    aqua:         [0,255,255],
    aquamarine:   [127,255,212],
    azure:        [240,255,255],
    beige:        [245,245,220],
    bisque:       [255,228,196],
    black:        [0,0,0],
    blanchedalmond:[255,235,205],
    blue:         [0,0,255],
    blueviolet:   [138,43,226],
    brown:        [165,42,42],
    burlywood:    [222,184,135],
    cadetblue:    [95,158,160],
    chartreuse:   [127,255,0],
    chocolate:    [210,105,30],
    coral:        [255,127,80],
    cornflowerblue:[100,149,237],
    cornsilk:     [255,248,220],
    crimson:      [220,20,60],
    cyan:         [0,255,255],
    darkblue:     [0,0,139],
    darkcyan:     [0,139,139],
    darkgoldenrod:[184,134,11],
    darkgray:     [169,169,169],
    darkgreen:    [0,100,0],
    darkgrey:     [169,169,169],
    darkkhaki:    [189,183,107],
    darkmagenta:  [139,0,139],
    darkolivegreen:[85,107,47],
    darkorange:    [255,140,0],
    darkorchid:   [153,50,204],
    darkred:      [139,0,0],
    darksalmon:   [233,150,122],
    darkseagreen: [143,188,143],
    darkslateblue:[72,61,139],
    darkslategray:[47,79,79],
    darkslategrey:[47,79,79],
    darkturquoise:[0,206,209],
    darkviolet:   [148,0,211],
    deeppink:     [255,20,147],
    deepskyblue:  [0,191,255],
    dimgray:      [105,105,105],
    dimgrey:      [105,105,105],
    dodgerblue:   [30,144,255],
    firebrick:    [178,34,34],
    floralwhite:  [255,250,240],
    forestgreen:  [34,139,34],
    fuchsia:      [255,0,255],
    gainsboro:    [220,220,220],
    ghostwhite:   [248,248,255],
    gold:         [255,215,0],
    goldenrod:    [218,165,32],
    gray:         [128,128,128],
    grey:         [128,128,128],
    green:        [0,128,0],
    greenyellow:  [173,255,47],
    honeydew:     [240,255,240],
    hotpink:      [255,105,180],
    indianred:    [205,92,92],
    indigo:       [75,0,130],
    ivory:        [255,255,240],
    khaki:        [240,230,140],
    lavender:     [230,230,250],
    lavenderblush:[255,240,245],
    lawngreen:    [124,252,0],
    lemonchiffon: [255,250,205],
    lightblue:    [173,216,230],
    lightcoral:   [240,128,128],
    lightcyan:    [224,255,255],
    lightgoldenrodyellow:[250,250,210],
    lightgray:    [211,211,211],
    lightgreen:   [144,238,144],
    lightgrey:    [211,211,211],
    lightpink:    [255,182,193],
    lightsalmon:  [255,160,122],
    lightseagree: [32,178,170],
    lightskyblue: [135,206,250],
    lightslategray:[119,136,153],
    lightslategrey:[119,136,153],
    lightsteelblue:[176,196,222],
    lightyellow:  [255,255,224],
    lime:         [0,255,0],
    limegreen:    [50,205,50],
    linen:        [250,240,230],
    magenta:      [255,0,255],
    maroon:       [128,0,0],
    mediumaquamarine:[102,205,170],
    mediumblue:    [0,0,205],
    mediumorchid:  [186,85,211],
    mediumpurple:  [147,112,219],
    mediumseagreen:[60,179,113],
    mediumslateblue:[123,104,238],
    mediumspringgreen:[0,250,154],
    mediumturquoise:[72,209,204],
    mediumvioletred:[199,21,133],
    midnightblue:  [25,25,112],
    mintcream:     [245,255,250],
    mistyrose:     [255,228,225],
    moccasin:      [255,228,181],
    navajowhite:   [255,222,173],
    navy:          [0,0,128],
    oldlace:       [253,245,230],
    olive:         [128,128,0],
    olivedrab:     [107,142,35],
    orange:        [255,165,0],
    orangered:     [255,69,0],
    orchid:        [218,112,214],
    palegoldenrod: [238,232,170],
    palegreen:     [152,251,152],
    paleturquoise: [175,238,238],
    palevioletred: [219,112,147],
    papayawhip:    [255,239,213],
    peachpuff:     [255,218,185],
    peru:          [205,133,63],
    pink:          [255,192,203],
    plum:          [221,160,221],
    powderblue:    [176,224,230],
    purple:        [128,0,128],
    red:           [255,0,0],
    rosybrown:     [188,143,143],
    royalblue:     [65,105,225],
    saddlebrown:   [139,69,19],
    salmon:        [250,128,114],
    sandybrown:    [244,164,96],
    seagreen:      [46,139,87],
    seashell:      [255,245,238],
    sienna:        [160,82,45],
    silver:        [192,192,192],
    skyblue:       [135,206,235],
    slateblue:     [106,90,205],
    slategray:     [112,128,144],
    slategrey:     [112,128,144],
    snow:          [255,250,250],
    springgreen:   [0,255,127],
    steelblue:     [70,130,180],
    tan:           [210,180,140],
    teal:          [0,128,128],
    thistle:       [216,191,216],
    tomato:        [255,99,71],
    turquoise:     [64,224,208],
    violet:        [238,130,238],
    wheat:         [245,222,179],
    white:         [255,255,255],
    whitesmoke:    [245,245,245],
    yellow:        [255,255,0],
    yellowgreen:   [154,205,50]
  },
  
  setAdd: function (ele, to) {
    /*additive属性がsum (加法アニメーション)の場合*/
    if (ele.getAttributeNS(null, "additive") === "sum") {
      var attrValue = ele.parentNode.getAttributeNS(null, this.attrName);
      ele.addEventListener("beginEvent", function(evt) {
        to.forEach( function(x) {
          x.to__.setAdditive(attrValue);
        } )
      }, false);
    }
  },
  setAccum: function (ele, to) {
    /*accumulate属性がsum (蓄積アニメーション)の場合*/
    if (ele.getAttributeNS(null, "accumulate") === "sum") {
      ele.addEventListener("repeatEvent", function(evt) {
        to.forEach( function(x) {
          x.to__.call();
          x.to__.setAccumulate(evt.detail);
        } )
      }, false);
    }
  },
  
  /*displayなど補間をしなくてもよい属性への対処するためのメソッド*/
  setString: function() {
    if (/^(?:display|class|edgeMode|(gradient|marker|pattern|maskContent|mask|patternContent|primitive)Units|in|in2|method|mode|operator|preserveAspectRatio|result|spacing|spreadMethod|stitchTiles|target|type|xlink:href|yChannelSelector|color-interpolation|(clip|fill)-rule|cursor|filter|font-(family|stretch|style|variant)|image-rendering|marker-(end|mid|start)|mask|overflow|pointer-events|shape-rendering|stroke-(linecap|linejoin)|text-(anchor|decoration|rendering)|visibility)$/.test(this.attrName) ) {
      this.mode = "discrete";
    }
  },
  
  /*小数点以下の桁数を指定するため、setValuesメソッドをオーバライドする*/
  degits: 1,
  setValues: function() {
    var s = this.$attribute.setValues.apply(this, arguments),
        degits = this.degits;
    s && s.forEach(function(x) {
      x.to__.degit = degits;
    } );
    degits = void 0;
    return s;
  },
  
  /*#から始まる文字列#aabbcc、例えばを、rgb(.., .., ..,)形式へと変換するためのメソッド
   *　initメソッドで使われる*/
  toRGB: function(rgbColor) {
           var keyword = this._keywords[rgbColor];
           if (keyword) {
             return "rgb(" + keyword.join(", ") + ")";
           }
           if (rgbColor && (rgbColor[0] === "#")) {  //#を含む場合
              var s = "rgb(",
                  _parseInt = parseInt;
              if (rgbColor.length < 5) {
                var r = rgbColor[1],
                    g = rgbColor[2],
                    b = rgbColor[3],
                rgbColor = "#" + r + r + g + g + b + b;
              }
              rgbColor.match(/\#(\w{2})(\w{2})(\w{2})/);
              s += _parseInt(RegExp.$1, 16)
                + ", "
                + _parseInt(RegExp.$2, 16)
                + ", "
                + _parseInt(RegExp.$3, 16)
                + ")";
              r = g = b = void 0;
              return s;
           }
           return rgbColor;
        }

/*initメソッドに追加処理
 * onメソッドについては、base.jsを参照のこと*/
} ).on ("init", function(ele) {
  var to, 
      keyTime = 0,
      /*関数toRGBはrgbColor形式への変換処理で使う*/
      toRGB = function(x) { return x; };
  if (/^(?:fill|stroke|stop-color|color)$/.test(this.attrName)) {
    /*通常は、小数点以下の桁数を既定値の1桁とする
     *ただし、fill属性など色を扱う属性であれば、
     *RGB形式では、補間に小数を使わないため、0桁に設定
     * （なお、この作業は、setKeyメソッドの前に済ませておく必要がある）*/
    this.degits = 0;
    /*たとえば、fill属性などである場合には、rgbColor形式への変換処理をする*/
    toRGB = this.toRGB.bind(this);
  }
  if (ele) {
    this.mode = ele.getAttributeNS(null, "calcMode") || "linear";
    this.setString();
    to = this.setKey(ele);
  }
  if (to) {
    var aa = new Array(to.length);
    for (var i=0;i<to.length;++i) {
      var x = to[i];
      x.to__.string = toRGB(x.to__.string);
      x.to__.from__.string = toRGB(x.to__.from__.string);
      var s = x.call();
      /*x.keyTimeプロパティは区間を示しているため、区切り時刻に変換しておく
       * startKeyTimeプロパティは区間のスタート時点
       * endKeyTimeプロパティは区間のエンド地点*/
      s.startKeyTime = keyTime;
      keyTime = s.endKeyTime = keyTime + x.keyTime;
      aa[i] = s;
    }
    this.funcs = aa.filter( function(s) {
       if (!this.timeline.isResolved) {
         /*begin属性などにイベントを設定していた（未解決の）場合、後のs(0.1)がうまく作動せず、
          * 例外を出してしまうため、ここで返しておく*/
         return true;
       }
       /*splineモードで、かつ、アニメーションが無効な関数の場合は配列から外しておく
        * 無効な場合に関しては、$calcModeオブジェクトのcallメソッドを参照*/
       return (this.mode !== "spline") || !s.isNotAnimate;
    }, this );
    this.setAdd(ele, to);
    this.setAccum(ele, to);
  }
  keyTime = keywords = toRGB = void 0;
} )
/*$animateTranformElementオブジェクト
 * animateTransform要素に関連するオブジェクト*/
 .up("$animateTransformElement")
 .mix({
   /*__transformListで何番目のアイテムかを示すプロパティ*/
   numberOfList: -1,
   
   /*type属性の値*/
   type: "translate",
   
   /*attributeName属性の値はtransformで固定*/
   attrName: "transform",
   
   /*明確にisCSSプロパティを設定しておくことで、プロトタイプチェーンを使わせず最適化*/
   isCSS: false,
   
   /*additive属性の値がsumのときにtrue*/
   isSum: false,
   
   /*transform属性の値に使われる数値は精密であることが求められるため、
    *小数点以下の桁数を決めるdegitsプロパティの値も大きくしておく*/
   degits: 15,
   
   /*tocallメソッド(後述の$motionElementオブジェクトも含む）で使うメソッド
    * __transformListの値を結合して、文字列として返す*/
   joinList: function(s) {
     var playList = this.element.__transformList;
     for (var i=0;i<playList.length;++i) {
       var item = playList[i],
           value = item.value;
       if (item.isSum) {
         s += " " + value;
       } else if (item.isPlaying) {
         /*他のanimateTransform要素がadditive属性の値にreplaceをすでに設定していた、
          *かつ、その要素がアニメーション再生中の場合はsを初期化*/
         s = value;
       }
     }
     return s.trim();
   },
   
   /*$animateElementオブジェクトのtocallメソッドをオーバライド*/
   tocall: function (advance) {
     if (this.numberOfList < 0) {
       throw new Error("Number of The List Error");
     }
     var currentItem = this.element.__transformList[this.numberOfList];
     currentItem.value = this.type+ "(" +this.$animateElement.tocall.call(this, advance)+ ")";
     currentItem.isPlaying = true;
     currentItem.isSum = this.isSum;
     return this.joinList(this.defaultValue || "");
   },
   
   /*後の_setFrameメソッドで使うダミー*/
   __setAttribute: function(){},
   
   _setFrame: function($list) {
    var currentFrame = $list.currentFrame;
     /*__transformListの中で、自分より後の項目に再生中のものがあれば、
      *アニメーションを再生させないで、後に続く項目に任せる*/
     var list = this.element.__transformList,
         isPostActive = false,
         length = list.length;
     if ( (length !== 1) && (this.numberOfList < (length - 1) ) ) {
       /*リストの項目が一つだけであったり、自分自身が最後尾であれば、アニメーションを再生する
        * また、後に続く項目で再生中のものがあったら、今回は再生しない*/
       for (var i=this.numberOfList+1;i<length;++i) {
         if (list[i].isPlaying) {
           isPostActive = true;
         }
       }
     }
     /*__setAttributeはダミーなので、アニメーションが再生されない*/
     this.setAttribute = isPostActive ? this.__setAttribute
                                    : this.$animateElement.setAttribute;
     /*上書きされたメソッドを呼び出す*/
     this.$animateElement._setFrame.call(this, $list);
   },
   
   _setEndFrame: function($list) {
     var list = this.element.__transformList;
     if (!list) {
       return;
     }
     var item = list[this.numberOfList];
     if (this.fill === "remove") {
       if (!item) {
         return;
       } else if (!this.isSum) {
         /*凍結処理をしないで、かつ、元の状態に戻して、効果が出ないようにする*/
         item.isPlaying = false;
       } else {
         /*凍結処理をしないで、かつ、効果を出すが、変形させないようにする*/
         item.value = "translate(0)";
       }
     } else {
       /*凍結処理をする
        * 自前のtocallメソッドはvalueプロパティを書きかえてしまうため、
        * 上書きメソッドを呼び出す*/
       item.value = this.type+ "(" 
                                      +this.$animateElement.tocall.call( this, this.getAdvanceEnd($list) )+ ")";
     }
     this.setAttribute( this.joinList(this.defaultValue || "") );
     item = void 0;
   },
   
    /*setAddメソッドのオーバライド
     * additive属性のsumに対する振る舞いが異なるため*/
    setAdd: function() {},
  })
 .on("init", function (ele) {
   if (!ele || !ele.parentNode) {
     return;
   }
   this.getAttr = this.$attribute.getAttr;
   this.type = this.getAttr("type", "translate");
   this.attrName = "transform";
   var parent = this.element;
   this.isDefault = parent.hasAttributeNS(null, "transform");
   this.defaultValue = parent.getAttributeNS(null, "transform") || "";
   this.isSum = (this.getAttr("additive", "replace") === "sum");
   if (!parent.__transformList) {
     parent.__transformList = [];
     this.numberOfList = -1;
   }
   if (this.hasAttrValues()
    && (this.numberOfList < 0) ) {
     /*もし、今まで、このオブジェクトで、initメソッドを実行していなければ*/
     this.numberOfList = parent.__transformList.length;
     /*isPlayingプロパティはアニメーション再生終了後でもtrueとなるので注意*/
     parent.__transformList.push( {isPlaying: false,
                                   value: "translate(0)",
                                   isSum: this.isSum
                                  } );
   }
  } )
  .up("$motionElement")
  .mix( function() {
    /*$animateTransformElementオブジェクトのでは、うまくいかないため、
     * setRFrameとsetEndFrameメソッドを再定義*/
    this._setFrame = this.$animateElement._setFrame;
    this._setEndFrame = this.$animateElement._setEndFrame;
  })
  .mix( {
    numberOfList: -1,
    mode: "paced",
        
    /*hasAttrValuesメソッドのオーバライド
     * path属性などに対応させるため*/
    hasAttrValues: function () {
      if (this.$attribute.hasAttrValues.call(this)) {
        return true;
      } else {
        return (this._ele.hasAttribute("keyPoints") || this._ele.hasAttribute("path")
                 || this._ele.getElementsByTagNameNS(this.path.namespaceURI, "mpath").length);
      }
    },
    
    path: document.createElementNS("http://www.w3.org/2000/svg", "path"),
    
    rotate: "0",
    
    /*tocallメソッドのオーバライド*/
    tocall: function(advance) {
      /*モーションは仕様の関係上、かならず、CTMの一番初めに置かれ、前置積となる
     * なお、__transformListプロパティはtransform属性の値であり、CTMを表現する。
     * 必ず、CTMの一番目に設定する*/
      return ("translate(" +this.$animateElement.tocall.call(this, advance)+ ") "
              + this.joinList(this.defaultValue || "")).trim();
    },
    
    /*図の現在の角度rを求めて、rotate(rの文字列（最後に括弧はいらない）で返すメソッド*/
    getRotate: function(path, advanceLength, rotate) {
      /*パスセグメントの数値を求めてから、動いている図形の傾き角度r(ラジアンではなく度数)を算出する*/
      var length = path.getPathSegAtLength(advanceLength),
          seg = path.pathSegList.getItem(length),
          command = seg.pathSegTypeAsLetter,
          /*pi*180ではEdgeでうまく作動しない（原因は不明）*/
          pi = Math.PI;
      if (command === "M") {
        /*次のセグメントがどのコマンドによるかで、計算方式が違う*/
        var nextSeg = path.pathSegList.getItem(length+1),
            nextCommand = nextSeg.pathSegTypeAsLetter;
        if (nextCommand === "M") {
          return "";
        } else if (nextCommand === "L") {
          return ") rotate(" +(Math.atan2(nextSeg.y-seg.y, nextSeg.x-seg.x)/pi*180 + rotate)+ "";
        } else if (nextCommand === "C") {
          return ") rotate(" +(Math.atan2(nextSeg.y1-seg.y, nextSeg.x1-seg.x)/pi*180 + rotate)+ "";
        }
      } else if ((command === "L") && (length-1 >= 0)) {
        var preSeg = path.pathSegList.getItem(length-1);
        return ") rotate(" +(Math.atan2(seg.y-preSeg.y, seg.x-preSeg.x)/pi*180 + rotate)+ "";
      } else if (command === "C") {
        /*3次ベジェ曲線を微分する方法はニュートン法など数値解析の必要があるので、
         * 以下の通り、別の方法を採用する。
         * 現在位置から一歩進んだ曲線上の点Bをとり、それを、現在の点Aと結んで線分ABとしたとき、
         * その直線の傾きからおおよその角度を求める*/
         var point = path.getPointAtLength(advanceLength),
             x = point.x,
             y = point.y;
         /*一歩進んだ点*/
         point = path.getPointAtLength(advanceLength+1);
         var rad = Math.atan2(point.y-y, point.x-x);
         if (rad === 0) {
           /*アニメーションの終了間際では、radが0となるため、
            * 代わりに、ベジェ曲線のハンドルの傾きから角度を求める*/
           rad = Math.atan2(seg.y-seg.y2, seg.x-seg.x2);
         }
         return ") rotate(" +(rad/pi*180 + rotate)+ "";
      } else {
        return "";
      }
    },
    
    /*this.$animateElement.tocallメソッドを置き換えるためのメソッド
     * mpath要素が指定されたときか、path属性のときにのみ使われる*/
    _tocallForPath: function(advance) {
      if (this.isKeyPoints) {
        /*keyPoints属性の値に従って、advance値を決定する。
         * なお、$animateElementはinitメソッドで書き換えているので、二重に呼び出す必要がある*/
        advance = +this.$animateElement.$animateElement.tocall.call(this, advance);
      }
      var path = this.path,
          advanceLength = advance * path.getTotalLength();
      /*全体の距離から、現在進めている距離を算出して、そこから、現在点を導き出す*/
      var point = path.getPointAtLength(advanceLength),
          rotate = 0; //追加すべき角度
      if (this.rotate === "0") {
        return point.x+ "," +point.y;
      } else if (this.rotate === "auto") {
        rotate = 0;
      } else if (this.rotate === "auto-reverse") {
        rotate = 180;
      } else {
        rotate = +this.rotate;
      }
      return point.x+ "," +point.y + this.getRotate(path, advanceLength, rotate);
    },
    
    /*setValuesメソッドのオーバライド*/
    setValues: function() {
      var keyPoints = this.getAttr("keyPoints", null),
          /*$animateElementプロパティは下記のinitメソッドで上書きされているため、
           * $animateElementを別方法で呼び出す必要がある*/
          superSetValues = this.$animateElement.$animateElement.setValues;
      if (keyPoints) {
        return superSetValues.call(this, keyPoints, null, null, null);
      } else {
        return superSetValues.apply(this, arguments);
      }
    }
  } )
  .on("init", function (ele) {
    if (!ele || !ele.parentNode) {
     return;
    }
    /*type属性で変更されないように（$animateTransformElementのinitメソッドを参照のこと）*/
    this.type = "translate";
    /*isSumプロパティは常にtrueにしておく。animateTransform要素とは挙動が違うため
     * また、$aniamteTransformElementのtocallメソッド参照*/
    this.isSum = true;
    this.mode = this.getAttr("mode", "paced");
    this.rotate = this.getAttr("rotate", "0");
    /*isKeyPointsプロパティはkeyPoints属性が設定されていたら真*/
    this.isKeyPoints = ele.hasAttributeNS(null, "keyPoints");
    if (this.isKeyPoints && !ele.hasAttributeNS(null, "path")) {
      /*keyPoints属性がある場合は、path属性に指定がなければ、
       * values属性などの値をpath属性に書いておく*/
       var values = this.getAttr( "values", this.getAttr("from", "")+" L "+this.getAttr("to", "") );
       ele.setAttributeNS( null, "path", "M " +values.replace(/;/g, " L ") );
    }
    this.path = this.path.cloneNode(true);
    var mpath = ele.getElementsByTagNameNS(this.path.namespaceURI, "mpath");
    /*$animateは後で、プロパティを書き換えるために使う。tocallメソッドも参照*/
    var $animate = this.$animateElement;
    if (mpath.length) {
      var p = ele.ownerDocument.getElementById(
        mpath[0].getAttributeNS("http://www.w3.org/1999/xlink", "href").slice(1)
      );
      p && this.path.setAttributeNS(null, "d", p.getAttributeNS(null, "d"));
      this.$animateElement = $animate.up().mix ( {tocall: this._tocallForPath} );
    } else if (ele.hasAttributeNS(null, "path")) {
      this.path.setAttributeNS(null, "d", ele.getAttributeNS(null, "path"));
      this.$animateElement = $animate.up().mix ( {tocall: this._tocallForPath} );
    }
  } );

base("$getDocument").mix ( function() {

  function getDocument() 
  {
    var svg = document.getElementsByTagName("object"),
        svgns = "http://www.w3.org/2000/svg";
    if (svg) {
      for (var i=0;i<svg.length;++i) {
        getElement( svg[i].getSVGDocument() );
      }
    }
    /*SVG文書から呼び出されたときも処理する*/
    getElement(document);
    /*idはアニメの中止ハンドル*/
    var id = __step(),
        idstop = function() {
          /*アニメーションを中止する関数*/
          window.cancelAnimationFrame && cancelAnimationFrame(id);
        };
    base("$frame").on("pauseAnimation", idstop);
    window.addEventListener("unload", idstop);
    
    /*文書からアニメーション関連要素を取り出して、オブジェクトを初期化*/
    function getElement (svgDoc) {
        var $set = base("$calcMode").$attribute.$setElement,
            $animate = $set.$animateElement,
            frame = base("$frame");
        init($set, "set");
        init($animate, "animate");
        init($animate.up(), "animateColor");
        init($animate.$animateTransformElement, "animateTransform");
        init($animate.$animateTransformElement.$motionElement, "animateMotion");
          /*リンクのハッシュ読み取りで、ハイパーリンクのイベント処理
         * たとえば、a要素のxlink:href="#hoge"で、<animate id="hoge"のとき、
         * animate要素がハイパーリンク作動と同時に動くようになる
         * 
         * ただし、SMIL アニメーションの仕様では、
         * animate要素の開始時刻まで、時を進める操作をするだけ*/
         svgDoc.defaultView.addEventListener("hashchange", function() {
             var hash = svgDoc.defaultView.location.hash.slice(1);
             svgDoc.getElementById(hash).beginElement();
           });
        function init (obj, name) {
          /*あとでframe.initializeメソッドで呼び出すために準備しておく*/
          var elist = svgDoc.getElementsByTagNameNS(svgns, name);
          obj.numberOfElemList = 0;
          if (elist.length > 0) {
            obj.elementList = elist;
            frame.objList.push(obj);
          }
          elist = obj = void 0;
        };
    };
  }
  
  window.addEventListener && window.addEventListener("load", getDocument);

  this.step = __step;
  function __step() {
    /*EdgeはhasFeatureメソッドでtrueを返す*/
    if (!document.implementation.hasFeature("http://www.w3.org/TR/SVG11/feature#Animation", "1.1")
        || (window.navigator.userAgent.toLowerCase().indexOf("edge") > 0)) {
      if (window.requestAnimationFrame && requestAnimationFrame) {
        /*IE11やEdgeなどSMILアニメーションに対応していないブラウザ用*/
        /*cancelはアニメーションの中止ハンドル*/
        var cancel = {
           handle: null
          };
        (function(frame) {
          var _cancel = cancel; /*cancelのエイリアス*/
          var step = function () {
            if (!this.isPaused) {
              frame++;
              try {
                this.initialize();
                this.setFrame(frame);
              } catch(e) {
              }
              _cancel.handle = requestAnimationFrame(step);
            }
          }.bind(base("$frame"));
          _cancel.handle = requestAnimationFrame(step);
        })(-1);
        return cancel;
      } else {
        setInterval( (function(frame) {
          var $f = base("$frame");
          return function () {
            frame++;
            $f.initialize();
            $f.setFrame(frame);
          };
        })(-1), 1 );
      }
    }
  }
} );
//#endif // _SMIL_IDL_
