/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;

/**
 * プリミティブ型の拡大変換および縮小変換と同等の処理をラッパーオブジェクトに対して提供します。<br>
 * また任意精度の数値型との拡大変換および縮小変換を拡張機能として提供します。
 */
final class Widening {

    static final BigInteger LONG_MAX_VALUE = BigInteger.valueOf(Long.MAX_VALUE);

    static final BigInteger LONG_MIN_VALUE = BigInteger.valueOf(Long.MIN_VALUE);

    static final BigInteger INT_MAX_VALUE = BigInteger.valueOf(Integer.MAX_VALUE);

    static final BigInteger INT_MIN_VALUE = BigInteger.valueOf(Integer.MIN_VALUE);

    static final BigInteger SHORT_MAX_VALUE = BigInteger.valueOf(Short.MAX_VALUE);

    static final BigInteger SHORT_MIN_VALUE = BigInteger.valueOf(Short.MIN_VALUE);

    static final BigInteger BYTE_MAX_VALUE = BigInteger.valueOf(Byte.MAX_VALUE);

    static final BigInteger BYTE_MIN_VALUE = BigInteger.valueOf(Byte.MIN_VALUE);

    /** BigDecimal 型を示す。 */
    public static final int TYPE_BIGDECIMAL = 0;

    /** BigInteger 型を示す。 */
    public static final int TYPE_BIGINTEGER = 1;

    /** Double 型を示す。 */
    public static final int TYPE_DOUBLE = 2;

    /** Float 型を示す。 */
    public static final int TYPE_FLOAT = 3;

    /** Long 型を示す。 */
    public static final int TYPE_LONG = 4;

    /** Integer 型を示す。 */
    public static final int TYPE_INTEGER = 5;

    /**
     * Widening Primitive Conversion. <br>
     * <br>
     * byte to short, int, long, float, or double. <br>
     * short to int, long, float, or double. <br>
     * char to int, long, float, or double. <br>
     * int to long, float, or double. <br>
     * long to float or double. <br>
     * float to double. <br>
     */

    /**
     * Narrowing Primitive Conversions. <br>
     * <br>
     * byte to char. <br>
     * short to byte or char. <br>
     * char to byte or short. <br>
     * int to byte, short, or char. <br>
     * long to byte, short, char, or int. <br>
     * float to byte, short, char, int, or long. <br>
     * double to byte, short, char, int, long, or float.
     */

    /*
     * WidenTypeCache
     */

    private static final Collection WidenShortTypeCache;
    static {
        WidenShortTypeCache = Collections.singleton(Byte.class);
    }

    private static final Collection WidenIntegerTypeCache;
    static {
        final Class[] types = new Class[] { Byte.class, Short.class };
        WidenIntegerTypeCache = Collections.unmodifiableCollection(new HashSet(Arrays.asList(types)));
    }

    private static final Collection WidenLongTypeCache;
    static {
        final Class[] types = new Class[] { Byte.class, Short.class, Integer.class };
        WidenLongTypeCache = Collections.unmodifiableCollection(new HashSet(Arrays.asList(types)));
    }

    private static final Collection WidenFloatTypeCache;
    static {
        final Class[] types = new Class[] { Byte.class, Short.class, Integer.class, Long.class };
        WidenFloatTypeCache = Collections.unmodifiableCollection(new HashSet(Arrays.asList(types)));
    }

    private static final Collection WidenDoubleTypeCache;
    static {
        final Class[] types = new Class[] { Byte.class, Short.class, Integer.class, Long.class, Float.class };
        WidenDoubleTypeCache = Collections.unmodifiableCollection(new HashSet(Arrays.asList(types)));
    }

    private static final Collection WidenBigIntegerTypeCache;
    static {
        final Class[] types = new Class[] { Byte.class, Short.class, Integer.class, Long.class };
        WidenBigIntegerTypeCache = Collections.unmodifiableCollection(new HashSet(Arrays.asList(types)));
    }

    private static final Collection WidenBigDecimalTypeCache;
    static {
        final Class[] types = new Class[] { Byte.class, Short.class, Integer.class, Long.class, Float.class, Double.class, BigInteger.class };
        WidenBigDecimalTypeCache = Collections.unmodifiableCollection(new HashSet(Arrays.asList(types)));
    }

    /*
     * NarrowTypeCache
     */

    private static final Collection NarrowByteTypeCache;
    static {
        final Class[] types = new Class[] { BigDecimal.class, BigInteger.class, Double.class, Float.class, Long.class, Integer.class, Short.class };
        NarrowByteTypeCache = Collections.unmodifiableCollection(new HashSet(Arrays.asList(types)));
    }

    private static final Collection NarrowShortTypeCache;
    static {
        final Class[] types = new Class[] { BigDecimal.class, BigInteger.class, Double.class, Float.class, Long.class, Integer.class };
        NarrowShortTypeCache = Collections.unmodifiableCollection(new HashSet(Arrays.asList(types)));
    }

    private static final Collection NarrowIntegerTypeCache;
    static {
        final Class[] types = new Class[] { BigDecimal.class, BigInteger.class, Double.class, Float.class, Long.class };
        NarrowIntegerTypeCache = Collections.unmodifiableCollection(new HashSet(Arrays.asList(types)));
    }

    private static final Collection NarrowLongTypeCache;
    static {
        final Class[] types = new Class[] { BigDecimal.class, BigInteger.class, Double.class, Float.class };
        NarrowLongTypeCache = Collections.unmodifiableCollection(new HashSet(Arrays.asList(types)));
    }

    private static final Collection NarrowFloatTypeCache;
    static {
        final Class[] types = new Class[] { BigDecimal.class, BigInteger.class, Double.class };
        NarrowFloatTypeCache = Collections.unmodifiableCollection(new HashSet(Arrays.asList(types)));
    }

    private static final Collection NarrowDoubleTypeCache;
    static {
        final Class[] types = new Class[] { BigDecimal.class, BigInteger.class };
        NarrowDoubleTypeCache = Collections.unmodifiableCollection(new HashSet(Arrays.asList(types)));
    }

    private static final Collection NarrowBigIntegerTypeCache;
    static {
        NarrowBigIntegerTypeCache = Collections.singleton(BigDecimal.class);
    }

    /*
     * Short
     */

    /**
     * Short への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static Short widenShort(Byte num) {
        return toShort(num);
    }

    /*
     * Integer
     */

    /**
     * Integer への拡大変換を行い返却します。
     * 
     * @param ch
     *            変換元
     * @return 拡大変換した数値
     */
    public static Integer widenInteger(Character ch) {
        return new Integer(ch.charValue());
    }

    /**
     * Integer への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static Integer widenInteger(Byte num) {
        return toInteger(num);
    }

    /**
     * Integer への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static Integer widenInteger(Short num) {
        return toInteger(num);
    }

    /*
     * Long
     */

    /**
     * Long への拡大変換を行い返却します。
     * 
     * @param ch
     *            変換元
     * @return 拡大変換した数値
     */
    public static Long widenLong(Character ch) {
        return new Long(ch.charValue());
    }

    /**
     * Long への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static Long widenLong(Byte num) {
        return toLong(num);
    }

    /**
     * Long への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static Long widenLong(Short num) {
        return toLong(num);
    }

    /**
     * Long への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static Long widenLong(Integer num) {
        return toLong(num);
    }

    /**
     * Long への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static BigInteger widenBigInteger(Long num) {
        return toBigInteger(num);
    }

    /*
     * BigInteger
     */
    /**
     * BigInteger への拡大変換を行い返却します。
     * 
     * @param ch
     *            変換元
     * @return 拡大変換した数値
     */
    public static BigInteger widenBigInteger(Character ch) {
        return BigInteger.valueOf(ch.charValue());
    }

    /**
     * BigInteger への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static BigInteger widenBigInteger(Byte num) {
        return toBigInteger(num);
    }

    /**
     * BigInteger への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static BigInteger widenBigInteger(Short num) {
        return toBigInteger(num);
    }

    /**
     * BigInteger への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static BigInteger widenBigInteger(Integer num) {
        return toBigInteger(num);
    }

    /*
     * Float
     */

    /**
     * Float への拡大変換を行い返却します。
     * 
     * @param ch
     *            変換元
     * @return 拡大変換した数値
     */
    public static Float widenFloat(Character ch) {
        return new Float(ch.charValue());
    }

    /**
     * Float への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static Float widenFloat(Byte num) {
        return toFloat(num);
    }

    /**
     * Float への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static Float widenFloat(Short num) {
        return toFloat(num);
    }

    /**
     * Float への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static Float widenFloat(Integer num) {
        return toFloat(num);
    }

    /**
     * Float への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static Float widenFloat(Long num) {
        return toFloat(num);
    }

    /*
     * Double
     */

    /**
     * Double への拡大変換を行い返却します。
     * 
     * @param ch
     *            変換元
     * @return 拡大変換した数値
     */
    public static Double widenDouble(Character ch) {
        return new Double(ch.charValue());
    }

    /**
     * Double への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static Double widenDouble(Byte num) {
        return toDouble(num);
    }

    /**
     * Double への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static Double widenDouble(Short num) {
        return toDouble(num);
    }

    /**
     * Double への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static Double widenDouble(Integer num) {
        return toDouble(num);
    }

    /**
     * Double への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static Double widenDouble(Long num) {
        return toDouble(num);
    }

    /**
     * Double への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static Double widenDouble(Float num) {
        return toDouble(num);
    }

    /*
     * BigDecimal
     */
    /**
     * BigDecimal への拡大変換を行い返却します。
     * 
     * @param ch
     *            変換元
     * @return 拡大変換した数値
     */
    public static BigDecimal widenBigDecimal(Character ch) {
        return BigDecimal.valueOf(ch.charValue());
    }

    /**
     * BigDecimal への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static BigDecimal widenBigDecimal(Byte num) {
        return toBigDecimal(num);
    }

    /**
     * BigDecimal への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static BigDecimal widenBigDecimal(Short num) {
        return toBigDecimal(num);
    }

    /**
     * BigDecimal への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static BigDecimal widenBigDecimal(Integer num) {
        return toBigDecimal(num);
    }

    /**
     * BigDecimal への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static BigDecimal widenBigDecimal(Long num) {
        return toBigDecimal(num);
    }

    /**
     * BigDecimal への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static BigDecimal widenBigDecimal(Float num) {
        return toBigDecimal(num);
    }

    /**
     * BigDecimal への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static BigDecimal widenBigDecimal(Double num) {
        return toBigDecimal(num);
    }

    /**
     * BigDecimal への拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 拡大変換した数値
     */
    public static BigDecimal widenBigDecimal(BigInteger num) {
        return toBigDecimal(num);
    }

    /*
     * Number Type
     */

    /**
     * 指定されたクラス型に拡大変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @param forType
     *            変換する数値型
     * @return 拡大変換した数値
     * @throws IllegalArgumentException
     *             引数が指定の型への拡大変換に無効な数値型の場合
     */
    public static Number widenType(Number num, Class forType) {
        final Class numType = num.getClass();
        if (BigDecimal.class.equals(forType)) {
            if (WidenBigDecimalTypeCache.contains(numType)) {
                return toBigDecimal(num);
            }
        } else if (BigInteger.class.equals(forType)) {
            if (WidenBigIntegerTypeCache.contains(numType)) {
                return toBigInteger(num);
            }
        } else if (Double.class.equals(forType)) {
            if (WidenDoubleTypeCache.contains(numType)) {
                return toDouble(num);
            }
        } else if (Float.class.equals(forType)) {
            if (WidenFloatTypeCache.contains(numType)) {
                return toFloat(num);
            }
        } else if (Long.class.equals(forType)) {
            if (WidenLongTypeCache.contains(numType)) {
                return toLong(num);
            }
        } else if (Integer.class.equals(forType)) {
            if (WidenIntegerTypeCache.contains(numType)) {
                return toInteger(num);
            }
        } else if (Short.class.equals(forType)) {
            if (WidenShortTypeCache.contains(numType)) {
                return toShort(num);
            }
        }
        throw new IllegalArgumentException("widening error. " + num + "," + forType);
    }

    /*
     * Character
     */

    /**
     * Character への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Character narrowCharacter(Byte num) {
        return toCharacter(num);
    }

    /**
     * Character への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Character narrowCharacter(Short num) {
        return toCharacter(num);
    }

    /**
     * Character への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Character narrowCharacter(Integer num) {
        return toCharacter(num);
    }

    /**
     * Character への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Character narrowCharacter(Long num) {
        return toCharacter(num);
    }

    /**
     * Character への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Character narrowCharacter(Float num) {
        return toCharacter(num);
    }

    /**
     * Character への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Character narrowCharacter(Double num) {
        return toCharacter(num);
    }

    /**
     * Character への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Character narrowCharacter(BigInteger num) {
        return toCharacter(num);
    }

    /**
     * Character への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Character narrowCharacter(BigDecimal num) {
        return toCharacter(num);
    }

    /*
     * Byte
     */

    /**
     * Byte への縮小変換を行い返却します。
     * 
     * @param ch
     *            変換元
     * @return 縮小変換した数値
     */
    public static Byte narrowByte(Character ch) {
        return new Byte((byte) ch.charValue());
    }

    /**
     * Byte への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Byte narrowByte(Short num) {
        return toByte(num);
    }

    /**
     * Byte への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Byte narrowByte(Integer num) {
        return toByte(num);
    }

    /**
     * Byte への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Byte narrowByte(Long num) {
        return toByte(num);
    }

    /**
     * Byte への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Byte narrowByte(Float num) {
        return toByte(num);
    }

    /**
     * Byte への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Byte narrowByte(Double num) {
        return toByte(num);
    }

    /**
     * Byte への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Byte narrowByte(BigInteger num) {
        return toByte(num);
    }

    /**
     * Byte への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Byte narrowByte(BigDecimal num) {
        return toByte(num);
    }

    /*
     * Short
     */
    /**
     * Short への縮小変換を行い返却します。
     * 
     * @param ch
     *            変換元
     * @return 縮小変換した数値
     */
    public static Short narrowShort(Character ch) {
        return new Short((short) ch.charValue());
    }

    /**
     * Short への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Short narrowShort(Integer num) {
        return toShort(num);
    }

    /**
     * Short への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Short narrowShort(Long num) {
        return toShort(num);
    }

    /**
     * Short への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Short narrowShort(Float num) {
        return toShort(num);
    }

    /**
     * Short への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Short narrowShort(Double num) {
        return toShort(num);
    }

    /**
     * Short への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Short narrowShort(BigInteger num) {
        return toShort(num);
    }

    /**
     * Short への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Short narrowShort(BigDecimal num) {
        return toShort(num);
    }

    /*
     * Integer
     */
    /**
     * Integer への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Integer narrowInteger(Long num) {
        return toInteger(num);
    }

    /**
     * Integer への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Integer narrowInteger(Float num) {
        return toInteger(num);
    }

    /**
     * Integer への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Integer narrowInteger(Double num) {
        return toInteger(num);
    }

    /**
     * Integer への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Integer narrowInteger(BigInteger num) {
        return toInteger(num);
    }

    /**
     * Integer への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Integer narrowInteger(BigDecimal num) {
        return toInteger(num);
    }

    /*
     * Long
     */
    /**
     * Long への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Long narrowLong(Float num) {
        return toLong(num);
    }

    /**
     * Long への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Long narrowLong(Double num) {
        return toLong(num);
    }

    /**
     * Long への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Long narrowLong(BigInteger num) {
        return toLong(num);
    }

    /**
     * Long への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Long narrowLong(BigDecimal num) {
        return toLong(num);
    }

    /*
     * Float
     */
    /**
     * Float への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Float narrowFloat(Double num) {
        return toFloat(num);
    }

    /**
     * Float への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Float narrowFloat(BigInteger num) {
        return toFloat(num);
    }

    /**
     * Float への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Float narrowFloat(BigDecimal num) {
        return toFloat(num);
    }

    /*
     * Double
     */

    /**
     * Double への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Double narrowDouble(BigInteger num) {
        return toDouble(num);
    }

    /**
     * Double への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static Double narrowDouble(BigDecimal num) {
        return toDouble(num);
    }

    /*
     * BigInteger
     */

    /**
     * BigInteger への縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @return 縮小変換した数値
     */
    public static BigInteger narrowBigInteger(BigDecimal num) {
        return toBigInteger(num);
    }

    /*
     * Number Type
     */

    /**
     * 指定されたクラス型に縮小変換を行い返却します。
     * 
     * @param num
     *            変換元
     * @param forType
     *            変換する数値型
     * @return 縮小変換した数値
     * @throws IllegalArgumentException
     *             引数が指定の型への縮小変換に無効な数値型の場合
     */
    public static Number narrowType(Number num, Class forType) {
        final Class numType = num.getClass();
        if (Byte.class.equals(forType)) {
            if (NarrowByteTypeCache.contains(numType)) {
                return toByte(num);
            }
        } else if (Short.class.equals(forType)) {
            if (NarrowShortTypeCache.contains(numType)) {
                return toShort(num);
            }
        } else if (Integer.class.equals(forType)) {
            if (NarrowIntegerTypeCache.contains(numType)) {
                return toInteger(num);
            }
        } else if (Long.class.equals(forType)) {
            if (NarrowLongTypeCache.contains(numType)) {
                return toLong(num);
            }
        } else if (Float.class.equals(forType)) {
            if (NarrowFloatTypeCache.contains(numType)) {
                return toFloat(num);
            }
        } else if (Double.class.equals(forType)) {
            if (NarrowDoubleTypeCache.contains(numType)) {
                return toDouble(num);
            }
        } else if (BigInteger.class.equals(forType)) {
            if (NarrowBigIntegerTypeCache.contains(numType)) {
                return toBigInteger(num);
            }
        }
        throw new IllegalArgumentException("narrowing error. " + num + "," + forType);
    }

    /*
     * isNarrow
     */

    /**
     * 精度を落とさずに Byte への縮小変換が可能か検証します。
     * 
     * @param num
     *            検証する数値
     * @return 精度を落とさずに変換が可能の場合は true
     */
    public static boolean isNarrowByte(Short num) {
        final int n = num.intValue();
        return (Byte.MIN_VALUE <= n && n <= Byte.MAX_VALUE);
    }

    /**
     * 精度を落とさずに Byte への縮小変換が可能か検証します。
     * 
     * @param num
     *            検証する数値
     * @return 精度を落とさずに変換が可能の場合は true
     */
    public static boolean isNarrowByte(Integer num) {
        final int n = num.intValue();
        return (Byte.MIN_VALUE <= n && n <= Byte.MAX_VALUE);
    }

    /**
     * 精度を落とさずに Short への縮小変換が可能か検証します。
     * 
     * @param num
     *            検証する数値
     * @return 精度を落とさずに変換が可能の場合は true
     */
    public static boolean isNarrowShort(Integer num) {
        final int n = num.intValue();
        return (Short.MIN_VALUE <= n && n <= Short.MAX_VALUE);
    }

    /**
     * 精度を落とさずに Byte への縮小変換が可能か検証します。
     * 
     * @param num
     *            検証する数値
     * @return 精度を落とさずに変換が可能の場合は true
     */
    public static boolean isNarrowByte(Long num) {
        final long n = num.longValue();
        return (Byte.MIN_VALUE <= n && n <= Byte.MAX_VALUE);
    }

    /**
     * 精度を落とさずに Short への縮小変換が可能か検証します。
     * 
     * @param num
     *            検証する数値
     * @return 精度を落とさずに変換が可能の場合は true
     */
    public static boolean isNarrowShort(Long num) {
        final long n = num.longValue();
        return (Short.MIN_VALUE <= n && n <= Short.MAX_VALUE);
    }

    /**
     * 精度を落とさずに Integer への縮小変換が可能か検証します。
     * 
     * @param num
     *            検証する数値
     * @return 精度を落とさずに変換が可能の場合は true
     */
    public static boolean isNarrowInteger(Long num) {
        final long n = num.longValue();
        return (Integer.MIN_VALUE <= n && n <= Integer.MAX_VALUE);
    }

    /**
     * 精度を落とさずに Byte への縮小変換が可能か検証します。
     * 
     * @param num
     *            検証する数値
     * @return 精度を落とさずに変換が可能の場合は true
     */
    public static boolean isNarrowByte(BigInteger num) {
        return (num.compareTo(BYTE_MIN_VALUE) >= 0 && num.compareTo(BYTE_MAX_VALUE) <= 0);
    }

    /**
     * 精度を落とさずに Short への縮小変換が可能か検証します。
     * 
     * @param num
     *            検証する数値
     * @return 精度を落とさずに変換が可能の場合は true
     */
    public static boolean isNarrowShort(BigInteger num) {
        return (num.compareTo(SHORT_MIN_VALUE) >= 0 && num.compareTo(SHORT_MAX_VALUE) <= 0);
    }

    /**
     * 精度を落とさずに Integer への縮小変換が可能か検証します。
     * 
     * @param num
     *            検証する数値
     * @return 精度を落とさずに変換が可能の場合は true
     */
    public static boolean isNarrowInteger(BigInteger num) {
        return (num.compareTo(INT_MIN_VALUE) >= 0 && num.compareTo(INT_MAX_VALUE) <= 0);
    }

    /**
     * 精度を落とさずに Integer への縮小変換が可能か検証します。
     * 
     * @param num
     *            検証する数値
     * @return 精度を落とさずに変換が可能の場合は true
     */
    public static boolean isNarrowLong(BigInteger num) {
        return (num.compareTo(LONG_MIN_VALUE) >= 0 && num.compareTo(LONG_MAX_VALUE) <= 0);
    }

    /*
     * Arithmetic Widening
     */

    /**
     * 算術計算のための拡大変換を行い配列の内要素に結果を格納し、変換型を返却します。<br>
     * 引数には算術計算の対象となる２つの数値を格納して渡します。<br>
     * 変換の規則は、任意制度の数値型以外は Java のプリミティブ変換と同様です。 <br>
     * <br>
     * このメソッドでは機能拡張として任意精度の数値型がサポートされており変換規則は以下とします。<br>
     * 1. 双方が BigInteger の場合は変換されないものとする。<br>
     * 2. 双方が BigDecimal の場合は変換されないものとする。<br>
     * 3. 片方が BigInteger であり、片方が (Byte|Short|Integer|Long) の場合は、BigInteger に変換する。<br>
     * 4. 片方が BigInteger であり、片方が (Float|Double) の場合は、双方を BigDecimal に変換する。<br>
     * 5. 片方が BigDecimal であり、片方が (Byte|Short|Integer|Long|Float|Double|BigInteger) の場合は、BigDecimal に変換する。<br>
     * 
     * @param arithmetic
     *            算術計算の対象となる変換元の数値
     * @return 変換結果の数値型
     * @throws IllegalArgumentException
     *             引数のサイズが２つより小さい場合、または引数が無効な数値型の場合
     */
    public static int arithmetic(Number[] arithmetic) {

        if (arithmetic == null) {
            throw new IllegalArgumentException("widening error. arithmetic is null.");
        }
        if (arithmetic.length < 2) {
            throw new IllegalArgumentException("widening error. arithmetic.length=" + arithmetic.length);
        }

        final Number n1 = arithmetic[0];
        final Number n2 = arithmetic[1];
        // BigDecimal
        if (isBigDecimalOperand(n1, n2)) {
            arithmetic[0] = toBigDecimal(n1);
            arithmetic[1] = toBigDecimal(n2);
            return TYPE_BIGDECIMAL;
        } else
        // BigInteger
        if (n1 instanceof BigInteger || n2 instanceof BigInteger) {
            arithmetic[0] = toBigInteger(n1);
            arithmetic[1] = toBigInteger(n2);
            return TYPE_BIGINTEGER;
        } else
        // Double
        if (n1 instanceof Double || n2 instanceof Double) {
            arithmetic[0] = toDouble(n1);
            arithmetic[1] = toDouble(n2);
            return TYPE_DOUBLE;
        } else
        // Float
        if (n1 instanceof Float || n2 instanceof Float) {
            arithmetic[0] = toFloat(n1);
            arithmetic[1] = toFloat(n2);
            return TYPE_FLOAT;
        } else
        // Long
        if (n1 instanceof Long || n2 instanceof Long) {
            arithmetic[0] = toLong(n1);
            arithmetic[1] = toLong(n2);
            return TYPE_LONG;
        } else
        // Integer
        if (isIntegerOperand(n1, n2)) {
            arithmetic[0] = toInteger(n1);
            arithmetic[1] = toInteger(n2);
            return TYPE_INTEGER;
        } else {
            throw new IllegalArgumentException("widening error. arithmetic= [" + n1 + "," + n2 + "]");
        }
    }

    private static boolean isIntegerOperand(Number n1, Number n2) {
        return isIntegerOperand(n1) && isIntegerOperand(n2);
    }

    private static boolean isIntegerOperand(Number n) {
        return (n instanceof Integer || n instanceof Short || n instanceof Byte);
    }

    private static boolean isBigDecimalOperand(Number n1, Number n2) {
        // BigDecimal
        if (n1 instanceof BigDecimal || n2 instanceof BigDecimal) {
            return true;
        } else
        // BigInteger or (Double | Float)
        if (n1 instanceof BigInteger && (n2 instanceof Double || n2 instanceof Float)) {
            return true;
        } else
        // BigInteger or (Double | Float)
        if (n2 instanceof BigInteger && (n1 instanceof Double || n1 instanceof Float)) {
            return true;
        }
        return false;
    }

    /*
     * Type Conversion
     */

    private static Character toCharacter(Number num) {
        return new Character((char) num.intValue());
    }

    private static Byte toByte(Number num) {
        return (num instanceof Byte) ? (Byte) num : new Byte(num.byteValue());
    }

    private static Short toShort(Number num) {
        return (num instanceof Short) ? (Short) num : new Short(num.shortValue());
    }

    private static Integer toInteger(Number num) {
        return (num instanceof Integer) ? (Integer) num : new Integer(num.intValue());
    }

    private static Long toLong(Number num) {
        return (num instanceof Long) ? (Long) num : new Long(num.longValue());
    }

    private static Float toFloat(Number num) {
        return (num instanceof Float) ? (Float) num : new Float(num.floatValue());
    }

    private static Double toDouble(Number num) {
        return (num instanceof Double) ? (Double) num : new Double(num.doubleValue());
    }

    private static BigInteger toBigInteger(Number num) {
        return (num instanceof BigInteger) ? (BigInteger) num : toBigDecimal(num).toBigInteger();
    }

    private static BigDecimal toBigDecimal(Number num) {
        if (num instanceof BigDecimal) {
            return (BigDecimal) num;
        } else if (num instanceof BigInteger) {
            return new BigDecimal((BigInteger) num);
        } else {
            return new BigDecimal(num.toString());
        }
    }

}
