/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.lang.reflect.Array;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * ユーティリティを提供します。
 */
class Utils {

    /**
     * 引数が null または空の文字シーケンスであるかを評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @return 引数が null または空の文字シーケンスの場合は true
     */
    static boolean isEmpty(CharSequence cs) {
        return (cs == null || cs.length() == 0);
    }

    /**
     * 引数に null 値が含まれるか評価します。
     * 
     * @param o
     *            評価する一番目の値
     * @param o2
     *            評価する二番目の値
     * @return 引数に null 値が含まれる場合は true
     */
    static boolean isOrNull(Object o, Object o2) {
        return (o == null || o2 == null);
    }

    /**
     * 引数に null 値が含まれるか評価します。
     * 
     * @param o
     *            評価する一番目の値
     * @param o2
     *            評価する二番目の値
     * @param o3
     *            評価する三番目の値
     * @return 引数に null 値が含まれる場合は true
     */
    static boolean isOrNull(Object o, Object o2, Object o3) {
        return (o == null || o2 == null || o3 == null);
    }

    /**
     * 指定の文字列に指定の文字が含まれているか評価します。
     * 
     * @param source
     *            評価する文字列
     * @param c
     *            検索する文字
     * @return 文字列に指定の文字が含まれている場合は true
     */
    static boolean isContains(String source, char c) {
        return (source != null && -1 < source.indexOf(c));
    }

    /**
     * 数値の絶対値が無限量か、または非数 (NaN) であるか評価します。
     * 
     * @param val
     *            評価する数値
     * @return 数値の絶対値が無限量か、または非数 (NaN) の場合は true
     */
    static boolean isInfiniteOrNaN(Float val) {
        return (val.isInfinite() || val.isNaN());
    }

    /**
     * 数値の絶対値が無限量か、または非数 (NaN) であるか評価します。
     * 
     * @param val
     *            評価する数値
     * @return 数値の絶対値が無限量か、または非数 (NaN) の場合は true
     */
    static boolean isInfiniteOrNaN(Double val) {
        return (val.isInfinite() || val.isNaN());
    }

    /*
     * class
     */

    /**
     * ログ出力用のオブジェクトのクラスの文字列を生成します。
     * 
     * @param o
     *            要素
     * @return ログ出力用のクラス文字列
     */
    static String clazz(Object o) {
        return (null == o) ? String.valueOf(null) : o.getClass().toString();
    }

    /**
     * ログ出力用のオブジェクトのクラスの文字列を生成します。
     * 
     * @param msg
     *            メッセージ
     * @param o
     *            要素
     * @return ログ出力用のクラス文字列
     */
    static String clazz(String msg, Object o) {
        return (new StringBuffer(msg)).append(clazz(o)).toString();
    }

    /*
     * log
     */

    /**
     * ログ出力用の文字列を生成します。
     * 
     * @param msg
     *            メッセージ
     * @param o
     *            要素
     * @return ログ出力用の文字列
     */
    static String log(String msg, Object o) {
        return (new StringBuffer(msg)).append(o).toString();
    }

    /**
     * ログ出力用の文字列を生成します。
     * 
     * @param msg
     *            メッセージ
     * @param i
     *            要素１
     * @return ログ出力用の文字列
     */
    static String log(String msg, int i) {
        return (new StringBuffer(msg)).append(i).toString();
    }

    /**
     * ログ出力用の文字列を生成します、第二引数以降はコンマで区切られます。
     * 
     * @param msg
     *            メッセージ
     * @param o
     *            要素１
     * @param i
     *            要素２
     * @return ログ出力用の文字列
     */
    static String log(String msg, Object o, int i) {
        return (new StringBuffer(msg)).append(o).append(", ").append(i).toString();
    }

    /**
     * ログ出力用の文字列を生成します、第二引数以降はコンマで区切られます。
     * 
     * @param msg
     *            メッセージ
     * @param o
     *            要素１
     * @param o1
     *            要素２
     * @return ログ出力用の文字列
     */
    static String log(String msg, Object o, Object o1) {
        return (new StringBuffer(msg)).append(o).append(", ").append(o1).toString();
    }

    /**
     * ログ出力用の文字列を生成します、第二引数以降はコンマで区切られます。
     * 
     * @param msg
     *            メッセージ
     * @param o
     *            要素１
     * @param o1
     *            要素２
     * @param o2
     *            要素３
     * @return ログ出力用の文字列
     */
    static String log(String msg, Object o, Object o1, Object o2) {
        return (new StringBuffer(msg)).append(o).append(", ").append(o1).append(", ").append(o2).toString();
    }

    /**
     * ログ出力用の文字列を生成します、第二引数以降はコンマで区切られます。
     * 
     * @param msg
     *            メッセージ
     * @param o
     *            要素１
     * @param o1
     *            要素２
     * @param o2
     *            要素３
     * @param o3
     *            要素４
     * @return ログ出力用の文字列
     */
    static String log(String msg, Object o, Object o1, Object o2, Object o3) {
        return (new StringBuffer(msg)).append(o).append(", ").append(o1).append(", ").append(o2).append(", ").append(o3).toString();
    }

    /* オブジェクト型の配列シンボル。 */
    static final char OBJECT_SYMBOL = 'L';

    /* プリミティブ型のクラスと省略名のマッピング。 */
    static final Map PRIMITIVE_FOR_TYPE_MAP;
    static {
        final HashMap m = new HashMap();
        // primitive
        m.put(Byte.TYPE.getName(), Byte.TYPE);
        m.put(Short.TYPE.getName(), Short.TYPE);
        m.put(Integer.TYPE.getName(), Integer.TYPE);
        m.put(Long.TYPE.getName(), Long.TYPE);
        m.put(Float.TYPE.getName(), Float.TYPE);
        m.put(Double.TYPE.getName(), Double.TYPE);
        m.put(Boolean.TYPE.getName(), Boolean.TYPE);
        m.put(Character.TYPE.getName(), Character.TYPE);
        m.put(Void.TYPE.getName(), Void.TYPE);
        PRIMITIVE_FOR_TYPE_MAP = Collections.unmodifiableMap(m);
    }

    /* プリミティブ型のクラスとクラス名のマッピング。 */
    static final Map PRIMITIVE_CLASS_NAME_MAP;
    static {
        final HashMap m = new HashMap();
        // primitive
        m.put(Byte.TYPE, "B");
        m.put(Short.TYPE, "S");
        m.put(Integer.TYPE, "I");
        m.put(Long.TYPE, "J");
        m.put(Float.TYPE, "F");
        m.put(Double.TYPE, "D");
        m.put(Boolean.TYPE, "Z");
        m.put(Character.TYPE, "C");
        m.put(Void.TYPE, "V");
        PRIMITIVE_CLASS_NAME_MAP = Collections.unmodifiableMap(m);
    }

    /**
     * パッケージ名を含まないクラス名を返却します。
     * 
     * @param clazz
     *            クラス
     * @return パッケージ名を含まないクラス名
     */
    static String getShortClassName(Class clazz) {
        final String s = clazz.getName();
        final int off = s.lastIndexOf('.');
        final int beginSize = (-1 < off) ? off + 1 : 0;
        return s.substring(beginSize, s.length());
    }

    /**
     * 指定のクラス名からクラスを検索し、クラス参照を返却します。 <br>
     * 取得に失敗した場合は null を返却します。
     * 
     * @param className
     *            クラス名
     * @param loader
     *            クラスローダ
     * @return クラス参照
     */
    static Class loadClass(String className, ClassLoader loader) {
        ClassLoader classLoader = loader;
        // Class.forName：指定クラスの ClassLoader
        try {
            if (classLoader != null) {
                return Class.forName(className, true, loader);
            } else {
                return Class.forName(className);
            }
        } catch (final Exception e) {
            // no op
        }
        // Current Thread の ClassLoader から取得
        try {
            classLoader = getContextClassLoader();
            if (classLoader != null) {
                return classLoader.loadClass(className);
            }
        } catch (final Exception e) {
            // no op
        }
        // システムクラスローダを使って検索します。
        try {
            classLoader = ClassLoader.getSystemClassLoader();
            return classLoader.loadClass(className);
        } catch (final Exception e) {
            // no op
        }
        return null;
    }

    /* 現在の Thread のコンテキスト ClassLoader を返却します。 */
    private static ClassLoader getContextClassLoader() {
        return Thread.currentThread().getContextClassLoader();
    }

    /**
     * コレクションを指定されたクラスの配列に変換します、プリミティブ型の生成を可能とします。
     * 
     * @param coll
     *            生成基のコレクション
     * @param type
     *            配列の基クラス
     * @return コレクションと同じ要素を持つ配列
     */
    static Object toArray(Collection coll, Class type) {
        final Object a = Array.newInstance(type, coll.size());
        if (type.isPrimitive()) {
            int i = 0;
            final Iterator iter = coll.iterator();
            while (iter.hasNext()) {
                Array.set(a, i++, iter.next());
            }
        } else {
            coll.toArray((Object[]) a);
        }
        return a;
    }

}
