/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * OGDLの実行環境を提供します。
 */
class OgdlRuntime {

    /* 単一の式を評価して返却します。 */
    static Object evaluate(OgdlEvent ev) {
        try {

            ev.shiftSpace();
            Object rvalue = OgdlParser.evaluate(ev);
            if (ev.hasNext()) {
                if (ev.isNotNextSpace()) {
                    rvalue = OgdlParser.evaluatePostfixExpression(ev, rvalue);
                }
                ev.shiftSpace();
            }
            return rvalue;

        } catch (final OgdlSyntaxException e) {
            throw e.throwFor(ev);
        } catch (final IndexOutOfBoundsException e) {
            throw new OgdlSyntaxException(ev, "syntax err.", e);
        } catch (final RuntimeException e) {
            throw new OgdlSyntaxException(ev, "runtime err", e);
        }
    }

    /* 囲み文字を検証して、式をコンマ区切りのリストとして評価して返却します。 */
    static List evaluateList(OgdlEvent ev) {
        try {

            ev.shiftSpace();
            final List rvalue = OgdlParser.evaluateEncloseProgressionList(ev.get(new ArrayList()));
            ev.shiftSpace();
            return rvalue;

        } catch (final OgdlSyntaxException e) {
            throw e.throwFor(ev);
        } catch (final IndexOutOfBoundsException e) {
            throw new OgdlSyntaxException(ev, "syntax err.", e);
        } catch (final RuntimeException e) {
            throw new OgdlSyntaxException(ev, "runtime err", e);
        }
    }

    /* 終了文字のみ検証して、式をコンマ区切りのリストとして評価して返却します。 */
    static List evaluateCloseList(OgdlEvent ev) {
        try {

            ev.shiftSpace();
            final List rvalue = OgdlParser.evaluateCloseProgressionList(ev.get(new ArrayList()));
            ev.shiftSpace();
            return rvalue;

        } catch (final OgdlSyntaxException e) {
            throw e.throwFor(ev);
        } catch (final IndexOutOfBoundsException e) {
            throw new OgdlSyntaxException(ev, "syntax err.", e);
        } catch (final RuntimeException e) {
            throw new OgdlSyntaxException(ev, "runtime err", e);
        }
    }

    /* 囲み文字を検証して、式をコンマ区切りのセットとして評価して返却します。 */
    static Set evaluateSet(OgdlEvent ev) {
        try {

            ev.shiftSpace();
            final Set rvalue = OgdlParser.evaluateEncloseProgressionSet(ev.get(new LinkedHashSet()));
            ev.shiftSpace();
            return rvalue;

        } catch (final OgdlSyntaxException e) {
            throw e.throwFor(ev);
        } catch (final IndexOutOfBoundsException e) {
            throw new OgdlSyntaxException(ev, "syntax err.", e);
        } catch (final RuntimeException e) {
            throw new OgdlSyntaxException(ev, "runtime err", e);
        }
    }

    /* 終了文字のみ検証して、式をコンマ区切りのセットとして評価して返却します。 */
    static Set evaluateCloseSet(OgdlEvent ev) {
        try {

            ev.shiftSpace();
            final Set rvalue = OgdlParser.evaluateCloseProgressionSet(ev.get(new LinkedHashSet()));
            ev.shiftSpace();
            return rvalue;

        } catch (final OgdlSyntaxException e) {
            throw e.throwFor(ev);
        } catch (final IndexOutOfBoundsException e) {
            throw new OgdlSyntaxException(ev, "syntax err.", e);
        } catch (final RuntimeException e) {
            throw new OgdlSyntaxException(ev, "runtime err", e);
        }
    }

    /* 囲み文字を検証して、式を '=' で対応付けられたマッピングをコンマ区切りで列挙したマップとして評価して返却します。 */
    static Map evaluateMap(OgdlEvent ev) {
        try {

            ev.shiftSpace();
            final Map rvalue = OgdlParser.evaluateEncloseMap(ev.get(new LinkedHashMap()));
            ev.shiftSpace();
            return rvalue;

        } catch (final OgdlSyntaxException e) {
            throw e.throwFor(ev);
        } catch (final IndexOutOfBoundsException e) {
            throw new OgdlSyntaxException(ev, "syntax err.", e);
        } catch (final RuntimeException e) {
            throw new OgdlSyntaxException(ev, "runtime err", e);
        }
    }

    /* 終了文字のみ検証して、式を '=' で対応付けられたマッピングをコンマ区切りで列挙したマップとして評価して返却します。 */
    static Map evaluateCloseMap(OgdlEvent ev) {
        try {

            ev.shiftSpace();
            final Map rvalue = OgdlParser.evaluateCloseMap(ev.get(new LinkedHashMap()));
            ev.shiftSpace();
            return rvalue;

        } catch (final OgdlSyntaxException e) {
            throw e.throwFor(ev);
        } catch (final IndexOutOfBoundsException e) {
            throw new OgdlSyntaxException(ev, "syntax err.", e);
        } catch (final RuntimeException e) {
            throw new OgdlSyntaxException(ev, "runtime err", e);
        }
    }

    static Class evaluateClass(OgdlEvent ev) {
        try {

            ev.shiftSpace();
            final Class rvalue = OgdlParser.evaluateClass(ev);
            ev.shiftSpace();
            return rvalue;

        } catch (final OgdlSyntaxException e) {
            throw e.throwFor(ev);
        } catch (final IndexOutOfBoundsException e) {
            throw new OgdlSyntaxException(ev, "syntax err.", e);
        } catch (final RuntimeException e) {
            throw new OgdlSyntaxException(ev, "runtime err", e);
        }
    }

}
