/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

/**
 * 基本的なデータ型の構文解析機能を提供します。
 */
class OgdlPrimitiveParser {

    /* 終了リテラル文字を検索して前置文字を削除した文字列を返却します。 */
    static String evaluateString(OgdlEvent ev) {
        if (OgdlSyntax.isEncloseOpenChar(ev)) {
            final String src = OgdlSyntax.cutEncloseString(ev);
            try {
                return ValueOf.decodeString(src);
            } catch (final IllegalArgumentException e) {
                throw new OgdlSyntaxException(ev, Utils.log("string format err. string=", src), e);
            }
        } else {
            throw new OgdlSyntaxException(ev, "is not enclose literal.");
        }
    }

    /* リテラル文字で囲まれた文字オブジェクトとして解析して返却します。 */
    static Character evaluateCharacter(OgdlEvent ev) {
        if (OgdlSyntax.isEncloseOpenChar(ev)) {
            final String src = OgdlSyntax.cutEncloseString(ev);
            try {
                return ValueOf.decodeCharacter(src);
            } catch (final IllegalArgumentException e) {
                throw new OgdlSyntaxException(ev, Utils.log("character format err. char=", src), e);
            }
        } else {
            throw new OgdlSyntaxException(ev, "is not enclose literal.");
        }
    }

    /* リテラル文字で囲まれた文字列を識別子とする、コンテキスト属性の参照を返却します。 */
    static Object evaluateReference(OgdlEvent ev) {
        if (OgdlSyntax.isEncloseOpenChar(ev)) {
            final String src = OgdlSyntax.cutEncloseString(ev);
            return ev.context.getAttribute(src);
        } else {
            final String s = OgdlSyntax.cutMemberName(ev);
            return ev.context.getAttribute(s);
        }
    }

    /* ISO書式の日付オブジェクトとして解析して返却します。 */
    static Object evaluateDateTime(OgdlEvent ev) throws OgdlSyntaxException {
        if (OgdlSyntax.isEncloseOpenChar(ev)) {
            final String src = OgdlSyntax.cutEncloseString(ev);
            try {
                return ValueOf.decodeDateTime(src);
            } catch (final IllegalArgumentException e) {
                throw new OgdlSyntaxException(ev, Utils.log("date time format err. format=", src), e);
            }
        } else {
            throw new OgdlSyntaxException(ev, "is not enclose literal.");
        }
    }

    /* リテラル文字で囲まれた正規表現パターンとして解析して返却します。 */
    static Pattern evaluateRegexPattern(OgdlEvent ev) throws OgdlSyntaxException {
        if (OgdlSyntax.isEncloseOpenChar(ev)) {
            final String src = OgdlSyntax.cutEncloseString(ev);
            try {
                return Pattern.compile(src);
            } catch (final PatternSyntaxException e) {
                throw new OgdlSyntaxException(ev, Utils.log("regex pattern err. pattern=", src), e);
            }
        } else {
            throw new OgdlSyntaxException(ev, "is not enclose literal.");
        }
    }

    /* 数値として解析して返却します。 */
    static Number evaluateNumber(OgdlEvent ev) {
        return decodeNumber(ev, OgdlSyntax.cutNumberString(ev));
    }

    /* Integer を切り出して返却します。 */
    static Integer evaluateInteger(OgdlEvent ev) throws OgdlSyntaxException {
        return ValueOf.decodeInteger(OgdlSyntax.cutNumberString(ev));
    }

    /* 指定されたクラス名をリードして返却します。 */
    static Class evaluateClassByExtendName(OgdlEvent ev) {
        return loadClassByExtendClassName(ev, OgdlSyntax.cutExtendClassName(ev));
    }

    /*
     * private
     */

    /* 拡張規則で文字列をJava数値型に変換します。 */
    private static Number decodeNumber(OgdlEvent ev, String s) throws OgdlSyntaxException {
        try {
            return ValueOf.decodeNumber(s);
        } catch (final NumberFormatException e) {
            throw new OgdlSyntaxException(ev, Utils.log("value is not Number format. format=", s), e);
        }
    }

    /* Java構文（例：java.lang.Integer, int, Integer, Integer[]）からクラスをリードし返却します。 */
    private static Class loadClassByExtendClassName(OgdlEvent ev, String extendClassName) {

        // 拡張の配列表記の検証 ”String[][] 等”
        final StringBuffer symbol = new StringBuffer();
        int off = extendClassName.length();
        while (off >= 2 && extendClassName.charAt(off - 1) == ']' && extendClassName.charAt(off - 2) == '[') {
            off -= 2;
            symbol.append('[');
        }
        final boolean isArray = (symbol.length() > 0);

        // 拡張表記でクラスを検索(クラス名の配列部を切る)
        final String rootName = ((isArray) ? extendClassName.substring(0, off) : extendClassName);
        final Class clazz = findClass(rootName, ev);
        if (isArray) {
            final StringBuffer className = new StringBuffer();
            // 配列の表記を付加する
            if (clazz != null) {
                final String primitiveName = (String) Utils.PRIMITIVE_CLASS_NAME_MAP.get(clazz);
                if (primitiveName == null) {
                    // オブジェクト型
                    className.append(symbol).append(Utils.OBJECT_SYMBOL).append(clazz.getName()).append(';');
                } else {
                    // プリミティブ型
                    className.append(symbol).append(primitiveName);
                }
            } else {
                // オブジェクト型
                className.append(symbol).append(Utils.OBJECT_SYMBOL).append(rootName).append(';');
            }
            // 通常のルールでクラスをロードする
            final Class loadClass = Utils.loadClass(className.toString(), ev.loader);
            if (loadClass != null) {
                return loadClass;
            }
        } else {
            // 配列型以外は返却
            if (clazz != null) {
                return clazz;
            }
            // 通常のルールでクラスをロードする
            final Class loadClass = Utils.loadClass(rootName, ev.loader);
            if (loadClass != null) {
                return loadClass;
            }
        }
        throw new OgdlSyntaxException(ev, Utils.log("not find class. class=", extendClassName));
    }

    private static Class findClass(String className, OgdlEvent ev) {
        Class clazz = (Class) Utils.PRIMITIVE_FOR_TYPE_MAP.get(className);
        if (clazz == null) {
            if (!Utils.isContains(className, '.')) {
                try {
                    clazz = Class.forName("java.lang." + className);
                } catch (final ClassNotFoundException e) {
                    // no op
                }
            }
        }
        if (clazz == null) {
            clazz = ev.context.forImport(className);
        }
        return clazz;
    }

}
