/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.beans.IndexedPropertyDescriptor;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * イントロスペクションの構文解析機能を提供します。
 */
class OgdlIntrospectParser {

    /* データ要素へアクセスする式を実行して返却します。 */
    static Object evaluateObjectNavigate(OgdlEvent ev) {
        // index, Field, Method
        Object o = ev.target;
        while (ev.hasNextBy(1)) {

            final int open = ev.charAt();
            switch (open) {
            case '[':
                final Object key = readIndexingKeys(ev);
                o = readIndexingValue(ev, key, o);
                break;
            case '.':
                ev.shift();// add separator
                final String memberName = OgdlSyntax.cutMemberName(ev);
                final Class clazz = o.getClass();
                // method or property or field
                if (ev.isCharAt('(')) {// method 'method(arg..)'
                    o = invokeMethod(ev, memberName, o, clazz);
                } else if (ev.isCharAt('[')) {// index property or field or property
                    o = readIndexPropertyOrFieldOrProperty(ev, memberName, o, clazz);
                } else {// field or property or Array.length
                    o = readFieldOrProperty(memberName, o, clazz);
                }
                break;
            case '#':
                ev.shift();// add separator
                final String staticMemberName = OgdlSyntax.cutMemberName(ev);
                // static method or static field
                if (ev.isCharAt('(')) {// static method 'method(arg..)'
                    o = invokeStaticMethod(ev, staticMemberName, o);
                } else {// static field
                    o = readStaticFieldValue(ev, staticMemberName, o);
                }
                break;
            default:
                return o;
            }
        }
        return o;
    }

    /* インスタンス生成式を実行して値を返却します。 */
    static Object evaluateNewInstance(OgdlEvent ev) {

        final Class clazz = OgdlPrimitiveParser.evaluateClassByExtendName(ev);
        final int next = ev.charAt();

        final Object obj;
        // switch create type
        switch (next) {
        case '#':
            // Factory Method
            ev.shift();
            final String memberName = OgdlSyntax.cutMemberName(ev);
            if (ev.isCharAt('(')) {
                final Object[] args = evaluateEncloseArguments(ev.get('(', ')'));
                obj = BeanIntrospectHelper.invokeMethod(clazz, null, memberName, args);
                ev.shiftSpace();
                // set properties
                if (ev.isCharAt('{')) {
                    writeInitProperties(ev.get('{', '}'), obj);
                }
            } else {
                throw new OgdlSyntaxException(ev, "is not arguments open literal.");
            }
            break;
        case '(':
            // Constructor
            final Object[] args = evaluateEncloseArguments(ev.get('(', ')'));
            obj = BeanIntrospectHelper.newInstance(clazz, args);
            ev.shiftSpace();
            // init properties
            if (ev.isCharAt('{')) {
                writeInitProperties(ev.get('{', '}'), obj);
            }
            break;
        case '[':
            // Array "class[n][n]"
            final Integer[] dimensions = evaluateDimensionArraySizes(ev);
            try {
                obj = Array.newInstance(clazz, Boxing.unbox(dimensions));
            } catch (final IllegalArgumentException e) {
                throw new OgdlSyntaxException(ev, "arrays newInstance err.", e);
            } catch (final NegativeArraySizeException e) {
                throw new OgdlSyntaxException(ev, "arrays newInstance err.", e);
            }
            break;
        default:
            // Array "class[][]"
            if (!clazz.isArray()) {
                throw new OgdlSyntaxException(ev, "is not array type. " + clazz);
            }
            ev.shiftSpace();
            // init arrays
            if (ev.isCharAt('{')) {
                // 数値領域（等差数列や時間領域）リストで生成する。
                final List list = (List) OgdlCollectionParser.evaluateEncloseProgressionCollection(ev.get('{', '}', new LinkedList()));
                try {
                    obj = Utils.toArray(list, clazz.getComponentType());// 配列への格納
                } catch (final IllegalArgumentException e) {
                    throw new OgdlSyntaxException(ev, "array item type mismatch.", e);
                }
            } else {
                throw new OgdlSyntaxException(ev, "is not initialize array literal.");
            }
        }

        return obj;
    }

    static Object evaluateConstructor(OgdlEvent ev) {
        final Class type = OgdlPrimitiveParser.evaluateClassByExtendName(ev);
        final Class[] argsType = evaluateEncloseTypeArray(ev.get('(', ')'));
        return BeanIntrospectHelper.getConstructor(type, argsType);
    }

    static Object evaluateMethod(OgdlEvent ev) {
        final Class type = OgdlPrimitiveParser.evaluateClassByExtendName(ev);
        if (ev.isNotCharAt('#')) {
            throw new OgdlSyntaxException(ev, "method separator err. ");
        }
        ev.shift();
        final String methodName = OgdlSyntax.cutMemberName(ev);
        final Class[] argsType = evaluateEncloseTypeArray(ev.get('(', ')'));
        return BeanIntrospectHelper.getMethod(type, methodName, argsType);
    }

    static Object evaluateField(OgdlEvent ev) {
        final Class type = OgdlPrimitiveParser.evaluateClassByExtendName(ev);
        if (ev.isNotCharAt('#')) {
            throw new OgdlSyntaxException(ev, "field separator err. ");
        }
        ev.shift();
        final String fieldName = OgdlSyntax.cutMemberName(ev);
        return BeanIntrospectHelper.getField(type, fieldName);
    }

    /*
     * private
     */

    /* コンマ区切りのクラス名のリスト書式のクラス配列として返却します。 */
    private static Class[] evaluateEncloseTypeArray(OgdlEvent ev) {
        if (ev.isNotOpen()) {
            throw new OgdlSyntaxException(ev, "is not open literal.");
        }
        ev.shiftAndShiftSpace();
        final List list = new ArrayList();
        final int close = ev.close;
        if (ev.isNotClose(close)) {
            while (ev.hasNext()) {
                list.add(OgdlPrimitiveParser.evaluateClassByExtendName(ev));
                ev.shiftSpace();
                if (ev.isClose(close)) {
                    break;
                }
                if (ev.isNotSeparator()) {
                    throw new OgdlSyntaxException(ev, "is not separator.");
                }
                ev.shiftAndShiftSpace();
            }
            ev.shiftSpace();
            if (ev.isNotClose(close)) {
                throw new OgdlSyntaxException(ev, "is not close literal.");
            }
        }
        final Class[] types = (Class[]) list.toArray(new Class[0]);
        ev.shift();// next index
        return types;
    }

    /* 配列生成式の配列数の初期化部分（例：[2][4]）を解析して返却します。 */
    private static Integer[] evaluateDimensionArraySizes(OgdlEvent ev) {
        final Collection output = new LinkedList();
        while (ev.isCharAt('[')) {
            ev.shiftAndShiftSpace();// + '['
            output.add(OgdlPrimitiveParser.evaluateInteger(ev));
            ev.shiftSpace();
            if (ev.isNotCharAt(']')) {
                throw new OgdlSyntaxException(ev, "is not close literal.");
            }
            ev.shift();// + ']'
        }
        return (Integer[]) output.toArray(new Integer[output.size()]);
    }

    private static Object readFieldOrProperty(String memberName, Object o, Class clazz) {
        final PropertyDescriptor pd = BeanIntrospectHelper.getPropertyDescriptor(clazz, memberName);
        Object r = null;
        if (pd == null) {
            // is Array.length
            if (OgdlSyntax.isArraylength(clazz, memberName)) {
                r = Boxing.box(Array.getLength(o));
            } else {
                r = BeanIntrospectHelper.getFieldValue(clazz, o, memberName);
            }
        } else {
            r = BeanIntrospectHelper.getProperty(o, pd.getReadMethod());
        }
        return r;
    }

    private static Object readStaticFieldValue(OgdlEvent ev, String memberName, Object o) {
        if (o instanceof Class) {
            // static field
            return BeanIntrospectHelper.getFieldValue((Class) o, null, memberName);
        } else {
            throw new OgdlSyntaxException(ev, "is not Class class.");
        }
    }

    private static Object readIndexedValue(OgdlEvent ev) {
        ev.shift();// add indexer
        final Object inx = OgdlRuntime.evaluate(ev);
        if (ev.isNotCharAt(']')) {
            throw new OgdlSyntaxException(ev, "is not indexing close literal.");
        }
        if (!(inx instanceof Integer)) {
            throw new OgdlSyntaxException(ev, Utils.clazz("not indexing type. type=", inx));
        }
        ev.shift();// add indexer
        return inx;
    }

    private static Object readIndexingKeys(OgdlEvent ev) {
        ev.shift();// add indexer
        final Object key = OgdlRuntime.evaluate(ev);
        if (ev.isNotCharAt(']')) {
            throw new OgdlSyntaxException(ev, "is not indexing close literal.");
        }
        ev.shift();// add indexer
        return key;
    }

    private static int toKeyAsIndex(OgdlEvent ev, Object key) {
        if (!(key instanceof Integer)) {
            throw new OgdlSyntaxException(ev, Utils.clazz("not indexing type. type=", key));
        }
        return ((Integer) key).intValue();
    }

    private static Object readIndexingValue(OgdlEvent ev, Object key, Object o) {
        // index|map|field|Property
        if (o instanceof Map) {
            return ((Map) o).get(key);
        } else if (key instanceof String) {
            final String memberName = (String) key;
            return readFieldOrProperty(memberName, o, o.getClass());
        } else if (o instanceof List) {
            final int index = toKeyAsIndex(ev, key);
            return ((List) o).get(index);
        } else if (o.getClass().isArray()) {
            final int index = toKeyAsIndex(ev, key);
            return Array.get(o, index);
        } else if (o instanceof CharSequence) {
            final int index = toKeyAsIndex(ev, key);
            return Boxing.box(((CharSequence) o).charAt(index));
        } else {
            throw new OgdlSyntaxException(ev, Utils.clazz("not indexing type. type=", o));
        }
    }

    private static Object readIndexPropertyOrFieldOrProperty(OgdlEvent ev, String memberName, Object o, Class clazz) {
        final IndexedPropertyDescriptor ipd = BeanIntrospectHelper.getIndexedPropertyDescriptor(clazz, memberName);
        if (ipd != null) {
            final Object inx = readIndexedValue(ev);// get index
            return BeanIntrospectHelper.getIndexedProperty(o, ipd.getIndexedReadMethod(), (Integer) inx);
        } else {
            return readFieldOrProperty(memberName, o, clazz);
        }
    }

    private static Object invokeMethod(OgdlEvent ev, String memberName, Object o, Class clazz) {
        final Object[] paramValues = evaluateEncloseArguments(ev.get('(', ')'));
        return BeanIntrospectHelper.invokeMethod(clazz, o, memberName, paramValues);
    }

    private static Object invokeStaticMethod(OgdlEvent ev, String memberName, Object o) {
        final Object[] paramValues = evaluateEncloseArguments(ev.get('(', ')'));
        if (o instanceof Class) {
            return BeanIntrospectHelper.invokeMethod((Class) o, null, memberName, paramValues);
        } else {
            throw new OgdlSyntaxException(ev, "is not Class class.");
        }
    }

    /* メソッド引数の書式として解析してオブジェクト型配列として返却します。 イベントに格納されるリテラル文字で区切られる前提で解析されます。 */
    private static Object[] evaluateEncloseArguments(OgdlEvent ev) {
        final Object args = OgdlParser.evaluateEncloseCollection(ev.get(new LinkedList()));
        return ((List) args).toArray();
    }

    /* プロパティ設定式として解析して、そのプロパティを格納します。 */
    private static void writeInitProperties(OgdlEvent ev, Object obj) {
        final Class clazz = obj.getClass();
        if (ev.isNotOpen()) {
            throw new OgdlSyntaxException(ev, "is not open literal.");
        }
        ev.shift();
        final int close = ev.close;
        if (ev.isNotClose(close)) {
            while (ev.hasNext()) {
                ev.shiftSpace();
                final String memberName = OgdlSyntax.cutMemberName(ev);
                // method or property or field
                if (ev.isCharAt('(')) {// method 'method(arg..)'
                    invokeInitMethod(ev, memberName, obj, clazz);
                } else if (ev.isCharAt('[')) {// index property
                    invokeInitIndexedProperty(ev, memberName, obj, clazz);
                } else {// property
                    invokeInitProperty(ev, memberName, obj, clazz);
                }
                ev.shiftSpace();
                if (ev.isClose(close)) {
                    break;
                }
                if (ev.isNotSeparator()) {
                    throw new OgdlSyntaxException(ev, "is not separator.");
                }
                ev.shiftAndShiftSpace();
            }
            ev.shiftSpace();
            if (ev.isNotClose(close)) {
                throw new OgdlSyntaxException(ev, "is not close literal.");
            }
        }
        ev.shiftAndShiftSpace();
    }

    private static void invokeInitIndexedProperty(OgdlEvent ev, String memberName, Object o, Class clazz) {
        final IndexedPropertyDescriptor ipd = BeanIntrospectHelper.getIndexedPropertyDescriptor(clazz, memberName);
        if (ipd != null) {
            final Object inx = readIndexedValue(ev);
            ev.shiftSpace();
            if (ev.isNotMapSeparator()) {
                throw new OgdlSyntaxException(ev, "is not property separator.");
            }
            ev.shiftAndShiftSpace();
            BeanIntrospectHelper.setIndexedProperty(o, ipd.getIndexedWriteMethod(), (Integer) inx, OgdlRuntime.evaluate(ev));
        } else {
            throw new OgdlSyntaxException(ev, "is not Indexed property.");
        }
    }

    private static void invokeInitMethod(OgdlEvent ev, String memberName, Object o, Class clazz) {
        final Object[] paramValues = evaluateEncloseArguments(ev.get('(', ')'));
        BeanIntrospectHelper.invokeMethod(clazz, o, memberName, paramValues);
    }

    private static void invokeInitProperty(OgdlEvent ev, String memberName, Object o, Class clazz) {
        final PropertyDescriptor pd = BeanIntrospectHelper.getPropertyDescriptor(clazz, memberName);
        if (pd != null) {
            ev.shiftSpace();
            if (ev.isNotMapSeparator()) {
                throw new OgdlSyntaxException(ev, "is not property separator.");
            }
            ev.shiftAndShiftSpace();
            BeanIntrospectHelper.setProperty(o, pd.getWriteMethod(), OgdlRuntime.evaluate(ev));
        } else {
            throw new OgdlSyntaxException(ev, Utils.log("is not property. ", memberName));
        }
    }
}
