/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

/**
 * OGDLの実行イベントを提供します。 解析の実行単位で生成されます。
 */
class OgdlEvent {

    /* 式構文 */
    final String ptn;

    /* 式構文の長さ */
    final int len;

    /* 式構文の解析開始インデックス */
    final OgdlParseIndex inx;

    /* クラスローダ */
    final ClassLoader loader;

    /* コンテキスト */
    final OgdlContext context;

    /* 式の処理対象の値（一時変数） */
    Object target;

    /* 式の開始文字（一時変数） */
    int open;

    /* 式の終了文字（一時変数） */
    int close;

    OgdlEvent(String pattern, OgdlParseIndex index, ClassLoader loader, OgdlContext context) {
        this.ptn = pattern;
        this.len = pattern.length();
        this.inx = index;
        this.loader = loader;
        this.context = context;
    }

    /**
     * 指定のプロパティを更新して自身の参照を返却します。
     * 
     * @param targetValue
     *            式の処理対象の値
     * @return 自身の参照
     */
    OgdlEvent get(Object targetValue) {
        this.target = targetValue;
        return this;
    }

    /**
     * 指定のプロパティを更新して自身の参照を返却します。
     * 
     * @param openChar
     *            開始を示すリテラル文字
     * @param closeChar
     *            終了を示すリテラル文字
     * @param targetValue
     *            式の処理対象の値
     * @return 自身の参照
     */
    OgdlEvent get(int openChar, int closeChar, Object targetValue) {
        this.open = openChar;
        this.close = closeChar;
        this.target = targetValue;
        return this;
    }

    /**
     * 指定のプロパティを更新して自身の参照を返却します。
     * 
     * @param openChar
     *            開始を示すリテラル文字
     * @param closeChar
     *            終了を示すリテラル文字
     * @return 自身の参照
     */
    OgdlEvent get(int openChar, int closeChar) {
        this.open = openChar;
        this.close = closeChar;
        return this;
    }

    /*
     * 解析ユーティリティ
     */

    boolean isCharAt(int c) {
        return (inx.get() < len && ptn.charAt(inx.get()) == c);
    }

    boolean isCharAtTo(int c, int index) {
        return (index < len && ptn.charAt(index) == c);
    }

    boolean isNotCharAt(int c) {
        return (inx.get() < len && ptn.charAt(inx.get()) != c);
    }

    boolean isNotCharAtTo(int c, int index) {
        return (index < len && ptn.charAt(index) != c);
    }

    boolean isClose(int closeChar) {
        return (ptn.charAt(inx.get()) == closeChar);
    }

    boolean isNotClose(int closeChar) {
        return (ptn.charAt(inx.get()) != closeChar);
    }

    boolean isNotOpen() {
        return (ptn.charAt(inx.get()) != open);
    }

    /* データ集合の区切り文字（論理否定） */
    boolean isNotSeparator() {
        return (ptn.charAt(inx.get()) != ',');
    }

    /* 写像を対応付ける区切り文字（論理否定） */
    boolean isNotMapSeparator() {
        return (ptn.charAt(inx.get()) != '=');
    }

    boolean isNotNextSpace() {
        return (inx.get() < len) && !(ptn.charAt(inx.get()) <= ' ');
    }

    boolean isPrefix(String prefix) {
        return ptn.startsWith(prefix, inx.get());
    }

    boolean isPrefixTo(String prefix, int index) {
        return ptn.startsWith(prefix, index);
    }

    int index() {
        return inx.get();
    }

    boolean hasNext() {
        return (inx.get() < len);
    }

    boolean hasNextBy(int add) {
        return (inx.get() + add < len);
    }

    boolean hasNextTo(int index) {
        return (index < len);
    }

    int charAt() {
        return ptn.charAt(inx.get());
    }

    int charAtBy(int add) {
        return ptn.charAt(inx.get() + add);
    }

    int charAtTo(int index) {
        return ptn.charAt(index);
    }

    int charAtUpper() {
        return OgdlSyntax.toUpper(ptn.charAt(inx.get()));
    }

    int charAtUpperBy(int add) {
        return OgdlSyntax.toUpper(ptn.charAt(inx.get() + add));
    }

    int shift() {
        return inx.increment();
    }

    int shiftBy(int add) {
        return inx.add(add);
    }

    void shiftSpace() {
        inx.set(OgdlSyntax.skipSpace(this));
    }

    void shiftAndShiftSpace() {
        inx.increment();
        inx.set(OgdlSyntax.skipSpace(this));
    }

    void setIndex(int index) {
        this.inx.set(index);
    }

    String cutForShift(int endIndex) {
        final String sub = ptn.substring(inx.get(), endIndex);
        inx.set(endIndex);
        return sub;
    }

    String cutForEnd(int endIndex) {
        return ptn.substring(inx.get(), endIndex);
    }

    /**
     * "{pattern=this.pattern,offset=this.offset,target=this.target}" の表現で返却します。
     * 
     * @return このオブジェクトの文字列表現
     * @see java.lang.Object#toString()
     */
    public String toString() {
        final StringBuffer sb = new StringBuffer();
        sb.append("{");
        sb.append("pattern=");
        sb.append(this.ptn);
        sb.append(",offset=");
        sb.append(this.inx);
        sb.append(",target=");
        sb.append(this.target);
        sb.append("}");
        return sb.toString();
    }

}