/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

/**
 * {@link shohaku.ogdl.OgdlContext} の機能を実装するクラスに機能を混ぜ込む、Mix-In（組込み機能）を提供します。<br>
 * このクラスは OgdlContext を実装したクラスのプライベートメンバーとして組み込まれて使用されます。
 */
public class OgdlContextMixIn implements OgdlContext {

    private final Map _attributesMap;

    private final Map _classesMap;

    private final Map _functionsMap;

    private ClassLoader _classLoader;

    /**
     * 属性値を格納するマップと、クラス参照を格納するマップ、組込み関数を格納するマップを格納して初期化します。
     * 
     * @param attributes
     *            属性値を格納するマップ、機能を使用しない場合は null
     * @param classes
     *            クラス参照を格納するマップ、機能を使用しない場合は null
     * @param functions
     *            組込み関数を格納するマップ、機能を使用しない場合は null
     */
    public OgdlContextMixIn(Map attributes, Map classes, Map functions) {
        this._attributesMap = attributes;
        this._classesMap = classes;
        this._functionsMap = functions;
    }

    public boolean addImport(Class clazz) {
        return (null != _classesMap.put(Utils.getShortClassName(clazz), clazz));
    }

    public Class forImport(String shortName) {
        return (Class) _classesMap.get(shortName);
    }

    public void addFunctionsMap(String prefix, Map functionsMap) {
        if (!Utils.isEmpty(prefix)) {
            for (final Iterator i = functionsMap.entrySet().iterator(); i.hasNext();) {
                final Map.Entry e = (Map.Entry) i.next();
                final String funcName = (String) e.getKey();
                final Collection func = (Collection) e.getValue();
                _functionsMap.put(prefix + ':' + funcName, func);
            }
        } else {
            _functionsMap.putAll(functionsMap);
        }
    }

    public Collection forFunctions(String name) {
        return (Collection) _functionsMap.get(name);
    }

    public Iterator attributeNames() {
        return _attributesMap.keySet().iterator();
    }

    public Object getAttribute(String name) {
        return _attributesMap.get(name);
    }

    public void setAttribute(String name, Object value) {
        _attributesMap.put(name, value);
    }

    public Object removeAttribute(String name) {
        return _attributesMap.remove(name);
    }

    public boolean containsAttribute(String name) {
        return _attributesMap.containsKey(name);
    }

    public ClassLoader getClassLoader() {
        return _classLoader;
    }

    public void setClassLoader(ClassLoader classLoader) {
        this._classLoader = classLoader;
    }

    /**
     * クラス参照を格納するマップを返却します。
     * 
     * @return クラス参照を格納するマップ
     */
    public Map getClassesMap() {
        return this._classesMap;
    }

    /**
     * 組込み関数を格納するマップを返却します。
     * 
     * @return 組込み関数を格納するマップ
     */
    public Map getFunctionsMap() {
        return this._functionsMap;
    }

    /**
     * 属性値を格納するマップを返却します。
     * 
     * @return 属性値を格納するマップ
     */
    public Map getAttributesMap() {
        return this._attributesMap;
    }

}
