/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.SortedSet;

/**
 * コレクションの構文解析機能を提供します。
 */
class OgdlCollectionParser {

    static Object evaluateArray(OgdlEvent ev) {
        if (isSynchronizedView(ev)) {
            return evaluateSynchronizedArrayAsList(ev);
        }
        if (isUnmodifiableView(ev)) {
            return evaluateUnmodifiableArrayAsList(ev);
        }
        if (isFixedSizeView(ev)) {
            return evaluateFixedSizeArrayAsList(ev);
        }
        if (isUndecoratedView(ev)) {
            return evaluateUndecoratedArray(ev);
        }
        return evaluateUndecoratedArray(ev);
    }

    static Object evaluateCollection(OgdlEvent ev) {

        if (isSynchronizedView(ev)) {
            return evaluateSynchronizedCollection(ev);
        }
        if (isUnmodifiableView(ev)) {
            return evaluateUnmodifiableCollection(ev);
        }
        if (isUndecoratedView(ev)) {
            return evaluateUndecoratedCollection(ev);
        }
        return evaluateUnmodifiableCollection(ev);
    }

    static Object evaluateSortedMap(OgdlEvent ev) {

        if (isSynchronizedView(ev)) {
            return evaluateSynchronizedSortedMap(ev);
        }
        if (isUnmodifiableView(ev)) {
            return evaluateUnmodifiableSortedMap(ev);
        }
        if (isUndecoratedView(ev)) {
            return evaluateUndecoratedSortedMap(ev);
        }
        return evaluateUnmodifiableSortedMap(ev);
    }

    static Object evaluateMap(OgdlEvent ev) {

        if (isSynchronizedView(ev)) {
            return evaluateSynchronizedMap(ev);
        }
        if (isUnmodifiableView(ev)) {
            return evaluateUnmodifiableMap(ev);
        }
        if (isUndecoratedView(ev)) {
            return evaluateUndecoratedMap(ev);
        }
        return evaluateUnmodifiableMap(ev);
    }

    static Object evaluateSortedSet(OgdlEvent ev) {

        if (isSynchronizedView(ev)) {
            return evaluateSynchronizedSortedSet(ev);
        }
        if (isUnmodifiableView(ev)) {
            return evaluateUnmodifiableSortedSet(ev);
        }
        if (isUndecoratedView(ev)) {
            return evaluateUndecoratedSortedSet(ev);
        }
        return evaluateUnmodifiableSortedSet(ev);
    }

    static Object evaluateSet(OgdlEvent ev) {

        if (isSynchronizedView(ev)) {
            return evaluateSynchronizedSet(ev);
        }
        if (isUnmodifiableView(ev)) {
            return evaluateUnmodifiableSet(ev);
        }
        if (isUndecoratedView(ev)) {
            return evaluateUndecoratedSet(ev);
        }
        return evaluateUnmodifiableSet(ev);
    }

    static Object evaluateList(OgdlEvent ev) {

        if (isSynchronizedView(ev)) {
            return evaluateSynchronizedList(ev);
        }
        if (isUnmodifiableView(ev)) {
            return evaluateUnmodifiableList(ev);
        }
        if (isUndecoratedView(ev)) {
            return evaluateUndecoratedList(ev);
        }
        return evaluateUnmodifiableList(ev);
    }

    /* 同期コレクションの指定の場合は true を返却します。 */
    private static boolean isSynchronizedView(OgdlEvent ev) {
        if (ev.isPrefix("syn:")) {
            ev.shiftBy(4);
            return true;
        }
        return false;
    }

    /* 不変コレクションの指定の場合は true を返却します。 */
    private static boolean isUnmodifiableView(OgdlEvent ev) {
        if (ev.isPrefix("unm:")) {
            ev.shiftBy(4);
            return true;
        }
        return false;
    }

    /* 非装飾コレクションの指定の場合は true を返却します。 */
    private static boolean isUndecoratedView(OgdlEvent ev) {
        if (ev.isPrefix("und:")) {
            ev.shiftBy(4);
            return true;
        }
        return false;
    }

    /* 固定サイズコレクションの指定の場合は true を返却します。 */
    private static boolean isFixedSizeView(OgdlEvent ev) {
        if (ev.isPrefix("fix:")) {
            ev.shiftBy(4);
            return true;
        }
        return false;
    }

    /* コンマ区切りのコレクション書式として解析して返却します。 */
    static Collection evaluateEncloseCollection(OgdlEvent ev) {
        if (ev.isNotOpen()) {
            throw new OgdlSyntaxException(ev, "is not open literal.");
        }
        ev.shift();
        return evaluateCloseCollection(ev);
    }

    /* 開始文字を検証せずに、コンマ区切りのコレクション書式として解析して返却します。 */
    static Collection evaluateCloseCollection(OgdlEvent ev) {

        final Collection output = (Collection) ev.target;
        final int close = ev.close;
        if (ev.isNotClose(close)) {
            while (ev.hasNext()) {
                output.add(OgdlRuntime.evaluate(ev));
                if (ev.isClose(close)) {
                    break;
                }
                if (ev.isNotSeparator()) {
                    throw new OgdlSyntaxException(ev, "is not separator.");
                }
                ev.shift();// + ','
            }
            ev.shiftSpace();
            if (ev.isNotClose(close)) {
                throw new OgdlSyntaxException(ev, "is not close literal.");
            }
        }
        ev.shift();// next index
        return output;
    }

    /* マップ書式として解析して指定のマップに追加して返却します。 */
    static Map evaluateEncloseMap(OgdlEvent ev) {
        if (ev.isNotOpen()) {
            throw new OgdlSyntaxException(ev, "is not open literal.");
        }
        ev.shift();
        return evaluateCloseMap(ev);
    }

    /* 開始文字を検証せずに、マップ書式として解析して指定のマップに追加して返却します。 */
    static Map evaluateCloseMap(OgdlEvent ev) {

        final Map output = (Map) ev.target;
        final int close = ev.close;
        if (ev.isNotClose(close)) {
            while (ev.hasNext()) {

                // get key
                final Object key = OgdlRuntime.evaluate(ev);

                // key value separator
                if (ev.isNotMapSeparator()) {
                    throw new OgdlSyntaxException(ev, "is not map separator.");
                }
                ev.shiftAndShiftSpace();

                // get value
                final Object val = OgdlRuntime.evaluate(ev);

                // add Entry
                output.put(key, val);
                if (ev.isClose(close)) {
                    break;
                }
                // map entry separator
                if (ev.isNotSeparator()) {
                    throw new OgdlSyntaxException(ev, "is not separator.");
                }
                ev.shiftAndShiftSpace();
            }
            ev.shiftSpace();
            if (ev.isNotClose(close)) {
                throw new OgdlSyntaxException(ev, "is not close literal.");
            }
        }
        ev.shift();// next index
        return output;

    }

    /* 数値領域（等差数列や時間領域）式を有効とするコンマ区切りのリスト書式として引数のリストに格納します。 */
    static Collection evaluateEncloseProgressionCollection(OgdlEvent ev) {
        if (ev.isNotOpen()) {
            throw new OgdlSyntaxException(ev, "is not open literal.");
        }
        ev.shift();
        return evaluateCloseProgressionCollection(ev);
    }

    /* 開始文字を検証せずに、数値領域（等差数列や時間領域）式を有効とするコンマ区切りのリスト書式として解析して引数のリストに格納して返却します。 */
    static Collection evaluateCloseProgressionCollection(OgdlEvent ev) {

        final Collection output = (Collection) ev.target;
        final int close = ev.close;
        if (ev.isNotClose(close)) {
            while (ev.hasNext()) {
                final Object e = OgdlRuntime.evaluate(ev);
                if (ev.isClose(close)) {
                    output.add(e);
                    break;
                }

                // is range 'from:to'
                if (ev.isCharAt(':')) {
                    ev.shift();
                    final Object to = OgdlRuntime.evaluate(ev);
                    addListProgressionValue(ev, output, e, to);
                } else {
                    output.add(e);
                }

                ev.shiftSpace();
                if (ev.isClose(close)) {
                    break;
                }
                if (ev.isNotSeparator()) {
                    throw new OgdlSyntaxException(ev, "is not separator.");
                }
                ev.shift();// + ','
            }
            ev.shiftSpace();
            if (ev.isNotClose(close)) {
                throw new OgdlSyntaxException(ev, "is not close literal.");
            }
        }
        ev.shift();// next index
        return output;

    }

    /* 数値領域（等差数列や時間領域）式を解析して引数のリストに格納し次の式の開始位置を返却します。 */
    private static void addListProgressionValue(OgdlEvent ev, Collection output, Object from, Object to) {

        // get difference "from:to(difference)"
        String difference = null;
        ev.shiftSpace();
        if (ev.isCharAt('(')) {
            ev.shiftAndShiftSpace();
            difference = OgdlSyntax.cutDifferenceString(ev);
            ev.shiftSpace();
            if (ev.isNotCharAt(')')) {
                throw new OgdlSyntaxException(ev, "is not difference close literal.");
            }
            ev.shift();
            addNumberDomain(ev, output, from, to, difference);
        } else {
            addNumberDomain(ev, output, from, to);
        }
    }

    /*
     * evaluate collections
     */

    /* マップ生成式を実行して値を返却します。 */
    static Object evaluateDefaultMap(OgdlEvent ev) {
        final Map output = new LinkedHashMap();
        return Collections.unmodifiableMap(evaluateEncloseMap(ev.get('{', '}', output)));
    }

    /* リスト生成式を実行して値を返却します。 */
    static Object evaluateDefaultList(OgdlEvent ev) {
        final List output = new ArrayList();
        return Collections.unmodifiableList((List) evaluateEncloseProgressionCollection(ev.get('[', ']', output)));
    }

    /* セット生成式を実行して値を返却します。 */
    static Object evaluateDefaultSet(OgdlEvent ev) {
        final Set output = new LinkedHashSet();
        return Collections.unmodifiableSet((Set) evaluateEncloseProgressionCollection(ev.get('(', ')', output)));
    }

    /*
     * List
     */

    /* リスト生成式を実行して値を返却します。 */
    private static Object evaluateUndecoratedList(OgdlEvent ev) {

        List output = null;
        if (OgdlSyntax.isMemberHead(ev)) {

            final Object val = newCollectionInstance(ev);
            try {
                output = (List) val;
            } catch (final ClassCastException e) {
                throw new OgdlSyntaxException(ev, "not list type. " + val.getClass(), e);
            }

        } else {
            output = new ArrayList();
        }

        ev.shiftSpace();

        return evaluateEncloseProgressionCollection(ev.get('{', '}', output));
    }

    /* 同期リストの生成式を実行して値を返却します。 */
    private static Object evaluateSynchronizedList(OgdlEvent ev) {
        final Object val = evaluateUndecoratedList(ev);
        return Collections.synchronizedList((List) val);
    }

    /* 変更不可のリスト生成式を実行して値を返却します。 */
    private static Object evaluateUnmodifiableList(OgdlEvent ev) {
        final Object val = evaluateUndecoratedList(ev);
        return Collections.unmodifiableList((List) val);
    }

    /*
     * Set
     */

    /* セット生成式を実行して値を返却します。 */
    private static Object evaluateUndecoratedSet(OgdlEvent ev) {

        Set output = null;
        if (OgdlSyntax.isMemberHead(ev)) {

            final Object val = newCollectionInstance(ev);
            try {
                output = (Set) val;
            } catch (final ClassCastException e) {
                throw new OgdlSyntaxException(ev, "not set type. " + val.getClass(), e);
            }

        } else {
            output = new LinkedHashSet();
        }

        ev.shiftSpace();

        return evaluateEncloseProgressionCollection(ev.get('{', '}', output));
    }

    /* 同期セットの生成式を実行して値を返却します。 */
    private static Object evaluateSynchronizedSet(OgdlEvent ev) {
        final Object val = evaluateUndecoratedSet(ev);
        final Set set = Collections.synchronizedSet((Set) val);
        return set;
    }

    /* 変更不可のセット生成式を実行して値を返却します。 */
    private static Object evaluateUnmodifiableSet(OgdlEvent ev) {
        final Object val = evaluateUndecoratedSet(ev);
        final Set set = Collections.unmodifiableSet((Set) val);
        return set;
    }

    /*
     * SortedSet
     */

    /* ソートセットの生成式を実行して値を返却します。 */
    private static Object evaluateUndecoratedSortedSet(OgdlEvent ev) {
        final Object val = evaluateUndecoratedSet(ev);
        try {
            final SortedSet sortedSet = Collections.synchronizedSortedSet((SortedSet) val);
            return sortedSet;
        } catch (final ClassCastException e) {
            throw new OgdlSyntaxException(ev, "not SortedSet type. " + val.getClass(), e);
        }

    }

    /* 同期ソートセットの生成式を実行して値を返却します。 */
    private static Object evaluateSynchronizedSortedSet(OgdlEvent ev) {
        final Object val = evaluateUndecoratedSortedSet(ev);
        return Collections.synchronizedSortedSet((SortedSet) val);
    }

    /* 変更不可のソートセット生成式を実行して値を返却します。 */
    private static Object evaluateUnmodifiableSortedSet(OgdlEvent ev) {
        final Object val = evaluateUndecoratedSortedSet(ev);
        return Collections.unmodifiableSortedSet((SortedSet) val);
    }

    /*
     * Map
     */

    /* マップ生成式を実行して値を返却します。 */
    private static Map evaluateUndecoratedMap(OgdlEvent ev) {

        Map output = null;
        if (OgdlSyntax.isMemberHead(ev)) {

            final Object val = newCollectionInstance(ev);
            try {
                output = (Map) val;
            } catch (final ClassCastException e) {
                throw new OgdlSyntaxException(ev, "not map type. " + val.getClass(), e);
            }

        } else {
            output = new LinkedHashMap();
        }

        ev.shiftSpace();

        return evaluateEncloseMap(ev.get('{', '}', output));

    }

    /* 同期マップの生成式を実行して値を返却します。 */
    private static Object evaluateSynchronizedMap(OgdlEvent ev) {
        return Collections.synchronizedMap(evaluateUndecoratedMap(ev));

    }

    /* 変更不可のマップ生成式を実行して値を返却します。 */
    private static Object evaluateUnmodifiableMap(OgdlEvent ev) {
        return Collections.unmodifiableMap(evaluateUndecoratedMap(ev));

    }

    /*
     * SortedMap
     */

    /* ソートマップの生成式を実行して値を返却します。 */
    private static Object evaluateUndecoratedSortedMap(OgdlEvent ev) {
        final Object val = evaluateUndecoratedMap(ev);
        try {
            final SortedMap sortedMap = Collections.synchronizedSortedMap((SortedMap) val);
            return sortedMap;
        } catch (final ClassCastException e) {
            throw new OgdlSyntaxException(ev, "not SortedMap type. " + val.getClass(), e);
        }

    }

    /* 同期ソートマップの生成式を実行して値を返却します。 */
    private static Object evaluateSynchronizedSortedMap(OgdlEvent ev) {
        final Object val = evaluateUndecoratedSortedMap(ev);
        return Collections.synchronizedSortedMap((SortedMap) val);
    }

    /* 変更不可のソートマップ生成式を実行して値を返却します。 */
    private static Object evaluateUnmodifiableSortedMap(OgdlEvent ev) {
        final Object val = evaluateUndecoratedSortedMap(ev);
        final SortedMap sortedMap = Collections.unmodifiableSortedMap((SortedMap) val);
        return sortedMap;
    }

    /*
     * Array
     */

    /* 配列をの生成式を実行して値を返却します。 */
    private static Object evaluateUndecoratedArray(OgdlEvent ev) {
        final Class clazz;
        if (OgdlSyntax.isMemberHead(ev)) {
            clazz = OgdlPrimitiveParser.evaluateClassByExtendName(ev);
        } else {
            clazz = Object.class;
        }
        ev.shiftSpace();

        // 範囲書式のリストで生成する。
        final List list = (List) evaluateEncloseProgressionCollection(ev.get('{', '}', new LinkedList()));
        Object output = null;
        try {
            output = Utils.toArray(list, clazz);// 配列への格納
        } catch (final IllegalArgumentException e) {
            throw new OgdlSyntaxException(ev, "array item type mismatch.", e);
        }
        return output;
    }

    /* 配列を元にするリストの生成式を実行して値を返却します。 */
    private static Object evaluateFixedSizeArrayAsList(OgdlEvent ev) {
        final Class clazz;
        if (OgdlSyntax.isMemberHead(ev)) {
            clazz = OgdlPrimitiveParser.evaluateClassByExtendName(ev);
        } else {
            clazz = Object.class;
        }
        ev.shiftSpace();

        // 範囲書式のリストで生成する。
        final List list = (List) evaluateEncloseProgressionCollection(ev.get('{', '}', new LinkedList()));
        Object output = null;
        try {
            // 配列を元にするリストへの格納
            if (clazz.isPrimitive()) {
                output = Arrays.asList((Object[]) Utils.toArray(list, Boxing.boxClass(clazz)));
            } else {
                output = Arrays.asList((Object[]) Utils.toArray(list, clazz));
            }
        } catch (final IllegalArgumentException e) {
            throw new OgdlSyntaxException(ev, "array item type mismatch.", e);
        }
        return output;
    }

    /* 配列を元にする同期リストの生成式を実行して値を返却します。 */
    private static Object evaluateSynchronizedArrayAsList(OgdlEvent ev) {
        final Object val = evaluateFixedSizeArrayAsList(ev);
        return Collections.synchronizedList((List) val);
    }

    /* 配列を元にする変更不可のリスト生成式を実行して値を返却します。 */
    private static Object evaluateUnmodifiableArrayAsList(OgdlEvent ev) {
        final Object val = evaluateFixedSizeArrayAsList(ev);
        return Collections.unmodifiableList((List) val);
    }

    /*
     * Collection
     */

    /* コレクション生成式を実行して値を返却します。 */
    private static Object evaluateUndecoratedCollection(OgdlEvent ev) {

        Collection output = null;
        if (!OgdlSyntax.isMemberHead(ev)) {
            throw new OgdlSyntaxException(ev, "collection class is empty.");
        }

        final Object val = newCollectionInstance(ev);
        try {
            output = (Collection) val;
        } catch (final ClassCastException e) {
            throw new OgdlSyntaxException(ev, "not collection type. " + val.getClass(), e);
        }

        ev.shiftSpace();

        return evaluateEncloseProgressionCollection(ev.get('{', '}', output));
    }

    /* 同期コレクションの生成式を実行して値を返却します。 */
    private static Object evaluateSynchronizedCollection(OgdlEvent ev) {
        final Object val = evaluateUndecoratedCollection(ev);
        final Collection coll = Collections.synchronizedCollection((Collection) val);
        return coll;
    }

    /* 変更不可のコレクション生成式を実行して値を返却します。 */
    private static Object evaluateUnmodifiableCollection(OgdlEvent ev) {
        final Object val = evaluateUndecoratedCollection(ev);
        final Collection coll = Collections.unmodifiableCollection((Collection) val);
        return coll;
    }

    /*
     * data domain
     */

    /* 1.0 の BigDecimal */
    private static final BigDecimal BIGDECIMAL_ONE = new BigDecimal(BigInteger.ONE, 0);

    /* 数値領域（等差数列や時間領域）をデフォルトの公差で追加して返却します。 */
    private static void addNumberDomain(OgdlEvent ev, Collection coll, Object from, Object to) {
        if (from instanceof Number && to instanceof Number) {
            if (from instanceof Byte && to instanceof Byte) {
                Progressions.addArithmeticProgression(coll, (Byte) from, (Byte) to, (byte) 1);
            } else if (from instanceof Short && to instanceof Short) {
                Progressions.addArithmeticProgression(coll, (Short) from, (Short) to, (short) 1);
            } else if (from instanceof Integer && to instanceof Integer) {
                Progressions.addArithmeticProgression(coll, (Integer) from, (Integer) to, 1);
            } else if (from instanceof Long && to instanceof Long) {
                Progressions.addArithmeticProgression(coll, (Long) from, (Long) to, 1L);
            } else if (from instanceof Float && to instanceof Float) {
                Progressions.addArithmeticProgression(coll, (Float) from, (Float) to, 1);
            } else if (from instanceof Double && to instanceof Double) {
                Progressions.addArithmeticProgression(coll, (Double) from, (Double) to, 1L);
            } else if (from instanceof BigInteger && to instanceof BigInteger) {
                Progressions.addArithmeticProgression(coll, (BigInteger) from, (BigInteger) to, BigInteger.ONE);
            } else if (from instanceof BigDecimal && to instanceof BigDecimal) {
                Progressions.addArithmeticProgression(coll, (BigDecimal) from, (BigDecimal) to, BIGDECIMAL_ONE);
            } else {
                throw new OgdlSyntaxException(ev, "no domain type. ");
            }
        } else if (from instanceof Character && to instanceof Character) {
            Progressions.addArithmeticProgression(coll, (Character) from, (Character) to, 1);
        } else if (from instanceof Date && to instanceof Date) {
            Progressions.addDateTimeProgression(coll, (Date) from, (Date) to, Calendar.DATE, 1);
        } else {
            throw new OgdlSyntaxException(ev, "no domain type. ");
        }
    }

    /* 数値領域（等差数列や時間領域）を指定の公差で追加して返却します。 */
    private static void addNumberDomain(OgdlEvent ev, Collection coll, Object from, Object to, String difference) {
        try {
            if (from instanceof Number && to instanceof Number) {
                if (from instanceof Byte && to instanceof Byte) {
                    Progressions.addArithmeticProgression(coll, (Byte) from, (Byte) to, Byte.parseByte(difference));
                } else if (from instanceof Short && to instanceof Short) {
                    Progressions.addArithmeticProgression(coll, (Short) from, (Short) to, Short.parseShort(difference));
                } else if (from instanceof Integer && to instanceof Integer) {
                    Progressions.addArithmeticProgression(coll, (Integer) from, (Integer) to, Integer.parseInt(difference));
                } else if (from instanceof Long && to instanceof Long) {
                    Progressions.addArithmeticProgression(coll, (Long) from, (Long) to, Long.parseLong(difference));
                } else if (from instanceof Float && to instanceof Float) {
                    Progressions.addArithmeticProgression(coll, (Float) from, (Float) to, Integer.parseInt(difference));
                } else if (from instanceof Double && to instanceof Double) {
                    Progressions.addArithmeticProgression(coll, (Double) from, (Double) to, Long.parseLong(difference));
                } else if (from instanceof BigInteger && to instanceof BigInteger) {
                    Progressions.addArithmeticProgression(coll, (BigInteger) from, (BigInteger) to, new BigInteger(difference));
                } else if (from instanceof BigDecimal && to instanceof BigDecimal) {
                    Progressions.addArithmeticProgression(coll, (BigDecimal) from, (BigDecimal) to, new BigDecimal(difference));
                } else {
                    throw new OgdlSyntaxException(ev, "no domain type. ");
                }
            } else if (from instanceof Character && to instanceof Character) {
                Progressions.addArithmeticProgression(coll, (Character) from, (Character) to, Integer.parseInt(difference));
            } else if (from instanceof Date && to instanceof Date) {
                final int[] differences = toDateDifferences(ev, difference);
                if (differences[2] == 1) {
                    Progressions.addDateTimeInMillisProgression(coll, (Date) from, (Date) to, differences[0], differences[1]);
                } else {
                    Progressions.addDateTimeProgression(coll, (Date) from, (Date) to, differences[0], differences[1]);
                }
            } else {
                throw new OgdlSyntaxException(ev, "no domain type. ");
            }
        } catch (final NumberFormatException e) {
            throw new OgdlSyntaxException(ev, "number syntax err.", e);
        }
    }

    private static int[] toDateDifferences(OgdlEvent ev, String difference) {

        final String[] splits = difference.split("\\.");
        if (splits.length != 2 && splits.length != 3) {
            throw new OgdlSyntaxException(ev, Utils.log("range difference syntax err. difference=", difference));
        }
        final int[] differences = new int[] { Calendar.DATE, 1, 0 };
        if (splits.length == 3) {
            differences[2] = (splits[2].equalsIgnoreCase("L") ? 1 : 0);
        }
        differences[1] = Integer.parseInt(splits[1]);
        final Integer ofield = (Integer) getCalendarConstants().get(splits[0]);
        if (ofield != null) {
            differences[0] = ofield.intValue();
        } else {
            throw new OgdlSyntaxException(ev, Utils.log("not is calendar field. field=", splits[0]));
        }
        return differences;
    }

    private static Object newCollectionInstance(OgdlEvent ev) {

        final Class type = OgdlPrimitiveParser.evaluateClassByExtendName(ev);
        return BeanIntrospectHelper.newInstance(type);

    }

    private static Map calendarConstants;

    private static Map getCalendarConstants() {
        final Map consts = calendarConstants;
        return (consts != null) ? consts : (calendarConstants = BeanIntrospectHelper.getConstantFieldMap(Calendar.class));
    }

}
