/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 文字列をOGDL式として評価する機能を提供します。<br>
 * OGDLの詳細な仕様は概要説明を参照してください。
 */
public class Ogdl {

    /* コンテキスト */
    private OgdlContext _context;

    /**
     * 初期化します。
     */
    public Ogdl() {
        super();
    }

    /**
     * 文字列の先頭から末尾までを、一つの式として評価して返却します。<br>
     * 式の終端が文字列の末尾でない場合は OgdlSyntaxException を発生させます。 <br>
     * また文字列の前後の空白文字は無視されます。
     * 
     * @param pattern
     *            式構文
     * @return オブジェクト
     * @throws OgdlSyntaxException
     *             構文例外
     */
    public Object evaluate(String pattern) throws OgdlSyntaxException {
        final OgdlEvent ev = newOgdlEvent(pattern, new OgdlParseIndex(0));
        final Object rvalue = OgdlRuntime.evaluate(ev);
        if (ev.hasNext()) {
            throw new OgdlSyntaxException(ev, "is not end index.");
        }
        return rvalue;
    }

    /**
     * 指定された文字列の位置から、一つの式を評価して返却します。<br>
     * このメソッドは、式の終端が文字列の末尾でなくとも、例外を発生させません。<br>
     * 式の終端インデックスは引数の OgdlParseIndex に格納されて返却されます。<br>
     * また文字列の前後の空白文字は無視されます。
     * 
     * @param pattern
     *            式構文
     * @param offset
     *            開始インデックス
     * @return オブジェクト
     * @throws OgdlSyntaxException
     *             構文例外
     */
    public Object evaluate(String pattern, OgdlParseIndex offset) throws OgdlSyntaxException {
        final OgdlEvent ev = newOgdlEvent(pattern, offset);
        return OgdlRuntime.evaluate(ev);
    }

    /**
     * 指定された文字列の位置から、一つの式を評価して返却します。<br>
     * このメソッドは、式の終端が文字列の末尾でなくとも、例外を発生させません。<br>
     * 式の終端インデックスは引数の OgdlParseIndex に格納されて返却されます。<br>
     * また文字列の前後の空白文字は無視されます。
     * 
     * @param pattern
     *            式構文
     * @param offset
     *            開始インデックス
     * @return クラス
     * @throws OgdlSyntaxException
     *             構文例外
     */
    public Class evaluateClass(String pattern, OgdlParseIndex offset) throws OgdlSyntaxException {
        final OgdlEvent ev = newOgdlEvent(pattern, offset);
        return OgdlRuntime.evaluateClass(ev);
    }

    /**
     * 指定された文字列の位置から、コンマ区切りのリストとして評価して返却します。<br>
     * このメソッドは開始文字と終了文字の一致を検証します、正しく開始文字と終了文字で式の解析出来ない場合は例外が発生します。<br>
     * また文字列の前後の空白文字は無視されます。<br>
     * 式の終端インデックスは引数の OgdlParseIndex に格納されて返却されます。<br>
     * このメソッドは、式の終端が文字列の末尾でなくとも、例外を発生させません。
     * 
     * @param pattern
     *            式構文
     * @param offset
     *            開始インデックス
     * @param open
     *            開始を示すリテラル文字
     * @param close
     *            終了を示すリテラル文字
     * @return リスト
     * @throws OgdlSyntaxException
     *             構文例外
     */
    public List evaluateList(String pattern, OgdlParseIndex offset, char open, char close) throws OgdlSyntaxException {
        final OgdlEvent ev = newOgdlEvent(pattern, offset);
        return OgdlRuntime.evaluateList(ev.get(open, close));
    }

    /**
     * 指定された文字列の位置から、コンマ区切りのリストとして評価して返却します。<br>
     * このメソッドは終了文字の一致を検証します、正しく終了文字で式の終端を解析出来ない場合は例外が発生します。<br>
     * また文字列の前後の空白文字は無視されます。<br>
     * 式の終端インデックスは引数の OgdlParseIndex に格納されて返却されます。<br>
     * このメソッドは、式の終端が文字列の末尾でなくとも、例外を発生させません。
     * 
     * @param pattern
     *            式構文
     * @param offset
     *            開始インデックス
     * @param close
     *            終了を示すリテラル文字
     * @return リスト
     * @throws OgdlSyntaxException
     *             構文例外
     */
    public List evaluateList(String pattern, OgdlParseIndex offset, char close) throws OgdlSyntaxException {
        final OgdlEvent ev = newOgdlEvent(pattern, offset);
        return OgdlRuntime.evaluateCloseList(ev.get('\u0000', close));
    }

    /**
     * 指定された文字列の位置から、コンマ区切りのセットとして評価して返却します。<br>
     * このメソッドは開始文字と終了文字の一致を検証します、正しく開始文字と終了文字で式の解析出来ない場合は例外が発生します。<br>
     * また文字列の前後の空白文字は無視されます。<br>
     * 式の終端インデックスは引数の OgdlParseIndex に格納されて返却されます。<br>
     * このメソッドは、式の終端が文字列の末尾でなくとも、例外を発生させません。
     * 
     * @param pattern
     *            式構文
     * @param offset
     *            開始インデックス
     * @param open
     *            開始を示すリテラル文字
     * @param close
     *            終了を示すリテラル文字
     * @return セット
     * @throws OgdlSyntaxException
     *             構文例外
     */
    public Set evaluateSet(String pattern, OgdlParseIndex offset, char open, char close) throws OgdlSyntaxException {
        final OgdlEvent ev = newOgdlEvent(pattern, offset);
        return OgdlRuntime.evaluateSet(ev.get(open, close));
    }

    /**
     * 指定された文字列の位置から、コンマ区切りのセットとして評価して返却します。<br>
     * このメソッドは終了文字の一致を検証します、正しく終了文字で式の終端を解析出来ない場合は例外が発生します。<br>
     * また文字列の前後の空白文字は無視されます。<br>
     * 式の終端インデックスは引数の OgdlParseIndex に格納されて返却されます。<br>
     * このメソッドは、式の終端が文字列の末尾でなくとも、例外を発生させません。
     * 
     * @param pattern
     *            式構文
     * @param offset
     *            開始インデックス
     * @param close
     *            終了を示すリテラル文字
     * @return セット
     * @throws OgdlSyntaxException
     *             構文例外
     */
    public Set evaluateSet(String pattern, OgdlParseIndex offset, char close) throws OgdlSyntaxException {
        final OgdlEvent ev = newOgdlEvent(pattern, offset);
        return OgdlRuntime.evaluateCloseSet(ev.get('\u0000', close));
    }

    /**
     * 指定された文字列の位置から、'=' で対応付けられたマッピングをコンマ区切りで列挙したマップとして評価して返却します。<br>
     * このメソッドは開始文字と終了文字の一致を検証します、正しく開始文字と終了文字で式の解析出来ない場合は例外が発生します。<br>
     * また文字列の前後の空白文字は無視されます。<br>
     * 式の終端インデックスは引数の OgdlParseIndex に格納されて返却されます。<br>
     * このメソッドは、式の終端が文字列の末尾でなくとも、例外を発生させません。
     * 
     * @param pattern
     *            式構文
     * @param offset
     *            開始インデックス
     * @param open
     *            開始を示すリテラル文字
     * @param close
     *            終了を示すリテラル文字
     * @return マップ
     * @throws OgdlSyntaxException
     *             構文例外
     */
    public Map evaluateMap(String pattern, OgdlParseIndex offset, char open, char close) throws OgdlSyntaxException {
        final OgdlEvent ev = newOgdlEvent(pattern, offset);
        return OgdlRuntime.evaluateMap(ev.get(open, close));
    }

    /**
     * 指定された文字列の位置から、'=' で対応付けられたマッピングをコンマ区切りで列挙したマップとして評価して返却します。<br>
     * このメソッドは終了文字の一致を検証します、正しく終了文字で式の終端を解析出来ない場合は例外が発生します。<br>
     * また文字列の前後の空白文字は無視されます。<br>
     * 式の終端インデックスは引数の OgdlParseIndex に格納されて返却されます。<br>
     * このメソッドは、式の終端が文字列の末尾でなくとも、例外を発生させません。
     * 
     * @param pattern
     *            式構文
     * @param offset
     *            開始インデックス
     * @param close
     *            終了を示すリテラル文字
     * @return マップ
     * @throws OgdlSyntaxException
     *             構文例外
     */
    public Map evaluateMap(String pattern, OgdlParseIndex offset, char close) throws OgdlSyntaxException {
        final OgdlEvent ev = newOgdlEvent(pattern, offset);
        return OgdlRuntime.evaluateCloseMap(ev.get('\u0000', close));
    }

    /**
     * コンテキストを取得します。
     * 
     * @return コンテキスト
     */
    public OgdlContext getContext() {
        return _context;
    }

    /**
     * コンテキストを格納します。
     * 
     * @param context
     *            コンテキスト
     */
    public void setContext(OgdlContext context) {
        this._context = context;
    }

    /**
     * OGDLイベントを生成して返却します。
     * 
     * @param pattern
     *            式構文
     * @param offset
     *            開始インデックス
     * @return OGDLイベント
     */
    OgdlEvent newOgdlEvent(String pattern, OgdlParseIndex offset) {
        return new OgdlEvent(pattern, offset, findClassLoader(), findContext());
    }

    /**
     * 実行可能なクラスローダを検索し取得します。
     * 
     * @return クラスローダ
     */
    ClassLoader findClassLoader() {
        return (_context.getClassLoader() == null) ? getClass().getClassLoader() : _context.getClassLoader();
    }

    /**
     * 実行可能なコンテキストを検索し取得します。
     * 
     * @return コンテキスト
     */
    OgdlContext findContext() {
        return (_context == null) ? OgdlHelper.getOgdlContext() : _context;
    }

}
