/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

/**
 * プリミティブ型とラッパークラスを相互に交換する機能を提供します。 明示的な Boxing/Unboxing 機能です。<br>
 * プリミティブ配列のリストビュー、配列変換等の有用な機能も多く提供しますが、JDK 1.5 以降では無用な機能を含みます。
 */
final class Boxing {

    /*
     * class
     */

    /**
     * プリミティブ型のクラスをラッパー型のクラスに変換し返却します。 <br>
     * 引数がプリミティブ型でない場合 IllegalArgumentException を発生します。
     * 
     * @param primitiveType
     *            プリミティブ型のクラス
     * @return ラッパー型のクラス
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static Class boxClass(Class primitiveType) {

        if (primitiveType == null) {
            throw new NullPointerException();
        }

        if (Boolean.TYPE.equals(primitiveType)) {
            return Boolean.class;
        } else if (Character.TYPE.equals(primitiveType)) {
            return Character.class;
        } else if (Byte.TYPE.equals(primitiveType)) {
            return Byte.class;
        } else if (Short.TYPE.equals(primitiveType)) {
            return Short.class;
        } else if (Integer.TYPE.equals(primitiveType)) {
            return Integer.class;
        } else if (Long.TYPE.equals(primitiveType)) {
            return Long.class;
        } else if (Float.TYPE.equals(primitiveType)) {
            return Float.class;
        } else if (Double.TYPE.equals(primitiveType)) {
            return Double.class;
        } else if (Void.TYPE.equals(primitiveType)) {
            return Void.class;
        } else {
            throw new IllegalArgumentException("It is not a primitive type.");
        }
    }

    /**
     * ラッパー型のクラスをプリミティブ型のクラスに変換し返却します。 <br>
     * 引数がラッパークラスでない場合 IllegalArgumentException を発生します。
     * 
     * @param wrpperType
     *            ラッパー型のクラス
     * @return プリミティブ型のクラス
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static Class unboxClass(Class wrpperType) {

        if (wrpperType == null) {
            throw new NullPointerException();
        }

        if (Boolean.class.equals(wrpperType)) {
            return Boolean.TYPE;
        } else if (Character.class.equals(wrpperType)) {
            return Character.TYPE;
        } else if (Byte.class.equals(wrpperType)) {
            return Byte.TYPE;
        } else if (Short.class.equals(wrpperType)) {
            return Short.TYPE;
        } else if (Integer.class.equals(wrpperType)) {
            return Integer.TYPE;
        } else if (Long.class.equals(wrpperType)) {
            return Long.TYPE;
        } else if (Float.class.equals(wrpperType)) {
            return Float.TYPE;
        } else if (Double.class.equals(wrpperType)) {
            return Double.TYPE;
        } else if (Void.class.equals(wrpperType)) {
            return Void.TYPE;
        } else {
            throw new IllegalArgumentException("It is not a primitive wrpper type.");
        }
    }

    /*
     * wrapper object to primitive (throws NullPointerException)
     */

    /**
     * プリミティブ値に変換して返却します。
     * 
     * @param src
     *            変換元の値
     * @return 変換されたプリミティブ値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static char unbox(Character src) {
        return src.charValue();
    }

    /**
     * プリミティブ値に変換して返却します。
     * 
     * @param src
     *            変換元の値
     * @return 変換されたプリミティブ値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static boolean unbox(Boolean src) {
        return src.booleanValue();
    }

    /**
     * プリミティブ値に変換して返却します。
     * 
     * @param src
     *            変換元の値
     * @return 変換されたプリミティブ値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static byte unbox(Byte src) {
        return src.byteValue();
    }

    /**
     * プリミティブ値に変換して返却します。
     * 
     * @param src
     *            変換元の値
     * @return 変換されたプリミティブ値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static short unbox(Short src) {
        return src.shortValue();
    }

    /**
     * プリミティブ値に変換して返却します。
     * 
     * @param src
     *            変換元の値
     * @return 変換されたプリミティブ値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static int unbox(Integer src) {
        return src.intValue();
    }

    /**
     * プリミティブ値に変換して返却します。
     * 
     * @param src
     *            変換元の値
     * @return 変換されたプリミティブ値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static long unbox(Long src) {
        return src.longValue();
    }

    /**
     * プリミティブ値に変換して返却します。
     * 
     * @param src
     *            変換元の値
     * @return 変換されたプリミティブ値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static float unbox(Float src) {
        return src.floatValue();
    }

    /**
     * プリミティブ値に変換して返却します。
     * 
     * @param src
     *            変換元の値
     * @return 変換されたプリミティブ値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static double unbox(Double src) {
        return src.doubleValue();
    }

    /*
     * wrapper object to primitive (null as nullValue)
     */

    /**
     * null の場合 nullValue を以外はプリミティブ値に変換して返却します。
     * 
     * @param src
     *            変換元の値
     * @param nullValue
     *            変換元が null の場合に設定する値
     * @return 変換されたプリミティブ値
     */
    public static char unbox(Character src, char nullValue) {
        return (src == null) ? nullValue : src.charValue();
    }

    /**
     * null の場合 nullValue を以外はプリミティブ値に変換して返却します。
     * 
     * @param src
     *            変換元の値
     * @param nullValue
     *            変換元が null の場合に設定する値
     * @return 変換されたプリミティブ値
     */
    public static boolean unbox(Boolean src, boolean nullValue) {
        return (src == null) ? nullValue : src.booleanValue();
    }

    /**
     * null の場合 nullValue を以外はプリミティブ値に変換して返却します。
     * 
     * @param src
     *            変換元の値
     * @param nullValue
     *            変換元が null の場合に設定する値
     * @return 変換されたプリミティブ値
     */
    public static byte unbox(Byte src, byte nullValue) {
        return (src == null) ? nullValue : src.byteValue();
    }

    /**
     * null の場合 nullValue を以外はプリミティブ値に変換して返却します。
     * 
     * @param src
     *            変換元の値
     * @param nullValue
     *            変換元が null の場合に設定する値
     * @return 変換されたプリミティブ値
     */
    public static short unbox(Short src, short nullValue) {
        return (src == null) ? nullValue : src.shortValue();
    }

    /**
     * null の場合 nullValue を以外はプリミティブ値に変換して返却します。
     * 
     * @param src
     *            変換元の値
     * @param nullValue
     *            変換元が null の場合に設定する値
     * @return 変換されたプリミティブ値
     */
    public static int unbox(Integer src, int nullValue) {
        return (src == null) ? nullValue : src.intValue();
    }

    /**
     * null の場合 nullValue を以外はプリミティブ値に変換して返却します。
     * 
     * @param src
     *            変換元の値
     * @param nullValue
     *            変換元が null の場合に設定する値
     * @return 変換されたプリミティブ値
     */
    public static long unbox(Long src, long nullValue) {
        return (src == null) ? nullValue : src.longValue();
    }

    /**
     * null の場合 nullValue を以外はプリミティブ値に変換して返却します。
     * 
     * @param src
     *            変換元の値
     * @param nullValue
     *            変換元が null の場合に設定する値
     * @return 変換されたプリミティブ値
     */
    public static float unbox(Float src, float nullValue) {
        return (src == null) ? nullValue : src.floatValue();
    }

    /**
     * null の場合 nullValue を以外はプリミティブ値に変換して返却します。
     * 
     * @param src
     *            変換元の値
     * @param nullValue
     *            変換元が null の場合に設定する値
     * @return 変換されたプリミティブ値
     */
    public static double unbox(Double src, double nullValue) {
        return (src == null) ? nullValue : src.doubleValue();
    }

    /*
     * primitive to wrapper object
     */

    /*
     * 2005/11/06 - static 変数の初期化以前にメソッドから値が返却される場合があり得るため、 クラスローダーの仕様を利用してstatic 変数を確実に生成する。
     */

    private static class CharacterCache {
        static final Character cache[] = new Character[127 + 1];

        static {
            for (int i = 0; i < cache.length; i++) {
                cache[i] = new Character((char) i);
            }
        }
    }

    /**
     * プリミティブ型をラッパー型に変換して返却します。
     * 
     * @param src
     *            変換元のプリミティブ型
     * @return 変換されたラッパー型
     */
    public static Character box(char src) {
        if (src <= 127) {
            return CharacterCache.cache[src];
        }
        return new Character(src);
    }

    /**
     * プリミティブ型をラッパー型に変換して返却します。
     * 
     * @param src
     *            変換元のプリミティブ型
     * @return 変換されたラッパー型
     */
    public static Boolean box(boolean src) {
        return Boolean.valueOf(src);
    }

    private static class ByteCache {
        static final Byte cache[] = new Byte[-(-128) + 127 + 1];

        static {
            for (int i = 0; i < cache.length; i++) {
                cache[i] = new Byte((byte) (i - 128));
            }
        }
    }

    /**
     * プリミティブ型をラッパー型に変換して返却します。
     * 
     * @param src
     *            変換元のプリミティブ型
     * @return 変換されたラッパー型
     */
    public static Byte box(byte src) {
        return ByteCache.cache[src + 128];// (src + offset)
    }

    private static class ShortCache {
        static final Short cache[] = new Short[-(-128) + 127 + 1];

        static {
            for (int i = 0; i < cache.length; i++) {
                cache[i] = new Short((short) (i - 128));
            }
        }
    }

    /**
     * プリミティブ型をラッパー型に変換して返却します。
     * 
     * @param src
     *            変換元のプリミティブ型
     * @return 変換されたラッパー型
     */
    public static Short box(short src) {
        final int asInt = src;
        if (asInt >= -128 && asInt <= 127) {
            return ShortCache.cache[asInt + 128];// (src + offset)
        }
        return new Short(src);
    }

    private static class IntegerCache {
        static final Integer cache[] = new Integer[-(-128) + 127 + 1];

        static {
            for (int i = 0; i < cache.length; i++) {
                cache[i] = new Integer(i - 128);
            }
        }
    }

    /**
     * プリミティブ型をラッパー型に変換して返却します。
     * 
     * @param src
     *            変換元のプリミティブ型
     * @return 変換されたラッパー型
     */
    public static Integer box(int src) {
        if (src >= -128 && src <= 127) {
            return IntegerCache.cache[src + 128];// (src + offset)
        }
        return new Integer(src);
    }

    private static class LongCache {
        static final Long cache[] = new Long[-(-128) + 127 + 1];

        static {
            for (int i = 0; i < cache.length; i++) {
                cache[i] = new Long(i - 128);
            }
        }
    }

    /**
     * プリミティブ型をラッパー型に変換して返却します。
     * 
     * @param src
     *            変換元のプリミティブ型
     * @return 変換されたラッパー型
     */
    public static Long box(long src) {
        if (src >= -128 && src <= 127) {
            return LongCache.cache[(int) src + 128];// (src + offset)
        }
        return new Long(src);
    }

    /**
     * プリミティブ型をラッパー型に変換して返却します。
     * 
     * @param src
     *            変換元のプリミティブ型
     * @return 変換されたラッパー型
     */
    public static Float box(float src) {
        return new Float(src);
    }

    /**
     * プリミティブ型をラッパー型に変換して返却します。
     * 
     * @param src
     *            変換元のプリミティブ型
     * @return 変換されたラッパー型
     */
    public static Double box(double src) {
        return new Double(src);
    }

    /*
     * wrapper object to primitive (throws NullPointerException)
     */

    /**
     * プリミティブ型の配列に変換して返却します。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @throws NullPointerException
     *             引数がまたは要素が null の場合
     */
    public static char[] unbox(Character[] src) {
        final int size = src.length;
        final char[] a = new char[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返却します。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @throws NullPointerException
     *             引数がまたは要素が null の場合
     */
    public static boolean[] unbox(Boolean[] src) {
        final int size = src.length;
        final boolean[] a = new boolean[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返却します。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @throws NullPointerException
     *             引数がまたは要素が null の場合
     */
    public static byte[] unbox(Byte[] src) {
        final int size = src.length;
        final byte[] a = new byte[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返却します。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @throws NullPointerException
     *             引数がまたは要素が null の場合
     */
    public static short[] unbox(Short[] src) {
        final int size = src.length;
        final short[] a = new short[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返却します。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @throws NullPointerException
     *             引数がまたは要素が null の場合
     */
    public static int[] unbox(Integer[] src) {
        final int size = src.length;
        final int[] a = new int[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返却します。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @throws NullPointerException
     *             引数がまたは要素が null の場合
     */
    public static long[] unbox(Long[] src) {
        final int size = src.length;
        final long[] a = new long[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返却します。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @throws NullPointerException
     *             引数がまたは要素が null の場合
     */
    public static float[] unbox(Float[] src) {
        final int size = src.length;
        final float[] a = new float[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返却します。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @throws NullPointerException
     *             引数がまたは要素が null の場合
     */
    public static double[] unbox(Double[] src) {
        final int size = src.length;
        final double[] a = new double[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i]);
        }
        return a;
    }

    /*
     * wrapper object to primitive (element null as nullValue)
     */

    /**
     * プリミティブ型の配列に変換して返却します、要素が null の場合 nullValue を割り当てます。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @param nullValue
     *            変換元の要素が null の場合に設定する値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static char[] unbox(Character[] src, char nullValue) {
        final int size = src.length;
        final char[] a = new char[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i], nullValue);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返却します、要素が null の場合 nullValue を割り当てます。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @param nullValue
     *            変換元の要素が null の場合に設定する値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static boolean[] unbox(Boolean[] src, boolean nullValue) {
        final int size = src.length;
        final boolean[] a = new boolean[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i], nullValue);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返却します、要素が null の場合 nullValue を割り当てます。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @param nullValue
     *            変換元の要素が null の場合に設定する値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static byte[] unbox(Byte[] src, byte nullValue) {
        final int size = src.length;
        final byte[] a = new byte[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i], nullValue);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返却します、要素が null の場合 nullValue を割り当てます。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @param nullValue
     *            変換元の要素が null の場合に設定する値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static short[] unbox(Short[] src, short nullValue) {
        final int size = src.length;
        final short[] a = new short[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i], nullValue);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返却します、要素が null の場合 nullValue を割り当てます。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @param nullValue
     *            変換元の要素が null の場合に設定する値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static int[] unbox(Integer[] src, int nullValue) {
        final int size = src.length;
        final int[] a = new int[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i], nullValue);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返却します、要素が null の場合 nullValue を割り当てます。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @param nullValue
     *            変換元の要素が null の場合に設定する値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static long[] unbox(Long[] src, long nullValue) {
        final int size = src.length;
        final long[] a = new long[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i], nullValue);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返却します、要素が null の場合 nullValue を割り当てます。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @param nullValue
     *            変換元の要素が null の場合に設定する値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static float[] unbox(Float[] src, float nullValue) {
        final int size = src.length;
        final float[] a = new float[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i], nullValue);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返却します、要素が null の場合 nullValue を割り当てます。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @param nullValue
     *            変換元の要素が null の場合に設定する値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static double[] unbox(Double[] src, double nullValue) {
        final int size = src.length;
        final double[] a = new double[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i], nullValue);
        }
        return a;
    }

    /*
     * primitive to wrapper object
     */

    /**
     * プリミティブ型の配列をラッパー型の配列に変換して返却します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換されたラッパー型の配列
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static Character[] box(char[] src) {
        final int len = src.length;
        final Character[] a = new Character[len];
        for (int i = 0; i < len; i++) {
            a[i] = box(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列をラッパー型の配列に変換して返却します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換されたラッパー型の配列
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static Boolean[] box(boolean[] src) {
        final int len = src.length;
        final Boolean[] a = new Boolean[len];
        for (int i = 0; i < len; i++) {
            a[i] = box(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列をラッパー型の配列に変換して返却します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換されたラッパー型の配列
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static Byte[] box(byte[] src) {
        final int len = src.length;
        final Byte[] a = new Byte[len];
        for (int i = 0; i < len; i++) {
            a[i] = box(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列をラッパー型の配列に変換して返却します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換されたラッパー型の配列
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static Short[] box(short[] src) {
        final int len = src.length;
        final Short[] a = new Short[len];
        for (int i = 0; i < len; i++) {
            a[i] = box(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列をラッパー型の配列に変換して返却します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換されたラッパー型の配列
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static Integer[] box(int[] src) {
        final int len = src.length;
        final Integer[] a = new Integer[len];
        for (int i = 0; i < len; i++) {
            a[i] = box(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列をラッパー型の配列に変換して返却します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換されたラッパー型の配列
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static Long[] box(long[] src) {
        final int len = src.length;
        final Long[] a = new Long[len];
        for (int i = 0; i < len; i++) {
            a[i] = box(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列をラッパー型の配列に変換して返却します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換されたラッパー型の配列
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static Float[] box(float[] src) {
        final int len = src.length;
        final Float[] a = new Float[len];
        for (int i = 0; i < len; i++) {
            a[i] = box(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列をラッパー型の配列に変換して返却します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換されたラッパー型の配列
     * @throws NullPointerException
     *             引数が null の場合
     */
    public static Double[] box(double[] src) {
        final int len = src.length;
        final Double[] a = new Double[len];
        for (int i = 0; i < len; i++) {
            a[i] = box(src[i]);
        }
        return a;
    }

}
