/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.List;

/**
 * オブジェクトの算術演算機能を提供します。<br>
 * プリミティブ型の算術演算と同等の機能と、オブジェクト向けの拡張された演算子も提供します。
 */
final class Arithmetic {

    /*
     * Arithmetic Operator
     */

    /** 加算演算子または文字列連結演算子 */
    public static final int ARITHMETIC_ADDITION = 0x000;

    /** 減算演算子 */
    public static final int ARITHMETIC_SUBTRACTION = 0x001;

    /** 乗算演算子 */
    public static final int ARITHMETIC_MULTIPLICATION = 0x002;

    /** 除算演算子 */
    public static final int ARITHMETIC_DIVISION = 0x003;

    /** 剰余演算子 */
    public static final int ARITHMETIC_REMAINDER = 0x004;

    /** 文字列構築演算子 */
    public static final int ARITHMETIC_STRING_BUILDING = 0x005;

    /*
     * Bitwise Operator
     */

    /** ビットAND演算子 */
    public static final int BITWISE_AND = 0x010;

    /** ビットXOR演算子 */
    public static final int BITWISE_XOR = 0x011;

    /** ビットOR演算子 */
    public static final int BITWISE_OR = 0x012;

    /*
     * Shift Operator
     */

    /** 左シフト演算子 */
    public static final int BIT_SHIFT_LEFT = 0x020;

    /** 符号付き右シフト演算子 */
    public static final int BIT_SHIFT_RIGHT = 0x021;

    /** 符号なし右シフト演算子 */
    public static final int BIT_SHIFT_LOGICAL_RIGHT = 0x022;

    /*
     * Comparative Operator
     */

    /** 等価演算子 */
    public static final int COMPARATIVE_EQUALITY = 0x030;

    /** 不等価演算子 */
    public static final int COMPARATIVE_INEQUALITY = 0x031;

    /** 比較演算子：小なり */
    public static final int COMPARATIVE_LESS_THAN = 0x032;

    /** 比較演算子：大なり */
    public static final int COMPARATIVE_GREATER_THAN = 0x033;

    /** 比較演算子：小なりまたは同一 */
    public static final int COMPARATIVE_LESS_EQUALITY = 0x034;

    /** 比較演算子：大なりまたは同一 */
    public static final int COMPARATIVE_GREATER_EQUALITY = 0x035;

    /*
     * Relational Operator
     */

    /** オブジェクト同値演算子 */
    public static final int RELATIONAL_OBJECT_EQUALITY = 0x041;

    /** オブジェクト不同値演算子 */
    public static final int RELATIONAL_OBJECT_INEQUALITY = 0x042;

    /** 参照同一演算子 */
    public static final int RELATIONAL_REFERENCE_EQUALITY = 0x043;

    /** 参照不同一演算子 */
    public static final int RELATIONAL_REFERENCE_INEQUALITY = 0x044;

    /** 型比較演算子 */
    public static final int RELATIONAL_INSTANCE_OF = 0x045;

    /*
     * 単項演算子
     */

    /**
     * 単項マイナス演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Integer minus(Character v) {
        return Boxing.box(-(v.charValue()));
    }

    /**
     * 単項マイナス演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Integer minus(Byte v) {
        return Boxing.box(-(v.byteValue()));
    }

    /**
     * 単項マイナス演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Integer minus(Short v) {
        return Boxing.box(-(v.shortValue()));
    }

    /**
     * 単項マイナス演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Integer minus(Integer v) {
        return Boxing.box(-(v.intValue()));
    }

    /**
     * 単項マイナス演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Long minus(Long v) {
        return Boxing.box(-(v.longValue()));
    }

    /**
     * 単項マイナス演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Float minus(Float v) {
        return Boxing.box(-(v.floatValue()));
    }

    /**
     * 単項マイナス演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Double minus(Double v) {
        return Boxing.box(-(v.doubleValue()));
    }

    /**
     * 単項マイナス演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static BigInteger minus(BigInteger v) {
        return v.negate();
    }

    /**
     * 単項マイナス演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static BigDecimal minus(BigDecimal v) {
        return v.negate();
    }

    /**
     * 単項プラス演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Integer plus(Character v) {
        return Boxing.box((int) v.charValue());
    }

    /**
     * 単項プラス演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Integer plus(Byte v) {
        return Boxing.box(v.intValue());
    }

    /**
     * 単項プラス演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Integer plus(Short v) {
        return Boxing.box(v.intValue());
    }

    /**
     * 単項プラス演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Integer plus(Integer v) {
        return v;
    }

    /**
     * 単項プラス演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Long plus(Long v) {
        return v;
    }

    /**
     * 単項プラス演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Float plus(Float v) {
        return v;
    }

    /**
     * 単項プラス演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Double plus(Double v) {
        return v;
    }

    /**
     * 単項プラス演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static BigInteger plus(BigInteger v) {
        return v;
    }

    /**
     * 単項プラス演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static BigDecimal plus(BigDecimal v) {
        return v;
    }

    /**
     * ビット否定演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Integer bnot(Character v) {
        return Boxing.box(~(v.charValue()));
    }

    /**
     * ビット否定演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Integer bnot(Byte v) {
        return Boxing.box(~(v.byteValue()));
    }

    /**
     * ビット否定演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Integer bnot(Short v) {
        return Boxing.box(~(v.shortValue()));
    }

    /**
     * ビット否定演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Integer bnot(Integer v) {
        return Boxing.box(~(v.intValue()));
    }

    /**
     * ビット否定演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Long bnot(Long v) {
        return Boxing.box(~(v.longValue()));
    }

    /**
     * ビット否定演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static BigInteger bnot(BigInteger v) {
        return v.not();
    }

    /**
     * 論理否定演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Boolean not(Boolean v) {
        return Boxing.box(!v.booleanValue());
    }

    /**
     * 単項プラス演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Object plus(Object v) {
        if (v instanceof Character) {
            return plus((Character) v);
        }
        if (v instanceof Byte) {
            return plus((Byte) v);
        }
        if (v instanceof Short) {
            return plus((Short) v);
        }
        if (v instanceof Integer) {
            return plus((Integer) v);
        }
        if (v instanceof Long) {
            return plus((Long) v);
        }
        if (v instanceof Float) {
            return plus((Float) v);
        }
        if (v instanceof Double) {
            return plus((Double) v);
        }
        if (v instanceof BigInteger) {
            return plus((BigInteger) v);
        }
        if (v instanceof BigDecimal) {
            return plus((BigDecimal) v);
        }
        throw new ArithmeticException("is not Number or Character. " + v);
    }

    /**
     * 単項マイナス演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Object minus(Object v) {
        if (v instanceof Character) {
            return minus((Character) v);
        }
        if (v instanceof Byte) {
            return minus((Byte) v);
        }
        if (v instanceof Short) {
            return minus((Short) v);
        }
        if (v instanceof Integer) {
            return minus((Integer) v);
        }
        if (v instanceof Long) {
            return minus((Long) v);
        }
        if (v instanceof Float) {
            return minus((Float) v);
        }
        if (v instanceof Double) {
            return minus((Double) v);
        }
        if (v instanceof BigInteger) {
            return minus((BigInteger) v);
        }
        if (v instanceof BigDecimal) {
            return minus((BigDecimal) v);
        }
        throw new ArithmeticException("is not Number or Character. " + v);
    }

    /**
     * ビット否定演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Object bitwiseNot(Object v) {
        if (v instanceof Character) {
            return bnot((Character) v);
        }
        if (v instanceof Byte) {
            return bnot((Byte) v);
        }
        if (v instanceof Short) {
            return bnot((Short) v);
        }
        if (v instanceof Integer) {
            return bnot((Integer) v);
        }
        if (v instanceof Long) {
            return bnot((Long) v);
        }
        if (v instanceof BigInteger) {
            return bnot((BigInteger) v);
        }
        if (v instanceof Number) {
            throw new ArithmeticException("is not integer. " + v);
        }
        throw new ArithmeticException("is not Number or Character. " + v);
    }

    /**
     * 論理否定演算を実行し結果を返却します。
     * 
     * @param v
     *            値
     * @return 演算結果
     */
    public static Object logicalNot(Object v) {
        if (v instanceof Boolean) {
            return not((Boolean) v);
        }
        throw new ArithmeticException("is not Boolean. " + v);
    }

    /*
     * 二項演算子
     */

    /**
     * 引数が双方に数値か文字型の場合は一項に二項を加算し、片方か双方が文字列型の場合は一項に二項を文字列連結して結果を返却します。
     * 
     * @param v1
     *            値１
     * @param v2
     *            値２
     * @return 演算結果
     */
    public static Object add(Object v1, Object v2) {
        return ArithmeticHelper.add(v1, v2);
    }

    /**
     * 一項を二項で減算し結果を返却します。
     * 
     * @param v1
     *            値１
     * @param v2
     *            値２
     * @return 演算結果
     */
    public static Object subtract(Object v1, Object v2) {
        return ArithmeticHelper.arithmetic(ARITHMETIC_SUBTRACTION, v1, v2);
    }

    /**
     * 一項を二項で乗算し結果を返却します。
     * 
     * @param v1
     *            値１
     * @param v2
     *            値２
     * @return 演算結果
     */
    public static Object multiply(Object v1, Object v2) {
        return ArithmeticHelper.arithmetic(ARITHMETIC_MULTIPLICATION, v1, v2);
    }

    /**
     * 一項を二項で除算し結果を返却します。
     * 
     * @param v1
     *            値１
     * @param v2
     *            値２
     * @return 演算結果
     */
    public static Object div(Object v1, Object v2) {
        return ArithmeticHelper.arithmetic(ARITHMETIC_DIVISION, v1, v2);
    }

    /**
     * 一項に対する二項の剰余を計算し結果を返却します。
     * 
     * @param v
     *            計算元の値
     * @param modulus
     *            除数
     * @return 演算結果
     */
    public static Object mod(Object v, Object modulus) {
        return ArithmeticHelper.arithmetic(ARITHMETIC_REMAINDER, v, modulus);
    }

    /**
     * 一項に対する二項の左シフト演算を実行し結果を返却します。
     * 
     * @param v
     *            計算元の値
     * @param n
     *            シフト数
     * @return 演算結果
     */
    public static Object shl(Object v, Object n) {
        return ArithmeticHelper.shift(BIT_SHIFT_LEFT, v, n);
    }

    /**
     * 一項に対する二項の符号付き右シフト演算を実行し結果を返却します。
     * 
     * @param v
     *            計算元の値
     * @param n
     *            シフト数
     * @return 演算結果
     */
    public static Object shr(Object v, Object n) {
        return ArithmeticHelper.shift(BIT_SHIFT_RIGHT, v, n);
    }

    /**
     * 一項に対する二項の符号なし右シフト演算を実行し結果を返却します。
     * 
     * @param v
     *            計算元の値
     * @param n
     *            シフト数
     * @return 演算結果
     */
    public static Object ushr(Object v, Object n) {
        return ArithmeticHelper.shift(BIT_SHIFT_LOGICAL_RIGHT, v, n);
    }

    /**
     * 一項に対する二項のビットAND演算を実行し結果を返却します。
     * 
     * @param v1
     *            値１
     * @param v2
     *            値２
     * @return 演算結果
     */
    public static Object and(Object v1, Object v2) {
        return ArithmeticHelper.bitwise(BITWISE_AND, v1, v2);
    }

    /**
     * 一項に対する二項のビットOR演算を実行し結果を返却します。
     * 
     * @param v1
     *            値１
     * @param v2
     *            値２
     * @return 演算結果
     */
    public static Object or(Object v1, Object v2) {
        return ArithmeticHelper.bitwise(BITWISE_OR, v1, v2);
    }

    /**
     * 一項に対する二項のビットXOR演算を実行し結果を返却します。
     * 
     * @param v1
     *            値１
     * @param v2
     *            値２
     * @return 演算結果
     */
    public static Object xor(Object v1, Object v2) {
        return ArithmeticHelper.bitwise(BITWISE_XOR, v1, v2);
    }

    /**
     * 一項に対する二項の等価演算を実行し結果を返却します。<br>
     * 引数の双方が数値の場合は数値の拡張変換を行い、その結果の値を比較する。<br>
     * 引数は Comparable 型の必要がある。
     * 
     * @param v1
     *            値１
     * @param v2
     *            値２
     * @return 演算結果
     */
    public static Boolean eq(Object v1, Object v2) {
        return Boxing.box(ArithmeticHelper.compare(v1, v2) == 0);
    }

    /**
     * 一項に対する二項の不等価演算を実行し結果を返却します。<br>
     * 引数の双方が数値の場合は数値の拡張変換を行い、その結果の値を比較する。<br>
     * 引数は Comparable 型の必要がある。
     * 
     * @param v1
     *            値１
     * @param v2
     *            値２
     * @return 演算結果
     */
    public static Boolean ne(Object v1, Object v2) {
        return Boxing.box(ArithmeticHelper.compare(v1, v2) != 0);
    }

    /**
     * 一項に対する二項の小なり演算を実行し結果を返却します。<br>
     * 引数の双方が数値の場合は数値の拡張変換を行い、その結果の値を比較する。<br>
     * 引数は Comparable 型の必要がある。
     * 
     * @param v1
     *            値１
     * @param v2
     *            値２
     * @return 演算結果
     */
    public static Boolean lt(Object v1, Object v2) {
        return Boxing.box(ArithmeticHelper.compare(v1, v2) < 0);
    }

    /**
     * 一項に対する二項の大なり演算を実行し結果を返却します。<br>
     * 引数の双方が数値の場合は数値の拡張変換を行い、その結果の値を比較する。<br>
     * 引数は Comparable 型の必要がある。
     * 
     * @param v1
     *            値１
     * @param v2
     *            値２
     * @return 演算結果
     */
    public static Boolean gt(Object v1, Object v2) {
        return Boxing.box(ArithmeticHelper.compare(v1, v2) > 0);
    }

    /**
     * 一項に対する二項の等価または小なり演算を実行し結果を返却します。<br>
     * 引数の双方が数値の場合は数値の拡張変換を行い、その結果の値を比較する。<br>
     * 引数は Comparable 型の必要がある。
     * 
     * @param v1
     *            値１
     * @param v2
     *            値２
     * @return 演算結果
     */
    public static Boolean le(Object v1, Object v2) {
        return Boxing.box(ArithmeticHelper.compare(v1, v2) <= 0);
    }

    /**
     * 一項に対する二項の等価または大なり演算を実行し結果を返却します。<br>
     * 引数の双方が数値の場合は数値の拡張変換を行い、その結果の値を比較する。<br>
     * 引数は Comparable 型の必要がある。
     * 
     * @param v1
     *            値１
     * @param v2
     *            値２
     * @return 演算結果
     */
    public static Boolean ge(Object v1, Object v2) {
        return Boxing.box(ArithmeticHelper.compare(v1, v2) >= 0);
    }

    /**
     * 一項が二項へのキャスト可能か検証する、型比較演算を実行し結果を返却します。<br>
     * 二項は Class 型の必要がある。<br>
     * これは instanceof 演算子と等価。
     * 
     * 
     * @param v
     *            値１
     * @param clazz
     *            値２
     * @return 演算結果
     */
    public static Boolean instanceOf(Object v, Object clazz) {
        return Boxing.box(((Class) clazz).isInstance(v));
    }

    /**
     * 文字列構築演算を実行し結果を返却します。<br>
     * 引数が null の場合は文字列の要素には含まれません。
     * 
     * @param v1
     *            値１
     * @param v2
     *            値２
     * @return 演算結果
     */
    public static String concat(Object v1, Object v2) {
        return ArithmeticHelper.concatNvl(v1, v2);
    }

    /**
     * 一項に対する二項の等値演算を実行し結果を返却します。<br>
     * 片方の引数が null の場合は、false と評価します。<br>
     * また双方の引数が null の場合は、true と評価します。
     * 
     * @param v1
     *            値１
     * @param v2
     *            値２
     * @return 演算結果
     */
    public static Boolean oeq(Object v1, Object v2) {
        return ArithmeticHelper.oeq(v1, v2);
    }

    /**
     * 一項に対する二項の不等値演算を実行し結果を返却します。<br>
     * 片方の引数が null の場合は、true と評価します。<br>
     * また双方の引数が null の場合は、false と評価します。
     * 
     * @param v1
     *            値１
     * @param v2
     *            値２
     * @return 演算結果
     */
    public static Boolean one(Object v1, Object v2) {
        return not(oeq(v1, v2));
    }

    /**
     * 一項に対する二項の参照同一演算を実行し結果を返却します。
     * 
     * @param v1
     *            値１
     * @param v2
     *            値２
     * @return 演算結果
     */
    public static Boolean req(Object v1, Object v2) {
        return Boxing.box(v1 == v2);
    }

    /**
     * 一項に対する二項の参照不同一演算を実行し結果を返却します。
     * 
     * @param v1
     *            値１
     * @param v2
     *            値２
     * @return 演算結果
     */
    public static Boolean rne(Object v1, Object v2) {
        return not(req(v1, v2));
    }

    /**
     * 演算式を実行します。
     * 
     * @param type
     *            オペレータタイプ
     * @param v1
     *            演算する値
     * @param v2
     *            演算する値
     * @return 演算結果
     */
    public static Object evaluate(int type, Object v1, Object v2) {
        switch (type) {
        case ARITHMETIC_ADDITION:
            return add(v1, v2);
        case ARITHMETIC_SUBTRACTION:
            return subtract(v1, v2);
        case ARITHMETIC_MULTIPLICATION:
            return multiply(v1, v2);
        case ARITHMETIC_DIVISION:
            return div(v1, v2);
        case ARITHMETIC_REMAINDER:
            return mod(v1, v2);
        case BITWISE_AND:
            return and(v1, v2);
        case BITWISE_XOR:
            return xor(v1, v2);
        case BITWISE_OR:
            return or(v1, v2);
        case BIT_SHIFT_LEFT:
            return shl(v1, v2);
        case BIT_SHIFT_RIGHT:
            return shr(v1, v2);
        case BIT_SHIFT_LOGICAL_RIGHT:
            return ushr(v1, v2);
        case COMPARATIVE_EQUALITY:
            return eq(v1, v2);
        case COMPARATIVE_INEQUALITY:
            return ne(v1, v2);
        case COMPARATIVE_LESS_THAN:
            return lt(v1, v2);
        case COMPARATIVE_GREATER_THAN:
            return gt(v1, v2);
        case COMPARATIVE_LESS_EQUALITY:
            return le(v1, v2);
        case COMPARATIVE_GREATER_EQUALITY:
            return ge(v1, v2);
        case RELATIONAL_INSTANCE_OF:
            return instanceOf(v1, v2);
        case ARITHMETIC_STRING_BUILDING:
            return concat(v1, v2);
        case RELATIONAL_OBJECT_EQUALITY:
            return oeq(v1, v2);
        case RELATIONAL_OBJECT_INEQUALITY:
            return one(v1, v2);
        case RELATIONAL_REFERENCE_EQUALITY:
            return req(v1, v2);
        case RELATIONAL_REFERENCE_INEQUALITY:
            return rne(v1, v2);
        default:
            throw new ArithmeticException("illegal operand type.");
        }
    }

    /*
     * リスト演算子
     */

    /**
     * 引数が双方に数値か文字型の場合は一項に二項を加算し、片方か双方が文字列型の場合は一項に二項を文字列連結して結果を返却します。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Object add(List vs) {
        return ArithmeticHelper.add(vs);
    }

    /**
     * 一項を二項で減算し結果を返却します。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Object subtract(List vs) {
        return ArithmeticHelper.arithmetic(ARITHMETIC_SUBTRACTION, vs);
    }

    /**
     * 一項を二項で乗算し結果を返却します。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Object multiply(List vs) {
        return ArithmeticHelper.arithmetic(ARITHMETIC_MULTIPLICATION, vs);
    }

    /**
     * 一項を二項で除算し結果を返却します。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Object div(List vs) {
        return ArithmeticHelper.arithmetic(ARITHMETIC_DIVISION, vs);
    }

    /**
     * 一項に対する二項の剰余を計算し結果を返却します。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Object mod(List vs) {
        return ArithmeticHelper.arithmetic(ARITHMETIC_REMAINDER, vs);
    }

    /**
     * 一項に対する二項の左シフト演算を実行し結果を返却します。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Object shl(List vs) {
        return ArithmeticHelper.shift(BIT_SHIFT_LEFT, vs);
    }

    /**
     * 一項に対する二項の符号付き右シフト演算を実行し結果を返却します。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Object shr(List vs) {
        return ArithmeticHelper.shift(BIT_SHIFT_RIGHT, vs);
    }

    /**
     * 一項に対する二項の符号なし右シフト演算を実行し結果を返却します。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Object ushr(List vs) {
        return ArithmeticHelper.shift(BIT_SHIFT_LOGICAL_RIGHT, vs);
    }

    /**
     * 一項に対する二項のビットAND演算を実行し結果を返却します。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Object and(List vs) {
        return ArithmeticHelper.bitwise(BITWISE_AND, vs);
    }

    /**
     * 一項に対する二項のビットOR演算を実行し結果を返却します。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Object or(List vs) {
        return ArithmeticHelper.bitwise(BITWISE_OR, vs);
    }

    /**
     * 一項に対する二項のビットXOR演算を実行し結果を返却します。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Object xor(List vs) {
        return ArithmeticHelper.bitwise(BITWISE_XOR, vs);
    }

    /**
     * 一項に対する二項の等価演算を実行し結果を返却します。<br>
     * 引数の双方が数値の場合は数値の拡張変換を行い、その結果の値を比較する。<br>
     * 引数は Comparable 型の必要がある。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Boolean eq(List vs) {
        return ArithmeticHelper.compare(COMPARATIVE_EQUALITY, vs);
    }

    /**
     * 一項に対する二項の不等価演算を実行し結果を返却します。<br>
     * 引数の双方が数値の場合は数値の拡張変換を行い、その結果の値を比較する。<br>
     * 引数は Comparable 型の必要がある。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Boolean ne(List vs) {
        return not(eq(vs));
    }

    /**
     * 一項に対する二項の小なり演算を実行し結果を返却します。<br>
     * 引数の双方が数値の場合は数値の拡張変換を行い、その結果の値を比較する。<br>
     * 引数は Comparable 型の必要がある。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Boolean lt(List vs) {
        return ArithmeticHelper.compare(COMPARATIVE_LESS_THAN, vs);
    }

    /**
     * 一項に対する二項の大なり演算を実行し結果を返却します。<br>
     * 引数の双方が数値の場合は数値の拡張変換を行い、その結果の値を比較する。<br>
     * 引数は Comparable 型の必要がある。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Boolean gt(List vs) {
        return ArithmeticHelper.compare(COMPARATIVE_GREATER_THAN, vs);
    }

    /**
     * 一項に対する二項の等価または小なり演算を実行し結果を返却します。<br>
     * 引数の双方が数値の場合は数値の拡張変換を行い、その結果の値を比較する。<br>
     * 引数は Comparable 型の必要がある。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Boolean le(List vs) {
        return ArithmeticHelper.compare(COMPARATIVE_LESS_EQUALITY, vs);
    }

    /**
     * 一項に対する二項の等価または大なり演算を実行し結果を返却します。<br>
     * 引数の双方が数値の場合は数値の拡張変換を行い、その結果の値を比較する。<br>
     * 引数は Comparable 型の必要がある。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Boolean ge(List vs) {
        return ArithmeticHelper.compare(COMPARATIVE_GREATER_EQUALITY, vs);
    }

    /**
     * 一項が二項へのキャスト可能か検証する、型比較演算を実行し結果を返却します。<br>
     * 二項は Class 型の必要がある。<br>
     * これは instanceof 演算子と等価。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Boolean instanceOf(List vs) {
        return ArithmeticHelper.instanceOf(vs);
    }

    /**
     * 文字列構築演算を実行し結果を返却します。<br>
     * 引数が null の場合は文字列の要素には含まれません。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static String concat(List vs) {
        return ArithmeticHelper.concatNvl(vs);
    }

    /**
     * 一項に対する二項の等値演算を実行し結果を返却します。<br>
     * 片方の引数が null の場合は、false と評価します。<br>
     * また双方の引数が null の場合は、true と評価します。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Boolean oeq(List vs) {
        return ArithmeticHelper.oeq(vs);
    }

    /**
     * 一項に対する二項の不等値演算を実行し結果を返却します。<br>
     * 片方の引数が null の場合は、true と評価します。<br>
     * また双方の引数が null の場合は、false と評価します。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Boolean one(List vs) {
        return not(oeq(vs));
    }

    /**
     * 一項に対する二項の参照同一演算を実行し結果を返却します。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Boolean req(List vs) {
        return ArithmeticHelper.req(vs);
    }

    /**
     * 一項に対する二項の参照不同一演算を実行し結果を返却します。
     * 
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Boolean rne(List vs) {
        return not(req(vs));
    }

    /**
     * 演算式を実行します。
     * 
     * @param type
     *            オペレータタイプ
     * @param vs
     *            演算する値のリスト
     * @return 演算結果
     */
    public static Object evaluate(int type, List vs) {
        ArithmeticHelper.ifIllegalArgumentsSize(vs);
        switch (type) {
        case ARITHMETIC_ADDITION:
            return add(vs);
        case ARITHMETIC_SUBTRACTION:
            return subtract(vs);
        case ARITHMETIC_MULTIPLICATION:
            return multiply(vs);
        case ARITHMETIC_DIVISION:
            return div(vs);
        case ARITHMETIC_REMAINDER:
            return mod(vs);
        case BITWISE_AND:
            return and(vs);
        case BITWISE_XOR:
            return xor(vs);
        case BITWISE_OR:
            return or(vs);
        case BIT_SHIFT_LEFT:
            return shl(vs);
        case BIT_SHIFT_RIGHT:
            return shr(vs);
        case BIT_SHIFT_LOGICAL_RIGHT:
            return ushr(vs);
        case COMPARATIVE_EQUALITY:
            return eq(vs);
        case COMPARATIVE_INEQUALITY:
            return ne(vs);
        case COMPARATIVE_LESS_THAN:
            return lt(vs);
        case COMPARATIVE_GREATER_THAN:
            return gt(vs);
        case COMPARATIVE_LESS_EQUALITY:
            return le(vs);
        case COMPARATIVE_GREATER_EQUALITY:
            return ge(vs);
        case RELATIONAL_INSTANCE_OF:
            return instanceOf(vs);
        case ARITHMETIC_STRING_BUILDING:
            return concat(vs);
        case RELATIONAL_OBJECT_EQUALITY:
            return oeq(vs);
        case RELATIONAL_OBJECT_INEQUALITY:
            return one(vs);
        case RELATIONAL_REFERENCE_EQUALITY:
            return req(vs);
        case RELATIONAL_REFERENCE_INEQUALITY:
            return rne(vs);
        default:
            throw new ArithmeticException("illegal operand type.");
        }
    }
}
