/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin.resourceset;

import java.util.Collections;
import java.util.Iterator;
import java.util.Map;

import shohaku.core.collections.IteratorUtils;
import shohaku.shoin.ResourceSet;

/**
 * マップを内部表現とするリソース集合を提供します。
 */
public class MapResourceSet implements ResourceSet {

    /* リソースを内包するマップ。 */
    private final Map lookup;

    /**
     * リソースを内包するマップを格納して初期化します。
     * 
     * @param resources
     *            リソースを内包するマップ
     */
    public MapResourceSet(Map resources) {
        this.lookup = resources;
    }

    /**
     * リソースを内包するマップを返却します。
     * 
     * @return リソースを内包するマップ
     */
    protected Map getResourceMap() {
        return lookup;
    }

    public Iterator getKeys() {
        return IteratorUtils.unmodifiableIterator(lookup.keySet().iterator());
    }

    public Object getObject(Object key) {
        if (key == null) {
            throw new NullPointerException("resource key is null");
        }
        return lookup.get(key);
    }

    public Object getObject(Object key, Object defaultValue) {
        if (key == null) {
            throw new NullPointerException("resource key is null");
        }
        Object o = getObject(key);
        if (o != null) {
            o = defaultValue;
        }
        return o;
    }

    public boolean containsKey(Object key) {
        if (key == null) {
            throw new NullPointerException("resource key is null");
        }
        return lookup.containsKey(key);
    }

    public int size() {
        return lookup.size();
    }

    public Map getMapView() {
        return Collections.unmodifiableMap(lookup);
    }

    /**
     * ラップされたマップの toString() をそのまま返します。
     * 
     * @return 文字列表現
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return lookup.toString();
    }

}