/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin.resourceset;

import java.util.Iterator;
import java.util.NoSuchElementException;

import shohaku.shoin.ResourceSet;

/**
 * 複数のリソース集合を束ねる反復子を提供します。
 */
public class CompositeResourceSetIterator implements Iterator {

    /* 基のリソース集合の配列 */
    private final ResourceSet[] resourceSets;

    /* 基のリソース集合の反復子 */
    private final Iterator[] iters;

    /* 走査中のリソース集合配列のインデックス */
    private int index = -1;

    /* 走査中の反復子 */
    private Iterator iter = null;

    /* 走査中の要素 */
    private Object next = null;

    /*
     * hasNext() の結果の2項値の意味をもち、 また null を hasNext() が検証される前の意味とする三項値。
     */
    private Boolean hasNext = null;

    /**
     * リソース集合の配列を格納して初期化します。
     * 
     * @param resourceSets
     *            リソース集合の配列
     */
    public CompositeResourceSetIterator(ResourceSet[] resourceSets) {
        final Iterator[] is = new Iterator[resourceSets.length];
        for (int i = 0; i < resourceSets.length; i++) {
            is[i] = resourceSets[i].getKeys();
        }
        this.iters = is;
        this.resourceSets = resourceSets;
    }

    public boolean hasNext() {
        if (this.hasNext != null) {
            return this.hasNext.booleanValue();
        }
        this.hasNext = Boolean.FALSE;
        if (iter == null) {
            if ((index + 1) < iters.length) {
                iter = iters[++index];
            } else {
                return this.hasNext.booleanValue();// return false
            }
        }
        if (iter.hasNext()) {
            getNext();
        }
        while (!iter.hasNext()) {
            if ((index + 1) < iters.length) {
                iter = iters[++index];
                if (getNext()) {
                    break;
                }
            } else {
                return this.hasNext.booleanValue();// return false
            }
        }
        return this.hasNext.booleanValue();
    }

    private boolean getNext() {
        while (iter.hasNext()) {
            Object key = iter.next();
            if (isNext(key)) {
                this.next = key;
                this.hasNext = Boolean.TRUE;
                return true;
            }
        }
        return false;
    }

    private boolean isNext(Object key) {
        for (int i = (index + 1); i < resourceSets.length; i++) {
            if (resourceSets[i].containsKey(key)) {
                return false;
            }
        }
        return true;
    }

    public Object next() {
        if (!hasNext()) {
            throw new NoSuchElementException();
        }
        this.hasNext = null;
        return this.next;
    }

    public void remove() {
        throw new UnsupportedOperationException();
    }
}