/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin.proxy;

import java.io.IOException;

import shohaku.core.resource.IOResource;
import shohaku.shoin.ResourceSet;
import shohaku.shoin.ResourceSetCreationException;

/**
 * IOリソースが更新されていると新たなリソース集合を返却する生成プロキシを提供します。
 */
public class ModifiedResourceSetFactoryProxy extends AbstractIOResourceSetFactoryProxy {

    private long[] lastModifieds;

    /**
     * プロパティを初期値で初期化します。
     */
    public ModifiedResourceSetFactoryProxy() {
        super();
    }

    /**
     * リソースが更新されていると新たなリソース集合を返却します。<br>
     * ただし初回は更新に関わりなく、新たなリソース集合を生成します。<br>
     * create フラグが true が指定されている場合も新規に生成されます。
     * 
     * @param create
     *            再生成を要求する場合は true
     * @return リソース集合
     * @throws ResourceSetCreationException
     *             リソース集合の生成に失敗した場合
     * @see shohaku.shoin.ResourceSetFactoryProxy#getResourceSet(boolean)
     */
    public ResourceSet getResourceSet(boolean create) throws ResourceSetCreationException {
        synchronized (this) {
            if (create || resourceSet == null || isModified()) {
                return (resourceSet = createResourceSet());
            }
            return resourceSet;
        }
    }

    private boolean isModified() throws ResourceSetCreationException {
        if (resourceSet == null) {
            return true;
        }
        final IOResource[] inreses = getIOResourceSetFactory().getIOResources();
        for (int i = 0; i < inreses.length; i++) {
            if (isModified(i, inreses[i])) {
                return true;
            }
        }
        return false;
    }

    private boolean isModified(int i, IOResource ir) throws ResourceSetCreationException {
        try {
            long oldlm = lastModifieds[i];
            long newlm = ir.getLastModified();
            lastModifieds[i] = newlm;
            return (oldlm < newlm);
        } catch (IOException e) {
            throw new ResourceSetCreationException("call getLastModified err.", e);
        }
    }

}