/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin.proxy;

import shohaku.shoin.IOResourceSetFactory;
import shohaku.shoin.ResourceSet;
import shohaku.shoin.ResourceSetCreationException;
import shohaku.shoin.ResourceSetFactoryProxy;

/**
 * IOリソースからリソース集合を生成するプロキシの抽象実装を提供します。
 */
public abstract class AbstractIOResourceSetFactoryProxy implements ResourceSetFactoryProxy {

    /** リソース集合のキャッシュ */
    protected ResourceSet resourceSet;

    /* リソース集合生成機能 */
    private IOResourceSetFactory ioResourceSetFactory;

    /**
     * プロパティを初期値で初期化します。
     */
    public AbstractIOResourceSetFactoryProxy() {
        super();
    }

    /**
     * create フラグが true の場合以外は、常に同一のリソース集合を返却します。
     * 
     * @param create
     *            再生成を要求する場合は true
     * @return リソース集合
     * @throws ResourceSetCreationException
     *             リソース集合の生成に失敗した場合
     * @see shohaku.shoin.ResourceSetFactoryProxy#getResourceSet(boolean)
     */
    public ResourceSet getResourceSet(boolean create) throws ResourceSetCreationException {
        synchronized (this) {
            if (create || resourceSet == null) {
                return (resourceSet = createResourceSet());
            }
            return resourceSet;
        }
    }

    /**
     * リソース集合を生成して返却します。拡張ポイントです。
     * 
     * @return 生成されたリソース集合
     * @throws ResourceSetCreationException
     *             リソース集合の生成に失敗した場合
     */
    protected ResourceSet createResourceSet() throws ResourceSetCreationException {
        return ioResourceSetFactory.getResourceSet();
    }

    /**
     * リソース集合生成機能を返却します。
     * 
     * @return リソース集合生成機能
     */
    public IOResourceSetFactory getIOResourceSetFactory() {
        synchronized (this) {
            return ioResourceSetFactory;
        }
    }

    /**
     * リソース集合生成機能を格納します。
     * 
     * @param factory
     *            リソース集合生成機能
     */
    public void setIOResourceSetFactory(IOResourceSetFactory factory) {
        synchronized (this) {
            this.ioResourceSetFactory = factory;
            this.resourceSet = null;
        }
    }
}
