/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin.factory;

import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import shohaku.core.lang.Concat;
import shohaku.shoin.ResourceKeyPrefixSupport;

/**
 * {@link shohaku.shoin.ResourceKeyPrefixSupport} の機能を実装するクラスに機能を混ぜ込む、Mix-In（組込み機能）を提供します。<br>
 * このクラスは ResourceKeyPrefixSupport を実装したクラスのメンバーとして組み込まれて使用されます。
 */
public class ResourceKeyPrefixMixIn implements ResourceKeyPrefixSupport {

    private String _prefix;

    private String[] _sourcesPrefix;

    public String getPrefix() {
        return _prefix;
    }

    public void setPrefix(String prefix) {
        this._prefix = prefix;
    }

    public String[] getSourcesPrefix() {
        return _sourcesPrefix;
    }

    public void setSourcesPrefix(String[] prefixs) {
        this._sourcesPrefix = prefixs;
    }

    /*
     * helper
     */

    /*
     * Prefix
     */

    /**
     * 接頭辞が指定されているか検証します。
     * 
     * @return 接頭辞が指定されている場合は true
     */
    public boolean hasPrefix() {
        return (null != _prefix);
    }

    /**
     * 接頭辞を繋げたキーを返却します。
     * 
     * @param key
     *            基のキー
     * @return 接頭辞を追加した文字列
     */
    public String join(Object key) {
        return Concat.get(prefix(), key);
    }

    /**
     * 接頭辞を繋げたキーを文字列配列に格納して返却します。
     * 
     * @param keys
     *            基のキー
     * @return 接頭辞を追加した文字列配列
     */
    public String[] join(Object[] keys) {
        return join(keys, prefix());
    }

    /**
     * 接頭辞を繋げたキーを文字列配列に格納して返却します。
     * 
     * @param keys
     *            基のキー
     * @return 接頭辞を追加した文字列配列
     */
    public String[] join(Collection keys) {
        return join(keys, prefix());
    }

    /**
     * 接頭辞を繋げたキーをマップに追加して返却します。
     * 
     * @param resources
     *            基のリソース
     * @return 接頭辞を追加したマップ
     */
    public Map join(Map resources) {
        final Map output = new LinkedHashMap(resources.size());
        return join(resources, output, prefix());
    }

    /**
     * 接頭辞を繋げたキーを指定のマップに追加して返却します。
     * 
     * @param resources
     *            基のリソース
     * @param output
     *            接頭辞を繋げたキーを追加するマップ
     * @return 接頭辞を追加したマップ
     */
    public Map join(Map resources, Map output) {
        return join(resources, output, prefix());
    }

    /* 実際の接頭辞を返却します */
    private String prefix() {
        return (hasPrefix()) ? _prefix : "";
    }

    /*
     * Sources Prefix
     */

    /**
     * 指定のインデックスの接頭辞を返却します。
     * 
     * @param index
     *            インデックス
     * @return 接頭辞
     * @throws IndexOutOfBoundsException
     *             インデックスが範囲外の場合
     */
    public String getSourcesPrefix(int index) {
        return _sourcesPrefix[index];
    }

    /**
     * 指定のインデックスに接頭辞を格納します。
     * 
     * @param index
     *            インデックス
     * @param prefix
     *            接頭辞
     * @throws IndexOutOfBoundsException
     *             インデックスが範囲外の場合
     */
    public void setSourcesPrefix(int index, String prefix) {
        _sourcesPrefix[index] = prefix;
    }

    /**
     * 指定のインデックスの接頭辞が指定されているか検証します。
     * 
     * @param index
     *            インデックス
     * @return 接頭辞が指定されている場合は true
     */
    public boolean hasPrefix(int index) {
        return (null != _sourcesPrefix && index < _sourcesPrefix.length && null != _sourcesPrefix[index]);
    }

    /**
     * 接頭辞を繋げたキーを返却します。
     * 
     * @param index
     *            インデックス
     * @param key
     *            基のキー
     * @return 接頭辞を追加した文字列
     */
    public String join(int index, Object key) {
        return Concat.get(sourcesPrefix(index), key);
    }

    /**
     * 接頭辞を繋げたキーを文字列配列に格納して返却します。
     * 
     * @param index
     *            インデックス
     * @param keys
     *            基のキー
     * @return 接頭辞を追加した文字列配列
     */
    public String[] join(int index, Object[] keys) {
        return join(keys, sourcesPrefix(index));
    }

    /**
     * 接頭辞を繋げたキーを文字列配列に格納して返却します。
     * 
     * @param index
     *            インデックス
     * @param keys
     *            基のキー
     * @return 接頭辞を追加した文字列配列
     */
    public String[] join(int index, Collection keys) {
        return join(keys, sourcesPrefix(index));
    }

    /**
     * 接頭辞を繋げたキーをマップに追加して返却します。
     * 
     * @param index
     *            インデックス
     * @param resources
     *            基のリソース
     * @return 接頭辞を追加したマップ
     */
    public Map join(int index, Map resources) {
        final Map output = new LinkedHashMap(resources.size());
        return join(resources, output, sourcesPrefix(index));
    }

    /**
     * 接頭辞を繋げたキーを指定のマップに追加して返却します。
     * 
     * @param index
     *            インデックス
     * @param resources
     *            基のリソース
     * @param output
     *            接頭辞を繋げたキーを追加するマップ
     * @return 接頭辞を追加したマップ
     */
    public Map join(int index, Map resources, Map output) {
        return join(resources, output, sourcesPrefix(index));
    }

    /**
     * 接頭辞を繋げたキーを指定のマップに追加して返却します。
     * 
     * @param index
     *            インデックス
     * @param resources
     *            基のリソース
     * @param output
     *            接頭辞を繋げたキーを追加するマップ
     * @param prefix
     *            接頭辞
     * @return 接頭辞を追加したマップ
     */
    public Map join(int index, Map resources, Map output, String prefix) {
        return join(resources, output, Concat.get(prefix, sourcesPrefix(index)));
    }

    /* 実際の接頭辞を返却します */
    private String sourcesPrefix(int index) {
        return (hasPrefix(index)) ? ((_sourcesPrefix[index] != null) ? _sourcesPrefix[index] : "") : "";
    }

    /**
     * 接頭辞を付加して、リソースを全てマップに追加します。
     * 
     * @param lookup
     *            リソースを格納するマップ
     * @param resources
     *            リソース
     * @param index
     *            インデックス
     */
    public void putAll(Map lookup, Map resources, int index) {
        if (hasPrefix()) {
            if (hasPrefix(index)) {
                join(index, resources, lookup, getPrefix());
            } else {
                join(resources, lookup);
            }
        } else {
            if (hasPrefix(index)) {
                join(index, resources, lookup);
            } else {
                lookup.putAll(resources);
            }
        }
    }

    /**
     * 指定のインデックスの接頭辞が指定されているか検証します。
     * 
     * @param index
     *            インデックス
     * @return 接頭辞が指定されている場合は true
     */
    public boolean hasPrefixs(int index) {
        return hasPrefix() && hasPrefix(index);
    }

    /**
     * インデックスの接頭辞を返却します。<br>
     * 指定されていない場合は空文字列を返却します。
     * 
     * @param index
     *            インデックス
     * @return 接頭辞
     */
    public String findPrefix(int index) {
        if (hasPrefix()) {
            if (hasPrefix(index)) {
                return Concat.get(getPrefix(), getSourcesPrefix(index));
            } else {
                return getPrefix();
            }
        } else {
            if (hasPrefix(index)) {
                return getSourcesPrefix(index);
            } else {
                return "";
            }
        }
    }

    /*
     * static
     */

    /* 接頭辞を繋げます */
    static Map join(Map resources, Map output, String prefix) {
        for (Iterator i = resources.entrySet().iterator(); i.hasNext();) {
            Map.Entry e = (Map.Entry) i.next();
            output.put(Concat.get(prefix, e.getKey()), e.getValue());
        }
        return output;
    }

    /* 接頭辞を繋げます */
    static String[] join(Collection keys, String prefix) {
        final String[] output = new String[keys.size()];
        int index = 0;
        for (Iterator i = keys.iterator(); i.hasNext();) {
            output[index++] = Concat.get(prefix, i.next());
        }
        return output;
    }

    /* 接頭辞を繋げます */
    static String[] join(Object[] keys, String prefix) {
        final String[] output = new String[keys.length];
        for (int i = 0; i < keys.length; i++) {
            output[i] = Concat.get(prefix, keys[i]);
        }
        return output;
    }

}
