/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.shoin.factory;

import java.util.Iterator;
import java.util.Map;

import shohaku.core.lang.Concat;
import shohaku.core.lang.Eval;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.TagNode;
import shohaku.ginkgo.ValueNode;
import shohaku.shoin.ResourceKeyPrefixSupport;
import shohaku.shoin.ResourceSet;
import shohaku.shoin.resourceset.ObjectCreationProxyResourceSet;

/**
 * XMLデータで定義された情報を基に、POJOを生成するファクトリを提供します。
 * <p>
 * 例：
 * 
 * <pre>
 * // 湖を表現するオブジェクト
 * public class Lake {
 *     private String name;
 *     public static Lake(String name) { this.name = name; }
 *     public String getName() { return name; }
 *     public void setName(String name) { this.name = name; }
 * }
 *
 * //定義ファイル component-factory.xml
 * &lt;component-factory&gt;
 *
 *    &lt;!-- コンポーネントの定義、グループ名="lake", シングルトン --&gt;
 *    &lt;components name="lake"&gt;
 *        &lt;component id="constructorId" name="constructor" class="Lake" instance="singleton"&gt;
 *            &lt;init&gt;
 *                &lt;arg&gt;&lt;string&gt;支笏湖&lt;/string&gt;&lt;/arg&gt;
 *            &lt;/init&gt;
 *        &lt;/component&gt;
 *        &lt;component id="propertyId" name="property" class="Lake" instance="singleton"&gt;
 *            &lt;property name="name"&gt;
 *                &lt;string&gt;阿寒湖&lt;/string&gt;
 *            &lt;/property&gt;
 *        &lt;/component&gt;
 *        &lt;component id="methodId" name="method" class="Lake" instance="singleton"&gt;
 *            &lt;method name="setName"&gt;
 *                &lt;arg&gt;&lt;string&gt;倶多楽湖&lt;/string&gt;&lt;/arg&gt;
 *            &lt;/method&gt;
 *        &lt;/component&gt;
 *    &lt;/components&gt;
 *
 *    &lt;!-- コンポーネントを参照する、グループ名="ref",  プロトタイプ（要求ごとに新規生成する）  --&gt;
 *    &lt;components name="ref"&gt;
 *        &lt;!-- コンポーネントを参照してリストを生成します --&gt;
 *        &lt;component name="list" class="java.util.ArrayList" instance="prototype"&gt;
 *            &lt;method name="add"&gt;
 *                &lt;arg&gt;&lt;ref&gt;constructorId&lt;/ref&gt;&lt;/arg&gt;
 *            &lt;/method&gt;
 *            &lt;method name="add"&gt;
 *                &lt;arg&gt;&lt;ref&gt;propertyId&lt;/ref&gt;&lt;/arg&gt;
 *            &lt;/method&gt;
 *            &lt;method name="add"&gt;
 *                &lt;arg&gt;&lt;ref&gt;methodId&lt;/ref&gt;&lt;/arg&gt;
 *            &lt;/method&gt;
 *        &lt;/component&gt;
 *    &lt;/components&gt;
 *
 * &lt;component-factory&gt;
 *
 * グループ名とコンポーネント名を ':' で区切りコンポーネント識別子として認識されます。
 * 一般的なファイルの読み取りとコンポーネント取得の手順は以下の様に為ります。 
 * ComponentFactory factory = new ComponentFactory();
 * IOResource inres = FeatureFactory.getLoader().getIOResource("component-factory.xml");
 * factory.setIOResources(new IOResource[] { inres });
 * 
 * ResourceSet resources = factory.getResourceSet();
 * Lake lake = (Lake) resources.getObject("lake:constructor");
 * List list = (List) resources.getObject("ref:list");
 * 
 * System.out.println("湖名=" + lake.getName());
 * System.out.println("リスト=" + list.toString());
 *
 * ＞＞湖名=支笏湖
 * ＞＞リスト=[支笏湖, 阿寒湖, 倶多楽湖]
 *
 * </pre>
 */
public class ComponentFactory extends AbstractGinkgoResourceSetFactory implements ResourceKeyPrefixSupport {

    /* 階層を分割する区切り文字 */
    private static final String DELIMITER = ":";

    /* ResourceKeyPrefixSupport Mix-in */
    private final ResourceKeyPrefixMixIn prefixsMixIn = new ResourceKeyPrefixMixIn();

    /**
     * プロパティを初期値で初期化します。
     */
    public ComponentFactory() {
        super();
    }

    /**
     * マップからリソース集合を生成して返却します。<br>
     * デフォルトでは ObjectCreationProxy 用のリソース集合を生成します。
     * 
     * @param lookup
     *            リソースを格納するマップ
     * @return リソース集合
     */
    protected ResourceSet newResourceSetObject(Map lookup) {
        return new ObjectCreationProxyResourceSet(lookup);
    }

    protected void initValues(Map lookup, TagNode root, int index) throws GinkgoException {
        final String prefix = prefixsMixIn.findPrefix(index);
        for (Iterator i = root.getTagContext().elementIterator("components"); i.hasNext();) {
            TagNode components = (TagNode) i.next();
            String groupName = components.getName();
            if (Eval.isBlank(groupName)) {
                throw new GinkgoException("components name is null.");
            }
            initComponents(components, prefix, groupName, lookup);
        }
    }

    /* コンポーネントの初期化および登録をする */
    private void initComponents(TagNode components, String prefix, String groupName, Map lookup) throws GinkgoException {
        for (Iterator i = components.getTagContext().elementIterator("component"); i.hasNext();) {
            TagNode component = (TagNode) i.next();
            String name = component.getName();
            if (Eval.isBlank(name)) {
                throw new GinkgoException("component name is null.");
            }
            lookup.put(Concat.get(prefix, groupName, DELIMITER, name), ((ValueNode) component).getNodeValue());
        }
    }

    /*
     * ResourceKeyPrefixSupport
     */

    public String getPrefix() {
        return prefixsMixIn.getPrefix();
    }

    public void setPrefix(String prefix) {
        prefixsMixIn.setPrefix(prefix);
    }

    public String[] getSourcesPrefix() {
        return prefixsMixIn.getSourcesPrefix();
    }

    public void setSourcesPrefix(String[] prefixs) {
        prefixsMixIn.setSourcesPrefix(prefixs);
    }
}
