/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin;

import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * 拡張可能な階層化リソースバンドルのキーの反復子を提供します。
 */
class XResourceBundleIterator implements Iterator {

    /** リソース集合。 */
    private ResourceSet resourceSet;

    /** 反復子。 */
    private Iterator iterator;

    /** 親の反復子。 */
    private Iterator parentIterator;

    /** 次の要素を保存します。 */
    private Object next = null;

    /**
     * リソース集合とその親の反復子を指定して初期化します。
     * 
     * @param resources
     *            リソース集合
     * @param parentIterator
     *            親の反復子
     */
    XResourceBundleIterator(ResourceSet resources, Iterator parentIterator) {
        this.resourceSet = resources;
        this.iterator = resources.getKeys();
        this.parentIterator = parentIterator;
    }

    /**
     * 繰り返し処理でさらに要素がある場合に true を返却します。 <br>
     * 親のXResourceBundleの反復子も再帰的に検索します。
     * 
     * @see java.util.Iterator#hasNext()
     */
    public boolean hasNext() {
        if (next == null) {
            if (iterator.hasNext()) {
                next = iterator.next();
            } else if (parentIterator != null) {
                while (next == null && parentIterator.hasNext()) {
                    next = parentIterator.next();
                    if (resourceSet.containsKey(next)) {
                        next = null;
                    }
                }
            }
        }
        return next != null;
    }

    /**
     * 繰り返し処理で次の要素を返却します。
     * 
     * @see java.util.Iterator#next()
     */
    public Object next() {
        if (hasNext()) {
            Object result = next;
            next = null;
            return result;
        } else {
            throw new NoSuchElementException();
        }
    }

    /**
     * サポートしていません。 基になるコレクションから、反復子によって最後に返された要素を削除します (任意のオペレーション).
     * 
     * @see java.util.Iterator#remove()
     */
    public void remove() {
        throw new UnsupportedOperationException("XResourceBundle#getKeyIterator() return Iterator un supported remove() method");
    }
}
