/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin;

import java.util.Iterator;
import java.util.Map;

import shohaku.shoin.resourceset.ResourceSetMapView;

/**
 * 拡張可能な階層化リソースバンドルの抽象実装を提供します。 <br>
 * この拡張可能なリソースバンドルは java.util.ResourceBundle 同様に地域対応に主眼を置きますが、その用途に制限を持ちません。 <br>
 * 論理的には階層化されたリソースは全て実装の対象とすることが出来ます。 <br>
 * <br>
 * リソースを定義するファイル形式には XResourceBundleCreater で実装出来る任意の形式を利用できます。<br>
 * 階層化されたリソースをキャッシュし束ねるには XResourceBundleCache を使用します。 <br>
 * またリソースを束ね階層化する基準とファイル名の規約には XResourceBundleBaseCalculator の実装として任意に定義出来ます。 <br>
 * そのため java.util.ResourceBundle に於ける java.util.Locale のファイル名規約のと同等のファイル接尾辞を設計し、<br>
 * それを示すクラスを定義する事で、容易に独自基準で階層化されたリソースバンドルを構築出来ます。 <br>
 */
public abstract class XResourceBundle implements ResourceSet {

    /** リソースを格納します。 */
    protected final ResourceSet lookup;

    /** 親バンドルを格納します。 */
    protected final XResourceBundle parent;

    /** 束縛基準を格納します。 */
    protected final XResourceBundleBase bundleBase;

    /**
     * リソースリストを初期化します。
     * 
     * @param parent
     *            親バンドル
     * @param bundleBase
     *            束縛基準
     * @param resources
     *            リソース
     */
    protected XResourceBundle(XResourceBundle parent, XResourceBundleBase bundleBase, ResourceSet resources) {
        this.parent = parent;
        this.bundleBase = bundleBase;
        this.lookup = resources;
    }

    /*
     * Get Resources
     */

    /**
     * このバンドルと上位階層の全てのリソースキーの反復子を返却します。 <br>
     * この反復子は削除不可です。
     * 
     * @return このバンドルと上位階層の全てのリソースキーの反復子
     */
    public Iterator getKeys() {
        Iterator parentIterator = (this.parent != null) ? this.parent.getKeys() : null;
        return new XResourceBundleIterator(lookup, parentIterator);
    }

    /**
     * リソースキーが示す値を返却します。指定されたキーが存在しない場合 null を返却します。
     * 
     * @param key
     *            リソースキー
     * @throws NullPointerException
     *             key が null の場合
     * @return リソースキーが示す値
     */
    public Object getObject(Object key) {
        if (key == null) {
            throw new NullPointerException("resource key is null");
        }
        Object obj = lookup.getObject(key);
        if (obj == null) {
            if (parent != null) {
                obj = parent.getObject(key);
            }
        }
        return obj;
    }

    /**
     * 指定された文字列型のキーが示す値を返却します。 <br>
     * 指定されたキーが存在しない場合 defaultValue を返却します。
     * 
     * @param key
     *            リソースキー
     * @param defaultValue
     *            リソースキー
     * @return リソースキーが示す値
     * @throws NullPointerException
     *             key が null の場合
     */
    public Object getObject(Object key, Object defaultValue) {
        if (containsKey(key)) {
            return getObject(key);
        } else {
            return defaultValue;
        }
    }

    /**
     * 指定されたキーがリソースセットに含まれている場合に true を返却します。 <br>
     * 親の拡張リソースバンドルも検索対象とします。
     * 
     * @param key
     *            リソースキー
     * @return 指定されたキーが含まれている場合は true
     * @throws NullPointerException
     *             key が null の場合
     */
    public boolean containsKey(Object key) {
        if (key == null) {
            throw new NullPointerException("resource key is null");
        }
        if (!lookup.containsKey(key)) {
            if (parent != null) {
                return parent.containsKey(key);
            }
            return false;
        }
        return true;
    }

    public int size() {
        return (parent == null) ? lookup.size() : (lookup.size() + parent.size());
    }

    /**
     * リソース集合のマップ表現を返却します。 <br>
     * 親の拡張リソースバンドルの情報も含みます。
     * 
     * @return リソース集合のマップ表現
     */
    public Map getMapView() {
        return new ResourceSetMapView(this);
    }

}