/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin;

import java.io.IOException;
import java.net.URISyntaxException;

import shohaku.core.collections.Cache;
import shohaku.core.collections.cache.AccessOrderHashCache;
import shohaku.core.lang.Concat;
import shohaku.core.lang.feature.FeatureFactory;
import shohaku.core.resource.IOResource;
import shohaku.core.resource.IOResourceLoader;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.NodeCompositeRule;
import shohaku.ginkgo.rule.XMLNodeCompositeRuleFactory;

/**
 * 松韻ライブラリの構成ファイル等のリソースをロードする機能を提供します。
 */
public class ShoinResourceLoader {

    /* デフォルトの最大キャッシュサイズ */
    static final int CACHE_MAXIMUM_SIZE = 127;

    /*
     * NodeCompositeRule
     */

    /* 構成ルールを弱参照でキャシュします。 */
    static final Cache nodeCompositeRuleCache = new AccessOrderHashCache(4, 0.75f, CACHE_MAXIMUM_SIZE);

    /**
     * Ginkgo API のデフォルトの構成ルールを生成して返却します。 <br>
     * カスタマイズされたデフォルトルールが定義されていない場合は、システムのデフォルトルールを使用します。
     * 
     * @param clazz
     *            構成リソースを生成するクラス
     * @param classLoader
     *            生成に使用するクラスローダ
     * @return デフォルトの構成ルール
     */
    public static NodeCompositeRule getDefaultNodeCompositeRule(Class clazz, ClassLoader classLoader) {

        // get snapshot cache data
        NodeCompositeRule rule = (NodeCompositeRule) nodeCompositeRuleCache.get(clazz);
        if (rule != null) {
            return rule;
        }

        final XMLNodeCompositeRuleFactory ruleFactory = getNodeCompositeRuleFactory(classLoader);
        final IOResourceLoader irloader = getIOResourceLoader(classLoader);
        final String baseName = getRuleFileBaseName(clazz);// 基底名取得
        // カスタム定義
        rule = loadCustomRule(baseName, ruleFactory, irloader);
        // デフォルト定義
        if (rule == null) {
            rule = getDefaultRule(baseName, clazz, ruleFactory, irloader);
        }

        // overwriting is ok.
        nodeCompositeRuleCache.put(clazz, rule);
        return rule;

    }

    /* 構成ルール生成機能を生成して返却します。 */
    private static XMLNodeCompositeRuleFactory getNodeCompositeRuleFactory(ClassLoader classLoader) {
        final XMLNodeCompositeRuleFactory ruleFactory = new XMLNodeCompositeRuleFactory();
        ruleFactory.setClassLoader(classLoader);
        return ruleFactory;
    }

    /* IOリソース生成機能を生成して返却します。 */
    private static IOResourceLoader getIOResourceLoader(ClassLoader classLoader) {
        final IOResourceLoader irloader = FeatureFactory.getLoader().getIOResourceLoader();
        irloader.setClassLoader(classLoader);
        return irloader;
    }

    /* カスタム定義を生成して返却します。 */
    private static NodeCompositeRule loadCustomRule(String baseName, XMLNodeCompositeRuleFactory ruleFactory, IOResourceLoader irloader) {
        NodeCompositeRule rule = null;
        final String customRulePath = Concat.get(baseName.replace('.', '/'), "-rule.xml");
        try {
            IOResource ioResource = irloader.getIOResource(customRulePath);
            if (ioResource.exists()) {
                ruleFactory.setIOResource(ioResource);
                rule = ruleFactory.create();
            }
        } catch (IOException e) {
            throw new GinkgoException("NodeCompositeRule create err.", e);
        } catch (URISyntaxException e) {
            throw new GinkgoException("NodeCompositeRule create err.", e);
        }
        return rule;
    }

    /* デフォルト定義を生成して返却します。 */
    private static NodeCompositeRule getDefaultRule(String baseName, Class clazz, XMLNodeCompositeRuleFactory ruleFactory, IOResourceLoader irloader) {
        NodeCompositeRule rule = null;
        final String packagePath = clazz.getPackage().getName();
        final String defaultRulePath = Concat.get(packagePath.replace('.', '/'), "/", baseName, "-default-rule.xml");
        try {
            IOResource ioResource = irloader.getIOResource(defaultRulePath);
            ruleFactory.setIOResource(ioResource);
            rule = ruleFactory.create();
        } catch (IOException e) {
            throw new GinkgoException("NodeCompositeRule create err.", e);
        } catch (URISyntaxException e) {
            throw new GinkgoException("NodeCompositeRule create err.", e);
        }
        return rule;
    }

    /* 構成ルールファイルの基底名を生成する、大文字を "'-' + 小文字" に変換する */
    private static String getRuleFileBaseName(Class clazz) {

        final StringBuffer buf = new StringBuffer();
        final String className = clazz.getName();
        // 拡張子を切捨てます。
        char[] chars = null;
        int off = className.lastIndexOf('.');
        if (0 <= off) {
            chars = className.substring(++off).toCharArray();
        } else {
            chars = className.toCharArray();
        }
        // 大文字を - を付加した小文字に変換します。
        for (int i = 0; i < chars.length; i++) {
            char ch = chars[i];
            if (0 < i && 'A' <= ch && ch <= 'Z') {
                if ((i + 1) < chars.length) {
                    char next = chars[i + 1];
                    if ('a' <= next && next <= 'z') {
                        buf.append('-');
                    }
                } else {
                    buf.append('-');
                }
            }
            buf.append(Character.toLowerCase(ch));
        }
        return buf.toString();

    }

}
