/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.shoin;

import shohaku.core.lang.ConfigurationException;
import shohaku.core.lang.feature.FeatureFactory;
import shohaku.core.resource.IOResource;
import shohaku.core.resource.IOResourceLoader;
import shohaku.shoin.factory.ShoinContextFactory;

/**
 * プロキシを経由してリソース集合を生成する、松韻ライブラリの集約機能を提供します。<br>
 * ファクトリやプロキシの各機能は独立して利用可能ですが、この機能を使用することで唯一のアクセスポイントからリソース集合の生成と管理を行えます。<br>
 * またファクトリやプロキシの定義は構成ファイルに置かれるため、柔軟に構成を変更出来ます。
 * <p>
 * XML例：
 * 
 * <pre>
 * &lt;shoin-context&gt;
 * 
 *     &lt;!-- 定数を定義できます。 --&gt;
 *     &lt;define&gt;
 *         &lt;type id="irs-type"&gt;shohaku.core.io.IOResource[]&lt;/type&gt;
 *     &lt;/define&gt;
 * 
 *     &lt;!-- ResourceSetFactoryProxy の集合を定義します。 --&gt;
 *     &lt;proxys&gt;
 * 
 *         &lt;proxy id="xmlprops" class="shohaku.shoin.proxy.StaticResourceSetFactoryProxy"&gt;
 *             &lt;property name="resourceSetFactory"&gt;
 *                 &lt;object class="shohaku.shoin.factory.XMLProperties"&gt;
 *                     &lt;property name="ioResources"&gt;
 *                         &lt;array type="shohaku.core.io.IOResource[]"&gt;
 *                             &lt;resource path="classpath:/test-xml-properties.xml" /&gt;
 *                         &lt;/array&gt;
 *                     &lt;/property&gt;
 *                 &lt;/object&gt;
 *             &lt;/property&gt;
 *         &lt;/proxy&gt;
 * 
 *         &lt;proxy id="constants" class="shohaku.shoin.proxy.StaticResourceSetFactoryProxy"&gt;
 *             &lt;property name="resourceSetFactory"&gt;
 *                 &lt;object class="shohaku.shoin.factory.ConstantsProperties"&gt;
 *                     &lt;property name="sources"&gt;
 *                         &lt;array type="Object[]"&gt;
 *                             &lt;type&gt;java.lang.Integer&lt;/type&gt;
 *                             &lt;type&gt;java.lang.Long&lt;/type&gt;
 *                         &lt;/array&gt;
 *                     &lt;/property&gt;
 *                     &lt;property name="prefix"&gt;
 *                         &lt;string&gt;num:&lt;/string&gt;
 *                     &lt;/property&gt;
 *                     &lt;property name="sourcesPrefix"&gt;
 *                         &lt;array type="String[]"&gt;
 *                             &lt;string&gt;int:&lt;/string&gt;
 *                             &lt;string&gt;long:&lt;/string&gt;
 *                         &lt;/array&gt;
 *                     &lt;/property&gt;
 *                 &lt;/object&gt;
 *             &lt;/property&gt;
 *         &lt;/proxy&gt;
 * 
 *     &lt;/proxys&gt;
 * 
 * &lt;/shoin-context&gt;
 * </pre>
 */
public class Shoin {

    /** デフォルトの設定ファイルのパス (classpath:/shoin-context.xml)。 */
    public static final String DEFAULT_CONFIG_PATH = "classpath:/shoin-context.xml";

    private final ShoinContext context;

    /**
     * デフォルトの設定でコンテキストを生成して初期化します。
     * 
     * @throws ConfigurationException
     *             構成情報の解析に失敗した場合
     */
    public Shoin() {
        this(Shoin.class.getClassLoader());
    }

    /**
     * 指定されたクラスローダとデフォルトのURLでコンテキストを生成して初期化します。
     * 
     * @param classLoader
     *            クラスローダ
     * @throws ConfigurationException
     *             構成情報の解析に失敗した場合
     */
    public Shoin(ClassLoader classLoader) {
        this(classLoader, DEFAULT_CONFIG_PATH);
    }

    /**
     * 指定されたクラスローダとURLでコンテキストを生成して初期化します。
     * 
     * @param classLoader
     *            クラスローダ
     * @param url
     *            リソースURL
     * @throws ConfigurationException
     *             構成情報の解析に失敗した場合
     */
    public Shoin(ClassLoader classLoader, String url) {

        final ShoinContextFactory factory = new ShoinContextFactory();
        try {

            IOResourceLoader loader = FeatureFactory.getLoader().getIOResourceLoader();
            loader.setClassLoader(classLoader);
            IOResource inres = loader.getIOResource(url);

            factory.setIOResources(new IOResource[] { inres });
            factory.setClassLoader(classLoader);
            this.context = new ShoinContext(factory.getResourceSet());

        } catch (Exception e) {
            throw new ConfigurationException("shoin configuration error. ", e);
        }
    }

    /**
     * 再生成を行わずに、識別子を持つリソース集合を返却します。<br>
     * 識別子が登録されていない場合は null が返却されます。
     * 
     * @param id
     *            識別子
     * @return 識別子を持つリソース集合
     * @throws ResourceSetCreationException
     *             リソース集合の生成に失敗した場合
     */
    public ResourceSet lookup(String id) throws ResourceSetCreationException {
        return this.context.lookup(id);
    }

    /**
     * 再生成を行うか指定して、識別子を持つリソース集合を返却します。<br>
     * 識別子が登録されていない場合は null が返却されます。
     * 
     * @param id
     *            識別子
     * @param create
     *            再生成を要求する場合は true
     * @return 識別子を持つリソース集合
     * @throws ResourceSetCreationException
     *             リソース集合の生成に失敗した場合
     */
    public ResourceSet lookup(String id, boolean create) throws ResourceSetCreationException {
        return this.context.lookup(id, create);
    }

}
