/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
/**
 * 
 */
package shohaku.sugina.util;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.Charset;

import shohaku.core.helpers.HCoder;
import shohaku.core.lang.NoSuchResourceException;
import shohaku.core.lang.feature.FeatureFactory;
import shohaku.core.util.XProperties;
import shohaku.ogdl.Ogdl;
import shohaku.ogdl.OgdlContext;
import shohaku.ogdl.OgdlHelper;
import shohaku.ogdl.OgdlSyntaxException;

/**
 * 
 * プロパティ値を、OGDL 式として解析するプロパティセットを提供します。<br>
 * <p>
 * 例：
 * 
 * <pre>
 * # &#64; で始まると処理の宣言と見なされます。
 * &#64;import=java.sql.Time　　　//クラスをOGDLにインポートします。
 * 
 * # $ で始まると、OGDL変数の宣言と為ります。
 * $sunday="日曜"　　　//sundayが変数名、値をOGDL式として評価
 * 
 * #コメント
 * season.map={ 1="春", 2="夏", 3="秋", 4="冬" }　//プロパティを定義します。キーは文字列、値をOGDL式として評価
 * </pre>
 * 
 * コメントや区切り文字等の仕様は java.util.Properties と同等です。<br>
 * OGDLの構文に関しては、以下の <a href="http://shohaku.sourceforge.jp/projects/ogdl/reference/reference.html" target="_blank">OGDL リファレンス</a> を参照してください。<br>
 * <br>
 */
public class OgdlXProperties extends XProperties {

    /* serialVersionUID */
    private static final long serialVersionUID = 4109039838422671403L;

    final OgdlContext context;

    final Ogdl ogdl;

    /**
     * 初期化します。
     * 
     */
    public OgdlXProperties() {
        this(OgdlHelper.getOgdlContext(), OgdlXProperties.class.getClassLoader());
    }

    /**
     * 初期化します。
     * 
     * @param context
     * @param loader
     */
    public OgdlXProperties(OgdlContext context, ClassLoader loader) {
        super();
        this.context = context;
        this.ogdl = new Ogdl();
        ogdl.setContext(context);
        context.setClassLoader(loader);
    }

    /* Ogdl式として解析してプロパティを登録します */
    protected void putProperty(String key, String value, boolean isEscapes) throws IOException {

        String k = key;
        String v = value;
        if (isEscapes) {
            k = HCoder.decodePropertiesEscapes(k);
            v = HCoder.decodePropertiesEscapes(v);
        }

        try {
            if (k.charAt(0) == '$') {
                context.setAttribute(k.substring(1), ogdl.evaluate(v));
            } else if (k.charAt(0) == '@') {
                if (k.equals("@import")) {
                    Class c = null;
                    try {
                        c = FeatureFactory.getLoader().getClass(v);
                    } catch (NoSuchResourceException e) {
                        throw new IOException(e.getMessage());
                    }
                    context.addImport(c);
                } else {
                    if (k.equals("@eval")) {
                        ogdl.evaluate(v);
                    }
                }
            } else {
                this.lookup.put(k, ogdl.evaluate(v));
            }
        } catch (OgdlSyntaxException e) {
            final IOException ie = new IOException("ogdl evaluation err.");
            ie.initCause(e);// 1.5以降は起因例外のコンストラクタがある
            throw ie;
        }
    }

    /**
     * サポートしていません、UnsupportedOperationException を発生させます。
     * 
     * @param outStream
     *            出力ストリーム
     * @param charset
     *            文字エンコーディング
     * @param header
     *            ヘッダー
     * @throws IOException
     *             IO例外
     */
    public void store(OutputStream outStream, Charset charset, String header) throws IOException {
        throw new UnsupportedOperationException("Un Supported OgdlXProperties#store().");
    }

    /**
     * サポートしていません、UnsupportedOperationException を発生させます。
     * 
     * @param outStream
     *            出力ストリーム
     * @param header
     *            ヘッダー
     * @throws IOException
     *             IO例外
     */
    public void store(OutputStream outStream, String header) throws IOException {
        throw new UnsupportedOperationException("Un Supported OgdlXProperties#store().");
    }

}