/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.sugina.beans;

import java.lang.reflect.Array;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import shohaku.core.helpers.HBeans;
import shohaku.core.helpers.HSeek;
import shohaku.core.lang.Eval;
import shohaku.core.lang.IntrospectionBeansException;
import shohaku.ogdl.Ogdl;
import shohaku.ogdl.OgdlHelper;
import shohaku.ogdl.OgdlParseIndex;
import shohaku.ogdl.OgdlSyntaxException;

/**
 * 簡易な記述式を基に、オブジェクトグラフをメソッドの連鎖として、再帰的に実行して最後の実行結果を戻り値として返却します。<br>
 * このオブジェクトはスレッドセーフであることが保証されています。<br>
 * MethodGraph.compile(String) から返されるインスタンスは不変オブジェクトです。<br>
 * インスタンス生成の以降は、全ての処理がメソッド内で完結します。<br>
 * <br>
 * 処理例：
 * 
 * <pre>
 *  // 以下の例は ((String[] ) bean.get("names"))[0].substring(0, 11) と同等です。
 *  
 *  MethodGraph methodGraph = MethodGraph.compile("get('names')[0].substring(0, 11)"); //コンパイル
 * 
 *  Map bean = new HashMap();
 *  bean.put("names", new String[] { "Aleksandros III Megas", "Napoléon Bonaparte", "jinkiz khān"});
 *  
 *  Object name = methodGraph.invoke(bean); //メソッドを再起的に呼び出し最後の結果を返す
 *  
 *  System.out.println("name=" + name);
 *  
 *  .....................
 *  name=Aleksandros  
 * </pre>
 * 
 * 構文内の引数やインデクサには、松柏ライブラリの提供する式言語の OGDL が使用されます。<br>
 * OGDL はデータ生成に特化した簡易な式言語です。 <br>
 * 参照：<a href="http://shohaku.sourceforge.jp/projects/ogdl/reference/reference.html" target="_blank">OGDL リファレンス</a><br>
 * <br>
 * 構文の説明： <br>
 */
public final class MethodGraph {

    /**
     * このモードが指定されると、パターン文字列が、構文式として認識できない書式で終了する場合に、それ以降の文字列を無視します。<br>
     * 例として構文 "property[0].property;abcdewshjp..." はこのモードが指定されない場合は構文エラーが発生しますが、<br>
     * このモードが指定された場合は、構文式以外の文字 ";" の地点で解析が終了しコンパイルが正常に終了します。
     */
    public static final int LAST_ESC = 0x01;

    // /**
    // * パターンの構文区切り前後で空白とコメントを使用できるようにします。<br>
    // * このモードでは、空白は無視され、# で始まる埋め込みコメントは行末まで無視されます。<br>
    // * このモードで有効となる区切り文字は、'.', '[', ']', '(', ')', ',' となります。
    // */
    // public static final int COMMENTS = 0x02;

    /* 書式を関数分割する記号文字。 */
    private static final char DIVIDE = '.';

    /* インデックス（配列、マップキー、プロパティ）アクセスの記号文字。 */
    private static final char INDEX_OP = '[';

    /* インデックス（配列、マップキー、プロパティ）アクセスの記号文字。 */
    private static final char INDEX_CL = ']';

    /* メソッドの引数を指定する記号文字。 */
    private static final char ARGS_OP = '(';

    /* メソッドの引数を指定する記号文字。 */
    private static final char ARGS_CL = ')';

    /* コンパイル元の書式パターン。 */
    private final String _pattern;

    /* コンパイルオプションのビットマスク。 */
    private final int _flags;

    /* 実行関数を格納する実行機能。 */
    private final Invoker _invoker;

    /**
     * 書式パターンをコンパイルし初期化します。
     * 
     * @param pattern
     *            書式パターンを定義する文字列
     * @param flags
     *            コンパイルオプションを示すフラグ。LAST_EC などを含むビットマスク
     * @throws MethodGraphSyntaxException
     *             書式パターンの構文が無効である場合
     * @throws IllegalArgumentException
     *             定義されたマッチフラグに対応しないビット値が flags に設定された場合
     */
    private MethodGraph(final String pattern, final int flags) {
        this._pattern = pattern;
        this._flags = flags;
        this._invoker = compileImpl(pattern);
    }

    /* コンパイルを実装します。 */
    private Invoker compileImpl(final String pattern) {

        if (Eval.isBlank(pattern)) {
            throw new MethodGraphSyntaxException("pattern is blank.");
        }

        final char[] openChars = new char[] { DIVIDE, INDEX_OP, ARGS_OP };

        final List invokers = new LinkedList();
        final OgdlParseIndex off = new OgdlParseIndex(0);
        final Ogdl el = new Ogdl();
        el.setContext(OgdlHelper.getOgdlContext());
        el.getContext().setClassLoader(MethodGraph.class.getClassLoader());
        try {

            while (off.get() < pattern.length()) {

                if (pattern.charAt(off.get()) == INDEX_OP) {
                    // evaluateList EL. index(List, Array) or map
                    List args = el.evaluateList(pattern, off, INDEX_OP, INDEX_CL);
                    if (args.size() != 1) {
                        throw new MethodGraphSyntaxException(log(pattern, off));
                    }
                    invokers.add(new CollectionInvoker(args.get(0), pattern));

                } else {
                    if (pattern.charAt(off.get()) == DIVIDE) {
                        off.increment();
                    } else if (off.get() != 0) {
                        if (has(LAST_ESC)) {
                            break;
                        }
                        throw new MethodGraphSyntaxException(log(pattern, off));
                    }

                    // method or property
                    int end = HSeek.orIndexOf(pattern, off.get(), openChars, 1);
                    if (-1 < end) {

                        if (pattern.charAt(end) == ARGS_OP) {// method 'method(arg...)'
                            String methodName = pattern.substring(off.get(), end);
                            off.set(end);
                            // evaluateList (arg...)]
                            List args = el.evaluateList(pattern, off, ARGS_OP, ARGS_CL);
                            invokers.add(new MethodInvoker(methodName, args.toArray(), pattern));

                        } else {// property 'property. or property[num]'
                            String propertyName = pattern.substring(off.get(), end);
                            invokers.add(new PropertyInvoker(propertyName));
                            off.set(end);
                        }

                    } else {
                        // property
                        String propertyName = pattern.substring(off.get());
                        invokers.add(new PropertyInvoker(propertyName));
                        off.set(pattern.length());
                    }
                }

            }
            return new RootInvoker((Invoker[]) invokers.toArray(new Invoker[invokers.size()]));
        } catch (OgdlSyntaxException e) {
            throw new MethodGraphSyntaxException(log(pattern, off), e);
        } catch (IndexOutOfBoundsException e) {
            throw new MethodGraphSyntaxException(log(pattern, off), e);
        } catch (NumberFormatException e) {
            throw new MethodGraphSyntaxException(log(pattern, off), e);
        }
    }

    private String log(String pattern, OgdlParseIndex off) {
        return "illegal pattern=" + pattern + ", offset=" + off;
    }

    /**
     * 引数のオブジェクトを対象にして処理を実行します。
     * 
     * @param target
     *            処理対象のオブジェクト
     * @return 処理の戻り値、void の場合 null
     * @throws IntrospectionBeansException
     *             メソッドの呼出に失敗した場合
     */
    public Object invoke(Object target) throws IntrospectionBeansException {
        return _invoker.invoke(target, null, false);
    }

    /**
     * 引数のオブジェクトを対象にして値の設定処理を実行します。<br>
     * パターンの最終的な処理対象が値設定可能である必要があります、メソッドには使用出来ません。
     * 
     * @param target
     *            処理対象のオブジェクト
     * @param value
     *            設定する値
     * @return 処理の戻り値、void の場合 null
     * @throws IntrospectionBeansException
     *             メソッドの呼出に失敗した場合
     */
    public Object invoke(Object target, Object value) throws IntrospectionBeansException {
        return _invoker.invoke(target, value, true);
    }

    /**
     * コンパイル元の書式パターンを返します。
     * 
     * @return コンパイル元の書式パターン
     */
    public String pattern() {
        return _pattern;
    }

    /**
     * 文字列表現を返却します。
     * 
     * @return 文字列表現
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return _pattern;
    }

    /*
     * private
     */

    /* 指定のフラグが指定されている場合は true を返却する。 */
    private boolean has(int f) {
        return (_flags & f) > 0;
    }

    // ***********************************************//
    // * compile
    // ***********************************************//

    /**
     * 書式パターンをコンパイルし MethodGraph インスタンスを返却します。
     * 
     * @param pattern
     *            書式パターンを定義する文字列
     * @return コンパイルされた MethodGraph インスタンス
     * @throws MethodGraphSyntaxException
     *             書式パターンの構文が無効である場合
     */
    public static MethodGraph compile(String pattern) {
        return compile(pattern, 0);
    }

    /**
     * 書式パターンをコンパイルし MethodGraph インスタンスを返却します。
     * 
     * @param pattern
     *            書式パターンを定義する文字列
     * @param flags
     *            コンパイルオプションを示すフラグ。LAST_EC などを含むビットマスク
     * @return コンパイルされた MethodGraph インスタンス
     * @throws MethodGraphSyntaxException
     *             書式パターンの構文が無効である場合
     * @throws IllegalArgumentException
     *             定義されたマッチフラグに対応しないビット値が flags に設定された場合
     */
    public static MethodGraph compile(String pattern, int flags) {
        return new MethodGraph(pattern, flags);
    }

    // ***********************************************//
    // * Invoker
    // ***********************************************//

    /* 実行関数を定義します。 */
    static interface Invoker {

        /**
         * 関数を実行します、スレッドセーフである必要が有ります。
         * 
         * @param target
         *            処理対象のオブジェクト
         * @param value
         *            格納処理の値
         * @param isSetter
         *            格納処理の場合は true
         * @return 実行関数の戻り値
         * @throws IntrospectionBeansException
         */
        Object invoke(Object target, Object value, boolean isSetter) throws IntrospectionBeansException;
    }

    /* 実行関数のルートを定義します。 */
    static class RootInvoker implements Invoker {

        private final Invoker[] _invokers;

        /* 実行するオブジェクトを実行順の配列で格納して初期化します。 */
        RootInvoker(Invoker[] invokers) {
            this._invokers = invokers;
        }

        public Object invoke(Object target, Object value, boolean isSetter) throws IntrospectionBeansException {
            Object t = target;
            int size = _invokers.length;
            for (int i = 0; i < size; i++) {
                if ((i + 1) < size) {
                    t = _invokers[i].invoke(t, value, false);
                } else {
                    t = _invokers[i].invoke(t, value, isSetter);
                }

            }
            return t;
        }
    }

    /* コレクションアクセスの実行関数を定義します。 */
    static class CollectionInvoker implements Invoker {

        private final Object _key;

        private final Integer _index;

        private final String _pattern;

        CollectionInvoker(Object key, String pattern) {
            this._key = key;
            this._index = (key instanceof Integer) ? (Integer) key : null;
            this._pattern = pattern;
        }

        public Object invoke(Object target, Object value, boolean isSetter) throws IntrospectionBeansException {
            if (!isSetter) {
                if (target instanceof Map) {
                    return ((Map) target).get(_key);
                } else if (_index != null) {
                    if (target instanceof List) {
                        return ((List) target).get(_index.intValue());
                    } else if (Eval.isArray(target)) {
                        return Array.get(target, _index.intValue());
                    }
                }
            } else {
                if (target instanceof Map) {
                    return ((Map) target).put(_key, value);
                } else if (_index != null) {
                    if (target instanceof List) {
                        return ((List) target).set(_index.intValue(), value);
                    } else if (Eval.isArray(target)) {
                        Object oldValue = Array.get(target, _index.intValue());
                        Array.set(target, _index.intValue(), value);
                        return oldValue;
                    }
                }
            }
            throw new IntrospectionBeansException("illegal arg type. " + _pattern + ", " + ", " + target + ", " + _key);
        }
    }

    /* メソッドアクセスの実行関数を定義します。 */
    static class MethodInvoker implements Invoker {

        private final String _methodName;

        private final Class[] _types;

        private final Object[] _args;

        private final String _pattern;

        MethodInvoker(String methodName, Object[] args, String pattern) {
            this._methodName = methodName;
            this._types = HBeans.toClassArray(args);
            this._args = args;
            this._pattern = pattern;
        }

        public Object invoke(Object target, Object value, boolean isSetter) throws IntrospectionBeansException {
            if (!isSetter) {
                return HBeans.invokeMethod(target, HBeans.getAssignmentMethod(target.getClass(), _methodName, _types), _args);
            }
            throw new IntrospectionBeansException("no supported set method. " + _pattern + ", " + _methodName);
        }
    }

    /* プロパティアクセスの実行関数を定義します。 */
    static class PropertyInvoker implements Invoker {

        private final String _propertyName;

        PropertyInvoker(String propertyName) {
            this._propertyName = propertyName;
        }

        public Object invoke(Object target, Object value, boolean isSetter) throws IntrospectionBeansException {
            if (!isSetter) {
                return HBeans.getProperty(target, _propertyName);
            }
            HBeans.setProperty(target, _propertyName, value);
            return null;
        }
    }

}
