/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.util;

import java.util.Iterator;
import java.util.Map;

/**
 * Ascii 文字列を構成するビットセットを提供します。 <br>
 * このビットセットは 32 ビットのマッピングのみ定義可能な有限ビットセットです。
 */
public class AsciiBitSet {

    /** 大文字アルファベット。 */
    public static final int UPPER = 0x00000001;

    /** 小文字アルファベット。 */
    public static final int LOWER = 0x00000002;

    /** ２進数字。 */
    public static final int BINARY = 0x00000004;

    /** ８進数字。 */
    public static final int OCTAL = 0x00000008;

    /** １０進数字。 */
    public static final int DIGIT = 0x00000010;

    /** １６進数字。 */
    public static final int HEX = 0x00000020;

    /** スペース文字（空文字と改行）。 */
    public static final int SPACE = 0x00000040;

    /** 記号文字。 */
    public static final int PUNCT = 0x00000080;

    /** 制御文字。 */
    public static final int CNTRL = 0x00000100;

    /** 空文字（タブ、スペース）。 */
    public static final int BLANK = 0x00000200;

    /** 印刷可能文字。 */
    public static final int PRINT = 0x00000400;

    /** アンダーバー。 */
    public static final int UNDER = 0x00000800;

    /** アルファベット。 */
    public static final int ALPHA = (UPPER | LOWER);

    /** アルファベットと数字。 */
    public static final int ALNUM = (UPPER | LOWER | DIGIT);

    /** グリフ文字。 */
    public static final int GRAPH = (PUNCT | UPPER | LOWER | DIGIT);

    /** 単語を構成する文字。 */
    public static final int WORD = (UPPER | LOWER | UNDER | DIGIT);

    /** ビットセットのマッピング。 */
    private final int[] cclazz = new int[0xFF];

    /** ビットセットのマッピング。 */
    protected final static int[] DEFAULT_CHAR_CLAZZ;
    static {
        final int[] cclazz = new int[0xFF];
        cclazz[0x00] = CNTRL; // (NUL)
        cclazz[0x01] = CNTRL; // (SOH)
        cclazz[0x02] = CNTRL; // (STX)
        cclazz[0x03] = CNTRL; // (ETX)
        cclazz[0x04] = CNTRL; // (EOT)
        cclazz[0x05] = CNTRL; // (ENQ)
        cclazz[0x06] = CNTRL; // (ACK)
        cclazz[0x07] = CNTRL; // (BEL)
        cclazz[0x08] = CNTRL; // (BS)
        cclazz[0x09] = SPACE + CNTRL + BLANK; // (HT)
        cclazz[0x0A] = SPACE + CNTRL; // (LF)
        cclazz[0x0B] = SPACE + CNTRL; // (VT)
        cclazz[0x0C] = SPACE + CNTRL; // (FF)
        cclazz[0x0D] = SPACE + CNTRL; // (CR)
        cclazz[0x0E] = CNTRL; // (SI)
        cclazz[0x0F] = CNTRL; // (SO)
        cclazz[0x10] = CNTRL; // (DLE)
        cclazz[0x11] = CNTRL; // (DC1)
        cclazz[0x12] = CNTRL; // (DC2)
        cclazz[0x13] = CNTRL; // (DC3)
        cclazz[0x14] = CNTRL; // (DC4)
        cclazz[0x15] = CNTRL; // (NAK)
        cclazz[0x16] = CNTRL; // (SYN)
        cclazz[0x17] = CNTRL; // (ETB)
        cclazz[0x18] = CNTRL; // (CAN)
        cclazz[0x19] = CNTRL; // (EM)
        cclazz[0x1A] = CNTRL; // (SUB)
        cclazz[0x1B] = CNTRL; // (ESC)
        cclazz[0x1C] = CNTRL; // (FS)
        cclazz[0x1D] = CNTRL; // (GS)
        cclazz[0x1E] = CNTRL; // (RS)
        cclazz[0x1F] = CNTRL; // (US)
        cclazz[0x20] = PRINT + SPACE + BLANK; // SPACE
        cclazz[0x21] = PRINT + PUNCT; // !
        cclazz[0x22] = PRINT + PUNCT; // "
        cclazz[0x23] = PRINT + PUNCT; // #
        cclazz[0x24] = PRINT + PUNCT; // $
        cclazz[0x25] = PRINT + PUNCT; // %
        cclazz[0x26] = PRINT + PUNCT; // &
        cclazz[0x27] = PRINT + PUNCT; // '
        cclazz[0x28] = PRINT + PUNCT; // (
        cclazz[0x29] = PRINT + PUNCT; // )
        cclazz[0x2A] = PRINT + PUNCT; // *
        cclazz[0x2B] = PRINT + PUNCT; // +
        cclazz[0x2C] = PRINT + PUNCT; // ,
        cclazz[0x2D] = PRINT + PUNCT; // -
        cclazz[0x2E] = PRINT + PUNCT; // .
        cclazz[0x2F] = PRINT + PUNCT; // /
        cclazz[0x30] = PRINT + DIGIT + HEX + OCTAL + BINARY; // 0
        cclazz[0x31] = PRINT + DIGIT + HEX + OCTAL + BINARY; // 1
        cclazz[0x32] = PRINT + DIGIT + HEX + OCTAL; // 2
        cclazz[0x33] = PRINT + DIGIT + HEX + OCTAL; // 3
        cclazz[0x34] = PRINT + DIGIT + HEX + OCTAL; // 4
        cclazz[0x35] = PRINT + DIGIT + HEX + OCTAL; // 5
        cclazz[0x36] = PRINT + DIGIT + HEX + OCTAL; // 6
        cclazz[0x37] = PRINT + DIGIT + HEX + OCTAL; // 7
        cclazz[0x38] = PRINT + DIGIT + HEX; // 8
        cclazz[0x39] = PRINT + DIGIT + HEX; // 9
        cclazz[0x3A] = PRINT + PUNCT; // :
        cclazz[0x3B] = PRINT + PUNCT; // ;
        cclazz[0x3C] = PRINT + PUNCT; // <
        cclazz[0x3D] = PRINT + PUNCT; // =
        cclazz[0x3E] = PRINT + PUNCT; // >
        cclazz[0x3F] = PRINT + PUNCT; // ?
        cclazz[0x40] = PRINT + PUNCT; // @
        cclazz[0x41] = PRINT + UPPER + HEX; // A
        cclazz[0x42] = PRINT + UPPER + HEX; // B
        cclazz[0x43] = PRINT + UPPER + HEX; // C
        cclazz[0x44] = PRINT + UPPER + HEX; // D
        cclazz[0x45] = PRINT + UPPER + HEX; // E
        cclazz[0x46] = PRINT + UPPER + HEX; // F
        cclazz[0x47] = PRINT + UPPER; // G
        cclazz[0x48] = PRINT + UPPER; // H
        cclazz[0x49] = PRINT + UPPER; // I
        cclazz[0x4A] = PRINT + UPPER; // J
        cclazz[0x4B] = PRINT + UPPER; // K
        cclazz[0x4C] = PRINT + UPPER; // L
        cclazz[0x4D] = PRINT + UPPER; // M
        cclazz[0x4E] = PRINT + UPPER; // N
        cclazz[0x4F] = PRINT + UPPER; // O
        cclazz[0x50] = PRINT + UPPER; // P
        cclazz[0x51] = PRINT + UPPER; // Q
        cclazz[0x52] = PRINT + UPPER; // R
        cclazz[0x53] = PRINT + UPPER; // S
        cclazz[0x54] = PRINT + UPPER; // T
        cclazz[0x55] = PRINT + UPPER; // U
        cclazz[0x56] = PRINT + UPPER; // V
        cclazz[0x57] = PRINT + UPPER; // W
        cclazz[0x58] = PRINT + UPPER + HEX; // X
        cclazz[0x59] = PRINT + UPPER; // Y
        cclazz[0x5A] = PRINT + UPPER; // Z
        cclazz[0x5B] = PRINT + PUNCT; // [
        cclazz[0x5C] = PRINT + PUNCT; // \
        cclazz[0x5D] = PRINT + PUNCT; // ]
        cclazz[0x5E] = PRINT + PUNCT; // ^
        cclazz[0x5F] = PRINT + PUNCT + UNDER; // _
        cclazz[0x60] = PRINT + PUNCT; // `
        cclazz[0x61] = PRINT + LOWER + HEX; // a
        cclazz[0x62] = PRINT + LOWER + HEX; // b
        cclazz[0x63] = PRINT + LOWER + HEX; // c
        cclazz[0x64] = PRINT + LOWER + HEX; // d
        cclazz[0x65] = PRINT + LOWER + HEX; // e
        cclazz[0x66] = PRINT + LOWER + HEX; // f
        cclazz[0x67] = PRINT + LOWER; // g
        cclazz[0x68] = PRINT + LOWER; // h
        cclazz[0x69] = PRINT + LOWER; // i
        cclazz[0x6A] = PRINT + LOWER; // j
        cclazz[0x6B] = PRINT + LOWER; // k
        cclazz[0x6C] = PRINT + LOWER; // l
        cclazz[0x6D] = PRINT + LOWER; // m
        cclazz[0x6E] = PRINT + LOWER; // n
        cclazz[0x6F] = PRINT + LOWER; // o
        cclazz[0x70] = PRINT + LOWER; // p
        cclazz[0x71] = PRINT + LOWER; // q
        cclazz[0x72] = PRINT + LOWER; // r
        cclazz[0x73] = PRINT + LOWER; // s
        cclazz[0x74] = PRINT + LOWER; // t
        cclazz[0x75] = PRINT + LOWER; // u
        cclazz[0x76] = PRINT + LOWER; // v
        cclazz[0x77] = PRINT + LOWER; // w
        cclazz[0x78] = PRINT + LOWER + HEX; // x
        cclazz[0x79] = PRINT + LOWER; // y
        cclazz[0x7A] = PRINT + LOWER; // z
        cclazz[0x7B] = PRINT + PUNCT; // {
        cclazz[0x7C] = PRINT + PUNCT; // |
        cclazz[0x7D] = PRINT + PUNCT; // }
        cclazz[0x7E] = PRINT + PUNCT; // ~
        cclazz[0x7F] = CNTRL; // (DEL)
        DEFAULT_CHAR_CLAZZ = cclazz;
    }

    /**
     * デフォルトで初期化します。
     */
    public AsciiBitSet() {
        init();
    }

    /**
     * 加算するビットセットを設定して初期化します。<br>
     * 引数は Map<Character, Integer> の構造である必要があります。
     * 
     * @param charInt
     *            加算するビットセット
     */
    public AsciiBitSet(Map charInt) {
        this();
        final Iterator i = charInt.entrySet().iterator();
        while (i.hasNext()) {
            final Map.Entry e = (Map.Entry) i.next();
            final Character ch = (Character) e.getKey();
            final Integer set = (Integer) e.getValue();
            add(ch.charValue(), set.intValue());
        }
    }

    /**
     * ビットセットを初期化します。
     */
    protected void init() {
        System.arraycopy(DEFAULT_CHAR_CLAZZ, 0, cclazz, 0, DEFAULT_CHAR_CLAZZ.length);
    }

    /**
     * 指定の文字のビットセットにクラスのマッピングを追加します。
     * 
     * @param ch
     *            対象の文字
     * @param set
     *            追加するビットセット
     * @return 追加されたビットセット
     */
    protected int add(int ch, int set) {
        if (!isAscii(ch)) {
            throw new IllegalArgumentException("is not Ascii. char=" + ch);
        }
        cclazz[ch] += set;
        return cclazz[ch];
    }

    /**
     * 指定の文字のビットセットにクラスのマッピングを返却します。<br>
     * アスキー文字以外が指定された場合は 0 を返却します。
     * 
     * @param ch
     *            対象の文字
     * @return ビットセット
     */
    public int getClazz(int ch) {
        return isAscii(ch) ? cclazz[ch] : 0;
    }

    /**
     * 指定の文字が指定のビットセットにクラスのマッピングされているか検証します。
     * 
     * @param ch
     *            対象の文字
     * @param clazz
     *            検証するクラス
     * @return マッピングされている場合は true
     */
    public boolean isClazz(int ch, int clazz) {
        return (getClazz(ch) & clazz) != 0;
    }

    /**
     * 指定の文字がアスキー文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return アスキー文字の場合は true
     */
    public boolean isAscii(int ch) {
        return ((ch & 0xFFFFFF80) == 0);
    }

    /**
     * 指定の文字がアルファベット文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return アルファベット文字の場合は true
     */
    public boolean isAlpha(int ch) {
        return isClazz(ch, ALPHA);
    }

    /**
     * 指定の文字が２進数字文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return ２進数字文字の場合は true
     */
    public boolean isBinaryDigit(int ch) {
        return isClazz(ch, BINARY);
    }

    /**
     * 指定の文字が８進数字文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return ８進数字文字の場合は true
     */
    public boolean isOctDigit(int ch) {
        return isClazz(ch, OCTAL);
    }

    /**
     * 指定の文字が１０進数字文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return １０進数字文字の場合は true
     */
    public boolean isDigit(int ch) {
        return isClazz(ch, DIGIT);
    }

    /**
     * 指定の文字が１６進数字文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return １６進数字文字の場合は true
     */
    public boolean isHexDigit(int ch) {
        return isClazz(ch, HEX);
    }

    /**
     * 指定の文字がアルファベット文字か数字文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return アルファベット文字か数字文字の場合は true
     */
    public boolean isAlnum(int ch) {
        return isClazz(ch, ALNUM);
    }

    /**
     * 指定の文字がグリフ文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return グリフ文字の場合は true
     */
    public boolean isGraph(int ch) {
        return isClazz(ch, GRAPH);
    }

    /**
     * 指定の文字が印刷可能文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return 印刷可能文字の場合は true
     */
    public boolean isPrint(int ch) {
        return isClazz(ch, PRINT);
    }

    /**
     * 指定の文字が記号文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return 記号文字の場合は true
     */
    public boolean isPunct(int ch) {
        return isClazz(ch, PUNCT);
    }

    /**
     * 指定の文字がスペース文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return スペース文字の場合は true
     */
    public boolean isSpace(int ch) {
        return isClazz(ch, SPACE);
    }

    /**
     * 指定の文字が制御文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return 制御文字の場合は true
     */
    public boolean isCntrl(int ch) {
        return isClazz(ch, CNTRL);
    }

    /**
     * 指定の文字が小文字アルファベット文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return 小文字アルファベット文字の場合は true
     */
    public boolean isLower(int ch) {
        return isClazz(ch, LOWER);
    }

    /**
     * 指定の文字が大文字アルファベット文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return 大文字アルファベット文字の場合は true
     */
    public boolean isUpper(int ch) {
        return isClazz(ch, UPPER);
    }

    /**
     * 指定の文字が単語を構成する文字か検証します。
     * 
     * @param ch
     *            検証する文字
     * @return 単語を構成する文字の場合は true
     */
    public boolean isWord(int ch) {
        return isClazz(ch, WORD);
    }

    /**
     * 指定の文字を小文字のアルファベットに変換します。<br>
     * 文字が大文字のアルファベット以外の場合は、そのまま返却されます。
     * 
     * @param ch
     *            変換する文字
     * @return 変換後の文字
     */
    public int toLower(int ch) {
        return isUpper(ch) ? (ch + 0x20) : ch;
    }

    /**
     * 指定の文字を大文字のアルファベットに変換します。<br>
     * 文字が小文字のアルファベット以外の場合は、そのまま返却されます。
     * 
     * @param ch
     *            変換する文字
     * @return 変換後の文字
     */
    public int toUpper(int ch) {
        return isLower(ch) ? (ch - 0x20) : ch;
    }

}