/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.resource;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;

/**
 * URLが示すリソースの入力ストリームを返すIOリソースを提供します。
 */
public class UrlIOResource implements IOResource {

    /** ファイル。 */
    private final URL url;

    /**
     * URL文字列からURLオブジェクトを作成して初期化します。
     * 
     * @param spec
     *            URL文字列
     * @throws MalformedURLException
     *             文字列に指定されたプロトコルが未知である場合
     */
    public UrlIOResource(String spec) throws MalformedURLException {
        this.url = new URL(spec);
    }

    /**
     * URLオブジェクトを格納して初期化します。
     * 
     * @param url
     *            URL
     */
    public UrlIOResource(URL url) {
        this.url = url;
    }

    /**
     * URLオブジェクトが示すリソースの入力ストリームを返却します。
     * 
     * @return 入力ストリーム
     * @throws IOException
     *             入出力例外が発生した場合
     */
    public InputStream getInputStream() throws IOException {
        final URLConnection con = this.url.openConnection();
        con.setUseCaches(false);// キャッシュ不要
        return con.getInputStream();
    }

    /**
     * URLオブジェクトが示すリソースの出力ストリームを返却します。
     * 
     * @return 出力ストリーム
     * @throws IOException
     *             入出力例外が発生した場合
     */
    public OutputStream getOutputStream() throws IOException {
        final URLConnection con = this.url.openConnection();
        con.setUseCaches(false);// キャッシュ不要
        return con.getOutputStream();
    }

    /**
     * URLオブジェクトが示すリソースが存在するか検証します。
     * 
     * @return リソースが存在する場合は true
     * @throws SecurityException
     *             セキュリティ例外
     */
    public boolean exists() {
        InputStream is = null;
        try {
            final URLConnection con = this.url.openConnection();
            con.setUseCaches(false);// キャッシュ不要
            is = con.getInputStream();
            return true;
        } catch (IOException e) {
            return false;
        } finally {
            try {
                if (is != null) {
                    is.close();
                }
            } catch (IOException e) {
                // no op
            }
        }
    }

    /**
     * URLオブジェクトが示すリソースの最終更新日時を返却します。
     * 
     * @return リソースの最終更新日時
     * @throws SecurityException
     *             セキュリティ例外
     */
    public long getLastModified() throws IOException {
        final URLConnection con = this.url.openConnection();
        con.setUseCaches(false);// キャッシュ不要
        return con.getLastModified();
    }

}