/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.resource;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;

import shohaku.core.lang.Concat;
import shohaku.core.lang.Eval;

/**
 * URIからIOリソースを生成するファクトリを提供します。<br>
 * ここで指定されるリソースパスは URI (Uniform Resource Identifier) として有効である必要があります。<br>
 * 詳細は java.net.URI のドキュメントを参照してください。
 * 
 * @see java.net.URI
 * @see java.net.URI#relativize(URI)
 */
public class IOResourceLoader {

    /** クラスパスの検索時のスキーマ："classpath" */
    static final String CLASSPATH_URI_SCHEME = "classpath";

    /** クラスパスの検索時の接頭辞："classpath:" */
    public static final String CLASSPATH_URI_PREFIX = CLASSPATH_URI_SCHEME + ":";

    /** ファイルシステムの検索時の接頭辞："file:" */
    public static final String FILE_URI_PREFIX = "file:";

    /** クラスパスの検索に使用するクラスローダ */
    private ClassLoader classLoader;

    /** URIに追加するプレフィックス */
    private String uriPrefix;

    /**
     * クラスローダとプレフィックスを null で初期化します。
     */
    public IOResourceLoader() {
        super();
    }

    /**
     * クラスパスの検索に使用するクラスローダを返却します。
     * 
     * @return クラスローダ
     */
    public ClassLoader getClassLoader() {
        return classLoader;
    }

    /**
     * クラスパスの検索に使用するクラスローダを格納します。
     * 
     * @param classLoader
     *            クラスローダ
     */
    public void setClassLoader(ClassLoader classLoader) {
        this.classLoader = classLoader;
    }

    /**
     * URIに追加するプレフィックスを返却します。
     * 
     * @return URIに追加するプレフィックス
     */
    public String getUriPrefix() {
        return uriPrefix;
    }

    /**
     * URIに追加するプレフィックスを格納します。
     * 
     * @param prefix
     *            URIに追加するプレフィックス
     */
    public void setUriPrefix(String prefix) {
        this.uriPrefix = prefix;
    }

    /**
     * URI文字列の示すリソースオブジェクトを返却します。<br>
     * この引数は URI (Uniform Resource Identifier) として有効である必要があります。<br>
     * 引数にスキーマが指定されいる場合は絶対パスとしてプレフィックスは付加しません。<br>
     * プレフィックスと引数から新しい URI の構築は、単純な文字列の連結で行われます。 <br>
     * <br>
     * このメソッドはスレッドセーフです、実行時点のプロパティのスナップショットがメソッド内で使用されます。
     * 
     * @param uri
     *            リソースの論理位置を示すURI文字列
     * @return IOリソース
     * @throws IOException
     *             IOリソースの生成に失敗した場合
     * @throws URISyntaxException
     *             引数のURIまたはプレフィックスが URI として不正の場合
     */
    public IOResource getIOResource(String uri) throws IOException, URISyntaxException {
        // get snapshot property
        final String prefix = getUriPrefix();
        final ClassLoader loader = getClassLoader();

        final URI resourceUri = getResourceUri(prefix, uri);
        final String scheme = resourceUri.getScheme();
        if (Eval.isPrefix(scheme, CLASSPATH_URI_SCHEME)) {
            final String classpath = resourceUri.toString().substring(CLASSPATH_URI_PREFIX.length());
            return new ClassPathIOResource(classpath, loader);
        }
        if (Eval.isPrefix(scheme, FILE_URI_PREFIX)) {
            return new FileSystemIOResource(resourceUri);
        }
        return new UrlIOResource(resourceUri.toString());
    }

    /**
     * リソースのURI文字列を構築して返却します。
     * 
     * @param prefix
     *            URIに追加するプレフィックス
     * @param uri
     *            基となるURI
     * @return リソースのURI文字列
     */
    protected URI getResourceUri(String prefix, String uri) throws URISyntaxException {
        final URI relativeUri = new URI(uri);
        if (relativeUri.isAbsolute() || Eval.isBlank(prefix)) {
            return relativeUri;
        }

        final String relative = relativeUri.toString();
        final StringBuffer sb = new StringBuffer(Concat.size(prefix, relative));
        sb.append(prefix);
        if (prefix.endsWith("/")) {
            if (relative.startsWith("/")) {
                sb.append(relative.substring(1, relative.length()));
            }
        } else {
            if (!relative.startsWith("/")) {
                sb.append("/");
            }
            sb.append(relative);
        }
        return new URI(sb.toString());

    }

}
