/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.resource;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;

/**
 * ファイルシステム上のリソースの入力ストリームを返すIOリソースを提供します。
 */
public class FileSystemIOResource implements IOResource {

    /** ファイル。 */
    private final File file;

    /**
     * リソースのパスを格納して初期化します。
     * 
     * @param path
     *            ファイルパス
     */
    public FileSystemIOResource(String path) {
        this(new File(path));
    }

    /**
     * リソースのファイルを格納して初期化します。
     * 
     * @param uri
     *            ファイルのURI
     */
    public FileSystemIOResource(URI uri) {
        this(new File(uri));
    }

    /**
     * リソースのファイルを格納して初期化します。
     * 
     * @param file
     *            ファイル
     */
    public FileSystemIOResource(File file) {
        this.file = file;
    }

    /**
     * ファイルの入力ストリームを返却します。
     * 
     * @return 入力ストリーム
     * @throws FileNotFoundException
     *             ファイルを開くことが出来ない場合
     * @throws SecurityException
     *             セキュリティ例外
     */
    public InputStream getInputStream() throws IOException {
        return new FileInputStream(this.file);
    }

    /**
     * ファイルの出力ストリームを返却します。
     * 
     * @return 出力ストリーム
     * @throws FileNotFoundException
     *             ファイルを開くことが出来ない場合
     * @throws SecurityException
     *             セキュリティ例外
     */
    public OutputStream getOutputStream() throws IOException {
        return new FileOutputStream(this.file);
    }

    /**
     * ファイルが存在するか検証します。
     * 
     * @return ファイルが存在する場合は true
     * @throws SecurityException
     *             セキュリティ例外
     */
    public boolean exists() {
        return this.file.exists();
    }

    /**
     * ファイルの最終更新日時を返却します。
     * 
     * @return ファイルの最終更新日時
     * @throws SecurityException
     *             セキュリティ例外
     */
    public long getLastModified() throws IOException {
        return file.lastModified();
    }

}
