/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.resource;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.security.AccessController;
import java.security.PrivilegedAction;

import shohaku.core.lang.Eval;
import shohaku.core.lang.NoSuchResourceException;
import shohaku.core.lang.feature.FeatureFactory;

/**
 * リソースのクラスパスとクラスローダから入力ストリームを返すIOリソースを提供します。
 */
public class ClassPathIOResource implements IOResource {

    /** クラスパス。 */
    private final String path;

    /** リソースを読み取る為のクラスローダ。 */
    private final ClassLoader classLoader;

    /** リソースを読み取る為のクラス。 */
    private final Class clazz;

    /**
     * クラスパスを格納して初期化します。
     * 
     * @param path
     *            クラスパス
     * @throws IllegalArgumentException
     *             クラスパスが空の場合
     */
    public ClassPathIOResource(String path) {
        this(path, null, null);
    }

    /**
     * クラスパスとクラスローダを格納して初期化します。
     * 
     * @param path
     *            クラスパス
     * @param classLoader
     *            リソースを読み取る為のクラスローダ、null の場合は無視されます。
     * @throws IllegalArgumentException
     *             クラスパスが空の場合
     */
    public ClassPathIOResource(String path, ClassLoader classLoader) {
        this(path, classLoader, null);
    }

    /**
     * クラスパスとリードオブジェクトを格納して初期化します。
     * 
     * @param path
     *            クラスパス
     * @param clazz
     *            リソースを読み取る為のクラス、null の場合は無視されます
     * @throws IllegalArgumentException
     *             クラスパスが空の場合
     */
    public ClassPathIOResource(String path, Class clazz) {
        this(path, null, clazz);
    }

    /**
     * クラスパスとリードオブジェクトを格納して初期化します。
     * 
     * @param path
     *            クラスパス
     * @param classLoader
     *            クラスローダ
     * @param clazz
     *            クラス
     */
    private ClassPathIOResource(String path, ClassLoader classLoader, Class clazz) {
        if (Eval.isBlank(path)) {
            throw new IllegalArgumentException("path is blank");
        }
        this.path = ((path.startsWith("/")) ? path.substring(1) : path);
        this.classLoader = classLoader;
        this.clazz = clazz;
    }

    /**
     * クラスパスが示すリソースの入力ストリームを返却します。
     * 
     * @return 入力ストリーム
     * @throws IOException
     *             リソースが発見出来ない場合
     */
    public InputStream getInputStream() throws IOException {
        // 特権で実行する
        final Object r = AccessController.doPrivileged(new PrivilegedAction() {
            public Object run() {
                try {
                    final URLConnection con = getURL().openConnection();
                    con.setUseCaches(false);// キャッシュ不要
                    return con.getInputStream();
                } catch (IOException e) {
                    return e;
                } catch (NoSuchResourceException e) {
                    final IOException fe = new FileNotFoundException("resource not exist.");
                    fe.initCause(e);
                    return fe;
                }
            }
        });
        if (r instanceof IOException) {
            throw (IOException) r;
        }
        return (InputStream) r;
    }

    /**
     * クラスパスが示すリソースの出力ストリームを返却します。
     * 
     * @return 出力ストリーム
     * @throws IOException
     *             入出力例外が発生した場合
     */
    public OutputStream getOutputStream() throws IOException {
        // 特権で実行する
        final Object r = AccessController.doPrivileged(new PrivilegedAction() {
            public Object run() {
                try {
                    // クラスローダから取得するURLConnectionは出力処理をサポートいないのでURIからアクセスする
                    final URI uri = new URI(ClassPathIOResource.this.getURL().toString());
                    return new FileOutputStream(new File(uri));
                } catch (IOException e) {
                    return e;
                } catch (URISyntaxException e) {
                    final IOException fe = new FileNotFoundException("resource not exist.");
                    fe.initCause(e);
                    return fe;
                } catch (NoSuchResourceException e) {
                    final IOException fe = new FileNotFoundException("resource not exist.");
                    fe.initCause(e);
                    return fe;
                }
            }
        });
        if (r instanceof IOException) {
            throw (IOException) r;
        }
        return (OutputStream) r;
    }

    /**
     * クラスパスが示すリソースが存在するか検証します。
     * 
     * @return リソースが存在する場合は true
     * @throws SecurityException
     *             セキュリティ例外
     */
    public boolean exists() {
        InputStream is = null;
        try {
            final URLConnection con = getURL().openConnection();
            con.setUseCaches(false);// キャッシュ不要
            is = con.getInputStream();
            return true;
        } catch (IOException e) {
            return false;
        } catch (NoSuchResourceException e) {
            return false;
        } finally {
            try {
                if (is != null) {
                    is.close();
                }
            } catch (IOException e) {
                // no op
            }
        }
    }

    /**
     * クラスパスが示すリソースの最終更新日時を返却します。
     * 
     * @return リソースの最終更新日時
     * @throws SecurityException
     *             セキュリティ例外
     */
    public long getLastModified() throws IOException {
        try {
            final URLConnection con = getURL().openConnection();
            con.setUseCaches(false);// キャッシュ不要
            return con.getLastModified();
        } catch (NoSuchResourceException e) {
            final IOException fe = new FileNotFoundException("resource not exist.");
            fe.initCause(e);
            throw fe;
        }
    }

    /**
     * クラスパスが示すリソースの URL オブジェクトを返却します。
     * 
     * @return URL オブジェクト
     * @throws NoSuchResourceException
     *             リソースが発見出来ない場合
     */
    URL getURL() throws NoSuchResourceException {
        if (this.classLoader != null) {
            return FeatureFactory.getLoader().getResource(this.path, this.classLoader);
        }
        if (this.clazz != null) {
            return FeatureFactory.getLoader().getResource(this.path, this.clazz);
        }
        return FeatureFactory.getLoader().getResource(this.path);
    }
}