/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.lang.feature.impl;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.net.URISyntaxException;
import java.net.URL;

import shohaku.core.lang.Eval;
import shohaku.core.lang.NoSuchResourceException;
import shohaku.core.lang.ObjectCreationException;
import shohaku.core.lang.ShohakuCoreSystem;
import shohaku.core.lang.SystemUtils;
import shohaku.core.lang.feature.ResourceLoader;
import shohaku.core.resource.IOResource;
import shohaku.core.resource.IOResourceLoader;

/**
 * リソースを読込む機能のデフォルト実装を提供します。
 */
public class DefaultResourceLoader implements ResourceLoader {

    static final String RESOURCE_PREFIX;
    static {
        // システムリソースの接頭辞
        final String syatemPrefix = SystemUtils.getSystemProperty("shohaku.resource.prefix");
        String prefix = syatemPrefix;
        if (Eval.isBlank(prefix)) {
            prefix = (String) ShohakuCoreSystem.getLibraryClassProperty(ResourceLoader.class, "resource.prefix");
        }
        if (Eval.isBlank(prefix)) {
            prefix = IOResourceLoader.CLASSPATH_URI_PREFIX;
        }
        RESOURCE_PREFIX = prefix;
    }

    /*
     * IOResource
     */

    public IOResourceLoader getRelativeIOResourceLoader() {
        IOResourceLoader ioResourceLoader = new IOResourceLoader();
        ioResourceLoader.setUriPrefix(RESOURCE_PREFIX);
        return ioResourceLoader;
    }

    public IOResourceLoader getIOResourceLoader() {
        IOResourceLoader ioResourceLoader = new IOResourceLoader();
        ioResourceLoader.setUriPrefix(IOResourceLoader.CLASSPATH_URI_PREFIX);
        return ioResourceLoader;
    }

    public IOResource getIOResource(String url) throws IOException, URISyntaxException {
        return getIOResource(url, null);
    }

    public IOResource getIOResource(String url, ClassLoader loader) throws IOException, URISyntaxException {
        IOResourceLoader ioResourceLoader = getIOResourceLoader();
        ClassLoader l = (loader != null) ? loader : ClassLoader.getSystemClassLoader();
        ioResourceLoader.setClassLoader(l);
        return ioResourceLoader.getIOResource(url);
    }

    /*
     * class
     */

    public Class getClass(String className) throws NoSuchResourceException {
        return getClass(className, null);
    }

    public Class getClass(String className, ClassLoader loader) throws NoSuchResourceException {
        ClassLoader classLoader = loader;
        // Class.forName：指定クラスの ClassLoader
        try {
            if (classLoader != null) {
                return Class.forName(className, true, loader);
            } else {
                return Class.forName(className);
            }
        } catch (Exception e) {
            // no op
        }
        // Current Thread の ClassLoader から取得
        try {
            classLoader = getContextClassLoader();
            if (classLoader != null) {
                return classLoader.loadClass(className);
            }
        } catch (Exception e) {
            // no op
        }
        // システムクラスローダを使って検索します。
        try {
            classLoader = ClassLoader.getSystemClassLoader();
            return classLoader.loadClass(className);
        } catch (Exception e) {
            // no op
        }
        throw new NoSuchResourceException("Not Find Class:" + className);
    }

    /*
     * URL
     */

    public URL getResource(String resource) throws NoSuchResourceException {
        return getResource(resource, ResourceLoader.class);
    }

    public URL getResource(String resource, Class c) throws NoSuchResourceException {
        return getResource(resource, c.getClassLoader());
    }

    public URL getResource(String resource, ClassLoader loader) throws NoSuchResourceException {
        ClassLoader classLoader = null;
        URL url = null;
        try {
            // 指定クラスの ClassLoader
            classLoader = loader;
            if (classLoader != null) {
                url = classLoader.getResource(resource);
                if (url != null) {
                    return url;
                }
            }
            // Current Thread の ClassLoader
            classLoader = getContextClassLoader();
            if (classLoader != null) {
                url = classLoader.getResource(resource);
                if (url != null) {
                    return url;
                }
            }
            // システムクラスローダを使って検索します。
            url = ClassLoader.getSystemResource(resource);
        } catch (Exception e) {
            // no op
        }
        if (url == null) {
            throw new NoSuchResourceException("Not Find Resource:" + resource);
        }
        return url;
    }

    /*
     * InputStream
     */

    public InputStream getResourceAsStream(String resource) throws NoSuchResourceException {
        return getResourceAsStream(resource, ResourceLoader.class);
    }

    public InputStream getResourceAsStream(String resource, Class c) throws NoSuchResourceException {
        return getResourceAsStream(resource, c.getClassLoader());
    }

    public InputStream getResourceAsStream(String resource, ClassLoader loader) throws NoSuchResourceException {
        ClassLoader classLoader = null;
        InputStream inStream = null;
        try {
            // 指定クラスの ClassLoader
            classLoader = loader;
            if (classLoader != null) {
                inStream = classLoader.getResourceAsStream(resource);
                if (inStream != null) {
                    return inStream;
                }
            }
            // Current Thread の ClassLoader
            classLoader = getContextClassLoader();
            if (classLoader != null) {
                inStream = classLoader.getResourceAsStream(resource);
                if (inStream != null) {
                    return inStream;
                }
            }
            // システムクラスローダを使って検索します。
            inStream = ClassLoader.getSystemResourceAsStream(resource);
        } catch (Exception e) {
            // no op
        }
        if (inStream == null) {
            throw new NoSuchResourceException("Not Find Resource:" + resource);
        }
        return inStream;
    }

    /*
     * instance
     */

    public Object getInstance(String className) throws ObjectCreationException {
        return getInstance(className, null);
    }

    public Object getInstance(String className, ClassLoader loader) throws ObjectCreationException {
        try {
            return getClass(className, loader).newInstance();
        } catch (Exception e) {
            throw new ObjectCreationException("class:'" + className, e);
        }
    }

    public Object getInstance(String className, ClassLoader loader, Class[] parameterTypes, Object[] parameterValues) throws ObjectCreationException {
        try {
            Class c = getClass(className, loader);
            Constructor constructor = c.getConstructor(parameterTypes);
            return constructor.newInstance(parameterValues);
        } catch (Exception e) {
            throw new ObjectCreationException("class:'" + className, e);
        }

    }

    public Object getInstance(Class c) throws ObjectCreationException {
        try {
            return c.newInstance();
        } catch (Exception e) {
            throw new ObjectCreationException("class:" + c, e);
        }
    }

    public Object getInstance(Class c, Class[] parameterTypes, Object[] parameterValues) throws ObjectCreationException {
        try {
            Constructor constructor = c.getConstructor(parameterTypes);
            return constructor.newInstance(parameterValues);
        } catch (Exception e) {
            throw new ObjectCreationException("class:" + c, e);
        }
    }

    /*
     * private
     */

    /* 現在の Thread のコンテキスト ClassLoader を返却します。 */
    private ClassLoader getContextClassLoader() {
        return Thread.currentThread().getContextClassLoader();
    }

}