/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.lang.feature.impl;

import java.io.InputStream;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;

import shohaku.core.lang.ShohakuCoreSystem;
import shohaku.core.lang.feature.LogFeature;

/**
 * コンソールを出力先とするログ機能を提供します。
 */
class ConsoleLogFeature implements LogFeature {

    /*
     * protected static final
     */

    /* デフォルトの日付フォーマット。 */
    protected static final String DEFAULT_DATE_TIME_FORMAT = "yyyy/MM/dd HH:mm:ss:SSS zzz";

    /*
     * protected static
     */

    /* ログ名をメッセージ出力に含めるかを示す。 */
    protected static boolean showLogName = false;

    /* ログ名を短縮形で出力するかを示す。 */
    protected static boolean showShortName = true;

    /* 時刻をメッセージ出力に含めるかを示す。 */
    protected static boolean showDateTime = false;

    /* 時刻をメッセージ出力する際の日付フォーマット。 */
    protected static String dateTimeFormat = DEFAULT_DATE_TIME_FORMAT;

    /* 使用する日付フォーマッタ。 */
    protected static DateFormat dateFormatter = null;

    /*
     * Log Level Constants
     */

    /** "Trace" レベルのログを示す。 */
    public static final int LOG_LEVEL_TRACE = 1;

    /** "Debug" レベルのログを示す。 */
    public static final int LOG_LEVEL_DEBUG = 2;

    /** "Info" レベルのログを示す。 */
    public static final int LOG_LEVEL_INFO = 3;

    /** "Warn" レベルのログを示す。 */
    public static final int LOG_LEVEL_WARN = 4;

    /** "Error" レベルのログを示す。 */
    public static final int LOG_LEVEL_ERROR = 5;

    /** "Fatal" レベルのログを示す。 */
    public static final int LOG_LEVEL_FATAL = 6;

    /** 全てのログレベルが有効であることを示す。 */
    public static final int LOG_LEVEL_ALL = (LOG_LEVEL_TRACE - 1);

    /** 全てのログレベルが無効であることを示す。 */
    public static final int LOG_LEVEL_OFF = (LOG_LEVEL_FATAL + 1);

    /*
     * Initializer
     */

    protected static Object getProperty(String name) {
        return ShohakuCoreSystem.getLibraryClassProperty(LogFeature.class, name);
    }

    protected static Object getProperty(String name, Object dephault) {
        Object prop = getProperty(name);
        return (prop == null) ? dephault : prop;
    }

    protected static String getStringProperty(String name) {
        Object prop = getProperty(name);
        return (prop == null) ? null : prop.toString();
    }

    protected static String getStringProperty(String name, String dephault) {
        Object prop = getProperty(name, dephault);
        return (prop == null) ? null : prop.toString();
    }

    protected static boolean getBooleanProperty(String name, boolean dephault) {
        Object prop = getProperty(name, Boolean.valueOf(dephault));
        if (prop instanceof Boolean) {
            return ((Boolean) prop).booleanValue();
        }
        return false;
    }

    static {

        showLogName = getBooleanProperty("showlogname", showLogName);
        showShortName = getBooleanProperty("showShortLogname", showShortName);
        showDateTime = getBooleanProperty("showdatetime", showDateTime);

        if (showDateTime) {
            dateTimeFormat = getStringProperty("dateTimeFormat", dateTimeFormat);
            try {
                dateFormatter = new SimpleDateFormat(dateTimeFormat);
            } catch (IllegalArgumentException e) {
                dateTimeFormat = DEFAULT_DATE_TIME_FORMAT;
                dateFormatter = new SimpleDateFormat(dateTimeFormat);
            }
        }
    }

    /*
     * Attributes
     */

    /* このログインスタンスに対する名前。 */
    protected String logName = null;

    /* 現在のログレベル。 */
    protected int currentLogLevel;

    /* このログインスタンスに対する短い名前。 */
    protected String shortLogName = null;

    /*
     * Constructor
     */

    /**
     * ログの名前を指定して初期化します。
     * 
     * @param name
     *            ログの名前
     */
    public ConsoleLogFeature(String name) {

        logName = name;

        setLevel(ConsoleLogFeature.LOG_LEVEL_INFO);

        // Set log level from properties
        String lvl = getStringProperty("log." + logName);
        int i = String.valueOf(name).lastIndexOf(".");
        String nm = name;
        while (null == lvl && i > -1) {
            nm = nm.substring(0, i);
            lvl = getStringProperty("log." + nm);
            i = String.valueOf(nm).lastIndexOf(".");
        }

        if (null == lvl) {
            lvl = getStringProperty("defaultlog");
        }

        if ("all".equalsIgnoreCase(lvl)) {
            setLevel(ConsoleLogFeature.LOG_LEVEL_ALL);
        } else if ("trace".equalsIgnoreCase(lvl)) {
            setLevel(ConsoleLogFeature.LOG_LEVEL_TRACE);
        } else if ("debug".equalsIgnoreCase(lvl)) {
            setLevel(ConsoleLogFeature.LOG_LEVEL_DEBUG);
        } else if ("info".equalsIgnoreCase(lvl)) {
            setLevel(ConsoleLogFeature.LOG_LEVEL_INFO);
        } else if ("warn".equalsIgnoreCase(lvl)) {
            setLevel(ConsoleLogFeature.LOG_LEVEL_WARN);
        } else if ("error".equalsIgnoreCase(lvl)) {
            setLevel(ConsoleLogFeature.LOG_LEVEL_ERROR);
        } else if ("fatal".equalsIgnoreCase(lvl)) {
            setLevel(ConsoleLogFeature.LOG_LEVEL_FATAL);
        } else if ("off".equalsIgnoreCase(lvl)) {
            setLevel(ConsoleLogFeature.LOG_LEVEL_OFF);
        }

    }

    /*
     * Properties
     */

    /**
     * 現在のログレベルを設定します。
     * 
     * @param currentLogLevel
     *            現在のログレベル
     */
    public void setLevel(int currentLogLevel) {
        this.currentLogLevel = currentLogLevel;

    }

    /**
     * 現在のログレベルを取得します。
     * 
     * @return 現在のログレベル
     */
    public int getLevel() {
        return currentLogLevel;
    }

    /*
     * Logging Methods
     */

    /* 各レベルのログの出力を実行します。 */
    protected void log(int type, Object message, Throwable t) {
        // Use a string buffer for better performance
        StringBuffer buf = new StringBuffer();

        // Append date-time if so configured
        if (showDateTime) {
            buf.append(dateFormatter.format(new Date()));
            buf.append(" ");
        }

        // Append a readable representation of the log level
        switch (type) {
        case ConsoleLogFeature.LOG_LEVEL_TRACE:
            buf.append("[TRACE] ");
            break;
        case ConsoleLogFeature.LOG_LEVEL_DEBUG:
            buf.append("[DEBUG] ");
            break;
        case ConsoleLogFeature.LOG_LEVEL_INFO:
            buf.append("[INFO] ");
            break;
        case ConsoleLogFeature.LOG_LEVEL_WARN:
            buf.append("[WARN] ");
            break;
        case ConsoleLogFeature.LOG_LEVEL_ERROR:
            buf.append("[ERROR] ");
            break;
        case ConsoleLogFeature.LOG_LEVEL_FATAL:
            buf.append("[FATAL] ");
            break;
        }

        // Append the name of the log instance if so
        // configured
        if (showShortName) {
            if (shortLogName == null) {
                // Cut all but the last component of the
                // name for both styles
                shortLogName = logName.substring(logName.lastIndexOf(".") + 1);
                shortLogName = shortLogName.substring(shortLogName.lastIndexOf("/") + 1);
            }
            buf.append(String.valueOf(shortLogName)).append(" - ");
        } else if (showLogName) {
            buf.append(String.valueOf(logName)).append(" - ");
        }

        // Append the message
        buf.append(String.valueOf(message));

        // Append stack trace if not null
        if (t != null) {
            buf.append(" <");
            buf.append(t.toString());
            buf.append(">");

            java.io.StringWriter sw = new java.io.StringWriter(1024);
            java.io.PrintWriter pw = new java.io.PrintWriter(sw);
            t.printStackTrace(pw);
            pw.close();
            buf.append(sw.toString());
        }

        // Print to the appropriate destination
        write(type, buf);

    }

    /* ログの書き込みを実行します。 */
    protected void write(int type, StringBuffer buffer) {
        if (type > ConsoleLogFeature.LOG_LEVEL_INFO) {
            // WARN, ERROR, FATAL : error
            System.err.println(buffer.toString());
        } else {
            // TRACE, DEBUG, INFO : output
            System.out.println(buffer.toString());
        }
    }

    /* 指定されたログレベルが現在有効か検証して返す。 */
    protected boolean isLevelEnabled(int logLevel) {
        // log level are numerically ordered so can use
        // introspect numeric
        // comparison
        return (logLevel >= currentLogLevel);
    }

    /*
     * Log Implementation
     */

    /**
     * Debug レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     */
    public final void debug(Object message) {
        if (isLevelEnabled(ConsoleLogFeature.LOG_LEVEL_DEBUG)) {
            log(ConsoleLogFeature.LOG_LEVEL_DEBUG, message, null);
        }
    }

    /**
     * Debug レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     * @param t
     *            エラーまたは例外
     */
    public final void debug(Object message, Throwable t) {
        if (isLevelEnabled(ConsoleLogFeature.LOG_LEVEL_DEBUG)) {
            log(ConsoleLogFeature.LOG_LEVEL_DEBUG, message, t);
        }
    }

    /**
     * Trace レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     */
    public final void trace(Object message) {
        if (isLevelEnabled(ConsoleLogFeature.LOG_LEVEL_TRACE)) {
            log(ConsoleLogFeature.LOG_LEVEL_TRACE, message, null);
        }
    }

    /**
     * Trace レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     * @param t
     *            エラーまたは例外
     */
    public final void trace(Object message, Throwable t) {
        if (isLevelEnabled(ConsoleLogFeature.LOG_LEVEL_TRACE)) {
            log(ConsoleLogFeature.LOG_LEVEL_TRACE, message, t);
        }
    }

    /**
     * Info レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     */
    public final void info(Object message) {
        if (isLevelEnabled(ConsoleLogFeature.LOG_LEVEL_INFO)) {
            log(ConsoleLogFeature.LOG_LEVEL_INFO, message, null);
        }
    }

    /**
     * Info レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     * @param t
     *            エラーまたは例外
     */
    public final void info(Object message, Throwable t) {
        if (isLevelEnabled(ConsoleLogFeature.LOG_LEVEL_INFO)) {
            log(ConsoleLogFeature.LOG_LEVEL_INFO, message, t);
        }
    }

    /**
     * Warn レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     */
    public final void warn(Object message) {
        if (isLevelEnabled(ConsoleLogFeature.LOG_LEVEL_WARN)) {
            log(ConsoleLogFeature.LOG_LEVEL_WARN, message, null);
        }
    }

    /**
     * Warn レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     * @param t
     *            エラーまたは例外
     */
    public final void warn(Object message, Throwable t) {
        if (isLevelEnabled(ConsoleLogFeature.LOG_LEVEL_WARN)) {
            log(ConsoleLogFeature.LOG_LEVEL_WARN, message, t);
        }
    }

    /**
     * Error レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     */
    public final void error(Object message) {
        if (isLevelEnabled(ConsoleLogFeature.LOG_LEVEL_ERROR)) {
            log(ConsoleLogFeature.LOG_LEVEL_ERROR, message, null);
        }
    }

    /**
     * Error レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     * @param t
     *            エラーまたは例外
     */
    public final void error(Object message, Throwable t) {
        if (isLevelEnabled(ConsoleLogFeature.LOG_LEVEL_ERROR)) {
            log(ConsoleLogFeature.LOG_LEVEL_ERROR, message, t);
        }
    }

    /**
     * Fatal レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     */
    public final void fatal(Object message) {
        if (isLevelEnabled(ConsoleLogFeature.LOG_LEVEL_FATAL)) {
            log(ConsoleLogFeature.LOG_LEVEL_FATAL, message, null);
        }
    }

    /**
     * Fatal レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     * @param t
     *            エラーまたは例外
     */
    public final void fatal(Object message, Throwable t) {
        if (isLevelEnabled(ConsoleLogFeature.LOG_LEVEL_FATAL)) {
            log(ConsoleLogFeature.LOG_LEVEL_FATAL, message, t);
        }
    }

    /**
     * Debug レベルが有効の場合は true を返却します。
     * 
     * @return Debug レベルが有効であるか
     */
    public final boolean isDebugEnabled() {
        return isLevelEnabled(ConsoleLogFeature.LOG_LEVEL_DEBUG);
    }

    /**
     * Error レベルが有効の場合は true を返却します。
     * 
     * @return Error レベルが有効であるか
     */
    public final boolean isErrorEnabled() {
        return isLevelEnabled(ConsoleLogFeature.LOG_LEVEL_ERROR);
    }

    /**
     * Fatal レベルが有効の場合は true を返却します。
     * 
     * @return Fatal レベルが有効であるか
     */
    public final boolean isFatalEnabled() {
        return isLevelEnabled(ConsoleLogFeature.LOG_LEVEL_FATAL);
    }

    /**
     * Info レベルが有効の場合は true を返却します。
     * 
     * @return Info レベルが有効であるか
     */
    public final boolean isInfoEnabled() {
        return isLevelEnabled(ConsoleLogFeature.LOG_LEVEL_INFO);
    }

    /**
     * Trace レベルが有効の場合は true を返却します。
     * 
     * @return Trace レベルが有効であるか
     */
    public final boolean isTraceEnabled() {
        return isLevelEnabled(ConsoleLogFeature.LOG_LEVEL_TRACE);
    }

    /**
     * Warn レベルが有効の場合は true を返却します。
     * 
     * @return Warn レベルが有効であるか
     */
    public final boolean isWarnEnabled() {
        return isLevelEnabled(ConsoleLogFeature.LOG_LEVEL_WARN);
    }

    /* 引数が示すリソースの入力ストリームを特権で取得して返却します。 */
    protected static InputStream getResourceAsStream(final String name) {
        return (InputStream) AccessController.doPrivileged(new PrivilegedAction() {
            public Object run() {
                ClassLoader threadCL = Thread.currentThread().getContextClassLoader();
                if (threadCL != null) {
                    return threadCL.getResourceAsStream(name);
                } else {
                    return ClassLoader.getSystemResourceAsStream(name);
                }
            }
        });
    }

}