/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.lang.feature;

import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.net.URL;

import shohaku.core.lang.NoSuchResourceException;
import shohaku.core.lang.ObjectCreationException;
import shohaku.core.resource.IOResource;
import shohaku.core.resource.IOResourceLoader;

/**
 * リソースを読込む機能を定義します。
 */
public interface ResourceLoader {

    /**
     * 相対パスでアクセスするIOリソース生成機能を返却します。<br>
     * <br>
     * これは IOResourceLoader の URL に追加するプレフィックスを、環境変数か外部ファイルで変更する機能です。<br>
     * ここで返される IOResourceLoader を使用すれば後でリソース位置を簡単に変更する事が出来ます。<br>
     * またこの機能は、松柏ライブラリを組み込む側のシステム用なので、松柏ライブラリ内部からは使用されません。<br>
     * <br>
     * プレフィックスは以下の基準で検索されます。<br>
     * 1. システムプロパティ "shohaku.resource.prefix" を検索する。<br>
     * 2. 松柏コアのライブラリプロパティ "@shohaku.core.lang.feature.ResourceLoader.resource.prefix" を検索する。<br>
     * <br>
     * 検索で見つからない場合は、実装クラスのデフォルトが使用されます。<br>
     * デフォルト実装では IOResourceLoader.CLASSPATH_URI_PREFIX を使用します。<br>
     * 
     * @return IOリソース生成機能
     * @see shohaku.core.resource.IOResourceLoader
     * @see shohaku.core.lang.ShohakuCoreSystem#getLibraryClassProperty(Class, String)
     */
    IOResourceLoader getRelativeIOResourceLoader();

    /**
     * デフォルトのIOリソース生成機能を返却します。<br>
     * デフォルトではIOリソースのプレフィックスにクラスパスが使用されます。
     * 
     * @return IOリソース生成機能
     */
    IOResourceLoader getIOResourceLoader();

    /**
     * デフォルトのIOリソース生成機能から、URL文字列の示すリソースオブジェクトを返却します。<br>
     * この引数は URI (Uniform Resource Identifier) として有効である必要があります。<br>
     * 引数にスキーマが指定されいる場合は絶対パスとしてプレフィックスは付加しません。
     * 
     * @param url
     *            リソースの論理位置を示すURL文字列
     * @return URLに対応するIOリソース
     * @throws IOException
     *             IOリソースの生成に失敗した場合
     * @throws URISyntaxException
     *             引数のURIまたはプレフィックスが URI として不正の場合
     * @see shohaku.core.resource.IOResourceLoader#getIOResource(String)
     */
    IOResource getIOResource(String url) throws IOException, URISyntaxException;

    /**
     * デフォルトのIOリソース生成機能から、URL文字列の示すリソースオブジェクトを返却します。<br>
     * この引数は URI (Uniform Resource Identifier) として有効である必要があります。<br>
     * 引数にスキーマが指定されいる場合は絶対パスとしてプレフィックスは付加しません。
     * 
     * @param url
     *            リソースの論理位置を示すURL文字列
     * @param loader
     *            リソースの検索に使用するクラスローダ、null を指定すると標準のクラスローダを使用します。
     * @return URLに対応するIOリソース
     * @throws IOException
     *             IOリソースの生成に失敗した場合
     * @throws URISyntaxException
     *             引数のURIまたはプレフィックスが URI として不正の場合
     * @see shohaku.core.resource.IOResourceLoader#getIOResource(String)
     */
    IOResource getIOResource(String url, ClassLoader loader) throws IOException, URISyntaxException;

    /**
     * 指定された名前のクラスをロードして返却します。 <br>
     * クラスが見つからなかった場合は NoSuchResourceException を発生します。
     * 
     * @param className
     *            クラス名
     * @return クラス
     * @throws NoSuchResourceException
     *             クラスが見つからなかった場合
     */
    Class getClass(String className) throws NoSuchResourceException;

    /**
     * 指定された名前のクラスをロードして返却します。 <br>
     * クラスが見つからなかった場合は NoSuchResourceException を発生します。
     * 
     * @param className
     *            クラス名
     * @param loader
     *            リソースの検索に使用するクラスローダ、null を指定すると標準のクラスローダを使用します
     * @return クラス
     * @throws NoSuchResourceException
     *             クラスが見つからなかった場合
     */
    Class getClass(String className, ClassLoader loader) throws NoSuchResourceException;

    /*
     * URL
     */

    /**
     * リソースパスが示すリソースのURLを取得して返却します。 <br>
     * リソースが見つからなかった場合は NoSuchResourceException を発生します。
     * 
     * @param resource
     *            リソースパス
     * @return リソースパスが示すURL オブジェクト
     * @throws NoSuchResourceException
     *             リソースが見つからなかった場合
     */
    URL getResource(String resource) throws NoSuchResourceException;

    /**
     * リソースパスが示すリソースのURLを取得して返却します。 <br>
     * リソースが見つからなかった場合は NoSuchResourceException を発生します。
     * 
     * @param resource
     *            リソースパス
     * @param clazz
     *            リソースの検索に使用するクラス情報
     * @return リソースパスが示すURL オブジェクト
     * @throws NoSuchResourceException
     *             リソースが見つからなかった場合
     */
    URL getResource(String resource, Class clazz) throws NoSuchResourceException;

    /**
     * リソースパスが示すリソースのURLを取得して返却します。 <br>
     * リソースが見つからなかった場合は NoSuchResourceException を発生します。
     * 
     * @param resource
     *            リソースパス
     * @param loader
     *            リソースの検索に使用するクラスローダ
     * @return リソースパスが示すURL オブジェクト
     * @throws NoSuchResourceException
     *             リソースが見つからなかった場合
     */
    URL getResource(String resource, ClassLoader loader) throws NoSuchResourceException;

    /*
     * InputStream
     */

    /**
     * リソースパスが示すリソースの入力ストリームを取得して返却します。 <br>
     * リソースが見つからなかった場合は NoSuchResourceException を発生します。
     * 
     * @param resource
     *            リソースパス
     * @return リソースパスが示す入力ストリーム
     * @throws NoSuchResourceException
     *             リソースが見つからなかった場合
     */
    InputStream getResourceAsStream(String resource) throws NoSuchResourceException;

    /**
     * リソースパスが示すリソースの入力ストリームを取得して返却します。 <br>
     * リソースが見つからなかった場合は NoSuchResourceException を発生します。
     * 
     * @param resource
     *            リソースパス
     * @param clazz
     *            リソースの検索に使用するクラス情報
     * @return リソースパスが示す入力ストリーム
     * @throws NoSuchResourceException
     *             リソースが見つからなかった場合
     */
    InputStream getResourceAsStream(String resource, Class clazz) throws NoSuchResourceException;

    /**
     * リソースパスが示すリソースの入力ストリームを取得して返却します。 <br>
     * リソースが見つからなかった場合は NoSuchResourceException を発生します。
     * 
     * @param resource
     *            リソースパス
     * @param loader
     *            リソースの検索に使用するクラスローダ
     * @return リソースパスが示す入力ストリーム
     * @throws NoSuchResourceException
     *             リソースが見つからなかった場合
     */
    InputStream getResourceAsStream(String resource, ClassLoader loader) throws NoSuchResourceException;

    /*
     * instance
     */

    /**
     * クラス名のインスタンスを生成して返却します。 <br>
     * 生成に失敗した場合は ObjectCreationException を発生します。
     * 
     * @param className
     *            生成するクラス名
     * @return 指定クラス名のインスタンス
     * @throws ObjectCreationException
     *             生成に失敗した場合
     */
    Object getInstance(String className) throws ObjectCreationException;

    /**
     * 指定されたクラス名のインスタンスを生成して返却します。 <br>
     * 生成に失敗した場合は ObjectCreationException を発生します。
     * 
     * @param className
     *            生成するクラス名
     * @param loader
     *            リソースの検索に使用するクラスローダ
     * @return 指定クラス名のインスタンス
     * @throws ObjectCreationException
     *             生成に失敗した場合
     */
    Object getInstance(String className, ClassLoader loader) throws ObjectCreationException;

    /**
     * 指定されたクラス名のパラメータ型が示すコンストラクタからインスタンスを生成して返却します。 <br>
     * 生成に失敗した場合は ObjectCreationException を発生します。
     * 
     * @param className
     *            生成するクラス名
     * @param loader
     *            リソースの検索に使用するクラスローダ
     * @param parameterTypes
     *            コンストラクタに渡すパラメータの型
     * @param parameterValues
     *            コンストラクタに渡すパラメータの値
     * @return 指定クラス名のインスタンス
     * @throws ObjectCreationException
     *             生成に失敗した場合
     */
    Object getInstance(String className, ClassLoader loader, Class[] parameterTypes, Object[] parameterValues) throws ObjectCreationException;

    /**
     * 指定されたクラスのインスタンスを生成して返却します。 <br>
     * 生成に失敗した場合は ObjectCreationException を発生します。
     * 
     * @param clazz
     *            生成するクラス
     * @return 指定クラスのインスタンス
     * @throws ObjectCreationException
     *             生成に失敗した場合
     */
    Object getInstance(Class clazz) throws ObjectCreationException;

    /**
     * 指定されたクラスのパラメータ型が示すコンストラクタからインスタンスを生成して返却します。 <br>
     * 生成に失敗した場合は ObjectCreationException を発生します。
     * 
     * @param clazz
     *            生成する
     * @param parameterTypes
     *            コンストラクタに渡すパラメータの型
     * @param parameterValues
     *            コンストラクタに渡すパラメータの値
     * @return 指定クラス名のインスタンス
     * @throws ObjectCreationException
     *             生成に失敗した場合
     */
    Object getInstance(Class clazz, Class[] parameterTypes, Object[] parameterValues) throws ObjectCreationException;

}