/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.lang.feature;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

import shohaku.core.lang.ConfigurationException;
import shohaku.core.lang.feature.impl.DefaultFeatureFactory;

/**
 * ライブラリの横断的な機能を生成するファクトリを提供します。<br>
 * <br>
 * 各機能の実装クラスはプロパティファイルに定義します。<br>
 * デフォルトプロパティのクラスパスは "shohaku-feature-defaults.properties" です。<br>
 * 拡張プロパティのクラスパスは "shohaku-feature-extends.properties" です。<br>
 * デフォルトプロパティの値を拡張プロパティで上書きできます。
 */
public abstract class FeatureFactory {

    /** 機能のインスタンスを保持する。 */
    protected final Map features = Collections.synchronizedMap(new HashMap());

    /**
     * 初期化ポイントです、実装クラスは公開された同シグニチャーのコンストラクタを必ず定義します。<br>
     * シングルトンのインスタンス生成時呼び出されます。
     * 
     * @param properties
     *            初期化パラメータ
     */
    protected FeatureFactory(Map properties) {
        initialize(properties);
    }

    /**
     * 初期化を実行します。拡張ポイントです。
     * 
     * @param properties
     *            初期化パラメータ
     */
    protected void initialize(Map properties) {
        for (Iterator i = properties.entrySet().iterator(); i.hasNext();) {
            Map.Entry e = (Map.Entry) i.next();
            String key = (String) e.getKey();
            String value = (String) e.getValue();
            Class id = loadClass(key);
            Object impl = getInstance(value);
            addFeature(id, impl);
        }
    }

    /**
     * リソースローダ機能を生成して返却します。
     * 
     * @return リソースローダ機能
     */
    public abstract ResourceLoader getResourceLoader();

    /**
     * ログ機能を生成して返却します。
     * 
     * @param clazz
     *            クラス
     * @return ログ機能
     */
    public abstract LogFeature getLogFeature(Class clazz);

    /**
     * ログ機能を生成して返却します。
     * 
     * @param name
     *            ログ名
     * @return ログ機能
     */
    public abstract LogFeature getLogFeature(String name);

    /**
     * 識別子の示す機能のインスタンスを返却します。
     * 
     * @param id
     *            識別子
     * @return 機能のインスタンス
     */
    public Object getFeature(Class id) {
        if (id == null) {
            throw new ConfigurationException("id is null.");
        }
        return features.get(id);
    }

    /**
     * 識別子の示す機能のインスタンスを追加します。
     * 
     * @param id
     *            識別子
     * @param newFeature
     *            追加する機能のインスタンス
     */
    public void addFeature(Class id, Object newFeature) {
        if (id == null) {
            throw new ConfigurationException("id is null.");
        }
        if (newFeature == null) {
            throw new ConfigurationException("feature is null.");
        }
        if (!id.isInstance(newFeature)) {
            throw new ConfigurationException("!(id Instanceof feature).");
        }
        features.put(id, newFeature);
    }

    /**
     * リソースローダ機能を生成して返却します。<br>
     * これは FeatureFactory.getFactory().getResourceLoader() と同意です。
     * 
     * @return リソースローダ機能
     */
    public static ResourceLoader getLoader() {
        return getFactory().getResourceLoader();
    }

    /**
     * ログ機能を生成して返却します。<br>
     * これは FeatureFactory.getFactory().getLogFeature(clazz) と同意です。
     * 
     * @param clazz
     *            クラス
     * @return ログ機能
     */
    public static LogFeature getLog(Class clazz) {
        return getFactory().getLogFeature(clazz);
    }

    /**
     * ログ機能を生成して返却します。<br>
     * これは FeatureFactory.getFactory().getLogFeature(name) と同意です。
     * 
     * @param name
     *            ログ名
     * @return ログ機能
     */
    public static LogFeature getLog(String name) {
        return getFactory().getLogFeature(name);
    }

    /**
     * ファクトリのインスタンスを返却します。
     * 
     * @return ファクトリのインスタンス
     */
    public static FeatureFactory getFactory() {
        return FactoryCache.featureFactory;
    }

    /*
     * helper
     */

    /* クラス名のインスタンスを生成して返却します。 */
    protected static Object getInstance(String name) {
        Class clazz = loadClass(name);
        if (clazz != null) {
            try {
                return clazz.newInstance();
            } catch (Exception e) {
                // no op
            }
        }
        return null;
    }

    protected static Class loadClass(String name) {
        Class clazz = null;
        // Current ClassLoader から取得
        try {
            clazz = Class.forName(name);
        } catch (Exception e) {
            // no op
        }
        if (clazz == null) {
            // Current Thread の ClassLoader から取得
            try {
                ClassLoader loader = Thread.currentThread().getContextClassLoader();
                if (loader != null) {
                    clazz = loader.loadClass(name);
                }
            } catch (Exception e) {
                // no op
            }
        }
        if (clazz == null) {
            // システムクラスローダを使って検索します。
            try {
                ClassLoader loader = ClassLoader.getSystemClassLoader();
                clazz = loader.loadClass(name);
            } catch (Exception e) {
                // no op
            }
        }
        return clazz;
    }

    /* リソースの入力ストリームを特権で取得して返却します。 */
    static InputStream getResourceAsStream(final String name) {
        return (InputStream) AccessController.doPrivileged(new PrivilegedAction() {
            public Object run() {
                ClassLoader threadCL = Thread.currentThread().getContextClassLoader();
                if (threadCL != null) {
                    return threadCL.getResourceAsStream(name);
                } else {
                    return ClassLoader.getSystemResourceAsStream(name);
                }
            }
        });
    }

    /* シングルトンインスタンスを保持するクラス */
    private static final class FactoryCache {
        static final FeatureFactory featureFactory;
        static {
            // 機能定義プロパティを取得
            final Properties defaultProperties = loadProperties("shohaku-feature-defaults.properties");
            final Properties extendProperties = loadProperties("shohaku-feature-extends.properties");

            Map properties = new HashMap(defaultProperties);
            properties.putAll(extendProperties);

            // ファクトリのインスタンスを取得
            final String className = (String) properties.get(FeatureFactory.class.getName());
            FeatureFactory _factory = null;
            if (className == null) {
                _factory = new DefaultFeatureFactory(properties);
            } else {
                properties.remove(FeatureFactory.class.getName());
                try {
                    Class c = FeatureFactory.loadClass(className);
                    Constructor con = c.getConstructor(new Class[] { Map.class });
                    _factory = (FeatureFactory) con.newInstance(new Object[] { properties });
                } catch (Exception e) {
                    throw new InstantiationError("shohaku FeatureFactory creation err. " + className);
                }
            }
            properties.remove(FeatureFactory.class.getName());
            featureFactory = _factory;
        }
    }

    static Properties loadProperties(String path) {
        final Properties properties = new Properties();
        final InputStream in = getResourceAsStream(path);
        if (null != in) {
            try {
                properties.load(in);
            } catch (java.io.IOException e) {
                // no op
            } finally {
                try {
                    in.close();
                } catch (IOException e) {
                    // no op
                }
            }
        }
        return properties;
    }

}
