/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.lang;

/**
 * 変更可能な Double 型を提供します。
 */
public final class VariableDouble implements VariableNumber, Comparable {

    /* serialVersionUID */
    private static final long serialVersionUID = 7582024230823225337L;

    /** 変数値. */
    private double value;

    /**
     * Double 型の初期値（0.0）で初期化します.
     */
    public VariableDouble() {
        this(0.0);
    }

    /**
     * 指定の値で初期化します.
     * 
     * @param value
     *            初期値
     */
    public VariableDouble(double value) {
        this.value = value;
    }

    /**
     * 文字列を Double.valueOf(s).doubleValue() で変換した値で初期化します.
     * 
     * @param s
     *            初期値
     * @throws NumberFormatException
     *             引数が double として認識出来ない場合
     */
    public VariableDouble(String s) throws NumberFormatException {
        this.value = Double.valueOf(s).doubleValue();
    }

    /**
     * 指定の値の Number.doubleValue() で初期化します.
     * 
     * @param value
     *            初期値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public VariableDouble(Number value) {
        this.value = value.doubleValue();
    }

    /**
     * 指定の値の VariableNumber.doubleValue() で初期化します.
     * 
     * @param value
     *            初期値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public VariableDouble(VariableNumber value) {
        this.value = value.doubleValue();
    }

    /*
     * math
     */

    /**
     * 現在の値を返却します。
     * 
     * @return 現在の値
     */
    public double get() {
        return this.value;
    }

    /**
     * 新たな値を格納します。
     * 
     * @param value
     *            格納する値
     */
    public void set(double value) {
        this.value = value;
    }

    /**
     * 格納される変数に対して引数を加算します。
     * 
     * @param val
     *            加算する値
     * @return 加算後の値
     */
    public double add(double val) {
        this.value += val;
        return this.value;
    }

    /**
     * 格納される変数に対して引数を減算します。
     * 
     * @param val
     *            減算する値
     * @return 減算後の値
     */
    public double subtract(double val) {
        this.value -= val;
        return this.value;
    }

    /**
     * 格納される変数に対して引数を乗算します。
     * 
     * @param val
     *            乗算する値
     * @return 乗算後の値
     */
    public double multiply(double val) {
        this.value *= val;
        return this.value;
    }

    /**
     * 格納される変数に対して引数を除算します。
     * 
     * @param val
     *            除算する値
     * @return 除算後の値
     */
    public double divide(double val) {
        this.value /= val;
        return this.value;
    }

    /**
     * 格納される変数に対して値 1 を増分します。
     * 
     * @return 増分後の値
     */
    public double increment() {
        this.value++;
        return this.value;
    }

    /**
     * 格納される変数に対して値 1 を減分します。
     * 
     * @return 減分後の値
     */
    public double decrement() {
        this.value--;
        return this.value;
    }

    /*
     * value
     */

    /**
     * 現在の値を返却します。
     * 
     * @return 現在の値
     */
    public Number getValue() {
        return Boxing.box(this.value);
    }

    /**
     * 指定の値の Number.doubleValue() を格納します。
     * 
     * @param value
     *            格納する値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public void setValue(Number value) {
        set(value.doubleValue());
    }

    /**
     * 指定の値の VariableNumber.doubleValue() を格納します。
     * 
     * @param value
     *            格納する値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public void setValue(VariableNumber value) {
        set(value.doubleValue());
    }

    /*
     * Number impl
     */

    /**
     * byte で返却します。
     * 
     * @return 格納する値の byte 表現
     */
    public byte byteValue() {
        return (byte) value;
    }

    /**
     * short で返却します。
     * 
     * @return 格納する値の short 表現
     */
    public short shortValue() {
        return (short) value;
    }

    /**
     * int で返却します。
     * 
     * @return 格納する値の int 表現
     */
    public int intValue() {
        return (int) value;
    }

    /**
     * long で返却します。
     * 
     * @return 格納する値の long 表現
     */
    public long longValue() {
        return (long) value;
    }

    /**
     * float で返却します。
     * 
     * @return 格納する値の float 表現
     */
    public float floatValue() {
        return (float) value;
    }

    /**
     * double で返却します。
     * 
     * @return 格納する値の double 表現
     */
    public double doubleValue() {
        return value;
    }

    /*
     * Comparable Impl
     */

    /**
     * 比較順序を返却します。
     * 
     * @param obj
     *            比較するオブジェクト
     * @return 引数の格納値が大きい場合負数、小さい場合整数、同値の場合零を返却します
     * @throws ClassCastException
     *             引数が VariableDouble 以外の場合
     */
    public int compareTo(Object obj) {
        VariableDouble anotherDouble = (VariableDouble) obj;
        double thisVal = this.value;
        double anotherVal = anotherDouble.value;
        return Double.compare(thisVal, anotherVal);
    }

    /*
     * Object
     */

    /**
     * 同値性を検証します、 VariableDouble 型であり格納値が同値の場合は true と見なします。
     * 
     * @param obj
     *            比較先
     * @return VariableDouble 型であり格納値が同値の場合は true
     */
    public boolean equals(Object obj) {
        if (obj instanceof VariableDouble) {
            return (Double.doubleToLongBits(((VariableDouble) obj).doubleValue()) == Double.doubleToLongBits(value));
        }
        return false;
    }

    /**
     * 現在の値をハッシュ値として返却します。
     * 
     * @return ハッシュ値
     */
    public int hashCode() {
        long bits = Double.doubleToLongBits(value);
        return (int) (bits ^ (bits >>> 32));
    }

    /**
     * 変数値の文字列表現（String.valueOf(VariableDouble.getValue())）を返却します。
     * 
     * @return 変数値の文字列表現
     */
    public String toString() {
        return String.valueOf(value);
    }

}