/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.lang;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 文字列を基本的なデータ型とその集合に変換する機能を提供します。<br>
 * 機能を最小限にまとめた、基本的なデータ構造を記述する簡易言語です。<br>
 * <br>
 * またデータ型を文字列に変換する機能も提供します。これによりデータの永続化も可能です。<br>
 * <br>
 * 構文例：以下はリストの要素に、数値、リスト、マップ、リストを入れ子にするセット、が定義されています。<br>
 * <code> [ 98741123L, [12G, 1234.67800G], {1234='火', 1234.56='炎'}, (`文`,`字`, ['春','夏','秋','冬']) ]</code><br>
 * <br>
 * 実行例：実行は以下のように簡単です。<br>
 * <code> Map map = (Map) ValueOf.decode("{1='火', 2='炎', 3='焔'}"); </code><br>
 * <br>
 * 以下に文法の概要を一覧します。<br>
 * 文法の詳細に関しては、対応する各データ型の関数を参照してください。<br>
 * <table border="0" cellpadding="6" cellspacing="1">
 * <tr>
 * <td colspan="4">文法概要： </td>
 * </tr>
 * <tr>
 * <td>1.</td>
 * <td> 数値</td>
 * <td>[-][0|0x|0b][0-9A-F.+-e][u|U|s|S|i|I|l|L|f|F|d|D|h|H|g|G]</td>
 * <td>例： 123U, -0x10FFL, 1-E4, 82385920.3242G</td>
 * </tr>
 * <tr>
 * <td>2.</td>
 * <td> 文字列</td>
 * <td>'string'</td>
 * <td>例： '春夏秋冬', '\u005Cu6625の桜'</td>
 * </tr>
 * <tr>
 * <td>3.</td>
 * <td> 文字</td>
 * <td>`char`</td>
 * <td>例： 'あ', '\u005Cu6625'</td>
 * </tr>
 * <tr>
 * <td>4.</td>
 * <td> 論理</td>
 * <td>true|false</td>
 * <td>例： true, false</td>
 * </tr>
 * <tr>
 * <td>5.</td>
 * <td> Null</td>
 * <td>null</td>
 * <td>例： null</td>
 * </tr>
 * <tr>
 * <td>6.</td>
 * <td> 日付</td>
 * <td>/yyyy-mm-dd[T| ]hh:mi:ss.zzz[+|-]00:00[l|L|d|D|g|G]/</td>
 * <td>例： /2007-05-23T09:51:29.345+09:00/, /2007-05-23 09:51:29/, /2007-05-23/</td>
 * </tr>
 * <tr>
 * <td>7.</td>
 * <td> リスト</td>
 * <td>[ element, ... ]</td>
 * <td>例： [ 12345.678D, /2007-05-23/, '春夏秋冬', `夏` ]</td>
 * </tr>
 * <tr>
 * <td>8.</td>
 * <td> セット</td>
 * <td>( element, ... )</td>
 * <td>例： ( 12345.678D, /2007-05-23/, '春夏秋冬', `夏` )</td>
 * </tr>
 * <tr>
 * <td>9.</td>
 * <td> マップ</td>
 * <td>{ kay=value, ... }</td>
 * <td>例： { 0=12345.678D, 1=/2007-05-23/, 2='春夏秋冬', 3=`夏` }</td>
 * </tr>
 * </table> <br>
 * <br>
 * 構文にマクロ関数と変数を追加した機能もオプションとして提供します。これにより複雑なデータ生成への対応も可能です。<br>
 * <br>
 * 構文例：以下はリストの要素に、マクロ関数 'max' とその引数に変数 'num' を渡し、次の要素に変数 'foo' を指定しています。<br>
 * <code> [ $max(0, #num#), #foo# ]</code><br>
 * <br>
 * 実行例：実行は以下のように簡単です（説明のためのソースです）。<br>
 * <code> Map context = { num=1, foo="火" }<br>
 * Map macros = { max= new Closure() {<br>
 * &nbsp;&nbsp;public Object evaluate(Object p) {<br>
 * &nbsp;&nbsp;&nbsp;&nbsp;final List args = (List) p;<br>
 * &nbsp;&nbsp;&nbsp;&nbsp;return Math.max(args.get(0).intValue(), args.get(1).intValue());<br>
 * &nbsp;&nbsp;}<br>
 * }<br>
 * List list = (List) ValueOf.macro("[ $max(0, #num#), #foo# ]", macros, context); <br>
 * out: list ==> [1, "火"] <br>
 * </code><br>
 * <br>
 * 以下にマクロ関数と変数の文法概要を一覧します。 <table border="0" cellpadding="6" cellspacing="1">
 * <tr>
 * <td colspan="4">文法概要： </td>
 * </tr>
 * <tr>
 * <td>1.</td>
 * <td> マクロ関数</td>
 * <td>$[_:a-zA-Z0-9]+(element, ...)</td>
 * <td>例： $indexOf(#str#, `a`), $String:concat('春', '冬')</td>
 * </tr>
 * <tr>
 * <td>2.</td>
 * <td> 変数</td>
 * <td>#[^#]+#</td>
 * <td>例： #name#, #a:0.Z#</td>
 * </tr>
 * </table> <br>
 */
public final class ValueOf {

    /**
     * 字句解析位置を格納する。
     */
    static class Position {
        final int len;

        int index;

        Position(int index, int len) {
            this.index = index;
            this.len = len;
        }
    }

    /* Null。 */
    private static final String NULL_SYMBOL = "null";

    /* 負数記号。 */
    private static final String NEGATIVE_SIGN = "-";

    /* 日付を解析、要素の抽出を行う正規表現。 */
    private static final Pattern dateTimePattern;

    /* 八進数数値のデータテーブル。 */
    private static final char[] hexDigit = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F' };

    static {
        // 2000-01-01[T| ]23:01:01.000 +00:00
        final StringBuffer p = new StringBuffer();
        p.append("^");
        p.append("(?:(\\d{4})-(\\d{2})-(\\d{2}))?");
        p.append("(T|\\u0020+)?");
        p.append("(?:(?:(\\d{2}):(\\d{2}):(\\d{2}))(?:(?:\\.(\\d{3}))?)?(?:(?:\\u0020*)(?:([-+])(\\d{2}):(\\d{2})))?)?(l|L|d|D|g|G)?");
        p.append("$");
        dateTimePattern = Pattern.compile(p.toString());
    }

    /*
     * decode
     */

    /**
     * データを生成して返却します。<br>
     * 文字列が定義するデータ型は構文から自動で解析します。
     * 
     * @param src
     *            生成元の文字列
     * @return 生成されたデータ
     * @throws IllegalArgumentException
     *             構文が不正の場合
     */
    public static Object decode(String src) {
        final Position pos = new Position(0, src.length());
        next(src, pos);
        final Object reValue = decodeVlaue(src, pos);
        next(src, pos);
        if ((pos.index + 1) < pos.len) {
            throw newIllegalArgumentEx("pattern err.", src, pos.index);
        }
        return reValue;
    }

    /**
     * 文字列を BigDecimal に変換します。<br>
     * 接尾辞（[g|G]）の表記法をサポートします。
     * 
     * @param src
     *            生成元の文字列
     * @return 数値オブジェクト
     */
    public static BigDecimal decodeBigDecimal(String src) {
        final int suffix = (src.endsWith("g") || src.endsWith("G")) ? src.length() - 1 : src.length();
        return new BigDecimal(src.substring(0, suffix));
    }

    /**
     * 拡張規則で文字列を BigInteger に変換します。<br>
     * 10進数，16進数（[0x|0X]n），又は8進数（[0]n），2進数（[0b|0B]n）, 接尾辞（[h|H]）の表記法をサポートします。
     * 
     * @param src
     *            生成元の文字列
     * @return 数値オブジェクト
     */
    public static BigInteger decodeBigInteger(String src) {
        int index = 0;
        boolean negative = false;
        if (src.startsWith(NEGATIVE_SIGN)) {
            negative = true;
            index++;
        }
        int radix = 10;
        if (src.startsWith("0x", index) || src.startsWith("0X", index)) {
            index += 2;
            radix = 16;
        } else if (src.startsWith("0b", index) || src.startsWith("0B", index)) {
            index += 2;
            radix = 2;
        } else if (src.startsWith("0", index) && src.length() > 1 + index) {
            index++;
            radix = 8;
        }
        final int suffix = (src.endsWith("h") || src.endsWith("H")) ? src.length() - 1 : src.length();
        final String _num = (negative) ? NEGATIVE_SIGN.concat(src.substring(index, suffix)) : src.substring(index, suffix);
        return new BigInteger(_num, radix);
    }

    /**
     * 拡張規則で文字列を Boolean に変換します。<br>
     * 小文字の true と false の表記法のみを有効とします。
     * 
     * @param src
     *            生成元の文字列
     * @return Boolean オブジェクト
     */
    public static Boolean decodeBoolean(String src) {
        if (src == null) {
            throw newIllegalArgumentEx("is not Boolean.", src, 0);
        }
        if (src.equals("false")) {
            return Boolean.FALSE;
        }
        if (src.equals("true")) {
            return Boolean.TRUE;
        }
        throw newIllegalArgumentEx("is not Boolean.", src, 0);
    }

    /**
     * 拡張規則で文字列を Byte に変換します。<br>
     * 10進数，16進数（[0x|0X]n），又は8進数（[0]n），2進数（[0b|0B]n）, 接尾辞（[u|U]）の表記法をサポートします。
     * 
     * @param src
     *            生成元の文字列
     * @return 数値オブジェクト
     */
    public static Byte decodeByte(String src) {
        int index = 0;
        boolean negative = false;
        if (src.startsWith(NEGATIVE_SIGN)) {
            negative = true;
            index++;
        }
        int radix = 10;
        if (src.startsWith("0x", index) || src.startsWith("0X", index)) {
            index += 2;
            radix = 16;
        } else if (src.startsWith("0b", index) || src.startsWith("0B", index)) {
            index += 2;
            radix = 2;
        } else if (src.startsWith("0", index) && src.length() > 1 + index) {
            index++;
            radix = 8;
        }
        final int suffix = (src.endsWith("u") || src.endsWith("U")) ? src.length() - 1 : src.length();
        final String _num = (negative) ? NEGATIVE_SIGN.concat(src.substring(index, suffix)) : src.substring(index, suffix);
        return Byte.valueOf(_num, radix);
    }

    /**
     * Unicode escapes の文字表現を文字型へ変換して返却します。
     * 
     * @param src
     *            Unicode escapes 形式の文字列
     * @return デコードされた文字
     * @throws IllegalArgumentException
     *             構文が不正の場合
     */
    public static Character decodeCharacter(String src) {
        if (src == null) {
            throw newIllegalArgumentEx("Malformed character encoding.", src, 0);
        }
        final int srcLength = src.length();
        if (srcLength == 0) {
            throw newIllegalArgumentEx("Malformed character encoding.", src, 0);
        }
        char c;
        c = src.charAt(0);
        if (c == '\\') {
            if (srcLength < 2) {
                throw newIllegalArgumentEx("Malformed character encoding.", src, 0);
            }
            c = src.charAt(1);
            if (c == 'u') {
                if (srcLength != 6) {
                    throw newIllegalArgumentEx("Malformed \\uxxxx encoding.", src, 1);
                }
                return Boxing.box(decodeHexCharacter(src, 2));
            } else {
                if (srcLength != 2) {
                    throw newIllegalArgumentEx("Malformed \\x encoding.", src, 1);
                }
                return Boxing.box(decodeCntrlCharacter(src, c, 1));
            }
        } else {
            if (srcLength != 1) {
                throw newIllegalArgumentEx("Malformed 'x' encoding.", src, 1);
            }
            return Boxing.box(c);
        }
    }

    /**
     * 文字列を日付オブジェクトに変換して返却します。<br>
     * 有効な書式は 'yyyy-mm-dd[T| ]hh:mi:ss.zzz[+|-]hh:mi[l|L|g|G|d|D]' です。<br>
     * 日付と時刻の両方または片方の指定が可能です。またミリ秒とタイムゾーン、型接尾辞は省略可能です。<br>
     * 型接尾辞 'l' or 'L' を指定すると日付の結果は、グリニッジ標準日時（1970-01-01）からのミリ秒を示す Long 型で返却されます。<br>
     * 型接尾辞 'g' or 'G' を指定すると結果は、グレゴリオカレンダー型で返却されます。<br>
     * 型接尾辞 'd' or 'D' を指定すると結果は、日付オブジェクトで返却されます。<br>
     * 省略時は、日付オブジェクトで返却されます。<br>
     * <br>
     * 有効な書式例：<br>
     * "2007-05-23T09:51:29.345+09:00"<br>
     * "2007-05-23 09:51:29.345-09:00L"<br>
     * "2007-05-23 09:51:29"<br>
     * "2007-05-23 +05:30"<br>
     * "2007-05-23"<br>
     * "09:51:29.345+09:00"<br>
     * "09:51:29 +09:00L"<br>
     * 
     * @param src
     *            生成元の文字列
     * @return 日付オブジェクト
     * @throws IllegalArgumentException
     *             構文が不正の場合
     */
    public static Object decodeDateTime(String src) {
        if (src.length() == 0) {
            throw newIllegalArgumentEx("pattern err.", src, 0);
        }
        // 書式が有効ならば、構文用の書式に束ねる
        final Matcher match = dateTimePattern.matcher(src);
        if (match.find()) {
            final String year = match.group(1);
            final String month = match.group(2);
            final String day = match.group(3);
            final String deli = match.group(4);
            final String hour = match.group(5);
            final String minute = match.group(6);
            final String second = match.group(7);
            final String nano = match.group(8);
            final String zoneSign = match.group(9);
            final String zoneHour = match.group(10);
            final String zoneMinute = match.group(11);
            final String type = match.group(12);
            // 日付と時刻が指定された場合は区切り必須
            if ((year != null && hour != null) && deli == null) {
                throw newIllegalArgumentEx("pattern err.", src, 0);
            }
            // 区切りの有る場合は日付と時刻は必須
            if (deli != null && (year == null || hour == null)) {
                throw newIllegalArgumentEx("pattern err.", src, 0);
            }

            final int y;
            final int m;
            final int d;
            if (year != null) {
                y = Integer.parseInt(year);
                m = Integer.parseInt(month) - 1;
                d = Integer.parseInt(day);
            } else {
                y = 1970;
                m = 0;
                d = 1;
            }
            final int h;
            final int i;
            final int s;
            final int n;
            if (hour != null) {
                h = Integer.parseInt(hour);
                i = Integer.parseInt(minute);
                s = Integer.parseInt(second);
                n = (nano != null) ? Integer.parseInt(nano) : 0;
            } else {
                h = 0;
                i = 0;
                s = 0;
                n = 0;
            }

            final GregorianCalendar cal = new GregorianCalendar();
            cal.set(Calendar.YEAR, y);
            cal.set(Calendar.MONTH, m);
            cal.set(Calendar.DATE, d);
            cal.set(Calendar.HOUR_OF_DAY, h);
            cal.set(Calendar.MINUTE, i);
            cal.set(Calendar.SECOND, s);
            cal.set(Calendar.MILLISECOND, n);
            if (zoneSign != null) {
                final int zh = Integer.parseInt(zoneHour);
                final int zm = Integer.parseInt(zoneMinute);
                final int zone = (zh * 60 * 60 * 1000) + (zm * 60 * 1000);
                final int z = (zoneSign.equals("+")) ? zone : -zone;
                cal.set(Calendar.ZONE_OFFSET, z);
            }
            if (type != null) {
                final int _type = toUpper(type.charAt(0));
                switch (_type) {
                case 0x44: // D
                    return cal.getTime();
                case 0x47: // G
                    return cal;
                case 0x4C: // L
                    return Boxing.box(cal.getTimeInMillis());
                default:
                    break;
                }
            }
            // type is null
            return cal.getTime();
        }
        throw newIllegalArgumentEx("pattern err.", src, 0);
    }

    /**
     * 文字列を Double に変換します。<br>
     * 接尾辞（[d|D]）の表記法をサポートします。
     * 
     * @param src
     *            生成元の文字列
     * @return 数値オブジェクト
     */
    public static Double decodeDouble(String src) {
        final int suffix = (src.endsWith("d") || src.endsWith("D")) ? src.length() - 1 : src.length();
        return Double.valueOf(src.substring(0, suffix));
    }

    /**
     * 文字列を Float に変換します。<br>
     * 接尾辞（[f|F]）の表記法をサポートします。
     * 
     * @param src
     *            生成元の文字列
     * @return 数値オブジェクト
     */
    public static Float decodeFloat(String src) {
        final int suffix = (src.endsWith("f") || src.endsWith("F")) ? src.length() - 1 : src.length();
        return Float.valueOf(src.substring(0, suffix));
    }

    /**
     * 拡張規則で文字列を Integer に変換します。<br>
     * 10進数，16進数（[0x|0X]n），又は8進数（[0]n），2進数（[0b|0B]n）, 接尾辞（[i|I]）の表記法をサポートします。
     * 
     * @param src
     *            生成元の文字列
     * @return 数値オブジェクト
     */
    public static Integer decodeInteger(String src) {
        int index = 0;
        boolean negative = false;
        if (src.startsWith(NEGATIVE_SIGN)) {
            negative = true;
            index++;
        }
        int radix = 10;
        if (src.startsWith("0x", index) || src.startsWith("0X", index)) {
            index += 2;
            radix = 16;
        } else if (src.startsWith("0b", index) || src.startsWith("0B", index)) {
            index += 2;
            radix = 2;
        } else if (src.startsWith("0", index) && src.length() > 1 + index) {
            index++;
            radix = 8;
        }
        final int suffix = (src.endsWith("i") || src.endsWith("I")) ? src.length() - 1 : src.length();
        final String _num = (negative) ? NEGATIVE_SIGN.concat(src.substring(index, suffix)) : src.substring(index, suffix);
        return Integer.valueOf(_num, radix);
    }

    /**
     * リストを生成して返却します。<br>
     * 文字列は [ element, element, ... ] の構文として解析します。<br>
     * 返却される List は変更不可のリストに装飾されます。<br>
     * またリストを入れ子に定義することも可能です。
     * 
     * @param src
     *            生成元の文字列
     * @return リスト
     * @throws IllegalArgumentException
     *             構文が不正の場合
     */
    public static List decodeList(String src) {
        return decodeList(src, new Position(0, src.length()));
    }

    /**
     * 拡張規則で文字列を Long に変換します。<br>
     * 10進数，16進数（[0x|0X]n），又は8進数（[0]n），2進数（[0b|0B]n）, 接尾辞（[l|L]）の表記法をサポートします。
     * 
     * @param src
     *            生成元の文字列
     * @return 数値オブジェクト
     */
    public static Long decodeLong(String src) {
        int index = 0;
        boolean negative = false;
        if (src.startsWith(NEGATIVE_SIGN)) {
            negative = true;
            index++;
        }
        int radix = 10;
        if (src.startsWith("0x", index) || src.startsWith("0X", index)) {
            index += 2;
            radix = 16;
        } else if (src.startsWith("0b", index) || src.startsWith("0B", index)) {
            index += 2;
            radix = 2;
        } else if (src.startsWith("0", index) && src.length() > 1 + index) {
            index++;
            radix = 8;
        }
        final int suffix = (src.endsWith("l") || src.endsWith("L")) ? src.length() - 1 : src.length();
        final String _num = (negative) ? NEGATIVE_SIGN.concat(src.substring(index, suffix)) : src.substring(index, suffix);
        return Long.valueOf(_num, radix);
    }

    /**
     * マップを生成して返却します。<br>
     * 文字列は { kay=value, kay=value, ... } の構文として解析します。<br>
     * 返却される Map は格納順序を保存しており、変更不可のマップに装飾されます。<br>
     * またマップを入れ子に定義することも可能です。
     * 
     * @param src
     *            生成元の文字列
     * @return マップ
     * @throws IllegalArgumentException
     *             構文が不正の場合
     */
    public static Map decodeMap(String src) {
        return decodeMap(src, new Position(0, src.length()));
    }

    /**
     * Java と同等の規則で文字列を Null に変換します。<br>
     * 小文字の null の表記法のみを有効とします。
     * 
     * @param src
     *            生成元の文字列
     * @return null
     */
    public static Object decodeNull(String src) {
        if (src == null) {
            throw newIllegalArgumentEx("is not Null.", src, 0);
        }
        if (src.equals(NULL_SYMBOL)) {
            return null;
        }
        throw newIllegalArgumentEx("is not Null.", src, 0);
    }

    /**
     * 拡張規則で文字列を数値オブジェクトに変換します。<br>
     * 整数に対しては、10進数，16進数（[0x|0X]n），又は8進数（[0]n），2進数（[0b|0B]n）の表記法をサポートします。<br>
     * また, 型指定子として接尾辞（[u|U|s|S|i|I|l|L|f|F|d|D|h|H|g|G]）の表記法をサポートします。<br>
     * <br>
     * 型指定子の表記規則：<br>
     * &nbsp;&nbsp;1.&nbsp;u or U = Byte<br>
     * &nbsp;&nbsp;2.&nbsp;s or S = Short<br>
     * &nbsp;&nbsp;3.&nbsp;i or I = Integer<br>
     * &nbsp;&nbsp;4.&nbsp;l or L = Long<br>
     * &nbsp;&nbsp;5.&nbsp;f or F = Float<br>
     * &nbsp;&nbsp;6.&nbsp;d or D = Double<br>
     * &nbsp;&nbsp;7.&nbsp;h or H = BigInteger<br>
     * &nbsp;&nbsp;8.&nbsp;g or G = BigDecimal<br>
     * &nbsp;&nbsp;9.&nbsp;接尾辞が省略された整数型 = Integer<br>
     * &nbsp;10.&nbsp;接尾辞が省略された浮動小数点型 = Double<br>
     * 
     * @param src
     *            生成元の文字列
     * @return 数値オブジェクト
     */
    public static Number decodeNumber(String src) {
        final int srcLength = src.length();
        if (srcLength == 1) {
            return Integer.valueOf(src, 10); // 一文字、十進数整数
        }
        // 符号と接尾辞を検出
        final int negative = (src.startsWith(NEGATIVE_SIGN)) ? 1 : 0;
        final int suffix = toUpper(src.charAt(srcLength - 1));
        // 浮動小数点表記か検証
        if (-1 < src.indexOf('.', negative)) {
            // 浮動小数点表記
            switch (suffix) {
            case 0x44: // D
                return Double.valueOf(numstr(negative, src, 0, true));
            case 0x46: // F
                return Float.valueOf(numstr(negative, src, 0, true));
            case 0x47: // G
                return new BigDecimal(numstr(negative, src, 0, true));
            default:
                return Double.valueOf(numstr(negative, src, 0, false));
            }
        }
        // 整数表記の二進数、八進数または十六進数か検証
        if (src.startsWith("0", negative)) {
            final int radixNm = toUpper(src.charAt(negative + 1));
            // 十六進数
            if ('X' == radixNm) {
                switch (suffix) {
                case 0x48: // H
                    return new BigInteger(numstr(negative, src, 16, true), 16);
                case 0x49: // I
                    return Integer.valueOf(numstr(negative, src, 16, true), 16);
                case 0x4C: // L
                    return Long.valueOf(numstr(negative, src, 16, true), 16);
                case 0x53: // S
                    return Short.valueOf(numstr(negative, src, 16, true), 16);
                case 0x55: // U
                    return Byte.valueOf(numstr(negative, src, 16, true), 16);
                default:
                    return Integer.valueOf(numstr(negative, src, 16, false), 16);
                }
            }
            // 二進数
            if ('B' == radixNm) {
                switch (suffix) {
                case 0x48: // H
                    return new BigInteger(numstr(negative, src, 2, true), 2);
                case 0x49: // I
                    return Integer.valueOf(numstr(negative, src, 2, true), 2);
                case 0x4C: // L
                    return Long.valueOf(numstr(negative, src, 2, true), 2);
                case 0x53: // S
                    return Short.valueOf(numstr(negative, src, 2, true), 2);
                case 0x55: // U
                    return Byte.valueOf(numstr(negative, src, 2, true), 2);
                default:
                    return Integer.valueOf(numstr(negative, src, 2, false), 2);
                }
            }
            // '0' + 接尾辞
            if (src.length() == 2 + negative) {
                switch (suffix) {
                case 0x44: // D
                    return new Double((negative == 0) ? 0.0 : -0.0);
                case 0x46: // F
                    return new Float((negative == 0) ? 0.0f : -0.0f);
                case 0x47: // G
                    return new BigDecimal(BigInteger.ZERO, 0);
                case 0x48: // H
                    return BigInteger.ZERO;
                case 0x49: // I
                    return new Integer(0);
                case 0x4C: // L
                    return new Long(0L);
                case 0x53: // S
                    return new Short((short) 0);
                case 0x55: // U
                    return new Byte((byte) 0);
                default:
                    break;
                }
            }
            // 八進数
            switch (suffix) {
            case 0x48: // H
                return new BigInteger(numstr(negative, src, 8, true), 8);
            case 0x49: // I
                return Integer.valueOf(numstr(negative, src, 8, true), 8);
            case 0x4C: // L
                return Long.valueOf(numstr(negative, src, 8, true), 8);
            case 0x53: // S
                return Short.valueOf(numstr(negative, src, 8, true), 8);
            case 0x55: // U
                return Byte.valueOf(numstr(negative, src, 8, true), 8);
            default:
                return Integer.valueOf(numstr(negative, src, 8, false), 8);
            }
        }

        // 整数表記の十進数、科学表記、無限大、非数
        switch (suffix) {
        case 0x44: // D
            return Double.valueOf(numstr(negative, src, 0, true));
        case 0x46: // F
            return Float.valueOf(numstr(negative, src, 0, true));
        case 0x47: // G
            return new BigDecimal(numstr(negative, src, 0, true));
        case 0x48: // H
            return new BigInteger(numstr(negative, src, 10, true), 10);
        case 0x49: // I
            return Integer.valueOf(numstr(negative, src, 10, true), 10);
        case 0x4C: // L
            return Long.valueOf(numstr(negative, src, 10, true), 10);
        case 0x53: // S
            return Short.valueOf(numstr(negative, src, 10, true), 10);
        case 0x55: // U
            return Byte.valueOf(numstr(negative, src, 10, true), 10);
        default:
            if (src.startsWith("I", negative)) {// 無限大
                return Double.valueOf(src);
            }
            if (src.startsWith("N", negative)) {// 非数
                return Double.valueOf(src);
            }
            return Integer.valueOf(numstr(negative, src, 10, false), 10);
        }
    }

    /**
     * セットを生成して返却します。<br>
     * 文字列は ( element, element, ... ) の構文として解析します。<br>
     * 返却される Set は格納順序を保存しており、変更不可のセットに装飾されます。<br>
     * またセットを入れ子に定義することも可能です。
     * 
     * @param src
     *            生成元の文字列
     * @return セット
     * @throws IllegalArgumentException
     *             構文が不正の場合
     */
    public static Set decodeSet(String src) {
        return decodeSet(src, new Position(0, src.length()));
    }

    /**
     * 拡張規則で文字列を Short に変換します。<br>
     * 10進数，16進数（[0x|0X]n），又は8進数（[0]n），2進数（[0b|0B]n）, 接尾辞（[s|S]）の表記法をサポートします。
     * 
     * @param src
     *            生成元の文字列
     * @return 数値オブジェクト
     */
    public static Short decodeShort(String src) {
        int index = 0;
        boolean negative = false;
        if (src.startsWith(NEGATIVE_SIGN)) {
            negative = true;
            index++;
        }
        int radix = 10;
        if (src.startsWith("0x", index) || src.startsWith("0X", index)) {
            index += 2;
            radix = 16;
        } else if (src.startsWith("0b", index) || src.startsWith("0B", index)) {
            index += 2;
            radix = 2;
        } else if (src.startsWith("0", index) && src.length() > 1 + index) {
            index++;
            radix = 8;
        }
        final int suffix = (src.endsWith("s") || src.endsWith("S")) ? src.length() - 1 : src.length();
        final String _num = (negative) ? NEGATIVE_SIGN.concat(src.substring(index, suffix)) : src.substring(index, suffix);
        return Short.valueOf(_num, radix);
    }

    /**
     * Unicode escapes の文字表現を文字列型へ変換して返却します。
     * 
     * @param src
     *            Unicode escapes 形式の文字列
     * @return デコードされた文字列
     * @throws IllegalArgumentException
     *             構文が不正の場合
     */
    public static String decodeString(String src) {
        if (src == null) {
            throw newIllegalArgumentEx("Malformed character encoding.", src, 0);
        }
        final int srcLength = src.length();
        final StringBuffer sb = new StringBuffer(srcLength);
        for (int i = 0; i < srcLength; i++) {
            final char c = src.charAt(i);
            if (c == '\\') {
                if (!(i + 1 < srcLength)) {
                    throw newIllegalArgumentEx("Malformed character encoding.", src, i);
                }
                final char next = src.charAt(i + 1);
                if (next == 'u') {
                    if (!(i + 5 < srcLength)) {
                        throw newIllegalArgumentEx("Malformed \\uxxxx encoding.", src, i);
                    }
                    sb.append(decodeHexCharacter(src, i + 2));
                    i += 5;
                } else {
                    if (!(i + 1 < srcLength)) {
                        throw newIllegalArgumentEx("Malformed \\x encoding.", src, i);
                    }
                    sb.append(decodeCntrlCharacter(src, next, i));
                    i += 1;
                }
            } else {
                sb.append(c);
            }
        }
        return sb.toString();
    }

    /*
     * encode
     */

    /**
     * オブジェクトから文字列を生成して返却します。
     * 
     * @param src
     *            生成元のオブジェクト
     * @return 生成された文字列
     * @throws IllegalArgumentException
     *             データが不正の場合
     */
    public static String encode(Object src) {
        final StringBuffer sb = new StringBuffer();
        encodeVlaue(sb, src);
        return sb.toString();
    }

    /**
     * 拡張規則で BigDecimal を文字列に変換します。<br>
     * 数値文字列に接尾辞 'G' を付加します。
     * 
     * @param src
     *            数値オブジェクト
     * @return 文字列表現
     */
    public static String encodeBigDecimal(BigDecimal src) {
        return encodeNumber(src, 'G');
    }

    /**
     * 拡張規則で BigInteger を文字列に変換します。<br>
     * 十進表記の数値文字列に接尾辞 'H' を付加します。
     * 
     * @param src
     *            数値オブジェクト
     * @return 文字列表現
     */
    public static String encodeBigInteger(BigInteger src) {
        return encodeNumber(src, 'H');
    }

    /**
     * Boolean を文字列に変換します。
     * 
     * @param src
     *            Boolean オブジェクト
     * @return 文字列表現
     */
    public static String encodeBoolean(Boolean src) {
        return src.toString();
    }

    /**
     * 拡張規則で Byte を文字列に変換します。<br>
     * 十進表記の数値文字列に接尾辞 'U' を付加します。
     * 
     * @param src
     *            数値オブジェクト
     * @return 文字列表現
     */
    public static String encodeByte(Byte src) {
        return encodeNumber(src, 'U');
    }

    /**
     * 文字を Unicode escapes の文字列表現へ変換して返却します。
     * 
     * @param src
     *            変換元の文字
     * @return エンコードされた文字列
     */
    public static String encodeCharacter(Character src) {
        final StringBuffer sb = new StringBuffer(7);
        final char c = src.charValue();
        addUnicodeEscapeChar(sb, c);
        return sb.toString();
    }

    /**
     * カレンダーの情報を日付時刻文字列に変換します。
     * 
     * @param cal
     * @return 日付時刻文字列
     */
    public static String encodeDateTime(Calendar cal) {
        return encodeDateTime(cal, 'G');
    }

    /**
     * 日付オブジェクトの情報を日付時刻文字列に変換します。
     * 
     * @param date
     * @return 日付時刻文字列
     */
    public static String encodeDateTime(Date date) {
        final Calendar cal = new GregorianCalendar();
        cal.clear();
        cal.setTimeInMillis(date.getTime());
        return encodeDateTime(cal, 'D');
    }

    /**
     * 1970-01-01 からのミリ秒の情報を日付時刻文字列に変換します。
     * 
     * @param millis
     *            1970-01-01 からのミリ秒
     * @return 日付時刻文字列
     */
    public static String encodeDateTime(Long millis) {
        final Calendar cal = new GregorianCalendar();
        cal.clear();
        cal.setTimeInMillis(millis.longValue());
        return encodeDateTime(cal, 'L');
    }

    /**
     * 拡張規則で Double を文字列に変換します。<br>
     * 数値文字列に接尾辞 'D' を付加します。
     * 
     * @param src
     *            数値オブジェクト
     * @return 文字列表現
     */
    public static String encodeDouble(Double src) {
        return encodeNumber(src, 'D');
    }

    /**
     * 拡張規則で Float を文字列に変換します。<br>
     * 数値文字列に接尾辞 'F' を付加します。
     * 
     * @param src
     *            数値オブジェクト
     * @return 文字列表現
     */
    public static String encodeFloat(Float src) {
        return encodeNumber(src, 'F');
    }

    /**
     * 拡張規則で Integer を文字列に変換します。<br>
     * 十進表記の数値文字列に接尾辞 'i' を付加します。
     * 
     * @param src
     *            数値オブジェクト
     * @return 文字列表現
     */
    public static String encodeInteger(Integer src) {
        return encodeNumber(src, 'i');
    }

    /**
     * リストを文字列表現へ変換して返却します。
     * 
     * @param src
     *            リスト
     * @return リストの文字列表現
     */
    public static String encodeList(List src) {
        final StringBuffer sb = new StringBuffer();
        encodeList(sb, src);
        return sb.toString();
    }

    /**
     * 拡張規則で Long を文字列に変換します。<br>
     * 十進表記の数値文字列に接尾辞 'L' を付加します。
     * 
     * @param src
     *            数値オブジェクト
     * @return 文字列表現
     */
    public static String encodeLong(Long src) {
        return encodeNumber(src, 'L');
    }

    /**
     * マップを文字列表現へ変換して返却します。
     * 
     * @param src
     *            マップ
     * @return マップの文字列表現
     */
    public static String encodeMap(Map src) {
        final StringBuffer sb = new StringBuffer();
        encodeMap(sb, src);
        return sb.toString();
    }

    /**
     * Null を示す文字列に変換します。
     * 
     * @return 文字列表現
     */
    public static String encodeNull() {
        return NULL_SYMBOL;
    }

    /**
     * 拡張規則で Number を文字列に変換します。<br>
     * 数値文字列に接尾辞を付加します。
     * 
     * @param src
     *            数値オブジェクト
     * @return 文字列表現
     */
    public static String encodeNumber(Number src) {
        if (src instanceof Byte) {
            return encodeByte((Byte) src);
        }
        if (src instanceof Short) {
            return encodeShort((Short) src);
        }
        if (src instanceof Integer) {
            return encodeInteger((Integer) src);
        }
        if (src instanceof Long) {
            return encodeLong((Long) src);
        }
        if (src instanceof Float) {
            return encodeFloat((Float) src);
        }
        if (src instanceof Double) {
            return encodeDouble((Double) src);
        }
        if (src instanceof BigInteger) {
            return encodeBigInteger((BigInteger) src);
        }
        if (src instanceof BigDecimal) {
            return encodeBigDecimal((BigDecimal) src);
        }
        throw newIllegalArgumentEx("Unsupported type.", src, 0);
    }

    /**
     * セットを文字列表現へ変換して返却します。
     * 
     * @param src
     *            セット
     * @return セットの文字列表現
     */
    public static String encodeSet(Set src) {
        final StringBuffer sb = new StringBuffer();
        encodeSet(sb, src);
        return sb.toString();
    }

    /**
     * 拡張規則で Short を文字列に変換します。<br>
     * 十進表記の数値文字列に接尾辞 'S' を付加します。
     * 
     * @param src
     *            数値オブジェクト
     * @return 文字列表現
     */
    public static String encodeShort(Short src) {
        return encodeNumber(src, 'S');
    }

    /**
     * 文字列を Unicode escapes の文字表現へ変換して返却します。
     * 
     * @param src
     *            変換元の文字列
     * @return エンコードされた文字列
     */
    public static String encodeString(String src) {
        final int srcLength = src.length();
        final StringBuffer sb = new StringBuffer(srcLength * 7);
        for (int i = 0; i < srcLength; i++) {
            final char c = src.charAt(i);
            addUnicodeEscapeChar(sb, c);
        }
        return sb.toString();
    }

    /**
     * データを生成して返却します、マクロ関数と変数を利用可能としています。<br>
     * 文字列が定義するデータ型は構文から自動で解析します。
     * 
     * @param src
     *            生成元の文字列
     * @param macros
     *            マクロ名とマクロのマップ、構造は Map&lt;String, Closure&gt; である必要があります。
     * @param context
     *            変数のマップ、構造は Map&lt;String, Object&gt; である必要があります。
     * @return 生成されたデータ
     * @throws IllegalArgumentException
     *             構文が不正の場合
     */
    public static Object macro(String src, Map macros, Map context) {
        final Position pos = new Position(0, src.length());
        next(src, pos);
        final Object reValue = macroVlaue(src, macros, context, pos);
        next(src, pos);
        if ((pos.index + 1) < pos.len) {
            throw newIllegalArgumentEx("pattern err.", src, pos.index);
        }
        return reValue;
    }

    /*
     * private
     */

    private static void addUnicodeEscapeChar(StringBuffer sb, char c) {
        // unicode escape sequences
        sb.append('\\');
        sb.append('u');
        sb.append(toHex((c >> 12) & 0xF));
        sb.append(toHex((c >> 8) & 0xF));
        sb.append(toHex((c >> 4) & 0xF));
        sb.append(toHex(c & 0xF));
    }

    /* 日付時刻文字列を切り出して返却します。 */
    private static String cutDateTimeString(String src, Position pos) {
        final int bigin = pos.index;
        int off = pos.index;
        if (!(off + 1 < pos.len)) {
            throw newIllegalArgumentEx("not find close char.", src, pos.index);
        }
        while (off < pos.len) {
            off++;
            if (isNotDateTimeChar(src.charAt(off))) {
                break;
            }
        }
        if (!(off < pos.len) || src.charAt(off) != 0x2F) {
            throw newIllegalArgumentEx("not find close char.", src, pos.index);
        }
        pos.index = ++off;
        return src.substring(bigin + 1, off - 1);
    }

    /* 数値文字列を切り出して返却します。 */
    private static String cutNumberString(String src, Position pos) {
        final int bigin = pos.index;
        int off = pos.index;
        while (off < pos.len) {
            if (isNotNumChar(src.charAt(off))) {
                break;
            }
            off++;
        }
        pos.index = off;
        return src.substring(bigin, off);
    }

    /* Boolean を生成して返却します。 */
    private static Boolean decodeBoolean(String src, Position pos) {
        final Boolean b;
        // 小文字の false のみ有効
        if (src.startsWith("false", pos.index)) {
            b = Boolean.FALSE;
            pos.index += 5;
        } else
        // 小文字の true のみ有効
        if (src.startsWith("true", pos.index)) {
            b = Boolean.TRUE;
            pos.index += 4;
        } else {
            throw newIllegalArgumentEx("is not Boolean.", src, pos.index);
        }
        return b;
    }

    /* Unicode escapes の文字表現を文字型へ変換して返却します。 */
    private static Character decodeCharacter(String src, Position pos) {
        if (src == null) {
            throw newIllegalArgumentEx("Malformed character encoding.", src, pos.index);
        }

        int off = pos.index;
        if (!(off + 1 < pos.len)) {
            throw newIllegalArgumentEx("not find close char.", src, pos.index);
        }
        char c;
        c = src.charAt(++off);
        if (c == '\\') {
            if (!(off + 1 < pos.len)) {
                throw newIllegalArgumentEx("Malformed character encoding.", src, pos.index);
            }
            final char next = src.charAt(off + 1);
            if (next == 'u') {
                if (!(off + 5 < pos.len)) {
                    throw newIllegalArgumentEx("Malformed \\uxxxx encoding.", src, pos.index);
                }
                c = decodeHexCharacter(src, off + 2);
                off += 5;
            } else {
                if (!(off + 1 < pos.len)) {
                    throw newIllegalArgumentEx("Malformed \\x encoding.", src, pos.index);
                }
                c = decodeCntrlCharacter(src, next, pos.index);
                off += 1;
            }
        }
        off++;
        if (!(off < pos.len) || src.charAt(off) != 0x60) {
            throw newIllegalArgumentEx("not find close char.", src, pos.index);
        }
        pos.index = ++off;
        return Boxing.box(c);
    }

    private static char decodeCntrlCharacter(CharSequence src, char c, int index) {
        switch (c) {
        case 0x22: // "
            return '\"';
        case 0x27: // '
            return '\'';
        case 0x30: // 0
            return '\0';
        case 0x31: // 1
            return '\1';
        case 0x32: // 2
            return '\2';
        case 0x33: // 3
            return '\3';
        case 0x34: // 4
            return '\4';
        case 0x35: // 5
            return '\5';
        case 0x36: // 6
            return '\6';
        case 0x37: // 7
            return '\7';
        case 0x5C: // \
            return '\\';
        case 0x62: // b
            return '\b';
        case 0x66: // f
            return '\f';
        case 0x6E: // n
            return '\n';
        case 0x72: // r
            return '\r';
        case 0x74: // t
            return '\t';
        default:
            throw newIllegalArgumentEx("Malformed \\x encoding.", src, index);
        }
    }

    /* 要素を生成して引数のコレクションに格納します。 */
    private static Collection decodeCollection(String src, Position pos, int open, int close, Collection coll) {
        next(src, pos);
        final int openChar = src.charAt(pos.index);
        if (openChar != open) {
            throw newIllegalArgumentEx("pattern err.", src, pos.index);
        }
        pos.index++;// add open char

        next(src, pos);
        final int closeChar = src.charAt(pos.index);
        if (closeChar == close) {
            pos.index++;// add close char
            return coll;
        }

        while (pos.index < pos.len) {
            coll.add(decodeVlaue(src, pos));
            next(src, pos);
            final int next = src.charAt(pos.index);
            if (next == close) {
                pos.index++;// add close char
                break;
            }
            if (next == ',') {
                pos.index++;
            } else {
                throw newIllegalArgumentEx("pattern err.", src, pos.index);
            }
            next(src, pos);
        }
        return coll;
    }

    private static char decodeHexCharacter(CharSequence src, int index) {
        // Read the xxxx
        int value = 0;
        char c;
        for (int i = index; i < (index + 4); i++) {
            c = src.charAt(i);
            switch (c) {
            case 0x30: // 0
            case 0x31: // 1
            case 0x32: // 2
            case 0x33: // 3
            case 0x34: // 4
            case 0x35: // 5
            case 0x36: // 6
            case 0x37: // 7
            case 0x38: // 8
            case 0x39: // 9
                value = (value << 4) + c - '0';
                break;
            case 0x41: // A
            case 0x42: // B
            case 0x43: // C
            case 0x44: // D
            case 0x45: // E
            case 0x46: // F
                value = (value << 4) + 10 + c - 'A';
                break;
            case 0x61: // a
            case 0x62: // b
            case 0x63: // c
            case 0x64: // d
            case 0x65: // e
            case 0x66: // f
                value = (value << 4) + 10 + c - 'a';
                break;
            default:
                throw newIllegalArgumentEx("Malformed \\uxxxx encoding.", src, index);
            }
        }
        return (char) value;
    }

    /* リストを生成して返却します。 */
    private static List decodeList(String src, Position pos) {
        final List list = new ArrayList();
        return Collections.unmodifiableList((List) decodeCollection(src, pos, '[', ']', list));
    }

    /* マップを生成して返却します。 */
    private static Map decodeMap(String src, Position pos) {
        final Map map = new LinkedHashMap();
        return Collections.unmodifiableMap(decodeMap(src, pos, '{', '}', '=', map));
    }

    /* 要素を生成して引数のマップに格納します。 */
    private static Map decodeMap(String src, Position pos, int open, int close, int delimiter, Map map) {

        next(src, pos);
        final int openChar = src.charAt(pos.index);
        if (openChar != open) {
            throw newIllegalArgumentEx("pattern err.", src, pos.index);
        }
        pos.index++;// add open char

        next(src, pos);
        final int closeChar = src.charAt(pos.index);
        if (closeChar == close) {
            pos.index++;// add close char
            return map;
        }

        while (pos.index < pos.len) {
            final Object key = decodeVlaue(src, pos);

            next(src, pos);
            final int delimiterChar = src.charAt(pos.index);
            if (delimiterChar != delimiter) {
                throw newIllegalArgumentEx("pattern err.", src, pos.index);
            }
            pos.index++;// add delimiter char

            next(src, pos);
            final Object value = decodeVlaue(src, pos);

            map.put(key, value);

            next(src, pos);
            final int next = src.charAt(pos.index);
            if (next == close) {
                pos.index++;// add close char
                break;
            }
            if (next == ',') {
                pos.index++;
            } else {
                throw newIllegalArgumentEx("pattern err.", src, pos.index);
            }
            next(src, pos);
        }
        return map;
    }

    /* Null を生成して返却します。 */
    private static Object decodeNull(String src, Position pos) {
        // 小文字の null のみ有効
        if (src.startsWith(NULL_SYMBOL, pos.index)) {
            pos.index += 4;
            return null;
        }
        throw newIllegalArgumentEx("is not null.", src, pos.index);
    }

    /* セットを生成して返却します。 */
    private static Set decodeSet(String src, Position pos) {
        final Set set = new LinkedHashSet();
        return Collections.unmodifiableSet((Set) decodeCollection(src, pos, '(', ')', set));
    }

    /* Unicode escapes の文字表現を文字列型へ変換して返却します。 */
    private static String decodeString(String src, Position pos) {
        if (src == null) {
            throw newIllegalArgumentEx("Malformed character encoding.", src, pos.index);
        }

        final StringBuffer sb = new StringBuffer();
        int off = pos.index;
        if (!(off + 1 < pos.len)) {
            throw newIllegalArgumentEx("not find close char.", src, pos.index);
        }
        while (off < pos.len) {
            final char c = src.charAt(++off);
            if (c == 0x27) {
                break;
            }
            if (c == '\\') {
                if (!(off + 2 < pos.len)) {
                    throw newIllegalArgumentEx("Malformed character encoding.", src, pos.index);
                }
                final char next = src.charAt(off + 1);
                if (next == 'u') {
                    if (!(off + 6 < pos.len)) {
                        throw newIllegalArgumentEx("Malformed \\uxxxx encoding.", src, pos.index);
                    }
                    sb.append(decodeHexCharacter(src, off + 2));
                    off += 5;
                } else {
                    if (!(off + 2 < pos.len)) {
                        throw newIllegalArgumentEx("Malformed \\x encoding.", src, pos.index);
                    }
                    sb.append(decodeCntrlCharacter(src, next, pos.index));
                    off += 1;
                }
            } else {
                sb.append(c);
            }
        }
        if (!(off < pos.len) || src.charAt(off) != 0x27) {
            throw newIllegalArgumentEx("not find close char.", src, pos.index);
        }
        pos.index = ++off;
        return sb.toString();
    }

    /* 要素を生成して返却します。 */
    private static Object decodeVlaue(String src, Position pos) {
        final int ch = src.charAt(pos.index);
        switch (ch) {
        case 0x27: // '
            return decodeString(src, pos);
        case 0x28: // (
            return decodeSet(src, pos);
        case 0x2F: // /
            return decodeDateTime(cutDateTimeString(src, pos));
        case 0x5B: // [
            return decodeList(src, pos);
        case 0x60: // `
            return decodeCharacter(src, pos);
        case 0x66: // f
            return decodeBoolean(src, pos);
        case 0x6E: // n
            return decodeNull(src, pos);
        case 0x74: // t
            return decodeBoolean(src, pos);
        case 0x7B: // {
            return decodeMap(src, pos);
        default:
            if (isBeginNumberChar(ch)) {
                return decodeNumber(cutNumberString(src, pos));
            }
        }
        throw newIllegalArgumentEx("pattern err.", src, pos.index);
    }

    /* コレクションから文字列を生成して追加します。 */
    private static void encodeCollection(StringBuffer sb, Collection src) {
        final Iterator i = src.iterator();
        while (i.hasNext()) {
            encodeVlaue(sb, i.next());
            if (i.hasNext()) {
                sb.append(',');
            }
        }
    }

    /* カレンダーの情報を日付時刻文字列に変換します。 */
    private static String encodeDateTime(Calendar cal, char type) {
        final int year = cal.get(Calendar.YEAR);
        final int month = cal.get(Calendar.MONTH) + 1;
        final int date = cal.get(Calendar.DAY_OF_MONTH);
        final int hour = cal.get(Calendar.HOUR_OF_DAY);
        final int minute = cal.get(Calendar.MINUTE);
        final int second = cal.get(Calendar.SECOND);
        final int nanos = cal.get(Calendar.MILLISECOND);
        final int zone = cal.get(Calendar.ZONE_OFFSET);
        final char zsign = (0 > zone) ? '-' : '+';
        final int zabs = Math.abs(zone);
        final int zhour = zabs / 3600000;
        final int zminute = (zabs - zhour * 3600000) / 60000;

        final StringBuffer sb = new StringBuffer(28);
        padZero(sb, year, 4);
        sb.append('-');
        padZero(sb, month, 2);
        sb.append('-');
        padZero(sb, date, 2);
        sb.append('T');
        padZero(sb, hour, 2);
        sb.append(':');
        padZero(sb, minute, 2);
        sb.append(':');
        padZero(sb, second, 2);
        sb.append('.');
        padZero(sb, nanos, 3);
        sb.append(zsign);
        padZero(sb, zhour, 2);
        sb.append(':');
        padZero(sb, zminute, 2);
        sb.append(type);
        return sb.toString();
    }

    /* リストから文字列を生成して追加します。 */
    private static void encodeList(StringBuffer sb, List src) {
        sb.append('[');
        encodeCollection(sb, src);
        sb.append(']');
    }

    /* マップから文字列を生成して追加します。 */
    private static void encodeMap(StringBuffer sb, Map src) {
        sb.append('{');
        final Iterator i = src.entrySet().iterator();
        while (i.hasNext()) {
            final Map.Entry e = (Map.Entry) i.next();
            encodeVlaue(sb, e.getKey());
            sb.append('=');
            encodeVlaue(sb, e.getValue());
            if (i.hasNext()) {
                sb.append(',');
            }
        }
        sb.append('}');
    }

    /* 拡張規則で Number を文字列に変換します。 */
    private static String encodeNumber(Number src, char suffix) {
        final String _src = src.toString();
        final int capacity = _src.length() + 1;
        return (new StringBuffer(capacity)).append(_src).append(suffix).toString();
    }

    /* セットから文字列を生成して追加します。 */
    private static void encodeSet(StringBuffer sb, Set src) {
        sb.append('(');
        encodeCollection(sb, src);
        sb.append(')');
    }

    /* 要素から文字列を生成して追加します。 */
    private static void encodeVlaue(StringBuffer sb, Object src) {
        if (src == null) {
            sb.append(NULL_SYMBOL);
            return;
        }
        if (src instanceof Number) {
            if (src instanceof Byte) {
                sb.append(encodeByte((Byte) src));
                return;
            }
            if (src instanceof Short) {
                sb.append(encodeShort((Short) src));
                return;
            }
            if (src instanceof Integer) {
                sb.append(encodeInteger((Integer) src));
                return;
            }
            if (src instanceof Long) {
                sb.append(encodeLong((Long) src));
                return;
            }
            if (src instanceof Float) {
                sb.append(encodeFloat((Float) src));
                return;
            }
            if (src instanceof Double) {
                sb.append(encodeDouble((Double) src));
                return;
            }
            if (src instanceof BigInteger) {
                sb.append(encodeBigInteger((BigInteger) src));
                return;
            }
            if (src instanceof BigDecimal) {
                sb.append(encodeBigDecimal((BigDecimal) src));
                return;
            }
            throw newIllegalArgumentEx("Unsupported type.", src, 0);
        }
        if (src instanceof String) {
            sb.append('\'');
            sb.append(encodeString((String) src));
            sb.append('\'');
            return;
        }
        if (src instanceof Character) {
            sb.append('`');
            sb.append(encodeCharacter((Character) src));
            sb.append('`');
            return;
        }
        if (src instanceof Calendar) {
            sb.append('/');
            sb.append(encodeDateTime((Calendar) src));
            sb.append('/');
            return;
        }
        if (src instanceof Boolean) {
            sb.append(encodeBoolean((Boolean) src));
            return;
        }
        if (src instanceof Date) {
            sb.append('/');
            sb.append(encodeDateTime((Date) src));
            sb.append('/');
            return;
        }
        if (src instanceof List) {
            encodeList(sb, (List) src);
            return;
        }
        if (src instanceof Set) {
            encodeSet(sb, (Set) src);
            return;
        }
        if (src instanceof Map) {
            encodeMap(sb, (Map) src);
            return;
        }
        throw newIllegalArgumentEx("Unsupported type.", src, 0);
    }

    private static boolean isBeginNumberChar(int ch) {
        switch (ch) {
        case 0x2D: // -
        case 0x30: // 0
        case 0x31: // 1
        case 0x32: // 2
        case 0x33: // 3
        case 0x34: // 4
        case 0x35: // 5
        case 0x36: // 6
        case 0x37: // 7
        case 0x38: // 8
        case 0x39: // 9
        case 0x49: // I
        case 0x4E: // N
            return true;
        default:
            return false;
        }
    }

    /* 日付時刻で使用される文字か検証して返却します。 */
    private static boolean isNotDateTimeChar(int ch) {
        switch (ch) {
        case 0x20: // SPACE
        case 0x2B: // +
        case 0x2D: // -
        case 0x2E: // .
        case 0x30: // 0
        case 0x31: // 1
        case 0x32: // 2
        case 0x33: // 3
        case 0x34: // 4
        case 0x35: // 5
        case 0x36: // 6
        case 0x37: // 7
        case 0x38: // 8
        case 0x39: // 9
        case 0x3A: // :
        case 0x44: // D
        case 0x47: // G
        case 0x4C: // L
        case 0x54: // T
        case 0x64: // d
        case 0x67: // g
        case 0x6C: // l
            return false;
        default:
            return true;
        }
    }

    /* 数値で使用される文字か検証して返却します。 */
    private static boolean isNotNumChar(int ch) {
        switch (ch) {
        case 0x2B: // +
        case 0x2D: // -
        case 0x2E: // .
        case 0x30: // 0
        case 0x31: // 1
        case 0x32: // 2
        case 0x33: // 3
        case 0x34: // 4
        case 0x35: // 5
        case 0x36: // 6
        case 0x37: // 7
        case 0x38: // 8
        case 0x39: // 9
        case 0x41: // A
        case 0x42: // B
        case 0x43: // C
        case 0x44: // D
        case 0x45: // E
        case 0x46: // F
        case 0x47: // G
        case 0x48: // H
        case 0x49: // I
        case 0x4C: // L
        case 0x4E: // N
        case 0x53: // S
        case 0x55: // U
        case 0x58: // X
        case 0x61: // a
        case 0x62: // b
        case 0x63: // c
        case 0x64: // d
        case 0x65: // e
        case 0x66: // f
        case 0x67: // g
        case 0x68: // h
        case 0x69: // i
        case 0x6C: // l
        case 0x6E: // n
        case 0x73: // s
        case 0x74: // t
        case 0x75: // u
        case 0x78: // x
        case 0x79: // y
            return false;
        default:
            return true;
        }
    }

    /* スペース文字か検証します。 */
    private static boolean isNotSpaceChar(int ch) {
        switch (ch) {
        case 0x09: // (HT)
        case 0x0A: // (LF)
        case 0x0B: // (VT)
        case 0x0C: // (FF)
        case 0x0D: // (CR)
        case 0x20: // SPACE
            return false;
        default:
            return true;
        }
    }

    /* マクロ を実行して返却します。 */
    private static Object macro(String src, Map macros, Map context, Position pos) {
        pos.index++;// add open char

        final int endIndex = macroIdEndIndexOf(src, pos);
        if (endIndex < 0) {
            throw newIllegalArgumentEx("not find args.", src, pos.index);
        }
        final String key = src.substring(pos.index, endIndex);
        final Object macro = macros.get(key);
        if (macro == null) {
            throw newIllegalArgumentEx("is not contains macro id.", src, pos.index);
        }
        if (!(macro instanceof Closure)) {
            throw newIllegalArgumentEx("illegal Macro Object.", src, pos.index);
        }
        pos.index = endIndex;
        next(src, pos);

        final int argsType = src.charAt(pos.index);
        if (argsType == 0x28) { // (
            final List listArgs = macroArgsList(src, macros, context, pos);
            return ((Closure) macro).evaluate(listArgs);
        }
        throw newIllegalArgumentEx("illegal Macro Arguments.", src, pos.index);
    }

    /* 要素を生成して引数のマップに格納します。 */
    private static List macroArgsList(String src, Map macros, Map context, Position pos) {
        final List args = new ArrayList(8);
        macroCollection(src, macros, context, pos, 0x28, 0x29, args);
        return args;
    }

    /* 要素を生成して引数のコレクションに格納します。 */
    private static Collection macroCollection(String src, Map macros, Map context, Position pos, int open, int close, Collection coll) {
        next(src, pos);
        final int openChar = src.charAt(pos.index);
        if (openChar != open) {
            throw newIllegalArgumentEx("pattern err.", src, pos.index);
        }
        pos.index++;// add open char

        next(src, pos);
        final int closeChar = src.charAt(pos.index);
        if (closeChar == close) {
            pos.index++;// add close char
            return coll;
        }

        while (pos.index < pos.len) {
            coll.add(macroVlaue(src, macros, context, pos));
            next(src, pos);
            final int next = src.charAt(pos.index);
            if (next == close) {
                pos.index++;// add close char
                break;
            }
            if (next == ',') {
                pos.index++;
            } else {
                throw newIllegalArgumentEx("pattern err.", src, pos.index);
            }
            next(src, pos);
        }
        return coll;
    }

    /* Boolean を生成して返却します。 */
    private static Object macroContext(String src, Map context, Position pos) {
        final int openChar = src.charAt(pos.index);
        if (openChar != 0x23) {
            throw newIllegalArgumentEx("not find open char.", src, pos.index);
        }
        pos.index++;// add open char

        final int endIndex = src.indexOf(0x23, pos.index);
        if (endIndex < 0) {
            throw newIllegalArgumentEx("not find close char.", src, pos.index);
        }

        final String key = src.substring(pos.index, endIndex);
        final Object val = context.get(key);

        // 変数が存在しない場合はエラー（処理量への配慮から事後検証する）
        if (val == null && !context.containsKey(key)) {
            throw newIllegalArgumentEx("is not contains context key.", src, pos.index);
        }

        pos.index = endIndex;
        pos.index++;// add close char
        return val;
    }

    private static int macroIdEndIndexOf(String src, Position pos) {
        for (int i = pos.index; i < pos.len; i++) {
            final int ch = src.charAt(i);
            switch (ch) {
            case 0x30: // 0
            case 0x31: // 1
            case 0x32: // 2
            case 0x33: // 3
            case 0x34: // 4
            case 0x35: // 5
            case 0x36: // 6
            case 0x37: // 7
            case 0x38: // 8
            case 0x39: // 9
            case 0x3A: // :
            case 0x41: // A
            case 0x42: // B
            case 0x43: // C
            case 0x44: // D
            case 0x45: // E
            case 0x46: // F
            case 0x47: // G
            case 0x48: // H
            case 0x49: // I
            case 0x4A: // J
            case 0x4B: // K
            case 0x4C: // L
            case 0x4D: // M
            case 0x4E: // N
            case 0x4F: // O
            case 0x50: // P
            case 0x51: // Q
            case 0x52: // R
            case 0x53: // S
            case 0x54: // T
            case 0x55: // U
            case 0x56: // V
            case 0x57: // W
            case 0x58: // X
            case 0x59: // Y
            case 0x5A: // Z
            case 0x5F: // _
            case 0x61: // a
            case 0x62: // b
            case 0x63: // c
            case 0x64: // d
            case 0x65: // e
            case 0x66: // f
            case 0x67: // g
            case 0x68: // h
            case 0x69: // i
            case 0x6A: // j
            case 0x6B: // k
            case 0x6C: // l
            case 0x6D: // m
            case 0x6E: // n
            case 0x6F: // o
            case 0x70: // p
            case 0x71: // q
            case 0x72: // r
            case 0x73: // s
            case 0x74: // t
            case 0x75: // u
            case 0x76: // v
            case 0x77: // w
            case 0x78: // x
            case 0x79: // y
            case 0x7A: // z
                break;
            default:
                if (i == pos.index) {
                    return -1;
                }
                return i;
            }
        }
        return -1;
    }

    /* リストを生成して返却します。 */
    private static List macroList(String src, Map macros, Map context, Position pos) {
        final List list = new ArrayList();
        return Collections.unmodifiableList((List) macroCollection(src, macros, context, pos, '[', ']', list));
    }

    /* マップを生成して返却します。 */
    private static Map macroMap(String src, Map macros, Map context, Position pos) {
        final Map map = new LinkedHashMap();
        return Collections.unmodifiableMap(macroMap(src, macros, context, pos, '{', '}', '=', map));
    }

    /* 要素を生成して引数のマップに格納します。 */
    private static Map macroMap(String src, Map macros, Map context, Position pos, int open, int close, int delimiter, Map map) {

        next(src, pos);
        final int openChar = src.charAt(pos.index);
        if (openChar != open) {
            throw newIllegalArgumentEx("pattern err.", src, pos.index);
        }
        pos.index++;// add open char

        next(src, pos);
        final int closeChar = src.charAt(pos.index);
        if (closeChar == close) {
            pos.index++;// add close char
            return map;
        }

        while (pos.index < pos.len) {
            final Object key = macroVlaue(src, macros, context, pos);

            next(src, pos);
            final int delimiterChar = src.charAt(pos.index);
            if (delimiterChar != delimiter) {
                throw newIllegalArgumentEx("pattern err.", src, pos.index);
            }
            pos.index++;// add delimiter char

            next(src, pos);
            final Object value = macroVlaue(src, macros, context, pos);

            map.put(key, value);

            next(src, pos);
            final int next = src.charAt(pos.index);
            if (next == close) {
                pos.index++;// add close char
                break;
            }
            if (next == ',') {
                pos.index++;
            } else {
                throw newIllegalArgumentEx("pattern err.", src, pos.index);
            }
            next(src, pos);
        }
        return map;
    }

    /* セットを生成して返却します。 */
    private static Set macroSet(String src, Map macros, Map context, Position pos) {
        final Set set = new LinkedHashSet();
        return Collections.unmodifiableSet((Set) macroCollection(src, macros, context, pos, '(', ')', set));
    }

    /* 要素を生成して返却します。 */
    private static Object macroVlaue(String src, Map macros, Map context, Position pos) {
        final int ch = src.charAt(pos.index);
        switch (ch) {
        case 0x23: // #
            return macroContext(src, context, pos);
        case 0x24: // $
            return macro(src, macros, context, pos);
        case 0x27: // '
            return decodeString(src, pos);
        case 0x28: // (
            return macroSet(src, macros, context, pos);
        case 0x2F: // /
            return decodeDateTime(cutDateTimeString(src, pos));
        case 0x5B: // [
            return macroList(src, macros, context, pos);
        case 0x60: // `
            return decodeCharacter(src, pos);
        case 0x66: // f
            return decodeBoolean(src, pos);
        case 0x6E: // n
            return decodeNull(src, pos);
        case 0x74: // t
            return decodeBoolean(src, pos);
        case 0x7B: // {
            return macroMap(src, macros, context, pos);
        default:
            if (isBeginNumberChar(ch)) {
                return decodeNumber(cutNumberString(src, pos));
            }
        }
        throw newIllegalArgumentEx("pattern err.", src, pos.index);
    }

    private static IllegalArgumentException newIllegalArgumentEx(String msg, Object src, int index) {
        throw new IllegalArgumentException(msg + "　src=" + src + ", pos=" + index);
    }

    /* 開始インデックス以降のスペースをスキップします。 */
    private static void next(String src, Position pos) {
        while (pos.index < pos.len) {
            if (isNotSpaceChar(src.charAt(pos.index))) {
                break;
            }
            pos.index++;
        }
    }

    /* Javaで認識できる形式で数値の文字列を構築する */
    private static String numstr(int negative, String src, int radix, boolean hasSuffix) {
        final int endIndex = (hasSuffix) ? (src.length() - 1) : src.length();
        final String _num;
        switch (radix) {
        case 2:
            _num = src.substring(negative + 2, endIndex);
            break;
        case 8:
            _num = src.substring(negative + 1, endIndex);
            break;
        case 10:
            _num = src.substring(negative + 0, endIndex);
            break;
        case 16:
            _num = src.substring(negative + 2, endIndex);
            break;
        default:
            return src.substring(0, endIndex);
        }
        return (negative == 1) ? NEGATIVE_SIGN.concat(_num) : _num;
    }

    /* 指定の桁数をゼロ埋めして文字列を追加します。 */
    private static void padZero(StringBuffer sb, int value, int n) {
        int size = 10;
        for (int i = 1; i < n; i++) {
            if (value < size) {
                for (int j = i; j < n; j++) {
                    sb.append('0');
                }
                break;
            }
            size *= 10;
        }
        sb.append(value);
    }

    /* 八進数の文字を返します。 */
    private static char toHex(int nibble) {
        return hexDigit[(nibble & 0xF)];
    }

    /* 指定の文字が小文字アルファベットの場合は大文字に変換します。 */
    private static int toUpper(int ch) {
        return ((ch - 0x61) | (0x7A - ch)) >= 0 ? (ch - 0x20) : ch;
    }

}
