/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.lang;

import java.io.IOException;
import java.io.InputStream;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

/**
 * ライブラリの実行環境の情報に関するユーティリティを提供します。
 */
public class SystemUtils {

    /** ファイルシステムのエンコード "file.encoding"。 */
    public static final String FILE_ENCODING = getSystemProperty("file.encoding");

    /** ファイルシステムのアドレスの区切り文字 "file.separator"。 */
    public static final String FILE_SEPARATOR = getSystemProperty("file.separator");

    /** Javaランタイムのバージョン "java.version"。 */
    public static final String JAVA_VERSION = getSystemProperty("java.version");

    /** システムの改行文字 "line.separator"。 */
    public static final String LINE_SEPARATOR = getSystemProperty("line.separator");

    /**
     * システムのプロパティリストを特権で取得します。
     * 
     * @return プロパティリスト
     */
    public static Properties getSystemProperties() {
        return (Properties) AccessController.doPrivileged(new PrivilegedAction() {
            public Object run() {
                return System.getProperties();
            }
        });
    }

    /**
     * 指定された名前の、システムのプロパティを特権で取得します。
     * 
     * @param key
     *            プロパティ名
     * @return プロパティ
     */
    public static String getSystemProperty(String key) {
        final String _key = key;
        return (String) AccessController.doPrivileged(new PrivilegedAction() {
            public Object run() {
                return System.getProperty(_key);
            }
        });
    }

    /**
     * ライブラリのバージョンを返却します。
     * 
     * @param clazz
     *            ライブラリ内のクラス
     * @return ライブラリのバージョン
     */
    public static String getLibraryVersion(Class clazz) {
        return clazz.getPackage().getImplementationVersion();
    }

    /**
     * 松伯ライブラリ固有のプロパティリストを読み取ります。<br>
     * デフォルトプロパティのクラスパスは "shohaku-" + libraryName + "-system-defaults.properties" です。<br>
     * 拡張プロパティのクラスパスは "shohaku-" + libraryName + "-system-extends.properties" です。<br>
     * libraryName はパッケージ名の "shohaku." に続く部分です。<br>
     * デフォルトプロパティの値を拡張プロパティで上書きできます。
     * 
     * @param clazz
     *            ライブラリ内のクラス
     * @return ライブラリ固有のプロパティリスト
     */
    public static Map getLibraryProperties(Class clazz) {
        // get library name
        String path = clazz.getName().substring("shohaku.".length());
        final String libraryName = path.substring(0, path.indexOf('.'));

        // load library Properties
        final Properties properties = new Properties();
        loadProperties(properties, Concat.get("shohaku-", libraryName, "-system-defaults.properties"));
        loadProperties(properties, Concat.get("shohaku-", libraryName, "-system-extends.properties"));

        // convert all
        return new HashMap(convertProperties(properties));
    }

    /* プロパティリストを読み取ります。 */
    static void loadProperties(Properties properties, String path) {
        final InputStream in = getResourceAsStream(path);
        if (null != in) {
            try {
                properties.load(in);
            } catch (java.io.IOException e) {
                // no op
            } finally {
                try {
                    in.close();
                } catch (IOException e) {
                    // no op
                }
            }
        }
    }

    /*
     * package
     */

    /* プロパティリストを変換して再格納します。 */
    static Map convertProperties(Map properties) {
        for (Iterator i = properties.entrySet().iterator(); i.hasNext();) {
            Map.Entry e = (Map.Entry) i.next();
            e.setValue(convertProperty((String) e.getValue()));
        }
        return properties;
    }

    /* プロパティを変換して返却します。 */
    static Object convertProperty(String property) {
        String prop = property.trim();
        if (prop.length() == 0) {
            return "";
        }
        // String. name="string"
        if (prop.startsWith("\"") && prop.endsWith("\"")) {
            return prop.substring(1, prop.length() - 1);
        }
        // List. name=[ "string", false, 1234, 'C' ], split->","
        if (prop.startsWith("[") && prop.endsWith("]")) {
            String listString = prop.substring(1, prop.length() - 1);
            String[] elements = listString.split(",");
            ArrayList list = new ArrayList(elements.length);
            for (int i = 0; i < elements.length; i++) {
                list.add(convertProperty(elements[i].trim()));
            }
            return list;
        }

        // Character. name='a'
        if (prop.length() == 3 && prop.startsWith("'") && prop.endsWith("'")) {
            return new Character(prop.charAt(1));
        }
        // Boolean. name=false, name=true
        if (prop.equalsIgnoreCase("true") || prop.equalsIgnoreCase("false")) {
            return Boolean.valueOf(prop.toLowerCase());
        }
        // Number. Byte=123B, Short=2345S, Integer=12345I or 12345, Float=123.45F, Double=123.45D or 123.45
        try {
            char suffix = Character.toUpperCase(prop.charAt((prop.length() - 1)));
            String num = prop.substring(0, (prop.length() - 1));
            switch (suffix) {
            case 'B':
                return Byte.valueOf(num);
            case 'S':
                return Short.valueOf(num);
            case 'I':
                return Integer.valueOf(num);
            case 'L':
                return Long.valueOf(num);
            case 'F':
                return Float.valueOf(num);
            case 'D':
                return Double.valueOf(num);
            default:
                if (-1 < prop.indexOf('.')) {
                    return Double.valueOf(prop);
                } else {
                    return Integer.valueOf(prop);
                }
            }
        } catch (NumberFormatException e) {
            // no op
        }

        // default type String
        return prop;
    }

    /* リソースの入力ストリームを特権で取得して返却します。 */
    static InputStream getResourceAsStream(final String name) {
        return (InputStream) AccessController.doPrivileged(new PrivilegedAction() {
            public Object run() {
                ClassLoader threadCL = Thread.currentThread().getContextClassLoader();
                if (threadCL != null) {
                    return threadCL.getResourceAsStream(name);
                } else {
                    return ClassLoader.getSystemResourceAsStream(name);
                }
            }
        });
    }

}
