/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.lang;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TimeZone;
import java.util.TreeMap;

/**
 * 頻度が高く繁雑な処理を手早く済ませるための関数群を提供します。
 */
public final class Fast {

    /*
     * 空値やスペースの制御
     */

    /**
     * 引数が null の場合は空文字列を返却し、以外の場合は文字列変換して返却します。<br>
     * nvl は Null Value Loss の略です。
     * 
     * @param o
     *            値
     * @return null の場合は空文字列、以外の場合は文字列表現
     */
    public static String nvl(Object o) {
        return (o == null) ? "" : String.valueOf(o);
    }

    /**
     * 引数が null の場合は第二引数を返却し、以外の場合は文字列変換して返却します。<br>
     * nvl は Null Value Loss の略です。
     * 
     * @param o
     *            値
     * @param substitute
     *            代返値
     * @return null の場合は空文字列、以外の場合は文字列表現
     */
    public static String nvl(Object o, String substitute) {
        return (o == null) ? substitute : String.valueOf(o);
    }

    /**
     * 引数が null または空かスペースのみの場合は空文字列を返却し、以外の場合は引数をそのまま返却します。<br>
     * bvl は Blank Value Loss の略です。
     * 
     * @param s
     *            値
     * @return null または空かスペースのみの場合は空文字列、以外の場合は引数
     */
    public static String bvl(String s) {
        return (s == null || trimLength(s) == 0) ? "" : s;
    }

    /**
     * 引数が null または空かスペースのみの場合は第二引数を返却し、以外の場合は第一引数をそのまま返却します。<br>
     * bvl は Blank Value Loss の略です。
     * 
     * @param s
     *            値
     * @param substitute
     *            代返値
     * @return null または空かスペースのみの場合は第二引数、以外の場合は第一引数
     */
    public static String bvl(String s, String substitute) {
        return (s == null || trimLength(s) == 0) ? substitute : s;
    }

    /**
     * 引数が null または空かJavaの基準による空白文字のみの場合は空文字列を返却し、以外の場合は第一引数をそのまま返却します。<br>
     * wvl は Whitespace Value Loss の略です。
     * 
     * @param s
     *            値
     * @return null または空かJavaの基準による空白文字のみの場合は空文字列、以外の場合は第一引数
     */
    public static String wvl(String s) {
        return (s == null || trimWhiteLength(s) == 0) ? "" : s;
    }

    /**
     * 引数が null または空かJavaの基準による空白文字のみの場合は第二引数を返却し、以外の場合は第一引数をそのまま返却します。<br>
     * wvl は Whitespace Value Loss の略です。
     * 
     * @param s
     *            値
     * @param substitute
     *            代返値
     * @return null または空かJavaの基準による空白文字のみの場合は第二引数、以外の場合は第一引数
     */
    public static String wvl(String s, String substitute) {
        return (s == null || trimWhiteLength(s) == 0) ? substitute : s;
    }

    /**
     * 文字シーケンスの前方と後方からJavaの基準による空白文字を除いた文字列数を返却します。
     * 
     * @param cs
     *            処理対象の文字シーケンス
     * @return 前方と後方からJavaの基準による空白文字を除いた文字列数
     */
    static int trimWhiteLength(CharSequence cs) {
        int st = 0;
        int ed = cs.length();
        while ((st < ed) && Character.isWhitespace(cs.charAt(st))) {
            st++;
        }
        while ((st < ed) && Character.isWhitespace(cs.charAt(ed - 1))) {
            ed--;
        }
        return (ed - st);
    }

    /**
     * 文字シーケンスの前方と後方から空白文字を除いた文字列数を返却します。
     * 
     * @param cs
     *            処理対象の文字シーケンス
     * @return 前方と後方から空白文字を除いた文字列数
     */
    static int trimLength(CharSequence cs) {
        int st = 0;
        int ed = cs.length();
        while ((st < ed) && (cs.charAt(st) <= ' ')) {
            st++;
        }
        while ((st < ed) && (cs.charAt(ed - 1) <= ' ')) {
            ed--;
        }
        return (ed - st);
    }

    /*
     * 基本的な処理
     */

    /**
     * 配列の要素をコピーします。<br>
     * 要素の先頭から要素数の小さい配列のサイズ分のみコピーします。
     * 
     * @param src
     *            コピー元の配列
     * @param dest
     *            コピー先の配列
     * @throws IndexOutOfBoundsException
     *             コピーによって配列の境界を越えたデータのアクセスが生じた場合
     * @throws ArrayStoreException
     *             型の不一致により、配列 src の要素が配列 dest に保存できなかった場合
     * @throws NullPointerException
     *             src または dest が null の場合
     */
    public static void arraycopy(Object src, Object dest) {
        System.arraycopy(src, 0, dest, 0, Math.min(Array.getLength(src), Array.getLength(dest)));
    }

    /*
     * コレクションの構築
     */

    /**
     * キーと値を交互に格納する配列から、不可変の固定マップを生成して返却します。
     * 
     * @param keyValue
     *            キーと値を交互に格納する配列
     * @return 不可変の固定マップ
     */
    public static Map fixedMap(Object[] keyValue) {
        final int size = keyValue.length;
        if ((size % 2) != 0) {
            throw new IllegalArgumentException("keyValue size err ((keyValue.length % 2) != 0).");
        }
        if (size == 0) {
            return Collections.EMPTY_MAP;
        }
        if (size == 2) {
            return Collections.singletonMap(keyValue[0], keyValue[1]);
        }
        final Map map = new LinkedHashMap(size / 2);
        for (int i = 0; i < size; i += 2) {
            map.put(keyValue[i], keyValue[i + 1]);
        }
        return Collections.unmodifiableMap(map);
    }

    /**
     * キーと値を交互に格納するコレクションから、不可変の固定マップを生成して返却します。
     * 
     * @param keyValue
     *            キーと値を交互に格納するコレクション
     * @return 不可変の固定マップ
     */
    public static Map fixedMap(Collection keyValue) {
        final int size = keyValue.size();
        if ((size % 2) != 0) {
            throw new IllegalArgumentException("keyValue size err ((keyValue.size() % 2) != 0).");
        }
        if (size == 0) {
            return Collections.EMPTY_MAP;
        }
        if (size == 2) {
            final Iterator i = keyValue.iterator();
            return Collections.singletonMap(i.next(), i.next());
        }
        final Map map = new LinkedHashMap(size / 2);
        final Iterator i = keyValue.iterator();
        while (i.hasNext()) {
            map.put(i.next(), i.next());
        }
        return Collections.unmodifiableMap(map);
    }

    /**
     * キーと値を交互に格納する配列から、参照一致で検索する不可変の固定マップを生成して返却します。
     * 
     * @param keyValue
     *            キーと値を交互に格納する配列
     * @return 不可変の固定マップ
     */
    public static Map fixedReferenceMap(Object[] keyValue) {
        final int size = keyValue.length;
        if ((size % 2) != 0) {
            throw new IllegalArgumentException("keyValue size err ((keyValue.length % 2) != 0).");
        }
        if (size == 0) {
            return Collections.EMPTY_MAP;
        }
        final Map map = new IdentityHashMap(size / 2);
        for (int i = 0; i < size; i += 2) {
            map.put(keyValue[i], keyValue[i + 1]);
        }
        return Collections.unmodifiableMap(map);
    }

    /**
     * キーと値を交互に格納するコレクションから、参照一致で検索する不可変の固定マップを生成して返却します。
     * 
     * @param keyValue
     *            キーと値を交互に格納するコレクション
     * @return 不可変の固定マップ
     */
    public static Map fixedReferenceMap(Collection keyValue) {
        final int size = keyValue.size();
        if ((size % 2) != 0) {
            throw new IllegalArgumentException("keyValue size err ((keyValue.size() % 2) != 0).");
        }
        if (size == 0) {
            return Collections.EMPTY_MAP;
        }
        final Map map = new IdentityHashMap(size / 2);
        final Iterator i = keyValue.iterator();
        while (i.hasNext()) {
            map.put(i.next(), i.next());
        }
        return Collections.unmodifiableMap(map);
    }

    /**
     * キーと値を交互に格納する配列から、自然順序でソートされた不可変の固定マップを生成して返却します。
     * 
     * @param keyValue
     *            キーと値を交互に格納する配列
     * @return 不可変の固定マップ
     */
    public static SortedMap fixedSortedMap(Object[] keyValue) {
        final int size = keyValue.length;
        if ((size % 2) != 0) {
            throw new IllegalArgumentException("keyValue size err ((keyValue.length % 2) != 0).");
        }
        final Map map = new HashMap(size / 2);
        for (int i = 0; i < size; i += 2) {
            map.put(keyValue[i], keyValue[i + 1]);
        }
        final SortedMap sortedMap = new TreeMap(map);
        return Collections.unmodifiableSortedMap(sortedMap);
    }

    /**
     * キーと値を交互に格納するコレクションから、自然順序でソートされた不可変の固定マップを生成して返却します。
     * 
     * @param keyValue
     *            キーと値を交互に格納するコレクション
     * @return 不可変の固定マップ
     */
    public static SortedMap fixedSortedMap(Collection keyValue) {
        final int size = keyValue.size();
        if ((size % 2) != 0) {
            throw new IllegalArgumentException("keyValue size err ((keyValue.size() % 2) != 0).");
        }
        final Map map = new HashMap(size / 2);
        final Iterator i = keyValue.iterator();
        while (i.hasNext()) {
            map.put(i.next(), i.next());
        }
        final SortedMap sortedMap = new TreeMap(map);
        return Collections.unmodifiableSortedMap(sortedMap);
    }

    /**
     * キーと値を交互に格納する配列から、指定のコンパレータでソートされた不可変の固定マップを生成して返却します。
     * 
     * @param keyValue
     *            キーと値を交互に格納する配列
     * 
     * @param c
     *            コンパレータ
     * @return 不可変の固定マップ
     */
    public static SortedMap fixedSortedMap(Object[] keyValue, Comparator c) {
        final int size = keyValue.length;
        if ((size % 2) != 0) {
            throw new IllegalArgumentException("keyValue size err ((keyValue.length % 2) != 0).");
        }
        final Map map = new HashMap(size / 2);
        for (int i = 0; i < size; i += 2) {
            map.put(keyValue[i], keyValue[i + 1]);
        }
        final SortedMap sortedMap = new TreeMap(c);
        sortedMap.putAll(map);
        return Collections.unmodifiableSortedMap(sortedMap);
    }

    /**
     * キーと値を交互に格納するコレクションから、指定のコンパレータでソートされた不可変の固定マップを生成して返却します。
     * 
     * @param keyValue
     *            キーと値を交互に格納するコレクション
     * 
     * @param c
     *            コンパレータ
     * @return 不可変の固定マップ
     */
    public static SortedMap fixedSortedMap(Collection keyValue, Comparator c) {
        final int size = keyValue.size();
        if ((size % 2) != 0) {
            throw new IllegalArgumentException("keyValue size err ((keyValue.size() % 2) != 0).");
        }
        final Map map = new HashMap(size / 2);
        final Iterator i = keyValue.iterator();
        while (i.hasNext()) {
            map.put(i.next(), i.next());
        }
        final SortedMap sortedMap = new TreeMap(c);
        sortedMap.putAll(map);
        return Collections.unmodifiableSortedMap(sortedMap);
    }

    /**
     * 配列から不可変の固定リストを生成して返却します。
     * 
     * @param a
     *            リストの要素となる配列
     * @return 不可変の固定リスト
     */
    public static List fixedList(Object[] a) {
        if (a.length == 0) {
            return Collections.EMPTY_LIST;
        }
        if (a.length == 1) {
            return Collections.singletonList(a[0]);
        }
        return Collections.unmodifiableList(asArrayList(a));
    }

    /**
     * 配列から不可変の固定セットを生成して返却します。
     * 
     * @param a
     *            セットの要素となる配列
     * @return 不可変の固定セット
     */
    public static Set fixedSet(Object[] a) {
        if (a.length == 0) {
            return Collections.EMPTY_SET;
        }
        if (a.length == 1) {
            return Collections.singleton(a[0]);
        }
        return Collections.unmodifiableSet(asLinkedHashSet(a));
    }

    /**
     * 配列から不可変の固定リストを内部実装とするコレクション生成して返却します。
     * 
     * @param a
     *            コレクションの要素となる配列
     * @return 不可変の固定コレクション
     */
    public static Collection fixedListCollection(Object[] a) {
        return Collections.unmodifiableCollection(asArrayList(a));
    }

    /**
     * 配列から不可変の固定セットを内部実装とするコレクション生成して返却します。
     * 
     * @param a
     *            コレクションの要素となる配列
     * @return 不可変の固定コレクション
     */
    public static Collection fixedSetCollection(Object[] a) {
        return Collections.unmodifiableCollection(asLinkedHashSet(a));
    }

    /**
     * 配列から不可変の重複のない固定リストを内部実装とするコレクション生成して返却します。<br>
     * この固定コレクションの反復処理は、fixedSetCollection(Object[]) より高速です。<br>
     * ただし Collection.contains(o) オペレーションは遅い可能性があります。<br>
     * これは fixedSetCollection(Object[]) で返されるセット実装がハッシュ構造の検索ロジックを提供するためです。<br>
     * コレクションを使用する用途が反復処理中心の場合はこのメソッドを選択すると良いと思われます。
     * 
     * @param a
     *            コレクションの要素となる配列
     * @return 不可変の重複のない固定コレクション
     */
    public static Collection fixedSetArrayCollection(Object[] a) {
        return Collections.unmodifiableCollection(asSetArrayList(a));
    }

    /**
     * 配列から重複のない java.util.ArrayList を生成して返却します。
     * 
     * @param a
     *            リストの要素となる配列
     * @return 重複のない配列リスト
     */
    public static ArrayList asSetArrayList(Object[] a) {
        return asArrayList(asHashSet(a).toArray());
    }

    /**
     * 配列から java.util.ArrayList を生成して返却します。
     * 
     * @param a
     *            リストの要素となる配列
     * @return 配列リスト
     */
    public static ArrayList asArrayList(Object[] a) {
        return new ArrayList(Arrays.asList(a));
    }

    /**
     * 配列から java.util.LinkedList を生成して返却します。
     * 
     * @param a
     *            リストの要素となる配列
     * @return 双方向循環リンクリスト
     */
    public static LinkedList asLinkedList(Object[] a) {
        return new LinkedList(Arrays.asList(a));
    }

    /**
     * 配列から java.util.HashSet を生成して返却します。
     * 
     * @param a
     *            セットの要素となる配列
     * @return ハッシュセット
     */
    public static HashSet asHashSet(Object[] a) {
        return new HashSet(Arrays.asList(a));
    }

    /**
     * 配列から java.util.LinkedHashSet を生成して返却します。
     * 
     * @param a
     *            セットの要素となる配列
     * @return リンク構造でエントリの順序を保持するハッシュセット
     */
    public static LinkedHashSet asLinkedHashSet(Object[] a) {
        return new LinkedHashSet(Arrays.asList(a));
    }

    /**
     * キーと値を交互に格納する配列から java.util.HashMap を生成して返却します。
     * 
     * @param keyValue
     *            キーと値を交互に格納する配列
     * @return ハッシュマップ
     */
    public static HashMap asHashMap(Object[] keyValue) {
        final int size = keyValue.length;
        if ((size % 2) != 0) {
            throw new IllegalArgumentException("keyValue size err ((keyValue.size() % 2) != 0).");
        }
        final HashMap map = new HashMap(size / 2);
        for (int i = 0; i < size; i += 2) {
            map.put(keyValue[i], keyValue[i + 1]);
        }
        return map;
    }

    /**
     * キーと値を交互に格納する配列から java.util.LinkedHashMap を生成して返却します。
     * 
     * @param keyValue
     *            キーと値を交互に格納する配列
     * @return リンク構造でエントリの順序を保持するハッシュマップ
     */
    public static LinkedHashMap asLinkedHashMap(Object[] keyValue) {
        final int size = keyValue.length;
        if ((size % 2) != 0) {
            throw new IllegalArgumentException("keyValue size err ((keyValue.size() % 2) != 0).");
        }
        final LinkedHashMap map = new LinkedHashMap(size / 2);
        for (int i = 0; i < size; i += 2) {
            map.put(keyValue[i], keyValue[i + 1]);
        }
        return map;
    }

    /**
     * 引数から配列を生成して返却します。
     * 
     * @param o1
     *            要素１
     * @param o2
     *            要素２
     * @param o3
     *            要素３
     * @param o4
     *            要素４
     * @param o5
     *            要素５
     * @param o6
     *            要素６
     * @param o7
     *            要素７
     * @param o8
     *            要素８
     * @param o9
     *            要素９
     * @param o10
     *            要素１０
     * @param o11
     *            要素１１
     * @param o12
     *            要素１２
     * @return 配列
     */
    public static Object[] asArray(Object o1, Object o2, Object o3, Object o4, Object o5, Object o6, Object o7, Object o8, Object o9, Object o10, Object o11, Object o12) {
        return new Object[] { o1, o2, o3, o4, o5, o6, o7, o8, o9, o10, o11, o12 };
    }

    /**
     * 引数から配列を生成して返却します。
     * 
     * @param o1
     *            要素１
     * @param o2
     *            要素２
     * @param o3
     *            要素３
     * @param o4
     *            要素４
     * @param o5
     *            要素５
     * @param o6
     *            要素６
     * @param o7
     *            要素７
     * @param o8
     *            要素８
     * @param o9
     *            要素９
     * @param o10
     *            要素１０
     * @param o11
     *            要素１１
     * @return 配列
     */
    public static Object[] asArray(Object o1, Object o2, Object o3, Object o4, Object o5, Object o6, Object o7, Object o8, Object o9, Object o10, Object o11) {
        return new Object[] { o1, o2, o3, o4, o5, o6, o7, o8, o9, o10, o11 };
    }

    /**
     * 引数から配列を生成して返却します。
     * 
     * @param o1
     *            要素１
     * @param o2
     *            要素２
     * @param o3
     *            要素３
     * @param o4
     *            要素４
     * @param o5
     *            要素５
     * @param o6
     *            要素６
     * @param o7
     *            要素７
     * @param o8
     *            要素８
     * @param o9
     *            要素９
     * @param o10
     *            要素１０
     * @return 配列
     */
    public static Object[] asArray(Object o1, Object o2, Object o3, Object o4, Object o5, Object o6, Object o7, Object o8, Object o9, Object o10) {
        return new Object[] { o1, o2, o3, o4, o5, o6, o7, o8, o9, o10 };
    }

    /**
     * 引数から配列を生成して返却します。
     * 
     * @param o1
     *            要素１
     * @param o2
     *            要素２
     * @param o3
     *            要素３
     * @param o4
     *            要素４
     * @param o5
     *            要素５
     * @param o6
     *            要素６
     * @param o7
     *            要素７
     * @param o8
     *            要素８
     * @param o9
     *            要素９
     * @return 配列
     */
    public static Object[] asArray(Object o1, Object o2, Object o3, Object o4, Object o5, Object o6, Object o7, Object o8, Object o9) {
        return new Object[] { o1, o2, o3, o4, o5, o6, o7, o8, o9 };
    }

    /**
     * 引数から配列を生成して返却します。
     * 
     * @param o1
     *            要素１
     * @param o2
     *            要素２
     * @param o3
     *            要素３
     * @param o4
     *            要素４
     * @param o5
     *            要素５
     * @param o6
     *            要素６
     * @param o7
     *            要素７
     * @param o8
     *            要素８
     * @return 配列
     */
    public static Object[] asArray(Object o1, Object o2, Object o3, Object o4, Object o5, Object o6, Object o7, Object o8) {
        return new Object[] { o1, o2, o3, o4, o5, o6, o7, o8 };
    }

    /**
     * 引数から配列を生成して返却します。
     * 
     * @param o1
     *            要素１
     * @param o2
     *            要素２
     * @param o3
     *            要素３
     * @param o4
     *            要素４
     * @param o5
     *            要素５
     * @param o6
     *            要素６
     * @param o7
     *            要素７
     * @return 配列
     */
    public static Object[] asArray(Object o1, Object o2, Object o3, Object o4, Object o5, Object o6, Object o7) {
        return new Object[] { o1, o2, o3, o4, o5, o6, o7 };
    }

    /**
     * 引数から配列を生成して返却します。
     * 
     * @param o1
     *            要素１
     * @param o2
     *            要素２
     * @param o3
     *            要素３
     * @param o4
     *            要素４
     * @param o5
     *            要素５
     * @param o6
     *            要素６
     * @return 配列
     */
    public static Object[] asArray(Object o1, Object o2, Object o3, Object o4, Object o5, Object o6) {
        return new Object[] { o1, o2, o3, o4, o5, o6 };
    }

    /**
     * 引数から配列を生成して返却します。
     * 
     * @param o1
     *            要素１
     * @param o2
     *            要素２
     * @param o3
     *            要素３
     * @param o4
     *            要素４
     * @param o5
     *            要素５
     * @return 配列
     */
    public static Object[] asArray(Object o1, Object o2, Object o3, Object o4, Object o5) {
        return new Object[] { o1, o2, o3, o4, o5 };
    }

    /**
     * 引数から配列を生成して返却します。
     * 
     * @param o1
     *            要素１
     * @param o2
     *            要素２
     * @param o3
     *            要素３
     * @param o4
     *            要素４
     * @return 配列
     */
    public static Object[] asArray(Object o1, Object o2, Object o3, Object o4) {
        return new Object[] { o1, o2, o3, o4 };
    }

    /**
     * 引数から配列を生成して返却します。
     * 
     * @param o1
     *            要素１
     * @param o2
     *            要素２
     * @param o3
     *            要素３
     * @return 配列
     */
    public static Object[] asArray(Object o1, Object o2, Object o3) {
        return new Object[] { o1, o2, o3 };
    }

    /**
     * 引数から配列を生成して返却します。
     * 
     * @param o1
     *            要素１
     * @param o2
     *            要素２
     * @return 配列
     */
    public static Object[] asArray(Object o1, Object o2) {
        return new Object[] { o1, o2 };
    }

    /**
     * 引数から配列を生成して返却します。
     * 
     * @param o1
     *            要素１
     * @return 配列
     */
    public static Object[] asArray(Object o1) {
        return new Object[] { o1 };
    }

    /**
     * サイズ０の空配列を生成して返却します。
     * 
     * @return サイズ０の空配列
     */
    public static Object[] asArray() {
        return new Object[0];
    }

    /*
     * 日付・時刻の構築
     */

    /**
     * カレンダーを生成して返却します。<br>
     * 月の指定が 1 から始まることに注意が必要です。
     * 
     * @param year
     *            年 (-9999 - 9999)
     * @param month
     *            月 (1 - 12)
     * @param date
     *            日 (1 - 31)
     * @param hour
     *            時 (0 - 23)
     * @param minute
     *            分 (0 - 59)
     * @param second
     *            秒 (0 - 59)
     * @param milliSecond
     *            ミリ秒 (0 - 999)
     * @param zoneOffset
     *            オフセット時刻
     * @return カレンダー
     */
    public static Calendar getDateTimeCalendar(int year, int month, int date, int hour, int minute, int second, int milliSecond, int zoneOffset) {
        final Calendar cal = new GregorianCalendar();
        cal.clear();
        cal.set(Calendar.YEAR, year);
        cal.set(Calendar.MONTH, (month - 1));
        cal.set(Calendar.DAY_OF_MONTH, date);
        cal.set(Calendar.HOUR_OF_DAY, hour);
        cal.set(Calendar.MINUTE, minute);
        cal.set(Calendar.SECOND, second);
        cal.set(Calendar.MILLISECOND, milliSecond);
        cal.set(Calendar.ZONE_OFFSET, zoneOffset);
        return cal;
    }

    /**
     * カレンダーを生成して返却します。<br>
     * 月の指定が 1 から始まることに注意が必要です。
     * 
     * @param year
     *            年 (-9999 - 9999)
     * @param month
     *            月 (1 - 12)
     * @param date
     *            日 (1 - 31)
     * @param hour
     *            時 (0 - 23)
     * @param minute
     *            分 (0 - 59)
     * @param second
     *            秒 (0 - 59)
     * @param milliSecond
     *            ミリ秒 (0 - 999)
     * @return カレンダー
     */
    public static Calendar getDateTimeCalendar(int year, int month, int date, int hour, int minute, int second, int milliSecond) {
        return getDateTimeCalendar(year, month, date, hour, minute, second, milliSecond, TimeZone.getDefault().getRawOffset());
    }

    /**
     * カレンダーを生成して返却します。<br>
     * 月の指定が 1 から始まることに注意が必要です。
     * 
     * @param year
     *            年 (-9999 - 9999)
     * @param month
     *            月 (1 - 12)
     * @param date
     *            日 (1 - 31)
     * @param hour
     *            時 (0 - 23)
     * @param minute
     *            分 (0 - 59)
     * @param second
     *            秒 (0 - 59)
     * @return カレンダー
     */
    public static Calendar getDateTimeCalendar(int year, int month, int date, int hour, int minute, int second) {
        return getDateTimeCalendar(year, month, date, hour, minute, second, 0, TimeZone.getDefault().getRawOffset());
    }

    /**
     * 日時のみ示す、カレンダーを生成して返却します。<br>
     * 日付部分には、1970 年 1 月 1 日（グリニッジ標準時）が割り当てられます。
     * 
     * @param hour
     *            時 (0 - 23)
     * @param minute
     *            分 (0 - 59)
     * @param second
     *            秒 (0 - 59)
     * @param milliSecond
     *            ミリ秒 (0 - 999)
     * @param zoneOffset
     *            オフセット時刻
     * @return カレンダー
     */
    public static Calendar getTimeCalendar(int hour, int minute, int second, int milliSecond, int zoneOffset) {
        return getDateTimeCalendar(1970, 1, 1, hour, minute, second, milliSecond, zoneOffset);
    }

    /**
     * 日時のみ示す、カレンダーを生成して返却します。<br>
     * 日付部分には、1970 年 1 月 1 日（グリニッジ標準時）が割り当てられます。
     * 
     * @param hour
     *            時 (0 - 23)
     * @param minute
     *            分 (0 - 59)
     * @param second
     *            秒 (0 - 59)
     * @param milliSecond
     *            ミリ秒 (0 - 999)
     * @return カレンダー
     */
    public static Calendar getTimeCalendar(int hour, int minute, int second, int milliSecond) {
        return getDateTimeCalendar(1970, 1, 1, hour, minute, second, milliSecond, TimeZone.getDefault().getRawOffset());
    }

    /**
     * 日時のみ示す、カレンダーを生成して返却します。<br>
     * 日付部分には、1970 年 1 月 1 日（グリニッジ標準時）が割り当てられます。
     * 
     * @param hour
     *            時 (0 - 23)
     * @param minute
     *            分 (0 - 59)
     * @param second
     *            秒 (0 - 59)
     * @return カレンダー
     */
    public static Calendar getTimeCalendar(int hour, int minute, int second) {
        return getDateTimeCalendar(1970, 1, 1, hour, minute, second, 0, TimeZone.getDefault().getRawOffset());
    }

    /**
     * カレンダーを生成して返却します。<br>
     * 月の指定が 1 から始まることに注意が必要です。
     * 
     * @param year
     *            年 (-9999 - 9999)
     * @param month
     *            月 (1 - 12)
     * @param date
     *            日 (1 - 31)
     * @return カレンダー
     */
    public static Calendar getDateCalendar(int year, int month, int date) {
        return getDateTimeCalendar(year, month, date, 0, 0, 0, 0, TimeZone.getDefault().getRawOffset());
    }

    /**
     * カレンダーを生成して返却します。<br>
     * 月の指定が 1 から始まることに注意が必要です。
     * 
     * @param year
     *            年 (-9999 - 9999)
     * @param month
     *            月 (1 - 12)
     * @param date
     *            日 (1 - 31)
     * @param zoneOffset
     *            オフセット時刻
     * @return カレンダー
     */
    public static Calendar getDateCalendar(int year, int month, int date, int zoneOffset) {
        return getDateTimeCalendar(year, month, date, 0, 0, 0, 0, zoneOffset);
    }

    /**
     * 日付オブジェクトを生成して返却します。<br>
     * 月の指定が 1 から始まることに注意が必要です。
     * 
     * @param year
     *            年 (-9999 - 9999)
     * @param month
     *            月 (1 - 12)
     * @param date
     *            日 (1 - 31)
     * @param hour
     *            時 (0 - 23)
     * @param minute
     *            分 (0 - 59)
     * @param second
     *            秒 (0 - 59)
     * @param milliSecond
     *            ミリ秒 (0 - 999)
     * @param zoneOffset
     *            オフセット時刻
     * @return 日付オブジェクト
     */
    public static Date getDateTime(int year, int month, int date, int hour, int minute, int second, int milliSecond, int zoneOffset) {
        final Calendar cal = getDateTimeCalendar(year, month, date, hour, minute, second, milliSecond, zoneOffset);
        return cal.getTime();
    }

    /**
     * 日付オブジェクトを生成して返却します。<br>
     * 月の指定が 1 から始まることに注意が必要です。
     * 
     * @param year
     *            年 (-9999 - 9999)
     * @param month
     *            月 (1 - 12)
     * @param date
     *            日 (1 - 31)
     * @param hour
     *            時 (0 - 23)
     * @param minute
     *            分 (0 - 59)
     * @param second
     *            秒 (0 - 59)
     * @param milliSecond
     *            ミリ秒 (0 - 999)
     * @return 日付オブジェクト
     */
    public static Date getDateTime(int year, int month, int date, int hour, int minute, int second, int milliSecond) {
        return getDateTime(year, month, date, hour, minute, second, milliSecond, TimeZone.getDefault().getRawOffset());
    }

    /**
     * 日付オブジェクトを生成して返却します。<br>
     * 月の指定が 1 から始まることに注意が必要です。
     * 
     * @param year
     *            年 (-9999 - 9999)
     * @param month
     *            月 (1 - 12)
     * @param date
     *            日 (1 - 31)
     * @param hour
     *            時 (0 - 23)
     * @param minute
     *            分 (0 - 59)
     * @param second
     *            秒 (0 - 59)
     * @return 日付オブジェクト
     */
    public static Date getDateTime(int year, int month, int date, int hour, int minute, int second) {
        return getDateTime(year, month, date, hour, minute, second, 0, TimeZone.getDefault().getRawOffset());
    }

    /**
     * 日時のみ示す、日付オブジェクトを生成して返却します。<br>
     * 日付部分には、1970 年 1 月 1 日（グリニッジ標準時）が割り当てられます。
     * 
     * @param hour
     *            時 (0 - 23)
     * @param minute
     *            分 (0 - 59)
     * @param second
     *            秒 (0 - 59)
     * @param milliSecond
     *            ミリ秒 (0 - 999)
     * @param zoneOffset
     *            オフセット時刻
     * @return 日付オブジェクト
     */
    public static Date getTime(int hour, int minute, int second, int milliSecond, int zoneOffset) {
        return getDateTime(1970, 1, 1, hour, minute, second, milliSecond, zoneOffset);
    }

    /**
     * 日時のみ示す、日付オブジェクトを生成して返却します。<br>
     * 日付部分には、1970 年 1 月 1 日（グリニッジ標準時）が割り当てられます。
     * 
     * @param hour
     *            時 (0 - 23)
     * @param minute
     *            分 (0 - 59)
     * @param second
     *            秒 (0 - 59)
     * @param milliSecond
     *            ミリ秒 (0 - 999)
     * @return 日付オブジェクト
     */
    public static Date getTime(int hour, int minute, int second, int milliSecond) {
        return getDateTime(1970, 1, 1, hour, minute, second, milliSecond, TimeZone.getDefault().getRawOffset());
    }

    /**
     * 日時のみ示す、日付オブジェクトを生成して返却します。<br>
     * 日付部分には、1970 年 1 月 1 日（グリニッジ標準時）が割り当てられます。
     * 
     * @param hour
     *            時 (0 - 23)
     * @param minute
     *            分 (0 - 59)
     * @param second
     *            秒 (0 - 59)
     * @return 日付オブジェクト
     */
    public static Date getTime(int hour, int minute, int second) {
        return getDateTime(1970, 1, 1, hour, minute, second, 0, TimeZone.getDefault().getRawOffset());
    }

    /**
     * 日付オブジェクトを生成して返却します。<br>
     * 月の指定が 1 から始まることに注意が必要です。
     * 
     * @param year
     *            年 (-9999 - 9999)
     * @param month
     *            月 (1 - 12)
     * @param date
     *            日 (1 - 31)
     * @return 日付オブジェクト
     */
    public static Date getDate(int year, int month, int date) {
        return getDateTime(year, month, date, 0, 0, 0, 0, TimeZone.getDefault().getRawOffset());
    }

    /**
     * 日付オブジェクトを生成して返却します。<br>
     * 月の指定が 1 から始まることに注意が必要です。
     * 
     * @param year
     *            年 (-9999 - 9999)
     * @param month
     *            月 (1 - 12)
     * @param date
     *            日 (1 - 31)
     * @param zoneOffset
     *            オフセット時刻
     * @return 日付オブジェクト
     */
    public static Date getDate(int year, int month, int date, int zoneOffset) {
        return getDateTime(year, month, date, 0, 0, 0, 0, zoneOffset);
    }

    /**
     * 日付データをミリ秒で示す Long 値で生成して返却します。<br>
     * 定数のなどの不変データを定義する場合は、日付オブジェクトよりも安全です。<br>
     * また月の指定が 1 から始まることに注意が必要です。
     * 
     * @param year
     *            年 (-9999 - 9999)
     * @param month
     *            月 (1 - 12)
     * @param date
     *            日 (1 - 31)
     * @param hour
     *            時 (0 - 23)
     * @param minute
     *            分 (0 - 59)
     * @param second
     *            秒 (0 - 59)
     * @param milliSecond
     *            ミリ秒 (0 - 999)
     * @param zoneOffset
     *            オフセット時刻
     * @return 日付データをミリ秒で示す Long 値
     */
    public static Long getDateTimeInMillis(int year, int month, int date, int hour, int minute, int second, int milliSecond, int zoneOffset) {
        final Calendar cal = getDateTimeCalendar(year, month, date, hour, minute, second, milliSecond, zoneOffset);
        return Boxing.box(cal.getTimeInMillis());
    }

    /**
     * 日付データをミリ秒で示す Long 値で生成して返却します。<br>
     * 定数のなどの不変データを定義する場合は、日付オブジェクトよりも安全です。<br>
     * また月の指定が 1 から始まることに注意が必要です。
     * 
     * @param year
     *            年 (-9999 - 9999)
     * @param month
     *            月 (1 - 12)
     * @param date
     *            日 (1 - 31)
     * @param hour
     *            時 (0 - 23)
     * @param minute
     *            分 (0 - 59)
     * @param second
     *            秒 (0 - 59)
     * @param milliSecond
     *            ミリ秒 (0 - 999)
     * @return 日付データをミリ秒で示す Long 値
     */
    public static Long getDateTimeInMillis(int year, int month, int date, int hour, int minute, int second, int milliSecond) {
        return getDateTimeInMillis(year, month, date, hour, minute, second, milliSecond, TimeZone.getDefault().getRawOffset());
    }

    /**
     * 日付データをミリ秒で示す Long 値で生成して返却します。<br>
     * 定数のなどの不変データを定義する場合は、日付オブジェクトよりも安全です。<br>
     * また月の指定が 1 から始まることに注意が必要です。
     * 
     * @param year
     *            年 (-9999 - 9999)
     * @param month
     *            月 (1 - 12)
     * @param date
     *            日 (1 - 31)
     * @param hour
     *            時 (0 - 23)
     * @param minute
     *            分 (0 - 59)
     * @param second
     *            秒 (0 - 59)
     * @return 日付データをミリ秒で示す Long 値
     */
    public static Long getDateTimeInMillis(int year, int month, int date, int hour, int minute, int second) {
        return getDateTimeInMillis(year, month, date, hour, minute, second, 0, TimeZone.getDefault().getRawOffset());
    }

    /**
     * 日時のみ示す、日付データをミリ秒で示す Long 値で生成して返却します。<br>
     * 日付部分には、1970 年 1 月 1 日（グリニッジ標準時）が割り当てられます。<br>
     * 定数のなどの不変データを定義する場合は、日付オブジェクトよりも安全です。
     * 
     * @param hour
     *            時 (0 - 23)
     * @param minute
     *            分 (0 - 59)
     * @param second
     *            秒 (0 - 59)
     * @param milliSecond
     *            ミリ秒 (0 - 999)
     * @param zoneOffset
     *            オフセット時刻
     * @return 日付データをミリ秒で示す Long 値
     */
    public static Long getTimeInMillis(int hour, int minute, int second, int milliSecond, int zoneOffset) {
        return getDateTimeInMillis(1970, 1, 1, hour, minute, second, milliSecond, zoneOffset);
    }

    /**
     * 日時のみ示す、日付データをミリ秒で示す Long 値で生成して返却します。<br>
     * 日付部分には、1970 年 1 月 1 日（グリニッジ標準時）が割り当てられます。<br>
     * 定数のなどの不変データを定義する場合は、日付オブジェクトよりも安全です。
     * 
     * @param hour
     *            時 (0 - 23)
     * @param minute
     *            分 (0 - 59)
     * @param second
     *            秒 (0 - 59)
     * @param milliSecond
     *            ミリ秒 (0 - 999)
     * @return 日付データをミリ秒で示す Long 値
     */
    public static Long getTimeInMillis(int hour, int minute, int second, int milliSecond) {
        return getDateTimeInMillis(1970, 1, 1, hour, minute, second, milliSecond, TimeZone.getDefault().getRawOffset());
    }

    /**
     * 日時のみ示す、日付データをミリ秒で示す Long 値で生成して返却します。<br>
     * 日付部分には、1970 年 1 月 1 日（グリニッジ標準時）が割り当てられます。<br>
     * 定数のなどの不変データを定義する場合は、日付オブジェクトよりも安全です。
     * 
     * @param hour
     *            時 (0 - 23)
     * @param minute
     *            分 (0 - 59)
     * @param second
     *            秒 (0 - 59)
     * @return 日付データをミリ秒で示す Long 値
     */
    public static Long getTimeInMillis(int hour, int minute, int second) {
        return getDateTimeInMillis(1970, 1, 1, hour, minute, second, 0, TimeZone.getDefault().getRawOffset());
    }

    /**
     * 日付データをミリ秒で示す Long 値で生成して返却します。<br>
     * 定数のなどの不変データを定義する場合は、日付オブジェクトよりも安全です。<br>
     * また月の指定が 1 から始まることに注意が必要です。
     * 
     * @param year
     *            年 (-9999 - 9999)
     * @param month
     *            月 (1 - 12)
     * @param date
     *            日 (1 - 31)
     * @return 日付データをミリ秒で示す Long 値
     */
    public static Long getDateInMillis(int year, int month, int date) {
        return getDateTimeInMillis(year, month, date, 0, 0, 0, 0, TimeZone.getDefault().getRawOffset());
    }

    /**
     * 日付データをミリ秒で示す Long 値で生成して返却します。<br>
     * 定数のなどの不変データを定義する場合は、日付オブジェクトよりも安全です。<br>
     * また月の指定が 1 から始まることに注意が必要です。
     * 
     * @param year
     *            年 (-9999 - 9999)
     * @param month
     *            月 (1 - 12)
     * @param date
     *            日 (1 - 31)
     * @param zoneOffset
     *            オフセット時刻
     * @return 日付データをミリ秒で示す Long 値
     */
    public static Long getDateInMillis(int year, int month, int date, int zoneOffset) {
        return getDateTimeInMillis(year, month, date, 0, 0, 0, 0, zoneOffset);
    }

}
