/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.lang;

import java.util.Collection;
import java.util.Iterator;

/**
 * 複数の値の正否を評価するショートカット関数群を提供します。
 */
public class EvalSet {

    /*
     * Empty
     */

    /**
     * 引数が全て null または空の文字シーケンスであるかを評価します。
     * 
     * @param cs
     *            評価する一番目の文字シーケンス
     * @param cs2
     *            評価する二番目の文字シーケンス
     * @return 引数が全て null または空の場合は true
     */
    public static boolean isAndEmpty(CharSequence cs, CharSequence cs2) {
        return (Eval.isEmpty(cs) && Eval.isEmpty(cs2));
    }

    /**
     * 引数が全て null または空の文字シーケンスであるかを評価します。
     * 
     * @param cs
     *            評価する一番目の文字シーケンス
     * @param cs2
     *            評価する二番目の文字シーケンス
     * @param cs3
     *            評価する三番目の文字シーケンス
     * @return 引数が全て null または空の場合は true
     */
    public static boolean isAndEmpty(CharSequence cs, CharSequence cs2, CharSequence cs3) {
        return (Eval.isEmpty(cs) && Eval.isEmpty(cs2) && Eval.isEmpty(cs3));
    }

    /**
     * 全ての配列の要素が null または空の文字シーケンスであるかを評価します。<br>
     * 要素は全て文字シーケンス（CharSequence）である必要があります。
     * 
     * @param seqs
     *            評価する配列
     * @return 全要素が null または空の場合は true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isAndEmpty(Object[] seqs) {
        for (int i = 0; i < seqs.length; i++) {
            if (!Eval.isEmpty((CharSequence) seqs[i])) {
                return false;
            }
        }
        return true;
    }

    /**
     * 全てのコレクションの要素が null または空の文字シーケンスであるかを評価します。<br>
     * 要素は全て文字シーケンス（CharSequence）である必要があります。
     * 
     * @param seqs
     *            評価するコレクション
     * @return 全要素が null または空の場合は true
     * @throws NullPointerException
     *             引数のコレクションが null の場合
     */
    public static boolean isAndEmpty(Collection seqs) {
        for (Iterator i = seqs.iterator(); i.hasNext();) {
            if (!Eval.isEmpty((CharSequence) i.next())) {
                return false;
            }
        }
        return true;
    }

    /**
     * 引数に null または空の文字シーケンスが含まれるか評価します。
     * 
     * @param cs
     *            評価する一番目の文字シーケンス
     * @param cs2
     *            評価する二番目の文字シーケンス
     * @return 引数に null または空の値が含まれる場合は true
     */
    public static boolean isOrEmpty(CharSequence cs, CharSequence cs2) {
        return (Eval.isEmpty(cs) || Eval.isEmpty(cs2));
    }

    /**
     * 引数に null または空の文字シーケンスが含まれるか評価します。
     * 
     * @param cs
     *            評価する一番目の文字シーケンス
     * @param cs2
     *            評価する二番目の文字シーケンス
     * @param cs3
     *            評価する三番目の文字シーケンス
     * @return 引数に null または空の値が含まれる場合は true
     */
    public static boolean isOrEmpty(CharSequence cs, CharSequence cs2, CharSequence cs3) {
        return (Eval.isEmpty(cs) || Eval.isEmpty(cs2) || Eval.isEmpty(cs3));
    }

    /**
     * 全ての配列の要素が null または空の文字シーケンスが含まれるか評価します。<br>
     * 要素は全て文字シーケンス（CharSequence）である必要があります。
     * 
     * @param seqs
     *            評価する配列
     * @return null または空の値が含まれる場合は true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isOrEmpty(Object[] seqs) {
        for (int i = 0; i < seqs.length; i++) {
            if (Eval.isEmpty((CharSequence) seqs[i])) {
                return true;
            }
        }
        return false;
    }

    /**
     * 全てのコレクションの要素が null または空の文字シーケンスが含まれるか評価します。<br>
     * 要素は全て文字シーケンス（CharSequence）である必要があります。
     * 
     * @param seqs
     *            評価するコレクション
     * @return null または空の値が含まれる場合は true
     * @throws NullPointerException
     *             引数のコレクションが null の場合
     */
    public static boolean isOrEmpty(Collection seqs) {
        for (Iterator i = seqs.iterator(); i.hasNext();) {
            if (Eval.isEmpty((CharSequence) i.next())) {
                return true;
            }
        }
        return false;
    }

    /*
     * Blank
     */

    /**
     * 引数が全て null または空かスペースのみの文字シーケンスであるかを評価します。
     * 
     * @param cs
     *            評価する一番目の文字シーケンス
     * @param cs2
     *            評価する二番目の文字シーケンス
     * @return 引数が全て null または空かスペースのみの場合は true
     */
    public static boolean isAndBlank(CharSequence cs, CharSequence cs2) {
        return (Eval.isBlank(cs) && Eval.isBlank(cs2));
    }

    /**
     * 引数が全て null または空かスペースのみの文字シーケンスであるかを評価します。
     * 
     * @param cs
     *            評価する一番目の文字シーケンス
     * @param cs2
     *            評価する二番目の文字シーケンス
     * @param cs3
     *            評価する三番目の文字シーケンス
     * @return 引数が全て null または空かスペースのみの場合は true
     */
    public static boolean isAndBlank(CharSequence cs, CharSequence cs2, CharSequence cs3) {
        return (Eval.isBlank(cs) && Eval.isBlank(cs2) && Eval.isBlank(cs3));
    }

    /**
     * 全ての配列の要素が null または空かスペースのみの文字シーケンスであるかを評価します。<br>
     * 要素は全て文字シーケンス（CharSequence）である必要があります。
     * 
     * @param seqs
     *            評価する配列
     * @return 全要素が null または空かスペースのみの場合は true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isAndBlank(Object[] seqs) {
        for (int i = 0; i < seqs.length; i++) {
            if (!Eval.isBlank((CharSequence) seqs[i])) {
                return false;
            }
        }
        return true;
    }

    /**
     * 全てのコレクションの要素が null または空かスペースのみの文字シーケンスであるかを評価します。<br>
     * 要素は全て文字シーケンス（CharSequence）である必要があります。
     * 
     * @param seqs
     *            評価するコレクション
     * @return 全要素が null または空かスペースのみの場合は true
     * @throws NullPointerException
     *             引数のコレクションが null の場合
     */
    public static boolean isAndBlank(Collection seqs) {
        for (Iterator i = seqs.iterator(); i.hasNext();) {
            if (!Eval.isBlank((CharSequence) i.next())) {
                return false;
            }
        }
        return true;
    }

    /**
     * 引数に null または空かスペースのみの文字シーケンスが含まれるか評価します。
     * 
     * @param cs
     *            評価する一番目の文字シーケンス
     * @param cs2
     *            評価する二番目の文字シーケンス
     * @return 引数に null または空かスペースのみの値が含まれる場合は true
     */
    public static boolean isOrBlank(CharSequence cs, CharSequence cs2) {
        return (Eval.isBlank(cs) || Eval.isBlank(cs2));
    }

    /**
     * 引数に null または空かスペースのみの文字シーケンスが含まれるか評価します。
     * 
     * @param cs
     *            評価する一番目の文字シーケンス
     * @param cs2
     *            評価する二番目の文字シーケンス
     * @param cs3
     *            評価する三番目の文字シーケンス
     * @return 引数に null または空かスペースのみの値が含まれる場合は true
     */
    public static boolean isOrBlank(CharSequence cs, CharSequence cs2, CharSequence cs3) {
        return (Eval.isBlank(cs) || Eval.isBlank(cs2) || Eval.isBlank(cs3));
    }

    /**
     * 全ての配列の要素が null または空かスペースのみの文字シーケンスが含まれるか評価します。<br>
     * 要素は全て文字シーケンス（CharSequence）である必要があります。
     * 
     * @param seqs
     *            評価する配列
     * @return null または空かスペースのみの要素が含まれる場合は true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isOrBlank(Object[] seqs) {
        for (int i = 0; i < seqs.length; i++) {
            if (Eval.isBlank((CharSequence) seqs[i])) {
                return true;
            }
        }
        return false;
    }

    /**
     * 全てのコレクションの要素が null または空かスペースのみの文字シーケンスが含まれるか評価します。<br>
     * 要素は全て文字シーケンス（CharSequence）である必要があります。
     * 
     * @param seqs
     *            評価するコレクション
     * @return null または空かスペースのみの要素が含まれる場合は true
     * @throws NullPointerException
     *             引数のコレクションが null の場合
     */
    public static boolean isOrBlank(Collection seqs) {
        for (Iterator i = seqs.iterator(); i.hasNext();) {
            if (Eval.isBlank((CharSequence) i.next())) {
                return true;
            }
        }
        return false;
    }

    /*
     * Null
     */

    /**
     * 引数が全て null 値であるかを評価します。
     * 
     * @param o
     *            評価する一番目の値
     * @param o2
     *            評価する二番目の値
     * @return 引数が全て null 値の場合は true
     */
    public static boolean isAndNull(Object o, Object o2) {
        return (o == null && o2 == null);
    }

    /**
     * 引数が全て null 値であるかを評価します。
     * 
     * @param o
     *            評価する一番目の値
     * @param o2
     *            評価する二番目の値
     * @param o3
     *            評価する三番目の値
     * @return 引数が全て null 値の場合は true
     */
    public static boolean isAndNull(Object o, Object o2, Object o3) {
        return (o == null && o2 == null && o3 == null);
    }

    /**
     * 引数が全て null 値であるかを評価します。
     * 
     * @param o
     *            評価する一番目の値
     * @param o2
     *            評価する二番目の値
     * @param o3
     *            評価する三番目の値
     * @param o4
     *            評価する四番目の値
     * @return 引数が全て null 値の場合は true
     */
    public static boolean isAndNull(Object o, Object o2, Object o3, Object o4) {
        return (o == null && o2 == null && o3 == null && o4 == null);
    }

    /**
     * 全ての配列の要素が null 値であるかを評価します。
     * 
     * @param objs
     *            評価する配列
     * @return 全要素が null 値の場合は true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isAndNull(Object[] objs) {
        for (int i = 0; i < objs.length; i++) {
            if (objs[i] != null) {
                return false;
            }
        }
        return true;
    }

    /**
     * 全てのコレクションの要素が null 値であるかを評価します。
     * 
     * @param objs
     *            評価するコレクション
     * @return 全要素が null 値の場合は true
     * @throws NullPointerException
     *             引数のコレクションが null の場合
     */
    public static boolean isAndNull(Collection objs) {
        for (Iterator i = objs.iterator(); i.hasNext();) {
            if (i.next() != null) {
                return false;
            }
        }
        return true;
    }

    /**
     * 引数に null 値が含まれるか評価します。
     * 
     * @param o
     *            評価する一番目の値
     * @param o2
     *            評価する二番目の値
     * @return 引数に null 値が含まれる場合は true
     */
    public static boolean isOrNull(Object o, Object o2) {
        return (o == null || o2 == null);
    }

    /**
     * 引数に null 値が含まれるか評価します。
     * 
     * @param o
     *            評価する一番目の値
     * @param o2
     *            評価する二番目の値
     * @param o3
     *            評価する三番目の値
     * @return 引数に null 値が含まれる場合は true
     */
    public static boolean isOrNull(Object o, Object o2, Object o3) {
        return (o == null || o2 == null || o3 == null);
    }

    /**
     * 引数に null 値が含まれるか評価します。
     * 
     * @param o
     *            評価する一番目の値
     * @param o2
     *            評価する二番目の値
     * @param o3
     *            評価する三番目の値
     * @param o4
     *            評価する四番目の値
     * @return 引数に null 値が含まれる場合は true
     */
    public static boolean isOrNull(Object o, Object o2, Object o3, Object o4) {
        return (o == null || o2 == null || o3 == null || o4 == null);
    }

    /**
     * 配列の要素に null 値が含まれるか評価します。
     * 
     * @param objs
     *            評価する配列
     * @return 要素に null 値が含まれる場合は true
     * @throws NullPointerException
     *             引数の配列が null の場合
     */
    public static boolean isOrNull(Object[] objs) {
        for (int i = 0; i < objs.length; i++) {
            if (objs[i] == null) {
                return true;
            }
        }
        return false;
    }

    /**
     * コレクションの要素に null 値が含まれるか評価します。
     * 
     * @param objs
     *            評価するコレクション
     * @return 要素に null 値が含まれる場合は true
     * @throws NullPointerException
     *             引数のコレクションが null の場合
     */
    public static boolean isOrNull(Collection objs) {
        for (Iterator i = objs.iterator(); i.hasNext();) {
            if (i.next() == null) {
                return true;
            }
        }
        return false;
    }

    /*
     * Equality
     */

    /**
     * 全ての指定の値が同値であるかを評価します。 <br>
     * 同値か双方 null の場合は true と評価します。
     * 
     * @param o
     *            評価基の値
     * @param o2
     *            評価先の値
     * @param o3
     *            評価先の値
     * @return 指定の値の内いずれかが同値である場合は true
     */
    public static boolean isAndEquals(Object o, Object o2, Object o3) {
        return (Eval.isEquals(o, o2) && Eval.isEquals(o, o3));
    }

    /**
     * 全ての指定の値が同値であるかを評価します。 <br>
     * 同値か双方 null の場合は true と評価します。
     * 
     * @param o
     *            評価基の値
     * @param o2
     *            評価先の値
     * @param o3
     *            評価先の値
     * @param o4
     *            評価先の値
     * @return 指定の値の内いずれかが同値である場合は true
     */
    public static boolean isAndEquals(Object o, Object o2, Object o3, Object o4) {
        return (Eval.isEquals(o, o2) && Eval.isEquals(o, o3) && Eval.isEquals(o, o4));
    }

    /**
     * 指定の値の内いずれかが同値であるかを評価します。 <br>
     * 同値か双方 null の場合は true と評価します。
     * 
     * @param o
     *            評価基の値
     * @param o2
     *            評価先の値
     * @param o3
     *            評価先の値
     * @return 指定の値の内いずれかが同値である場合は true
     */
    public static boolean isOrEquals(Object o, Object o2, Object o3) {
        return (Eval.isEquals(o, o2) || Eval.isEquals(o, o3));
    }

    /**
     * 指定の値の内いずれかが同値であるかを評価します。 <br>
     * 同値か双方 null の場合は true と評価します。
     * 
     * @param o
     *            評価基の値
     * @param o2
     *            評価先の値
     * @param o3
     *            評価先の値
     * @param o4
     *            評価先の値
     * @return 指定の値の内いずれかが同値である場合は true
     */
    public static boolean isOrEquals(Object o, Object o2, Object o3, Object o4) {
        return (Eval.isEquals(o, o2) || Eval.isEquals(o, o3) || Eval.isEquals(o, o4));
    }

    /*
     * StartsWith
     */

    /**
     * 指定の文字シーケンスの位置から、いずれかの接頭辞が一致するか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param prefixs
     *            接頭辞のコレクション
     * @param offset
     *            検索を開始する相対インデックス
     * @return 一致する場合は true
     */
    public static boolean isOrStartsWith(CharSequence cs, Collection prefixs, int offset) {
        if (Eval.isEmpty(cs)) {
            return false;
        }
        for (Iterator i = prefixs.iterator(); i.hasNext();) {
            final String prefix = (String) i.next();
            if (Eval.startsWith(cs, prefix, offset)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 指定の文字シーケンスに接尾辞の内いずれかが含まれるか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param suffixs
     *            接尾辞の配列
     * @return 一致する場合は true
     */
    public static boolean isOrSuffix(CharSequence cs, char[] suffixs) {
        if (Eval.isEmpty(cs)) {
            return false;
        }
        final char suffix = cs.charAt(cs.length() - 1);
        for (int i = 0; i < suffixs.length; i++) {
            if (suffix == suffixs[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 指定の文字シーケンスに接尾辞の内いずれかが含まれるか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param suffixs
     *            接頭辞のコレクション
     * @return 一致する場合は true
     */
    public static boolean isOrSuffix(CharSequence cs, Collection suffixs) {
        if (Eval.isEmpty(cs)) {
            return false;
        }
        for (Iterator i = suffixs.iterator(); i.hasNext();) {
            final String suffix = (String) i.next();
            if (Eval.suffix(cs, suffix)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 指定の文字シーケンスに接頭辞の内いずれかが含まれるか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param prefixs
     *            評価先の接頭辞の配列
     * @return 一致する場合は true
     */
    public static boolean isOrPrefix(CharSequence cs, char[] prefixs) {
        if (Eval.isEmpty(cs)) {
            return false;
        }
        final char prefix = cs.charAt(0);
        for (int i = 0; i < prefixs.length; i++) {
            if (prefix == prefixs[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 指定の文字シーケンスに接頭辞の内いずれかが含まれるか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param prefixs
     *            接頭辞のコレクション
     * @return 一致する場合は true
     */
    public static boolean isOrPrefix(CharSequence cs, Collection prefixs) {
        if (Eval.isEmpty(cs)) {
            return false;
        }
        for (Iterator i = prefixs.iterator(); i.hasNext();) {
            final String prefix = (String) i.next();
            if (Eval.prefix(cs, prefix)) {
                return true;
            }
        }
        return false;
    }

    /*
     * Assignable
     */

    /**
     * 指定されたオブジェクトが、全てのクラス型に割り当て可能か評価します。<br>
     * オブジェクトが null の場合は false を返却します。
     * 
     * @param o
     *            評価するオブジェクト
     * @param clazz
     *            評価先のクラス
     * @param clazz2
     *            評価先のクラス
     * @return 全てのクラス型に割り当て可能の場合は true
     */
    public static boolean isAndAssignable(Object o, Class clazz, Class clazz2) {
        return (Eval.isAssignable(o, clazz) && Eval.isAssignable(o, clazz2));
    }

    /**
     * 指定されたオブジェクトが、全てのクラス型に割り当て可能か評価します。<br>
     * オブジェクトが null の場合は false を返却します。
     * 
     * @param o
     *            評価するオブジェクト
     * @param clazz
     *            評価先のクラス
     * @param clazz2
     *            評価先のクラス
     * @param clazz3
     *            評価先のクラス
     * @return 全てのクラス型に割り当て可能の場合は true
     */
    public static boolean isAndAssignable(Object o, Class clazz, Class clazz2, Class clazz3) {
        return (Eval.isAssignable(o, clazz) && Eval.isAssignable(o, clazz2) && Eval.isAssignable(o, clazz3));
    }

    /**
     * 指定されたオブジェクトが、コレクション内の全てのクラス型に割り当て可能か評価します。<br>
     * オブジェクトが null の場合は false を返却します。
     * 
     * @param o
     *            評価するオブジェクト
     * @param clazzes
     *            評価先のクラスのコレクション
     * @return 全てのクラス型に割り当て可能の場合は true
     */
    public static boolean isAndAssignable(Object o, Collection clazzes) {
        for (Iterator i = clazzes.iterator(); i.hasNext();) {
            final Class clazz = (Class) i.next();
            if (!Eval.isAssignable(o, clazz)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定されたオブジェクトが、コレクション内のいずれかのクラス型に割り当て可能か評価します。<br>
     * オブジェクトが null の場合は false を返却します。
     * 
     * @param o
     *            評価するオブジェクト
     * @param clazz
     *            評価先のクラス
     * @param clazz2
     *            評価先のクラス
     * @return オブジェクトがいずれかのクラス型に割り当て可能の場合は true
     */
    public static boolean isOrAssignable(Object o, Class clazz, Class clazz2) {
        return (Eval.isAssignable(o, clazz) || Eval.isAssignable(o, clazz2));
    }

    /**
     * 指定されたオブジェクトが、コレクション内のいずれかのクラス型に割り当て可能か評価します。<br>
     * オブジェクトが null の場合は false を返却します。
     * 
     * @param o
     *            評価するオブジェクト
     * @param clazz
     *            評価先のクラス
     * @param clazz2
     *            評価先のクラス
     * @param clazz3
     *            評価先のクラス
     * @return オブジェクトがいずれかのクラス型に割り当て可能の場合は true
     */
    public static boolean isOrAssignable(Object o, Class clazz, Class clazz2, Class clazz3) {
        return (Eval.isAssignable(o, clazz) || Eval.isAssignable(o, clazz2) || Eval.isAssignable(o, clazz3));
    }

    /**
     * 指定されたオブジェクトが、コレクション内のいずれかのクラス型に割り当て可能か評価します。<br>
     * オブジェクトが null の場合は false を返却します。
     * 
     * @param o
     *            評価するオブジェクト
     * @param clazzes
     *            評価先のクラスのコレクション
     * @return いずれかのクラス型に割り当て可能の場合は true
     */
    public static boolean isOrAssignable(Object o, Collection clazzes) {
        for (Iterator i = clazzes.iterator(); i.hasNext();) {
            final Class clazz = (Class) i.next();
            if (Eval.isAssignable(o, clazz)) {
                return true;
            }
        }
        return false;
    }

    /*
     * Range All
     */

    /**
     * 指定された全ての数値が指定の範囲内か評価します。
     * 
     * @param a
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @param max
     *            範囲の最大値
     * @return 指定の範囲内の場合は true
     */
    public static boolean isRangeAll(byte[] a, byte min, byte max) {
        for (int i = 0; i < a.length; i++) {
            if (!Eval.isRange(a[i], min, max)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された全ての数値が指定の範囲内か評価します。
     * 
     * @param a
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @param max
     *            範囲の最大値
     * @return 指定の範囲内の場合は true
     */
    public static boolean isRangeAll(short[] a, short min, short max) {
        for (int i = 0; i < a.length; i++) {
            if (!Eval.isRange(a[i], min, max)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された全ての数値が指定の範囲内か評価します。
     * 
     * @param a
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @param max
     *            範囲の最大値
     * @return 指定の範囲内の場合は true
     */
    public static boolean isRangeAll(int[] a, int min, int max) {
        for (int i = 0; i < a.length; i++) {
            if (!Eval.isRange(a[i], min, max)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された全ての数値が指定の範囲内か評価します。
     * 
     * @param a
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @param max
     *            範囲の最大値
     * @return 指定の範囲内の場合は true
     */
    public static boolean isRangeAll(long[] a, long min, long max) {
        for (int i = 0; i < a.length; i++) {
            if (!Eval.isRange(a[i], min, max)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された全ての数値が指定の範囲内か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0f == -0.0f) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Float と同一基準で比較する場合は isBitsRangeAll(float, float, float) を使用してください。
     * 
     * @param a
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @param max
     *            範囲の最大値
     * @return 指定の範囲内の場合は true
     */
    public static boolean isRangeAll(float[] a, float min, float max) {
        for (int i = 0; i < a.length; i++) {
            if (!Eval.isRange(a[i], min, max)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された全ての数値が指定の範囲内か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0d == -0.0d) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Double と同一基準で比較する場合は isBitsRangeAll(double, double, double) を使用してください。
     * 
     * @param a
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @param max
     *            範囲の最大値
     * @return 指定の範囲内の場合は true
     */
    public static boolean isRangeAll(double[] a, double min, double max) {
        for (int i = 0; i < a.length; i++) {
            if (!Eval.isRange(a[i], min, max)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された全ての数値が指定の範囲内か、Float.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0f > -0.0f、Float.NaN == Float.NaN、Float.NaN > Float.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isRangeAll(float, float, float) を使用してください。
     * 
     * @param a
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @param max
     *            範囲の最大値
     * @return 指定の範囲内の場合は true
     */
    public static boolean isBitsRangeAll(float[] a, float min, float max) {
        for (int i = 0; i < a.length; i++) {
            if (!Eval.isBitsRange(a[i], min, max)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された全ての数値が指定の範囲内か、Double.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0d > -0.0d、Double.NaN == Double.NaN、Double.NaN > Double.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isRangeAll(double, double, double) を使用してください。
     * 
     * @param a
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @param max
     *            範囲の最大値
     * @return 指定の範囲内の場合は true
     */
    public static boolean isBitsRangeAll(double[] a, double min, double max) {
        for (int i = 0; i < a.length; i++) {
            if (!Eval.isBitsRange(a[i], min, max)) {
                return false;
            }
        }
        return true;
    }

}
