/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.lang;

import java.lang.reflect.Array;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * 値の正否を評価するショートカット関数群を提供します。
 */
public class Eval {

    /*
     * Empty
     */

    /**
     * 引数が null または空の文字シーケンスであるかを評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @return 引数が null または空の場合は true
     */
    public static boolean isEmpty(CharSequence cs) {
        return (cs == null || cs.length() == 0);
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param a
     *            評価する配列
     * @return 引数が null または空の状態の場合は true
     */
    public static boolean isEmpty(char[] a) {
        return (a == null || a.length == 0);
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param a
     *            評価する配列
     * @return 引数が null または空の状態の場合は true
     */
    public static boolean isEmpty(boolean[] a) {
        return (a == null || a.length == 0);
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param a
     *            評価する配列
     * @return 引数が null または空の状態の場合は true
     */
    public static boolean isEmpty(byte[] a) {
        return (a == null || a.length == 0);
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param a
     *            評価する配列
     * @return 引数が null または空の状態の場合は true
     */
    public static boolean isEmpty(short[] a) {
        return (a == null || a.length == 0);
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param a
     *            評価する配列
     * @return 引数が null または空の状態の場合は true
     */
    public static boolean isEmpty(int[] a) {
        return (a == null || a.length == 0);
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param a
     *            評価する配列
     * @return 引数が null または空の状態の場合は true
     */
    public static boolean isEmpty(long[] a) {
        return (a == null || a.length == 0);
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param a
     *            評価する配列
     * @return 引数が null または空の状態の場合は true
     */
    public static boolean isEmpty(float[] a) {
        return (a == null || a.length == 0);
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param a
     *            評価する配列
     * @return 引数が null または空の状態の場合は true
     */
    public static boolean isEmpty(double[] a) {
        return (a == null || a.length == 0);
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param a
     *            評価する配列
     * @return 引数が null または空の状態の場合は true
     */
    public static boolean isEmpty(Object[] a) {
        return (a == null || a.length == 0);
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param m
     *            評価するマップ
     * @return 引数が null または空の状態の場合は true
     */
    public static boolean isEmpty(Map m) {
        return (m == null || m.isEmpty());
    }

    /**
     * 引数が null または空の状態であるかを評価します。
     * 
     * @param coll
     *            評価するコレクション
     * @return 引数が null または空の状態の場合は true
     */
    public static boolean isEmpty(Collection coll) {
        return (coll == null || coll.isEmpty());
    }

    /*
     * Blank
     */

    /**
     * 引数が null または空かスペースのみの文字シーケンスであるかを評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @return 引数が null または空かスペースのみの場合は true
     */
    public static boolean isBlank(CharSequence cs) {
        return (cs == null || Fast.trimLength(cs) == 0);
    }

    /*
     * WhiteSpace
     */

    /**
     * 引数が null または空かJavaの基準による空白文字のみの文字シーケンスであるかを評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @return 引数が null または空かJavaの基準による空白文字のみの場合は true
     */
    public static boolean isWhiteSpace(CharSequence cs) {
        return (cs == null || Fast.trimWhiteLength(cs) == 0);
    }

    /*
     * Null
     */

    /**
     * 引数が null 値か評価します。
     * 
     * @param o
     *            評価する値
     * @return 引数が null 値の場合は true
     */
    public static boolean isNull(Object o) {
        return (o == null);
    }

    /**
     * 引数が存在するか評価します。<br>
     * null か空文字列以外の場合に存在すると判断します。
     * 
     * @param o
     *            評価する値
     * @return 引数が存在する場合は true
     */
    public static boolean isPresence(Object o) {
        return o == null || ((o instanceof CharSequence) && isBlank((CharSequence) o));
    }

    /*
     * Equality
     */

    /**
     * null の検証を含み同値性を比較します。 <br>
     * 双方が null の場合は true、片方が null の場合は false、以外は通常の同値性比較で評価します。
     * 
     * @param o
     *            評価基の値
     * @param o2
     *            評価先の値
     * @return 同値か双方 null の場合は true
     */
    public static boolean isEquals(Object o, Object o2) {
        return (o == o2 || (o != null && o.equals(o2)));
    }

    /**
     * 指定された 2 つの値の同値性を、Float.equals と同一基準で評価します。<br>
     * この比較基準は new Float(n).equals(new Float(n2)) と同等です。<br>
     * 同値演算子( == )と違って、このメソッドは NaN をそれ自身と同等と見なし、0.0f と -0.0f は同等と見なしません。
     * 
     * @param n
     *            1 つ目の値
     * @param n2
     *            2 つ目の値
     * @return 同値である場合は true
     */
    public static boolean isEquals(float n, float n2) {
        return (Float.floatToIntBits(n) != Float.floatToIntBits(n2));
    }

    /**
     * 指定された 2 つの値の同値性を、Double.equals と同一基準で評価します。<br>
     * この比較基準は new Double(n).equals(new Double(n2)) と同等です。<br>
     * 同値演算子( == )と違って、このメソッドは NaN をそれ自身と同等と見なし、0.0d と -0.0d は同等と見なしません。
     * 
     * @param n
     *            1 つ目の値
     * @param n2
     *            2 つ目の値
     * @return 同値である場合は true
     */
    public static boolean isEquals(double n, double n2) {
        return (Double.doubleToLongBits(n) != Double.doubleToLongBits(n2));
    }

    /**
     * 指定された 2 つの配列の同値性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が同値である場合です。<br>
     * また、2 つの配列参照が null の場合にも同値と見なされます。<br>
     * <br>
     * このメソッドは一貫性のための機能です、実際は Arrays.equals() に処理を転送します。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 同値である場合は true
     */
    public static boolean isEquals(boolean[] a, boolean[] a2) {
        return Arrays.equals(a, a2);
    }

    /**
     * 指定された 2 つの配列の同値性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が同値である場合です。<br>
     * また、2 つの配列参照が null の場合にも同値と見なされます。<br>
     * <br>
     * このメソッドは一貫性のための機能です、実際は Arrays.equals() に処理を転送します。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 同値である場合は true
     */
    public static boolean isEquals(char[] a, char[] a2) {
        return Arrays.equals(a, a2);
    }

    /**
     * 指定された 2 つの配列の同値性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が同値である場合です。<br>
     * また、2 つの配列参照が null の場合にも同値と見なされます。<br>
     * <br>
     * このメソッドは一貫性のための機能です、実際は Arrays.equals() に処理を転送します。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 同値である場合は true
     */
    public static boolean isEquals(byte[] a, byte[] a2) {
        return Arrays.equals(a, a2);
    }

    /**
     * 指定された 2 つの配列の同値性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が同値である場合です。<br>
     * また、2 つの配列参照が null の場合にも同値と見なされます。<br>
     * <br>
     * このメソッドは一貫性のための機能です、実際は Arrays.equals() に処理を転送します。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 同値である場合は true
     */
    public static boolean isEquals(short[] a, short a2[]) {
        return Arrays.equals(a, a2);
    }

    /**
     * 指定された 2 つの配列の同値性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が同値である場合です。<br>
     * また、2 つの配列参照が null の場合にも同値と見なされます。<br>
     * <br>
     * このメソッドは一貫性のための機能です、実際は Arrays.equals() に処理を転送します。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 同値である場合は true
     */
    public static boolean isEquals(int[] a, int[] a2) {
        return Arrays.equals(a, a2);
    }

    /**
     * 指定された 2 つの配列の同値性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が同値である場合です。<br>
     * また、2 つの配列参照が null の場合にも同値と見なされます。<br>
     * <br>
     * このメソッドは一貫性のための機能です、実際は Arrays.equals() に処理を転送します。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 同値である場合は true
     */
    public static boolean isEquals(long[] a, long[] a2) {
        return Arrays.equals(a, a2);
    }

    /**
     * 指定された 2 つの配列の同値性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が同値である場合です。<br>
     * また、2 つの配列参照が null の場合にも同値と見なされます。<br>
     * <br>
     * この比較基準は比較演算子と同等です。<br>
     * このメソッドは片方または双方が NaN の場合は false を返し、0.0f と -0.0f を同値とみなします。<br>
     * ラッパーの equals(Object) と同等の評価基準が必要な場合は isBitsEquals() を使用してください。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 同値である場合は true
     */
    public static boolean isEquals(float[] a, float[] a2) {
        if (a == a2) {
            return true;
        }
        if (a == null || a2 == null) {
            return false;
        }
        final int length = a.length;
        if (a2.length != length) {
            return false;
        }
        for (int i = 0; i < length; i++) {
            if (a[i] != a2[i]) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された 2 つの配列の同値性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が同値である場合です。<br>
     * また、2 つの配列参照が null の場合にも同値と見なされます。<br>
     * <br>
     * この比較基準は new Float(n).equals(new Float(n2)) と同等です。<br>
     * 同値演算子( == )と違って、このメソッドは NaN をそれ自身と同等と見なし、0.0f と -0.0f は同等と見なしません。<br>
     * プリミティブの同値演算子( == )と同等の評価基準が必要な場合は isEquals() を使用してください。 <br>
     * <br>
     * このメソッドは一貫性のための機能です、実際は Arrays.equals() に処理を転送します。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 同値である場合は true
     */
    public static boolean isBitsEquals(float[] a, float[] a2) {
        return Arrays.equals(a, a2);
    }

    /**
     * 指定された 2 つの配列の同値性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が同値である場合です。<br>
     * また、2 つの配列参照が null の場合にも同値と見なされます。<br>
     * <br>
     * この比較基準は比較演算子と同等です。<br>
     * このメソッドは片方または双方が NaN の場合は false を返し、0.0d と -0.0d を同値とみなします。<br>
     * ラッパーの equals(Object) と同等の評価基準が必要な場合は isBitsEquals() を使用してください。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 同値である場合は true
     */
    public static boolean isEquals(double[] a, double[] a2) {
        if (a == a2) {
            return true;
        }
        if (a == null || a2 == null) {
            return false;
        }
        final int length = a.length;
        if (a2.length != length) {
            return false;
        }
        for (int i = 0; i < length; i++) {
            if (a[i] != a2[i]) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された 2 つの配列の同値性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が同値である場合です。<br>
     * また、2 つの配列参照が null の場合にも同値と見なされます。<br>
     * <br>
     * この比較基準は new Double(n).equals(new Double(n2)) と同等です。<br>
     * 同値演算子( == )と違って、このメソッドは NaN をそれ自身と同等と見なし、0.0d と -0.0d は同等と見なしません。<br>
     * プリミティブの同値演算子( == )と同等の評価基準が必要な場合は isEquals() を使用してください。 <br>
     * このメソッドは一貫性のための機能です、実際は Arrays.equals() に処理を転送します。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 同値である場合は true
     */
    public static boolean isBitsEquals(double[] a, double[] a2) {
        return Arrays.equals(a, a2);
    }

    /**
     * 指定された 2 つの配列の同値性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が同値である場合です。<br>
     * また配列の同一位置の要素が、双方 null の場合は true、片方が null の場合は false と評価します。<br>
     * また、2 つの配列参照が null の場合にも同値と見なされます。<br>
     * <br>
     * このメソッドは一貫性のための機能です、実際は Arrays.equals() に処理を転送します。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 同値である場合は true
     */
    public static boolean isEquals(Object[] a, Object[] a2) {
        return Arrays.equals(a, a2);
    }

    /**
     * 指定された 2 つの配列の参照同一性を評価します。<br>
     * 両方の配列に同じ数の要素があり、同一位置のすべての要素が、参照が同一である場合です。<br>
     * また配列の同一位置の要素が、双方 null の場合は true、片方が null の場合は false と評価します。<br>
     * また、2 つの配列参照が null の場合にも同一と見なされます。
     * 
     * @param a
     *            1 つ目の配列
     * @param a2
     *            2 つ目の配列
     * @return 参照が同一と見なせる場合は true
     */
    public static boolean isRefEquals(Object[] a, Object[] a2) {
        if (a == a2) {
            return true;
        }
        if (a == null || a2 == null) {
            return false;
        }
        final int length = a.length;
        if (length != a2.length) {
            return false;
        }
        for (int i = 0; i < length; i++) {
            if (a[i] != a2[i]) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された 2 つの値が、双方配列の場合は配列の同値性比較を、以外の場合は通常の同値性比較を評価します。 <br>
     * また、2 つの値が null の場合にも同値と見なされます。 <br>
     * また配列の比較基準は Arrays.equals() と同等に成ります。<br>
     * 配列がプリミティブの浮動小数点の場合は同値演算子( == )と比較基準が違うため注意が必要です。
     * 
     * @param o
     *            評価基の値
     * @param o2
     *            評価先の値
     * @return 配列の場合の条件を含み、同値である場合は true
     */
    public static boolean isArrayEquals(Object o, Object o2) {
        if (o == o2) {
            return true;
        }
        if (o == null || o2 == null) {
            return false;
        }
        // Array
        if (o.getClass().isArray() && o2.getClass().isArray()) {
            final Class type1 = o.getClass().getComponentType();
            final Class type2 = o2.getClass().getComponentType();
            final boolean isPrimitive = (type1.isPrimitive() && type2.isPrimitive());
            // Primitive[] eq Primitive[] and eq component type
            if (isPrimitive && type1.equals(type2)) {
                if (Boolean.TYPE.equals(type1)) {
                    return Arrays.equals((boolean[]) o, (boolean[]) o2);
                } else if (Character.TYPE.equals(type1)) {
                    return Arrays.equals((char[]) o, (char[]) o2);
                } else if (Byte.TYPE.equals(type1)) {
                    return Arrays.equals((byte[]) o, (byte[]) o2);
                } else if (Short.TYPE.equals(type1)) {
                    return Arrays.equals((short[]) o, (short[]) o2);
                } else if (Integer.TYPE.equals(type1)) {
                    return Arrays.equals((int[]) o, (int[]) o2);
                } else if (Long.TYPE.equals(type1)) {
                    return Arrays.equals((long[]) o, (long[]) o2);
                } else if (Float.TYPE.equals(type1)) {
                    return Arrays.equals((float[]) o, (float[]) o2);
                } else if (Double.TYPE.equals(type1)) {
                    return Arrays.equals((double[]) o, (double[]) o2);
                }
            }
            // Object[] eq Object[]
            if (!isPrimitive) {
                return Arrays.equals((Object[]) o, (Object[]) o2);
            }
            // not eq component type
            return false;
        }
        // Not Array
        return o.equals(o2);
    }

    /*
     * Comparable
     */

    /**
     * 指定された値の比較結果が同値であるか評価します。<br>
     * つまり比較演算子の == と同等です。
     * 
     * @param from
     *            比較基の値
     * @param to
     *            比較先の値
     * @return 値の比較結果が同値の場合は true
     */
    public static boolean isCompareEqual(Comparable from, Comparable to) {
        return (from.compareTo(to) == 0);
    }

    /**
     * 指定された値の比較結果が非同値であるか評価します。<br>
     * つまり比較演算子の != と同等です。
     * 
     * @param from
     *            比較基の値
     * @param to
     *            比較先の値
     * @return 値の比較結果が同値の場合は true
     */
    public static boolean isCompareNotEqual(Comparable from, Comparable to) {
        return (from.compareTo(to) != 0);
    }

    /**
     * 指定された値の比較結果が第一引数の方が大きい事を示すか評価します。<br>
     * つまり比較演算子の > と同等です。
     * 
     * @param from
     *            比較基の値
     * @param to
     *            比較先の値
     * @return 値の比較結果が第一引数の方が大きい事を示す場合は true
     */
    public static boolean isCompareGreater(Comparable from, Comparable to) {
        return (from.compareTo(to) > 0);
    }

    /**
     * 指定された値の比較結果が、同値または第一引数の方が大きい事を示すか評価します。<br>
     * つまり比較演算子の >= と同等です。
     * 
     * @param from
     *            比較基の値
     * @param to
     *            比較先の値
     * @return 値の比較結果が第一引数の方が大きい事を示す場合は true
     */
    public static boolean isCompareGreaterEqual(Comparable from, Comparable to) {
        return (from.compareTo(to) >= 0);
    }

    /**
     * 指定された値の比較結果が第一引数の方が小さい事を示すか評価します。<br>
     * つまり比較演算子の < と同等です。
     * 
     * @param from
     *            比較基の値
     * @param to
     *            比較先の値
     * @return 値の比較結果が第一引数の方が小さい事を示す場合は true
     */
    public static boolean isCompareLess(Comparable from, Comparable to) {
        return (from.compareTo(to) < 0);
    }

    /**
     * 指定された値の比較結果が、同値または第一引数の方が小さい事を示すか評価します。<br>
     * つまり比較演算子の <= と同等です。
     * 
     * @param from
     *            比較基の値
     * @param to
     *            比較先の値
     * @return 値の比較結果が第一引数の方が小さい事を示す場合は true
     */
    public static boolean isCompareLessEqual(Comparable from, Comparable to) {
        return (from.compareTo(to) <= 0);
    }

    /*
     * Range
     */

    /**
     * 指定された数値が指定の範囲内か評価します。
     * 
     * @param n
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @param max
     *            範囲の最大値
     * @return 指定の範囲内の場合は true
     */
    public static boolean isRange(int n, int min, int max) {
        if (min > max) {
            throw new IllegalArgumentException("min > max (" + min + " > " + max + ")");
        }
        return (min <= n && n <= max);
    }

    /**
     * 指定された数値が指定の範囲内か評価します。
     * 
     * @param n
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @param max
     *            範囲の最大値
     * @return 指定の範囲内の場合は true
     */
    public static boolean isRange(long n, long min, long max) {
        if (min > max) {
            throw new IllegalArgumentException("min > max (" + min + " > " + max + ")");
        }
        return (min <= n && n <= max);
    }

    /**
     * 指定された数値が指定の範囲内か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0f == -0.0f) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Float と同一基準で比較する場合は isBitsRange(float, float, float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @param max
     *            範囲の最大値
     * @return 指定の範囲内の場合は true
     */
    public static boolean isRange(float n, float min, float max) {
        if (min > max) {
            throw new IllegalArgumentException("min > max (" + min + " > " + max + ")");
        }
        return (isMin(n, min) && isMax(n, max));
    }

    /**
     * 指定された数値が指定の範囲内か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0d == -0.0d) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Double と同一基準で比較する場合は isBitsRange(double, double, double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @param max
     *            範囲の最大値
     * @return 指定の範囲内の場合は true
     */
    public static boolean isRange(double n, double min, double max) {
        if (min > max) {
            throw new IllegalArgumentException("min > max (" + min + " > " + max + ")");
        }
        return (isMin(n, min) && isMax(n, max));
    }

    /**
     * 指定された数値が指定の範囲内か、Float.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0f > -0.0f、Float.NaN == Float.NaN、Float.NaN > Float.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isRange(float, float, float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @param max
     *            範囲の最大値
     * @return 指定の範囲内の場合は true
     */
    public static boolean isBitsRange(float n, float min, float max) {
        if (Float.compare(min, max) > 0) {
            throw new IllegalArgumentException("min > max (" + min + " > " + max + ")");
        }
        return (isBitsMin(n, min) && isBitsMax(n, max));
    }

    /**
     * 指定された数値が指定の範囲内か、Double.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0d > -0.0d、Double.NaN == Double.NaN、Double.NaN > Double.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isRange(double, double, double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @param max
     *            範囲の最大値
     * @return 指定の範囲内の場合は true
     */
    public static boolean isBitsRange(double n, double min, double max) {
        if (Double.compare(min, max) > 0) {
            throw new IllegalArgumentException("min > max (" + min + " > " + max + ")");
        }
        return (isBitsMin(n, min) && isBitsMax(n, max));
    }

    /**
     * 指定された数値が指定の最小値の範囲か評価します。
     * 
     * @param n
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @return 指定の最小値の範囲の場合は true
     */
    public static boolean isMin(int n, int min) {
        return (min <= n);
    }

    /**
     * 指定された数値が指定の最小値の範囲か評価します。
     * 
     * @param n
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @return 指定の最小値の範囲の場合は true
     */
    public static boolean isMin(long n, long min) {
        return (min <= n);
    }

    /**
     * 指定された数値が指定の最小値の範囲か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0f == -0.0f) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Float と同一基準で比較する場合は isBitsMin(float, float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @return 指定の最小値の範囲の場合は true
     */
    public static boolean isMin(float n, float min) {
        return (min <= n);
    }

    /**
     * 指定された数値が指定の最小値の範囲か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0d == -0.0d) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Double と同一基準で比較する場合は isBitsMin(double, double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @return 指定の最小値の範囲の場合は true
     */
    public static boolean isMin(double n, double min) {
        return (min <= n);
    }

    /**
     * 指定された数値が指定の最小値の範囲か、Float.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0f > -0.0f、Float.NaN == Float.NaN、Float.NaN > Float.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isMin(float, float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @return 指定の最小値の範囲の場合は true
     */
    public static boolean isBitsMin(float n, float min) {
        return (Float.compare(n, min) >= 0);
    }

    /**
     * 指定された数値が指定の最小値の範囲か、Double.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0d > -0.0d、Double.NaN == Double.NaN、Double.NaN > Double.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isMin(double, double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param min
     *            範囲の最小値
     * @return 指定の最小値の範囲の場合は true
     */
    public static boolean isBitsMin(double n, double min) {
        return (Double.compare(n, min) >= 0);
    }

    /**
     * 指定された数値が指定の最大値の範囲か評価します。
     * 
     * @param n
     *            評価する数値
     * @param max
     *            範囲の最大値
     * @return 指定の最大値の範囲の場合は true
     */
    public static boolean isMax(int n, int max) {
        return (n <= max);
    }

    /**
     * 指定された数値が指定の最大値の範囲か評価します。
     * 
     * @param n
     *            評価する数値
     * @param max
     *            範囲の最大値
     * @return 指定の最大値の範囲の場合は true
     */
    public static boolean isMax(long n, long max) {
        return (n <= max);
    }

    /**
     * 指定された数値が指定の最大値の範囲か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0f == -0.0f) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Float と同一基準で比較する場合は isBitsMax(float, float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param max
     *            範囲の最大値
     * @return 指定の最大値の範囲の場合は true
     */
    public static boolean isMax(float n, float max) {
        return (n <= max);
    }

    /**
     * 指定された数値が指定の最大値の範囲か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0d == -0.0d) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Double と同一基準で比較する場合は isBitsMax(double, double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param max
     *            範囲の最大値
     * @return 指定の最大値の範囲の場合は true
     */
    public static boolean isMax(double n, double max) {
        return (n <= max);
    }

    /**
     * 指定された数値が指定の最大値の範囲か、Float.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0f > -0.0f、Float.NaN == Float.NaN、Float.NaN > Float.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isMax(float, float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param max
     *            範囲の最大値
     * @return 指定の最大値の範囲の場合は true
     */
    public static boolean isBitsMax(float n, float max) {
        return (Float.compare(n, max) <= 0);
    }

    /**
     * 指定された数値が指定の最大値の範囲か、Double.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0d > -0.0d、Double.NaN == Double.NaN、Double.NaN > Double.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isMax(double, double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @param max
     *            範囲の最大値
     * @return 指定の最大値の範囲の場合は true
     */
    public static boolean isBitsMax(double n, double max) {
        return (Double.compare(n, max) <= 0);
    }

    /*
     * char sequence size
     */

    /**
     * 指定された文字シーケンスの長さが指定の長さと同一か評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param size
     *            文字シーケンスの長さ
     * @return 指定された文字シーケンスの長さが指定の長さと同一の場合は true
     */
    public static boolean isSize(CharSequence cs, int size) {
        return (cs.length() == size);
    }

    /**
     * 指定された文字シーケンスの長さが指定の範囲内か評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param min
     *            文字数の最小値
     * @param max
     *            文字数の最大値
     * @return 文字シーケンスの長さが範囲内の場合は true
     */
    public static boolean isRangeSize(CharSequence cs, int min, int max) {
        return (isRange(cs.length(), min, max));
    }

    /**
     * 指定された文字シーケンスの長さが指定の最小値の範囲内か評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param min
     *            文字数の最小値
     * @return 文字シーケンスの長さが最小値の範囲内の場合は true
     */
    public static boolean isMinSize(CharSequence cs, int min) {
        return (isMin(cs.length(), min));
    }

    /**
     * 指定された文字シーケンスの長さが指定の最大値の範囲内か評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param max
     *            文字数の最大値
     * @return 文字シーケンスの長さが最大値の範囲内の場合は true
     */
    public static boolean isMaxSize(CharSequence cs, int max) {
        return (isMax(cs.length(), max));
    }

    /*
     * Negative
     */

    /**
     * 指定された数値が負数か評価します。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が負数の場合は true
     */
    public static boolean isNegative(int n) {
        return (0 > n);
    }

    /**
     * 指定された数値が負数か評価します。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が負数の場合は true
     */
    public static boolean isNegative(long n) {
        return (0 > n);
    }

    /**
     * 指定された数値が負数か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0f == -0.0f) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Float と同一基準で比較する場合は isBitsNegative(float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が負数の場合は true
     */
    public static boolean isNegative(float n) {
        return (0.0f > n);
    }

    /**
     * 指定された数値が負数か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0d == -0.0d) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Double と同一基準で比較する場合は isBitsNegative(double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が負数の場合は true
     */
    public static boolean isNegative(double n) {
        return (0.0d > n);
    }

    /**
     * 指定された数値が負数か、Float.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0f > -0.0f、Float.NaN == Float.NaN、Float.NaN > Float.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isNegative(float, float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が正数の場合は true
     */
    public static boolean isBitsNegative(float n) {
        return (Float.compare(n, 0.0f) < 0);
    }

    /**
     * 指定された数値が負数か、Double.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0d > -0.0d、Double.NaN == Double.NaN、Double.NaN > Double.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isNegative(double, double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が正数の場合は true
     */
    public static boolean isBitsNegative(double n) {
        return (Double.compare(n, 0.0d) < 0);
    }

    /**
     * 指定された数値がゼロまたは負数か評価します。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは負数の場合は true
     */
    public static boolean isZeroOrNegative(int n) {
        return (0 >= n);
    }

    /**
     * 指定された数値がゼロまたは負数か評価します。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは負数の場合は true
     */
    public static boolean isZeroOrNegative(long n) {
        return (0 >= n);
    }

    /**
     * 指定された数値がゼロまたは負数か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0f == -0.0f) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Float と同一基準で比較する場合は isBitsZeroOrNegative(float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは負数の場合は true
     */
    public static boolean isZeroOrNegative(float n) {
        return (0.0f >= n);
    }

    /**
     * 指定された数値がゼロまたは負数か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0d == -0.0d) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Double と同一基準で比較する場合は isBitsZeroOrNegative(double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは負数の場合は true
     */
    public static boolean isZeroOrNegative(double n) {
        return (0.0d >= n);
    }

    /**
     * 指定された数値がゼロまたは負数か、Float.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0f > -0.0f、Float.NaN == Float.NaN、Float.NaN > Float.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isZeroOrNegative(float, float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは負数の場合は true
     */
    public static boolean isBitsZeroOrNegative(float n) {
        return (Float.compare(n, 0.0f) <= 0);
    }

    /**
     * 指定された数値がゼロまたは負数か、Double.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0d > -0.0d、Double.NaN == Double.NaN、Double.NaN > Double.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isZeroOrNegative(double, double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは負数の場合は true
     */
    public static boolean isBitsZeroOrNegative(double n) {
        return (Double.compare(n, 0.0d) <= 0);
    }

    /*
     * Positive
     */

    /**
     * 指定された数値が正数か評価します。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が正数の場合は true
     */
    public static boolean isPositive(int n) {
        return (0 < n);
    }

    /**
     * 指定された数値が正数か評価します。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が正数の場合は true
     */
    public static boolean isPositive(long n) {
        return (0 < n);
    }

    /**
     * 指定された数値が正数か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0f == -0.0f) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Float と同一基準で比較する場合は isBitsPositive(float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が正数の場合は true
     */
    public static boolean isPositive(float n) {
        return (0.0f < n);
    }

    /**
     * 指定された数値が正数か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0d == -0.0d) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Double と同一基準で比較する場合は isBitsPositive(double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が正数の場合は true
     */
    public static boolean isPositive(double n) {
        return (0.0d < n);
    }

    /**
     * 指定された数値が正数か、Float.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0f > -0.0f、Float.NaN == Float.NaN、Float.NaN > Float.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isPositive(float, float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が正数の場合は true
     */
    public static boolean isBitsPositive(float n) {
        return (Float.compare(n, -0.0f) > 0);
    }

    /**
     * 指定された数値が正数か、Double.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0d > -0.0d、Double.NaN == Double.NaN、Double.NaN > Double.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isPositive(double, double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値が正数の場合は true
     */
    public static boolean isBitsPositive(double n) {
        return (Double.compare(n, -0.0d) > 0);
    }

    /**
     * 指定された数値がゼロまたは正数か評価します。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは正数の場合は true
     */
    public static boolean isZeroOrPositive(int n) {
        return (0 <= n);
    }

    /**
     * 指定された数値がゼロまたは正数か評価します。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは正数の場合は true
     */
    public static boolean isZeroOrPositive(long n) {
        return (0 <= n);
    }

    /**
     * 指定された数値がゼロまたは正数か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0f == -0.0f) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Float と同一基準で比較する場合は isBitsZeroOrPositive(float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは正数の場合は true
     */
    public static boolean isZeroOrPositive(float n) {
        return (0.0f <= n);
    }

    /**
     * 指定された数値がゼロまたは正数か、数値比較演算子の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを同値として、また片方または双方が非数の場合は常に false と評価します。<br>
     * つまり (+0.0d == -0.0d) = true、(NaN <==> NaN) = false、(NaN <==> !NaN) = false、(!NaN <==> NaN) = false と評価します。<br>
     * java.lang.Double と同一基準で比較する場合は isBitsZeroOrPositive(double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは正数の場合は true
     */
    public static boolean isZeroOrPositive(double n) {
        return (0.0d <= n);
    }

    /**
     * 指定された数値がゼロまたは正数か、Float.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0f > -0.0f、Float.NaN == Float.NaN、Float.NaN > Float.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isZeroOrPositive(float, float) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは正数の場合は true
     */
    public static boolean isBitsZeroOrPositive(float n) {
        return (Float.compare(n, -0.0f) >= 0);
    }

    /**
     * 指定された数値がゼロまたは正数か、Double.compare と同様の比較基準で評価します。<br>
     * この比較基準では正のゼロと負のゼロを区別し、双方が非数の場合は同値、非数を正の無限大よりも大きいと見做します。<br>
     * また比較基準は、すべての NaN 値を単一の正規 NaN 値に収納します。<br>
     * つまり +0.0d > -0.0d、Double.NaN == Double.NaN、Double.NaN > Double.POSITIVE_INFINITY と評価します。 <br>
     * 比較演算子と同一基準で比較する場合は isZeroOrPositive(double, double) を使用してください。
     * 
     * @param n
     *            評価する数値
     * @return 指定された数値がゼロまたは正数の場合は true
     */
    public static boolean isBitsZeroOrPositive(double n) {
        return (Double.compare(n, -0.0d) >= 0);
    }

    /*
     * Infinite Or NaN
     */

    /**
     * 数値の絶対値が無限量か、または非数 (NaN) であるか評価します。
     * 
     * @param val
     *            評価する数値
     * @return 数値の絶対値が無限量か、または非数 (NaN) の場合は true
     */
    public static boolean isInfiniteOrNaN(Number val) {
        if (val instanceof Float) {
            return isInfiniteOrNaN((Float) val);
        }
        if (val instanceof Double) {
            return isInfiniteOrNaN((Double) val);
        }
        return false;
    }

    /**
     * 数値の絶対値が無限量か、または非数 (NaN) であるか評価します。
     * 
     * @param val
     *            評価する数値
     * @return 数値の絶対値が無限量か、または非数 (NaN) の場合は true
     */
    public static boolean isInfiniteOrNaN(Float val) {
        return (val.isInfinite() || val.isNaN());
    }

    /**
     * 数値の絶対値が無限量か、または非数 (NaN) であるか評価します。
     * 
     * @param val
     *            評価する数値
     * @return 数値の絶対値が無限量か、または非数 (NaN) の場合は true
     */
    public static boolean isInfiniteOrNaN(float val) {
        return (Float.isInfinite(val) || Float.isNaN(val));
    }

    /**
     * 数値の絶対値が無限量か、または非数 (NaN) であるか評価します。
     * 
     * @param val
     *            評価する数値
     * @return 数値の絶対値が無限量か、または非数 (NaN) の場合は true
     */
    public static boolean isInfiniteOrNaN(Double val) {
        return (val.isInfinite() || val.isNaN());
    }

    /**
     * 数値の絶対値が無限量か、または非数 (NaN) であるか評価します。
     * 
     * @param val
     *            評価する数値
     * @return 数値の絶対値が無限量か、または非数 (NaN) の場合は true
     */
    public static boolean isInfiniteOrNaN(double val) {
        return (Double.isInfinite(val) || Double.isNaN(val));
    }

    /*
     * prefix and suffix
     */

    /**
     * 指定の文字シーケンスが指定の接頭辞と接尾辞を持つか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param prefix
     *            評価先の接頭辞
     * @param suffix
     *            接尾辞
     * @return 接頭辞と接尾辞を持つ場合は true
     */
    public static boolean isEnclose(CharSequence cs, char prefix, char suffix) {
        if (isEmpty(cs) || cs.length() < 2) {
            return false;
        }
        return prefix == cs.charAt(0) && suffix == cs.charAt(cs.length() - 1);
    }

    /**
     * 指定の文字シーケンスが指定の接頭辞と接尾辞を持つか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param prefix
     *            評価先の接頭辞
     * @param suffix
     *            接尾辞
     * @return 接頭辞と接尾辞を持つ場合は true
     */
    public static boolean isEnclose(CharSequence cs, CharSequence prefix, CharSequence suffix) {
        if (isEmpty(cs)) {
            return false;
        }
        final int min = Concat.size(prefix, suffix);
        if (cs.length() < min) {
            return false;
        }
        return prefix(cs, prefix) && suffix(cs, suffix);
    }

    /**
     * 指定の文字シーケンスの位置から前方一致するか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param prefix
     *            評価先の接頭辞
     * @param offset
     *            検索を開始する相対インデックス
     * @return 一致する場合は true
     */
    public static boolean isStartsWith(CharSequence cs, CharSequence prefix, int offset) {
        if (isEmpty(cs)) {
            return false;
        }
        return startsWith(cs, prefix, offset);
    }

    /**
     * 指定の文字シーケンスが指定の接尾辞を持つか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param suffix
     *            接尾辞
     * @return 一致する場合は true
     */
    public static boolean isSuffix(CharSequence cs, CharSequence suffix) {
        if (isEmpty(cs)) {
            return false;
        }
        return suffix(cs, suffix);
    }

    /**
     * 指定の文字シーケンスが指定の接頭辞を持つか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param prefix
     *            接頭辞
     * @return 一致する場合は true
     */
    public static boolean isPrefix(CharSequence cs, CharSequence prefix) {
        if (isEmpty(cs)) {
            return false;
        }
        return prefix(cs, prefix);
    }

    /*
     * Type
     */

    /**
     * 指定されたオブジェクトが、クラス型に割り当て可能か評価します。<br>
     * オブジェクトが null の場合は false を返却します。
     * 
     * @param o
     *            評価するオブジェクト
     * @param clazz
     *            評価先のクラス
     * @return クラス型に割り当て可能の場合は true
     */
    public static boolean isAssignable(Object o, Class clazz) {
        return (clazz != null && clazz.isInstance(o));
    }

    /**
     * 要素の数が同値であるか評価します。
     * 
     * @param c1
     *            コレクション１
     * @param c2
     *            コレクション２
     * @return 要素の数が同値の場合は true
     */
    public static boolean isEqualsSize(Collection c1, Collection c2) {
        return (c1.size() == c2.size());
    }

    /**
     * 要素の数が同値であるか評価します。
     * 
     * @param m1
     *            マップ１
     * @param m2
     *            マップ２
     * @return 要素の数が同値の場合は true
     */
    public static boolean isEqualsSize(Map m1, Map m2) {
        return (m1.size() == m2.size());
    }

    /**
     * クラスが同一であるかを評価します。<br>
     * 片方または双方のクラスが null の場合は false を返却します。
     * 
     * @param o
     *            評価するオブジェクト
     * @param clazz
     *            比較するクラス
     * @return クラスが同一の場合は true
     */
    public static boolean isEqualsClass(Object o, Class clazz) {
        return (o != null && o.getClass().equals(clazz));
    }

    /**
     * 配列のコンポーネント型が指定のクラスと同一であるかを評価します。<br>
     * 配列が null の場合は false を返却します。
     * 
     * @param a
     *            評価する配列
     * @param clazz
     *            評価先のクラス
     * @return 配列のコンポーネント型が指定のクラスと同一の場合は true
     * @throws IllegalArgumentException
     *             引数が配列型では無い場合
     */
    public static boolean isEqualsComponentType(Object a, Class clazz) {
        if (a == null) {
            return false;
        }
        final Class arrayType = a.getClass();
        if (!arrayType.isArray()) {
            throw new IllegalArgumentException("is not Array");
        }
        return (arrayType.getComponentType().equals(clazz));
    }

    /**
     * 指定のオブジェクトが、配列型か評価します。<br>
     * 指定のオブジェクトが null の場合は false を返却します。
     * 
     * @param o
     *            評価するオブジェクト
     * @return 配列型の場合は true
     */
    public static boolean isArray(Object o) {
        return (o != null && o.getClass().isArray());
    }

    /**
     * 引数が配列型のクラスか評価します。<br>
     * 引数が null の場合は false を返却します。
     * 
     * @param clazz
     *            評価するクラス
     * @return 配列型の場合は true
     */
    public static boolean isArrayClass(Class clazz) {
        return (clazz != null && clazz.isArray());
    }

    /**
     * 要素の数が同値であるか評価します。
     * 
     * @param o1
     *            オブジェクト１
     * @param o2
     *            オブジェクト２
     * @return 要素の数が同値の場合は true
     */
    public static boolean isEqualLength(Object o1, Object o2) {
        return (Array.getLength(o1) == Array.getLength(o2));
    }

    /**
     * 指定された二次元配列の全ての要素の配列長が同一か評価します。
     * 
     * @param array2d
     *            評価するの配列を格納とする二次元配列
     * @return 指定された二次元配列の全ての要素の配列長が同一の場合は true
     * @throws IllegalArgumentException
     *             引数の要素が配列でないまたは引数が空配列の場合
     */
    public static boolean isEqualLength(Object[] array2d) {
        if (isEmpty(array2d)) {
            throw new IllegalArgumentException("array as empty");
        }
        final int array1dLength = Array.getLength(array2d[0]);
        for (int i = 1; i < array2d.length; i++) {
            if (array1dLength != Array.getLength(array2d[i])) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定された文字シーケンスのインデックスの文字と、指定の文字が一致するか評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param index
     *            評価するインデックス
     * @param c
     *            比較する文字
     * @return インデックスの文字と指定の文字が一致する場合は true
     */
    public static boolean isCharAt(CharSequence cs, int index, char c) {
        return (-1 < index && index < cs.length()) && cs.charAt(index) == c;
    }

    /*
     * Contains
     */

    /**
     * 引数 e が配列 a に含まれているか評価します。<br>
     * 配列 a が null の場合は false を返却します。
     * 
     * @param a
     *            評価基の配列
     * @param e
     *            検索する値
     * @return 引数 e が配列 a に含まれている場合は true
     */
    public static boolean isContains(boolean[] a, boolean e) {
        if (a == null) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            if (e == a[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数 e が配列 a に含まれているか評価します。<br>
     * 配列 a が null の場合は false を返却します。
     * 
     * @param a
     *            評価基の配列
     * @param e
     *            検索する値
     * @return 引数 e が配列 a に含まれている場合は true
     */
    public static boolean isContains(char[] a, char e) {
        if (a == null) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            if (e == a[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数 e が配列 a に含まれているか評価します。<br>
     * 配列 a が null の場合は false を返却します。
     * 
     * @param a
     *            評価基の配列
     * @param e
     *            検索する値
     * @return 引数 e が配列 a に含まれている場合は true
     */
    public static boolean isContains(byte[] a, byte e) {
        if (a == null) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            if (e == a[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数 e が配列 a に含まれているか評価します。<br>
     * 配列 a が null の場合は false を返却します。
     * 
     * @param a
     *            評価基の配列
     * @param e
     *            検索する値
     * @return 引数 e が配列 a に含まれている場合は true
     */
    public static boolean isContains(short[] a, short e) {
        if (a == null) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            if (e == a[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数 e が配列 a に含まれているか評価します。<br>
     * 配列 a が null の場合は false を返却します。
     * 
     * @param a
     *            評価基の配列
     * @param e
     *            検索する値
     * @return 引数 e が配列 a に含まれている場合は true
     */
    public static boolean isContains(int[] a, int e) {
        if (a == null) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            if (e == a[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数 e が配列 a に含まれているか評価します。<br>
     * 配列 a が null の場合は false を返却します。
     * 
     * @param a
     *            評価基の配列
     * @param e
     *            検索する値
     * @return 引数 e が配列 a に含まれている場合は true
     */
    public static boolean isContains(long[] a, long e) {
        if (a == null) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            if (e == a[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数 e が配列 a に含まれているか評価します。 <br>
     * この比較基準は比較演算子と同等です。<br>
     * このメソッドは片方または双方が NaN の場合は false を返し、0.0f と -0.0f を同値とみなします。<br>
     * ラッパーの equals(Object) と同等の評価基準が必要な場合は isBitsContains() を使用してください。<br>
     * また配列 a が null の場合は false を返却します。
     * 
     * @param a
     *            評価基の配列
     * @param e
     *            検索する値
     * @return 引数 e が配列 a に含まれている場合は true
     */
    public static boolean isContains(float[] a, float e) {
        if (a == null) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            if (e == a[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数 e が配列 a に含まれているか評価します。 <br>
     * この比較基準は new Float(n).equals(new Float(n2)) と同等です。<br>
     * 同値演算子( == )と違って、このメソッドは NaN をそれ自身と同等と見なし、0.0f と -0.0f は同等と見なしません。<br>
     * プリミティブの同値演算子( == )と同等の評価基準が必要な場合は isContains() を使用してください。<br>
     * また配列 a が null の場合は false を返却します。
     * 
     * @param a
     *            評価基の配列
     * @param e
     *            検索する値
     * @return 引数 e が配列 a に含まれている場合は true
     */
    public static boolean isBitsContains(float[] a, float e) {
        if (a == null) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            if (Float.floatToIntBits(e) == Float.floatToIntBits(a[i])) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数 e が配列 a に含まれているか評価します。 <br>
     * この比較基準は比較演算子と同等です。<br>
     * このメソッドは片方または双方が NaN の場合は false を返し、0.0d と -0.0d を同値とみなします。<br>
     * ラッパーの equals(Object) と同等の評価基準が必要な場合は isBitsContains() を使用してください。<br>
     * また配列 a が null の場合は false を返却します。
     * 
     * @param a
     *            評価基の配列
     * @param e
     *            検索する値
     * @return 引数 e が配列 a に含まれている場合は true
     */
    public static boolean isContains(double[] a, double e) {
        if (a == null) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            if (e == a[i]) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数 e が配列 a に含まれているか評価します。 <br>
     * この比較基準は new Double(n).equals(new Double(n2)) と同等です。<br>
     * 同値演算子( == )と違って、このメソッドは NaN をそれ自身と同等と見なし、0.0d と -0.0d は同等と見なしません。<br>
     * プリミティブの同値演算子( == )と同等の評価基準が必要な場合は isContains() を使用してください。<br>
     * また配列 a が null の場合は false を返却します。
     * 
     * @param a
     *            評価基の配列
     * @param e
     *            検索する値
     * @return 引数 e が配列 a に含まれている場合は true
     */
    public static boolean isBitsContains(double[] a, double e) {
        if (a == null) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            if (Double.doubleToLongBits(e) == Double.doubleToLongBits(a[i])) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数 o が配列 a に含まれているか評価します。<br>
     * 配列 a が null の場合は false を返却します。
     * 
     * @param a
     *            評価基の配列
     * @param o
     *            検索するオブジェクト
     * @return 引数 o が配列 a に含まれている場合は true
     */
    public static boolean isContains(Object[] a, Object o) {
        if (a == null) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            if (isEquals(o, a[i])) {
                return true;
            }
        }
        return false;
    }

    /**
     * コレクション coll に引数 o が含まれているか評価します。<br>
     * 引数 coll が null の場合は false を返却します。
     * 
     * @param coll
     *            評価基のコレクション
     * @param o
     *            検索するオブジェクト
     * @return コレクション coll に引数 o が含まれている場合は true
     */
    public static boolean isContains(Collection coll, Object o) {
        if (coll == null) {
            return false;
        }
        return coll.contains(o);
    }

    /**
     * 指定の文字列に指定の文字が含まれているか評価します。<br>
     * 文字列が null の場合は false を返却します。
     * 
     * @param s
     *            評価する文字列
     * @param c
     *            検索する文字
     * @return 文字列に指定の文字が含まれている場合は true
     */
    public static boolean isContains(String s, char c) {
        if (s == null) {
            return false;
        }
        return (-1 < s.indexOf(c));
    }

    /**
     * 指定の文字列に指定の文字列が含まれているか評価します。<br>
     * 文字列が null の場合は false を返却します。
     * 
     * @param s
     *            評価する文字列
     * @param target
     *            検索する文字列
     * @return 文字列に指定の文字列が含まれている場合は true
     */
    public static boolean isContains(String s, String target) {
        if (s == null) {
            return false;
        }
        return (-1 < s.indexOf(target));
    }

    /**
     * 指定の文字シーケンスに指定の文字が含まれているか評価します。<br>
     * 文字シーケンスが null の場合は false を返却します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param c
     *            検索する文字
     * @return 文字シーケンスに指定の文字が含まれている場合は true
     */
    public static boolean isContains(CharSequence cs, char c) {
        if (cs == null) {
            return false;
        }
        return (-1 < indexOf(cs, c));
    }

    /**
     * 指定の文字シーケンスに指定の文字シーケンスが含まれているか評価します。<br>
     * 文字シーケンスが null の場合は false を返却します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param target
     *            検索する文字シーケンス
     * @return 文字シーケンスに指定の要素が含まれている場合は true
     */
    public static boolean isContains(CharSequence cs, CharSequence target) {
        if (cs == null) {
            return false;
        }
        return (-1 < indexOf(cs, target));
    }

    /*
     * Comparable
     */

    /**
     * 引数 o と順序比較が一致する値が配列 a に含まれているか評価します。<br>
     * 配列 a の要素は全て null または Comparable である必要があります、要素が null の場合は無視されます。<br>
     * つまり要素 e に対して (e != null && e.compareTo(o) == 0) の基準で比較します。<br>
     * これは java.math.BigDecimal のように equals が Comparable.ompareTo(o) と同一の結果を返さない場合に有用です。<br>
     * （java.math.BigDecimal はスケールの一致も評価します、つまり 1.200 と 1.20 は非同値になります）<br>
     * また配列 a が null の場合は false を返却します。
     * 
     * @param a
     *            評価基の配列
     * @param o
     *            比較する値
     * @return 引数 o と順序比較が一致する値が配列 a に含まれている場合は true
     */
    public static boolean isCompareContains(Object[] a, Comparable o) {
        if (a == null) {
            return false;
        }
        for (int i = 0; i < a.length; i++) {
            final Comparable e = (Comparable) a[i];
            if (e != null && e.compareTo(o) == 0) {
                return true;
            }
        }
        return false;
    }

    /**
     * 引数 o と順序比較が一致する値がコレクション coll に含まれているか評価します。<br>
     * コレクション coll の要素は全て null または Comparable である必要があります、要素が null の場合は無視されます。<br>
     * つまり要素 e に対して (e != null && e.compareTo(o) == 0) の基準で比較します。<br>
     * これは java.math.BigDecimal のように equals が Comparable.ompareTo(o) と同一の結果を返さない場合に有用です。<br>
     * （java.math.BigDecimal はスケールの一致も評価します、つまり 1.200 と 1.20 は非同値になります）<br>
     * またコレクション coll が null の場合は false を返却します。
     * 
     * @param coll
     *            評価基のコレクション
     * @param o
     *            比較する値
     * @return 引数 o と順序比較が一致する値がコレクション coll に含まれている場合は true
     */
    public static boolean isCompareContains(Collection coll, Comparable o) {
        if (coll == null) {
            return false;
        }
        for (Iterator i = coll.iterator(); i.hasNext();) {
            final Comparable e = (Comparable) i.next();
            if (e != null && e.compareTo(o) == 0) {
                return true;
            }
        }
        return false;
    }

    /*
     * Index
     */

    /**
     * 指定された文字シーケンスの範囲内のインデックスか評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(CharSequence cs, int index) {
        return isInIndex(cs.length(), index);
    }

    /**
     * 指定された文字シーケンスの範囲内のインデックスか評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param minIndex
     *            開始インデックス
     * @param maxIndex
     *            終了インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(CharSequence cs, int minIndex, int maxIndex) {
        return isInIndex(cs.length(), minIndex, maxIndex);
    }

    /**
     * 指定されたリストの範囲内のインデックスか評価します。
     * 
     * @param list
     *            評価するリスト
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(List list, int index) {
        return isInIndex(list.size(), index);
    }

    /**
     * 指定されたリストの範囲内のインデックスか評価します。
     * 
     * @param list
     *            評価するリスト
     * @param minIndex
     *            開始インデックス
     * @param maxIndex
     *            終了インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(List list, int minIndex, int maxIndex) {
        return isInIndex(list.size(), minIndex, maxIndex);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(char[] a, int index) {
        return isInIndex(a.length, index);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param minIndex
     *            開始インデックス
     * @param maxIndex
     *            終了インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(char[] a, int minIndex, int maxIndex) {
        return isInIndex(a.length, minIndex, maxIndex);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(boolean[] a, int index) {
        return isInIndex(a.length, index);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param minIndex
     *            開始インデックス
     * @param maxIndex
     *            終了インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(boolean[] a, int minIndex, int maxIndex) {
        return isInIndex(a.length, minIndex, maxIndex);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(byte[] a, int index) {
        return isInIndex(a.length, index);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param minIndex
     *            開始インデックス
     * @param maxIndex
     *            終了インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(byte[] a, int minIndex, int maxIndex) {
        return isInIndex(a.length, minIndex, maxIndex);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(short[] a, int index) {
        return isInIndex(a.length, index);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param minIndex
     *            開始インデックス
     * @param maxIndex
     *            終了インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(short[] a, int minIndex, int maxIndex) {
        return isInIndex(a.length, minIndex, maxIndex);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(int[] a, int index) {
        return isInIndex(a.length, index);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param minIndex
     *            開始インデックス
     * @param maxIndex
     *            終了インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(int[] a, int minIndex, int maxIndex) {
        return isInIndex(a.length, minIndex, maxIndex);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(long[] a, int index) {
        return isInIndex(a.length, index);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param minIndex
     *            開始インデックス
     * @param maxIndex
     *            終了インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(long[] a, int minIndex, int maxIndex) {
        return isInIndex(a.length, minIndex, maxIndex);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(float[] a, int index) {
        return isInIndex(a.length, index);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param minIndex
     *            開始インデックス
     * @param maxIndex
     *            終了インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(float[] a, int minIndex, int maxIndex) {
        return isInIndex(a.length, minIndex, maxIndex);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(double[] a, int index) {
        return isInIndex(a.length, index);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param minIndex
     *            開始インデックス
     * @param maxIndex
     *            終了インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(double[] a, int minIndex, int maxIndex) {
        return isInIndex(a.length, minIndex, maxIndex);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param index
     *            インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(Object[] a, int index) {
        return isInIndex(a.length, index);
    }

    /**
     * 指定された配列の範囲内のインデックスか評価します。
     * 
     * @param a
     *            評価する配列
     * @param minIndex
     *            開始インデックス
     * @param maxIndex
     *            終了インデックス
     * @return 範囲内のインデックスの場合は true
     */
    public static boolean isInIndex(Object[] a, int minIndex, int maxIndex) {
        return isInIndex(a.length, minIndex, maxIndex);
    }

    /*
     * Helper
     */

    /* 指定の文字を検索して発見されたインデックスを返却します。 */
    static int indexOf(CharSequence cs, char c) {
        final int length = cs.length();
        for (int i = 0; i < length; i++) {
            if (cs.charAt(i) == c) {
                return i;
            }
        }
        return -1;
    }

    /* 指定の文字列を検索して発見されたインデックスを返却します。 */
    static int indexOf(CharSequence cs, CharSequence target) {
        final int endIndex = (cs.length() - 1);
        final int targetLength = target.length();
        for (int i = 0; i <= endIndex; i++) {
            if (i + targetLength > endIndex) {
                break;
            }
            int tlen = targetLength;
            int to = i;
            int po = 0;
            boolean match = true;
            while (--tlen >= 0) {
                if (cs.charAt(to++) != target.charAt(po++)) {
                    match = false;
                    break;
                }
            }
            if (match) {
                return i;
            }
        }
        return -1;
    }

    /* 接頭辞が一致するか検証します。 */
    static boolean suffix(CharSequence cs, CharSequence suffix) {
        final int end = (cs.length() - suffix.length());
        return startsWith(cs, suffix, end);
    }

    /* 接尾辞が一致するか検証します。 */
    static boolean prefix(CharSequence cs, CharSequence prefix) {
        return startsWith(cs, prefix, 0);
    }

    /* 指定の位置から検索して接頭辞が一致するか検証します。 */
    static boolean startsWith(CharSequence cs, CharSequence prefix, int offset) {
        int prefixLength = prefix.length();
        if ((offset < 0) || (offset > cs.length() - prefixLength)) {
            return false;
        }
        int targetOffset = offset;
        int prefixOffset = 0;
        while (--prefixLength >= 0) {
            if (cs.charAt(targetOffset++) != prefix.charAt(prefixOffset++)) {
                return false;
            }
        }
        return true;
    }

    /* 指定されたサイズの範囲内のインデックスか評価します。 */
    static boolean isInIndex(int size, int index) {
        return (-1 < index && index < size);
    }

    /* 指定されたサイズの範囲内のインデックスか評価します。 */
    static boolean isInIndex(int size, int minIndex, int maxIndex) {
        if (minIndex > maxIndex) {
            throw new IllegalArgumentException("min > max (" + minIndex + " > " + maxIndex + ")");
        }
        return (-1 < minIndex && minIndex < size) && (-1 < maxIndex && maxIndex < size);
    }

}
