/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.helpers;

import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

import shohaku.core.lang.VariableInteger;

/**
 * オブジェクトからログ文字列を生成するヘルパーメソッドを提供します。
 */
class LogHelper {

    /** 日時とミリ秒のフォーマット。 */
    static final String DATETIME_FORMAT = "yyyy-MM-dd HH:mm:ss.SSS";

    /** 値を出力する処理数の上限。 */
    static final int REPETITION_LIMIT = 255;

    /**
     * オブジェクトが配列の場合にリスト書式の文字列表現を生成し、以外は String.valueOf で生成した文字列を返却します。
     * 
     * @param o
     *            生成基オブジェクト
     * @return 文字列表現
     */
    static String array(Object o) {
        if (null == o) {
            return "null";
        }
        if (o.getClass().isArray()) {
            return toArrayString(o, new VariableInteger());
        }
        return String.valueOf(o);
    }

    /**
     * オブジェクトの詳細な文字列表現を構築します。 <br>
     * 指定されたオブジェクトが配列の場合は配列の内容の文字列表現を返却します。 <br>
     * 配列または Map, Collection の要素が配列の場合にその要素の内容の文字列表現を含んだ文字列を返却します。 <br>
     * 以外の場合は String.valueOf(o) により文字列に変換されます。 <br>
     * <br>
     * o が配列の場合は、角括弧 (「[]」) で囲まれた配列要素のリストで構成されます。 <br>
     * 隣接する要素は、文字「, 」(コンマおよび空白文字) で区切られます。 <br>
     * 要素はその型に対応する String.valueOf(#type# o) により文字列に変換されます。 <br>
     * <br>
     * o が null の場合 「null」 を返却します。
     * 
     * @param o
     *            文字列表現を返す配列
     * @return o の文字列表現
     */
    static String arrays(Object o) {
        if (null == o) {
            return "null";
        }
        final StringBuffer buff = new StringBuffer();
        append(o, buff, new VariableInteger());
        return buff.toString();
    }

    /*
     * private
     */

    private static String toArrayString(Object a, VariableInteger count) {
        Class type = a.getClass().getComponentType();
        if (type.isPrimitive()) {
            if (type == Character.TYPE) {
                return toString((char[]) a, count);
            } else if (type == Boolean.TYPE) {
                return toString((boolean[]) a, count);
            } else if (type == Byte.TYPE) {
                return toString((byte[]) a, count);
            } else if (type == Short.TYPE) {
                return toString((short[]) a, count);
            } else if (type == Integer.TYPE) {
                return toString((int[]) a, count);
            } else if (type == Long.TYPE) {
                return toString((long[]) a, count);
            } else if (type == Float.TYPE) {
                return toString((float[]) a, count);
            } else if (type == Double.TYPE) {
                return toString((double[]) a, count);
            }
        }
        return toString((Object[]) a, count);
    }

    private static String toString(long[] a, VariableInteger count) {
        if (a == null) {
            return "null";
        }

        final StringBuffer buff = new StringBuffer();
        appendClass(a, buff);
        if (a.length == 0) {
            return buff.append("[]").toString();
        }

        buff.append('[');
        buff.append(a[0]);
        count.increment();

        for (int i = 1; i < a.length; i++) {
            if (count.intValue() >= REPETITION_LIMIT) {
                buff.append("...");
                break;
            }
            buff.append(',');
            buff.append(a[i]);
            count.increment();
        }

        buff.append(']');
        return buff.toString();
    }

    private static String toString(int[] a, VariableInteger count) {
        if (a == null) {
            return "null";
        }
        final StringBuffer buff = new StringBuffer();
        appendClass(a, buff);
        if (a.length == 0) {
            return buff.append("[]").toString();
        }

        buff.append('[');
        buff.append(a[0]);
        count.increment();

        for (int i = 1; i < a.length; i++) {
            if (count.intValue() >= REPETITION_LIMIT) {
                buff.append("...");
                break;
            }
            buff.append(',');
            buff.append(a[i]);
            count.increment();
        }

        buff.append(']');
        return buff.toString();
    }

    private static String toString(short[] a, VariableInteger count) {
        if (a == null) {
            return "null";
        }
        final StringBuffer buff = new StringBuffer();
        appendClass(a, buff);
        if (a.length == 0) {
            return buff.append("[]").toString();
        }

        buff.append('[');
        buff.append(a[0]);
        count.increment();

        for (int i = 1; i < a.length; i++) {
            if (count.intValue() >= REPETITION_LIMIT) {
                buff.append("...");
                break;
            }
            buff.append(',');
            buff.append(a[i]);
            count.increment();
        }

        buff.append(']');
        return buff.toString();
    }

    private static String toString(char[] a, VariableInteger count) {
        if (a == null) {
            return "null";
        }
        final StringBuffer buff = new StringBuffer();
        appendClass(a, buff);
        if (a.length == 0) {
            return buff.append("[]").toString();
        }

        buff.append('[');
        buff.append(a[0]);
        count.increment();

        for (int i = 1; i < a.length; i++) {
            if (count.intValue() >= REPETITION_LIMIT) {
                buff.append("...");
                break;
            }
            buff.append(',');
            buff.append(a[i]);
            count.increment();
        }

        buff.append(']');
        return buff.toString();
    }

    private static String toString(byte[] a, VariableInteger count) {
        if (a == null) {
            return "null";
        }
        final StringBuffer buff = new StringBuffer();
        appendClass(a, buff);
        if (a.length == 0) {
            return buff.append("[]").toString();
        }

        buff.append('[');
        buff.append(a[0]);
        count.increment();

        for (int i = 1; i < a.length; i++) {
            if (count.intValue() >= REPETITION_LIMIT) {
                buff.append("...");
                break;
            }
            buff.append(',');
            buff.append(a[i]);
            count.increment();
        }

        buff.append(']');
        return buff.toString();
    }

    private static String toString(boolean[] a, VariableInteger count) {
        if (a == null) {
            return "null";
        }
        final StringBuffer buff = new StringBuffer();
        appendClass(a, buff);
        if (a.length == 0) {
            return buff.append("[]").toString();
        }

        buff.append('[');
        buff.append(a[0]);
        count.increment();

        for (int i = 1; i < a.length; i++) {
            if (count.intValue() >= REPETITION_LIMIT) {
                buff.append("...");
                break;
            }
            buff.append(',');
            buff.append(a[i]);
            count.increment();
        }

        buff.append(']');
        return buff.toString();
    }

    private static String toString(float[] a, VariableInteger count) {
        if (a == null) {
            return "null";
        }
        final StringBuffer buff = new StringBuffer();
        appendClass(a, buff);
        if (a.length == 0) {
            return buff.append("[]").toString();
        }

        buff.append('[');
        buff.append(a[0]);
        count.increment();

        for (int i = 1; i < a.length; i++) {
            if (count.intValue() >= REPETITION_LIMIT) {
                buff.append("...");
                break;
            }
            buff.append(',');
            buff.append(a[i]);
            count.increment();
        }

        buff.append(']');
        return buff.toString();
    }

    private static String toString(double[] a, VariableInteger count) {
        if (a == null) {
            return "null";
        }
        final StringBuffer buff = new StringBuffer();
        appendClass(a, buff);
        if (a.length == 0) {
            return buff.append("[]").toString();
        }

        buff.append('[');
        buff.append(a[0]);
        count.increment();

        for (int i = 1; i < a.length; i++) {
            if (count.intValue() >= REPETITION_LIMIT) {
                buff.append("...");
                break;
            }
            buff.append(',');
            buff.append(a[i]);
            count.increment();
        }

        buff.append(']');
        return buff.toString();
    }

    private static String toString(Object[] a, VariableInteger count) {
        if (a == null) {
            return "null";
        }
        final StringBuffer buff = new StringBuffer();
        appendClass(a, buff);
        if (a.length == 0) {
            return buff.append("[]").toString();
        }

        for (int i = 0; i < a.length; i++) {
            if (i == 0) {
                buff.append('[');
            } else {
                buff.append(',');
            }
            if (count.intValue() >= REPETITION_LIMIT) {
                buff.append("...");
                break;
            }
            buff.append(String.valueOf(a[i]));
            count.increment();
        }

        buff.append(']');
        return buff.toString();
    }

    private static String toDeepString(Object[] a, VariableInteger count) {
        if (a == null) {
            return "null";
        }
        final StringBuffer buff = new StringBuffer();
        appendClass(a, buff);
        if (a.length == 0) {
            return buff.append("[]").toString();
        }

        for (int i = 0; i < a.length; i++) {
            if (i == 0) {
                buff.append('[');
            } else {
                buff.append(',');
            }
            if (append(a[i], buff, count)) {
                break;
            }
        }

        buff.append(']');
        return buff.toString();
    }

    private static String toDeepString(Map map, VariableInteger count) {
        if (map == null) {
            return "null";
        }
        final StringBuffer buff = new StringBuffer();
        appendClass(map, buff);
        if (map.size() == 0) {
            return buff.append("{}").toString();
        }

        buff.append('{');
        Iterator i = map.entrySet().iterator();
        boolean hasNext = i.hasNext();
        while (hasNext) {
            Map.Entry e = (Map.Entry) i.next();
            Object key = e.getKey();
            Object value = e.getValue();
            if (key == map) {
                buff.append("(this Map)");
            } else {
                if (append(key, buff, count)) {
                    break;
                }
            }
            buff.append('=');
            if (value == map) {
                buff.append("(this Map)");
            } else {
                if (append(value, buff, count)) {
                    break;
                }
            }
            hasNext = i.hasNext();
            if (hasNext) {
                buff.append(',');
            }
        }
        buff.append('}');

        return buff.toString();
    }

    private static String toDeepString(Collection coll, VariableInteger count) {
        if (coll == null) {
            return "null";
        }
        final StringBuffer buff = new StringBuffer();
        appendClass(coll, buff);
        if (coll.size() == 0) {
            return buff.append("[]").toString();
        }

        buff.append('[');
        Iterator i = coll.iterator();
        boolean hasNext = i.hasNext();
        while (hasNext) {
            Object e = i.next();
            if (e == coll) {
                buff.append("(this Collection)");
            } else {
                if (append(e, buff, count)) {
                    break;
                }
            }
            hasNext = i.hasNext();
            if (hasNext) {
                buff.append(',');
            }
        }
        buff.append(']');

        return buff.toString();
    }

    private static boolean append(Object o, StringBuffer buff, VariableInteger count) {
        if (count.intValue() >= REPETITION_LIMIT) {
            buff.append("...");
            return true;
        }
        count.increment();
        if (o == null) {
            buff.append("null");
            return false;
        }
        if (o.getClass().isArray()) {
            if (o instanceof Object[]) {
                // Object[]
                buff.append(toDeepString((Object[]) o, count));
            } else {
                // Primitive
                buff.append(toArrayString(o, count));
            }
        } else if (o instanceof Map) {
            buff.append(toDeepString((Map) o, count));
        } else if (o instanceof Collection) {
            buff.append(toDeepString((Collection) o, count));
        } else if (o instanceof Timestamp) {
            buff.append(String.valueOf(o));
        } else if (o instanceof java.sql.Time) {
            buff.append(String.valueOf(o));
        } else if (o instanceof java.util.Date) {
            buff.append(HFmt.formatDate(o, DATETIME_FORMAT));
        } else if (o instanceof Calendar) {
            buff.append(HFmt.formatDate(o, DATETIME_FORMAT));
        } else {
            buff.append(String.valueOf(o));
        }
        return false;
    }

    private static void appendClass(Object o, StringBuffer buff) {
        buff.append('<');
        buff.append(o.getClass().getName());
        buff.append('>');
    }

}
