/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write max the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.helpers;

import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.IllegalCharsetNameException;
import java.nio.charset.UnsupportedCharsetException;
import java.text.DateFormatSymbols;
import java.text.DecimalFormatSymbols;
import java.util.Collection;
import java.util.Locale;

import shohaku.core.lang.RangeInt;

/**
 * 値を妥当性を評価するヘルパーメソッド群を提供します。
 */
public class HValid {

    /*
     * Chars
     */

    /**
     * 指定の文字のみで構成される文字シーケンスか評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param chars
     *            構成文字
     * @return 指定の文字のみで構成される文字シーケンスの場合は true
     */
    public static boolean isCharsOnly(CharSequence cs, char[] chars) {
        if (cs == null) {
            return false;
        }
        return (0 > HSeek.orOtherIndexOf(cs, chars, 1));
    }

    /**
     * 指定の文字のみで構成される文字シーケンスか評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param chars
     *            構成文字
     * @param fromIndex
     *            検索の開始位置を示すインデックス
     * @param toIndex
     *            検索の終了位置を示すインデックス
     * @return 指定の範囲内が指定の文字のみで構成される文字シーケンスの場合は true
     */
    public static boolean isCharsOnly(CharSequence cs, char[] chars, int fromIndex, int toIndex) {
        if (cs == null) {
            return false;
        }
        return (0 > HSeek.orOtherIndexOf(cs, toIndex, fromIndex, chars, 1));
    }

    /**
     * 指定の範囲内の文字のみで構成される文字シーケンスか評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param rangeChar
     *            有効な文字の範囲
     * @return 文字シーケンスの全要素が指定の範囲内の場合は true
     */
    public static boolean isCharsRange(CharSequence cs, RangeInt rangeChar) {
        if (cs == null) {
            return false;
        }
        final int len = cs.length();
        for (int i = 0; i < len; i++) {
            if (!rangeChar.contain(cs.charAt(i))) {
                return false;
            }
        }
        return true;
    }

    /**
     * 指定の範囲内の文字のみで構成される文字シーケンスか評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param rangeChars
     *            有効な文字の範囲
     * @return 文字シーケンスの全要素が指定の範囲内の場合は true
     */
    public static boolean isCharsRange(CharSequence cs, RangeInt[] rangeChars) {
        if (cs == null) {
            return false;
        }
        final int len = cs.length();
        boolean hit = false;
        for (int i = 0; i < len; i++) {
            final int ch = cs.charAt(i);
            for (int j = 0; j < rangeChars.length; j++) {
                if (rangeChars[j].contain(ch)) {
                    hit = true;
                    break;
                }
            }
            if (!hit) {
                return false;
            }
            hit = false;
        }
        return true;
    }

    /*
     * date
     */

    /**
     * 日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @param pattern
     *            書式パターン
     * @return 日付文字列として認識できる場合は true
     */
    public static boolean isDateTime(CharSequence date, String pattern) {
        return isDateTime(date, Locale.getDefault(), pattern);
    }

    /**
     * 日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @param locale
     *            ロケール
     * @param pattern
     *            書式パターン
     * @return 日付文字列として認識できる場合は true
     */
    public static boolean isDateTime(CharSequence date, Locale locale, String pattern) {
        return isDateTime(date, new DateFormatSymbols(locale), pattern);
    }

    /**
     * 日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @param symbols
     *            日付の記号セット
     * @param pattern
     *            書式パターン
     * @return 日付文字列として認識できる場合は true
     */
    public static boolean isDateTime(CharSequence date, DateFormatSymbols symbols, String pattern) {
        return isDateTime(date, symbols, pattern, false);
    }

    /**
     * 日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @param symbols
     *            日付の記号セット
     * @param pattern
     *            書式パターン
     * @param lenient
     *            日付/時刻解析を曖昧に行うか設定する、true=曖昧な解析
     * @return 日付文字列として認識できる場合は true
     */
    public static boolean isDateTime(CharSequence date, DateFormatSymbols symbols, String pattern, boolean lenient) {
        return (date == null) ? false : (null != HCnv.toDateTime(date, symbols, pattern, lenient));
    }

    /**
     * 一つ以上の書式パターンで日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @param patterns
     *            書式パターン
     * @return 一つ以上の書式パターンで日付文字列として認識できる場合は true
     */
    public static boolean isDateTime(CharSequence date, Collection patterns) {
        return isDateTime(date, Locale.getDefault(), patterns);
    }

    /**
     * 一つ以上の書式パターンで日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @param locale
     *            ロケール
     * @param patterns
     *            書式パターン
     * @return 一つ以上の書式パターンで日付文字列として認識できる場合は true
     */
    public static boolean isDateTime(CharSequence date, Locale locale, Collection patterns) {
        return isDateTime(date, new DateFormatSymbols(locale), patterns);
    }

    /**
     * 一つ以上の書式パターンで日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @param symbols
     *            日付の記号セット
     * @param patterns
     *            書式パターン
     * @return 一つ以上の書式パターンで日付文字列として認識できる場合は true
     */
    public static boolean isDateTime(CharSequence date, DateFormatSymbols symbols, Collection patterns) {
        return isDateTime(date, symbols, patterns, false);
    }

    /**
     * 一つ以上の書式パターンで日付文字列として認識できるか評価します。
     * 
     * @param date
     *            評価する日付文字列
     * @param symbols
     *            日付の記号セット
     * @param patterns
     *            書式パターン
     * @param lenient
     *            日付/時刻解析を曖昧に行うか設定する、true=曖昧な解析
     * @return 一つ以上の書式パターンで日付文字列として認識できる場合は true
     */
    public static boolean isDateTime(CharSequence date, DateFormatSymbols symbols, Collection patterns, boolean lenient) {
        return (date == null) ? false : (null != HCnv.toDateTime(date, symbols, patterns, lenient));
    }

    /*
     * Decimal
     */

    /**
     * 数値文字列として認識できるか評価します。
     * 
     * @param num
     *            評価する数値文字列
     * @param pattern
     *            書式パターン
     * @return 数値文字列として認識できる場合は true
     */
    public static boolean isDecimal(CharSequence num, String pattern) {
        return isDecimal(num, Locale.getDefault(), pattern);
    }

    /**
     * 数値文字列として認識できるか評価します。
     * 
     * @param num
     *            評価する数値文字列
     * @param locale
     *            ロケール
     * @param pattern
     *            書式パターン
     * @return 数値文字列として認識できる場合は true
     */
    public static boolean isDecimal(CharSequence num, Locale locale, String pattern) {
        return isDecimal(num, new DecimalFormatSymbols(locale), pattern);
    }

    /**
     * 数値文字列として認識できるか評価します。
     * 
     * @param num
     *            評価する数値文字列
     * @param symbols
     *            数値変換の記号セット
     * @param pattern
     *            書式パターン
     * @return 数値文字列として認識できる場合は true
     */
    public static boolean isDecimal(CharSequence num, DecimalFormatSymbols symbols, String pattern) {
        return (num == null) ? false : (null != HCnv.toDecimal(num, symbols, pattern));
    }

    /**
     * 一つ以上の書式パターンで数値文字列として認識できるか評価します。
     * 
     * @param num
     *            評価する数値文字列
     * @param patterns
     *            書式パターン
     * @return 一つ以上の書式パターンで数値文字列として認識できる場合は true
     */
    public static boolean isDecimal(CharSequence num, Collection patterns) {
        return isDecimal(num, Locale.getDefault(), patterns);
    }

    /**
     * 一つ以上の書式パターンで数値文字列として認識できるか評価します。
     * 
     * @param num
     *            評価する数値文字列
     * @param locale
     *            ロケール
     * @param patterns
     *            書式パターン
     * @return 一つ以上の書式パターンで数値文字列として認識できる場合は true
     */
    public static boolean isDecimal(CharSequence num, Locale locale, Collection patterns) {
        return isDecimal(num, new DecimalFormatSymbols(locale), patterns);
    }

    /**
     * 一つ以上の書式パターンで数値文字列として認識できるか評価します。
     * 
     * @param num
     *            評価する数値文字列
     * @param symbols
     *            数値変換の記号セット
     * @param patterns
     *            書式パターン
     * @return 一つ以上の書式パターンで数値文字列として認識できる場合は true
     */
    public static boolean isDecimal(CharSequence num, DecimalFormatSymbols symbols, Collection patterns) {
        return (num == null) ? false : (null != HCnv.toDecimal(num, symbols, patterns));
    }

    /*
     * Encode
     */

    /**
     * 指定の文字セットで指定された文字をエンコードできるかを評価します。
     * 
     * @param c
     *            評価する文字
     * @param charsetName
     *            要求された文字セットの名前 (標準名または別名)
     * @return 指定の文字セットで指定された文字をエンコードできる場合にかぎり true
     * @throws UnsupportedCharsetException
     *             指定された文字セット名が不当である場合
     * @throws IllegalCharsetNameException
     *             指定された文字セットを現在の Java 仮想マシンでは利用できない場合
     * @throws UnsupportedOperationException
     *             この文字セットがエンコードをサポートしない場合
     */
    public static boolean isEncode(char c, String charsetName) throws UnsupportedCharsetException, IllegalCharsetNameException, UnsupportedOperationException {
        return isEncode(c, Charset.forName(charsetName));
    }

    /**
     * 指定の文字セットで指定された文字をエンコードできるかを評価します。
     * 
     * @param c
     *            評価する文字
     * @param charset
     *            文字セット
     * @return 指定の文字セットで指定された文字をエンコードできる場合にかぎり true
     * @throws UnsupportedOperationException
     *             この文字セットがエンコードをサポートしない場合
     */
    public static boolean isEncode(char c, Charset charset) throws UnsupportedOperationException {
        final CharsetEncoder cEncoder = charset.newEncoder();
        return cEncoder.canEncode(c);
    }

    /**
     * 指定の文字セットで指定された文字シーケンスをエンコードできるかを評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param charsetName
     *            要求された文字セットの名前 (標準名または別名)
     * @return 指定の文字セットで指定された文字をエンコードできる場合にかぎり true
     * @throws UnsupportedCharsetException
     *             指定された文字セット名が不当である場合
     * @throws IllegalCharsetNameException
     *             指定された文字セットを現在の Java 仮想マシンでは利用できない場合
     * @throws UnsupportedOperationException
     *             この文字セットがエンコードをサポートしない場合
     */
    public static boolean isEncode(CharSequence cs, String charsetName) throws UnsupportedCharsetException, IllegalCharsetNameException, UnsupportedOperationException {
        return isEncode(cs, Charset.forName(charsetName));
    }

    /**
     * 指定の文字セットで指定された文字シーケンスをエンコードできるかを評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @param charset
     *            文字セット
     * @return 指定の文字セットで指定された文字をエンコードできる場合にかぎり true
     * @throws UnsupportedOperationException
     *             この文字セットがエンコードをサポートしない場合
     */
    public static boolean isEncode(CharSequence cs, Charset charset) throws UnsupportedOperationException {
        final CharsetEncoder cEncoder = charset.newEncoder();
        return cEncoder.canEncode(cs);
    }

}
