/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.helpers;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import shohaku.core.lang.Eval;
import shohaku.core.lang.RangeInt;

/**
 * 集合や複数のオブジェクトから、要素を探索するヘルパーメソッド群を提供します。
 */
public class HSeek {

    /*
     * evaluation
     */

    /**
     * null では無い引数を引数順の優先順位で返却します。<br>
     * 最後の引数以外が null の場合は最後の引数が返却されます。
     * 
     * @param e
     *            第一要素
     * @param e2
     *            第二要素
     * @return null では無い引数または最後の引数
     */
    public static Object notNull(Object e, Object e2) {
        return (e != null) ? e : e2;
    }

    /**
     * null では無い引数を引数順の優先順位で返却します。<br>
     * 最後の引数以外が null の場合は最後の引数が返却されます。
     * 
     * @param e
     *            第一要素
     * @param e2
     *            第二要素
     * @param e3
     *            第三要素
     * @return null では無い引数または最後の引数
     */
    public static Object notNull(Object e, Object e2, Object e3) {
        return (e != null) ? e : ((e2 != null) ? e2 : e3);
    }

    /**
     * 空文字列では無い引数を引数順の優先順位で返却します。<br>
     * 最後の引数以外が空文字列の場合は最後の引数が返却されます。
     * 
     * @param e
     *            第一要素
     * @param e2
     *            第二要素
     * @return 空文字列では無い引数または最後の引数
     */
    public static String notEmpty(String e, String e2) {
        return (!Eval.isEmpty(e)) ? e : e2;
    }

    /**
     * 空文字列では無い引数を引数順の優先順位で返却します。<br>
     * 最後の引数以外が空文字列の場合は最後の引数が返却されます。
     * 
     * @param e
     *            第一要素
     * @param e2
     *            第二要素
     * @param e3
     *            第三要素
     * @return 空文字列では無い引数または最後の引数
     */
    public static String notEmpty(String e, String e2, String e3) {
        return (!Eval.isEmpty(e)) ? e : ((!Eval.isEmpty(e2)) ? e2 : e3);
    }

    /**
     * 空白文字列では無い引数を引数順の優先順位で返却します。<br>
     * 最後の引数以外が空白文字列の場合は最後の引数が返却されます。
     * 
     * @param e
     *            第一要素
     * @param e2
     *            第二要素
     * @return 空白文字列では無い引数または最後の引数
     */
    public static String notBlank(String e, String e2) {
        return (!Eval.isBlank(e)) ? e : e2;
    }

    /**
     * 空白文字列では無い引数を引数順の優先順位で返却します。<br>
     * 最後の引数以外が空白文字列の場合は最後の引数が返却されます。
     * 
     * @param e
     *            第一要素
     * @param e2
     *            第二要素
     * @param e3
     *            第三要素
     * @return 空白文字列では無い引数または最後の引数
     */
    public static String notBlank(String e, String e2, String e3) {
        return (!Eval.isBlank(e)) ? e : ((!Eval.isBlank(e2)) ? e2 : e3);
    }

    /*
     * skip
     */

    /**
     * 引数に一致する文字をスキップしたインデックスを返却します。
     * 
     * @param cs
     *            文字シーケンス
     * @param range
     *            スキップ対象の文字の範囲
     * @param fromIndex
     *            探索の開始位置
     * @return 引数に一致する文字をスキップしたインデックス
     */
    public static int skip(CharSequence cs, RangeInt range, int fromIndex) {
        int inx = fromIndex;
        while (inx < cs.length()) {
            if (!range.contain(cs.charAt(inx))) {
                break;
            }
            inx++;
        }
        return inx;
    }

    /**
     * 引数に一致する文字をスキップしたインデックスを返却します。
     * 
     * @param cs
     *            文字シーケンス
     * @param ranges
     *            スキップ対象の文字の範囲
     * @param fromIndex
     *            探索の開始位置
     * @return 引数に一致する文字をスキップしたインデックス
     */
    public static int skip(CharSequence cs, RangeInt[] ranges, int fromIndex) {
        int inx = fromIndex;
        while (inx < cs.length()) {
            char c = cs.charAt(inx);
            boolean hit = false;
            for (int i = 0; i < ranges.length; i++) {
                if (ranges[i].contain(c)) {
                    hit = true;
                    break;
                }
            }
            if (!hit) {
                break;
            }
            inx++;
        }
        return inx;
    }

    /**
     * 引数に一致する文字をスキップしたインデックスを返却します。
     * 
     * @param cs
     *            文字シーケンス
     * @param c
     *            スキップ対象の文字
     * @param fromIndex
     *            探索の開始位置
     * @return 引数に一致する文字をスキップしたインデックス
     */
    public static int skip(CharSequence cs, char c, int fromIndex) {
        int inx = fromIndex;
        while (inx < cs.length()) {
            if (cs.charAt(inx) != c) {
                break;
            }
            inx++;
        }
        return inx;
    }

    /**
     * 引数により大きい文字をスキップしたインデックスを返却します。
     * 
     * @param cs
     *            文字シーケンス
     * @param c
     *            スキップ対象の文字
     * @param fromIndex
     *            探索の開始位置
     * @return 引数に一致する文字をスキップしたインデックス
     */
    public static int skipGreater(CharSequence cs, int c, int fromIndex) {
        int inx = fromIndex;
        while (inx < cs.length()) {
            if (!(cs.charAt(inx) > c)) {
                break;
            }
            inx++;
        }
        return inx;
    }

    /**
     * 引数により小さい文字をスキップしたインデックスを返却します。
     * 
     * @param cs
     *            文字シーケンス
     * @param c
     *            スキップ対象の文字
     * @param fromIndex
     *            探索の開始位置
     * @return 引数に一致する文字をスキップしたインデックス
     */
    public static int skipLess(CharSequence cs, int c, int fromIndex) {
        int inx = fromIndex;
        while (inx < cs.length()) {
            if (!(cs.charAt(inx) < c)) {
                break;
            }
            inx++;
        }
        return inx;
    }

    /**
     * 空白文字をスキップしたインデックスを返却します。
     * 
     * @param cs
     *            文字シーケンス
     * @param fromIndex
     *            探索の開始位置
     * @return 空白文字をスキップしたインデックス
     */
    public static int skipSp(CharSequence cs, int fromIndex) {
        int inx = fromIndex;
        while (inx < cs.length()) {
            if (cs.charAt(inx) > ' ') {
                break;
            }
            inx++;
        }
        return inx;
    }

    /**
     * 引数に一致する文字列をスキップしたインデックスを返却します。
     * 
     * @param cs
     *            文字シーケンス
     * @param chars
     *            スキップ対象の文字の配列
     * @param fromIndex
     *            探索の開始位置
     * @return 引数に一致する文字をスキップしたインデックス
     */
    public static int skip(CharSequence cs, CharSequence chars, int fromIndex) {
        final int charslen = chars.length();
        int inx = fromIndex;
        while (inx < cs.length()) {
            if (!Eval.isStartsWith(cs, chars, inx)) {
                break;
            }
            inx += charslen;
        }
        return inx;
    }

    /**
     * 引数に一致する文字をスキップしたインデックスを返却します。
     * 
     * @param cs
     *            文字シーケンス
     * @param chars
     *            スキップ対象の文字の配列
     * @param fromIndex
     *            探索の開始位置
     * @return 引数に一致する文字をスキップしたインデックス
     */
    public static int skip(CharSequence cs, char[] chars, int fromIndex) {
        int inx = fromIndex;
        while (inx < cs.length()) {
            char c = cs.charAt(inx);
            boolean hit = false;
            for (int i = 0; i < chars.length; i++) {
                if (c == chars[i]) {
                    hit = true;
                    break;
                }
            }
            if (!hit) {
                break;
            }
            inx++;
        }
        return inx;
    }

    /**
     * 指定の値と一致するコレクションの要素のカウント数を返却します。
     * 
     * @param c
     *            走査対象のコレクション
     * @param o
     *            検索する値
     * @return 指定の値と一致するコレクションの要素のカウント数
     */
    public static int count(Collection c, Object o) {
        if (c instanceof Set) {
            return (c.contains(o)) ? 1 : 0;
        } else {
            int count = 0;
            for (Iterator i = c.iterator(); i.hasNext();) {
                if (Eval.isEquals(i.next(), o)) {
                    count++;
                }
            }
            return count;
        }
    }

    /*
     * indexOf
     */

    /**
     * 文字シーケンス内の指定の文字と一致する文字を検索してそのインデックスを返します。<br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param cs
     *            検索対象の文字シーケンス
     * @param searchChar
     *            検索する文字
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 発見出来た場合そのインデックス、以外は -1
     */
    public static int indexOf(CharSequence cs, char searchChar, int count) {
        return indexOf(cs, 0, (cs.length() - 1), searchChar, count);
    }

    /**
     * 文字シーケンス内の指定の文字と一致する文字を指定の開始位置から検索してそのインデックスを返します。<br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param cs
     *            検索対象の文字シーケンス
     * @param fromIndex
     *            検索の開始位置を示すインデックス
     * @param searchChar
     *            検索する文字
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 発見出来た場合そのインデックス、以外は -1
     */
    public static int indexOf(CharSequence cs, int fromIndex, char searchChar, int count) {
        return indexOf(cs, fromIndex, (cs.length() - 1), searchChar, count);
    }

    /**
     * 文字シーケンス内の指定の文字と一致する文字を指定の開始位置と終了位置の間を検索してそのインデックスを返します。<br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param cs
     *            検索対象の文字シーケンス
     * @param fromIndex
     *            検索の開始位置を示すインデックス
     * @param toIndex
     *            検索の終了位置を示すインデックス
     * @param searchChar
     *            検索する文字
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 発見出来た場合そのインデックス、以外は -1
     */
    public static int indexOf(CharSequence cs, int fromIndex, int toIndex, char searchChar, int count) {
        if (!Eval.isInIndex(cs, fromIndex, toIndex)) {
            throw new IllegalArgumentException(HLog.range("index err. ", cs.length(), fromIndex, toIndex));
        }

        int _count = 0;
        final int limit = Math.abs(count);
        if (0 < count) {
            for (int i = fromIndex; i <= toIndex; i++) {
                if (cs.charAt(i) == searchChar) {
                    if (++_count == limit) {
                        return i;
                    }
                }
            }
        } else if (0 > count) {
            for (int i = toIndex; i >= fromIndex; i--) {
                if (cs.charAt(i) == searchChar) {
                    if (++_count == limit) {
                        return i;
                    }
                }
            }
        }
        return -1;
    }

    /**
     * 文字シーケンス内の指定の文字の何れかと一致する文字を検索してそのインデックスを返します。<br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param cs
     *            検索対象の文字シーケンス
     * @param searchChars
     *            検索する文字
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 発見出来た場合そのインデックス、以外は -1
     */
    public static int orIndexOf(CharSequence cs, char[] searchChars, int count) {
        return orIndexOf(cs, 0, (cs.length() - 1), searchChars, count);
    }

    /**
     * 文字シーケンス内の指定の文字の何れかと一致する文字を指定の開始位置から検索してそのインデックスを返します。<br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param cs
     *            検索対象の文字シーケンス
     * @param fromIndex
     *            検索の開始位置を示すインデックス
     * @param searchChars
     *            検索する文字
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 発見出来た場合そのインデックス、以外は -1
     */
    public static int orIndexOf(CharSequence cs, int fromIndex, char[] searchChars, int count) {
        return orIndexOf(cs, fromIndex, (cs.length() - 1), searchChars, count);
    }

    /**
     * 文字シーケンス内の指定の文字の何れかと一致する文字を指定の開始位置と終了位置の間を検索してそのインデックスを返します。<br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param cs
     *            検索対象の文字シーケンス
     * @param fromIndex
     *            検索の開始位置を示すインデックス
     * @param toIndex
     *            検索の終了位置を示すインデックス
     * @param searchChars
     *            検索する文字
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 発見出来た場合そのインデックス、以外は -1
     */
    public static int orIndexOf(CharSequence cs, int fromIndex, int toIndex, char[] searchChars, int count) {
        if (!Eval.isInIndex(cs, fromIndex, toIndex)) {
            throw new IllegalArgumentException(HLog.range("index err. ", cs.length(), fromIndex, toIndex));
        }

        int _count = 0;
        final int limit = Math.abs(count);
        if (0 < count) {
            for (int i = fromIndex; i <= toIndex; i++) {
                char c = cs.charAt(i);
                for (int j = 0; j < searchChars.length; j++) {
                    if (c == searchChars[j]) {
                        if (++_count == limit) {
                            return i;
                        }
                    }
                }
            }
        } else if (0 > count) {
            for (int i = toIndex; i >= fromIndex; i--) {
                char c = cs.charAt(i);
                for (int j = 0; j < searchChars.length; j++) {
                    if (c == searchChars[j]) {
                        if (++_count == limit) {
                            return i;
                        }
                    }
                }
            }
        }
        return -1;
    }

    /**
     * 文字シーケンス内の、指定の文字以外の何れかと一致する文字を検索してそのインデックスを返します。<br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param cs
     *            検索対象の文字シーケンス
     * @param searchChars
     *            検索する文字
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 発見出来た場合そのインデックス、以外は -1
     */
    public static int orOtherIndexOf(CharSequence cs, char[] searchChars, int count) {
        return orOtherIndexOf(cs, 0, (cs.length() - 1), searchChars, count);
    }

    /**
     * 文字シーケンス内の開始位置から、指定の文字以外の何れかと一致する文字を検索してそのインデックスを返します。<br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param cs
     *            検索対象の文字シーケンス
     * @param fromIndex
     *            検索の開始位置を示すインデックス
     * @param searchChars
     *            検索する文字
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 発見出来た場合そのインデックス、以外は -1
     */
    public static int orOtherIndexOf(CharSequence cs, int fromIndex, char[] searchChars, int count) {
        return orOtherIndexOf(cs, fromIndex, (cs.length() - 1), searchChars, count);
    }

    /**
     * 文字シーケンス内の開始位置と終了位置の範囲で、指定の文字以外の何れかと一致する文字を検索してそのインデックスを返します。<br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param cs
     *            検索対象の文字シーケンス
     * @param fromIndex
     *            検索の開始位置を示すインデックス
     * @param toIndex
     *            検索の終了位置を示すインデックス
     * @param searchChars
     *            検索する文字
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 発見出来た場合そのインデックス、以外は -1
     */
    public static int orOtherIndexOf(CharSequence cs, int fromIndex, int toIndex, char[] searchChars, int count) {
        if (!Eval.isInIndex(cs, fromIndex, toIndex)) {
            throw new IllegalArgumentException(HLog.range("index err. ", cs.length(), fromIndex, toIndex));
        }

        int _count = 0;
        final int limit = Math.abs(count);
        if (0 < count) {
            boolean hit = false;
            for (int i = fromIndex; i <= toIndex; i++) {
                char c = cs.charAt(i);
                for (int j = 0; j < searchChars.length; j++) {
                    if (c == searchChars[j]) {
                        hit = true;
                        break;
                    }
                }
                if (!hit) {
                    if (++_count == limit) {
                        return i;
                    }
                }
                hit = false;
            }
        } else if (0 > count) {
            boolean hit = false;
            for (int i = toIndex; i >= fromIndex; i--) {
                char c = cs.charAt(i);
                for (int j = 0; j < searchChars.length; j++) {
                    if (c == searchChars[j]) {
                        hit = true;
                        break;
                    }
                }
                if (!hit) {
                    if (++_count == limit) {
                        return i;
                    }
                }
                hit = false;
            }
        }
        return -1;
    }

    /**
     * 文字シーケンス内の指定の文字と一致する文字シーケンスを検索してその開始位置のインデックスを返します。<br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param cs
     *            検索対象の文字シーケンス
     * @param searchChars
     *            検索する文字シーケンス
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 発見出来た場合その開始位置のインデックス、以外は -1
     */
    public static int indexOf(CharSequence cs, CharSequence searchChars, int count) {
        return indexOf(cs, 0, (cs.length() - 1), searchChars, count);
    }

    /**
     * 文字シーケンス内の指定の文字と一致する文字シーケンスを指定の開始位置から検索してその開始位置のインデックスを返します。<br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param cs
     *            検索対象の文字シーケンス
     * @param fromIndex
     *            検索の開始位置を示すインデックス
     * @param searchChars
     *            検索する文字シーケンス
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 発見出来た場合その開始位置のインデックス、以外は -1
     */
    public static int indexOf(CharSequence cs, int fromIndex, CharSequence searchChars, int count) {
        return indexOf(cs, fromIndex, (cs.length() - 1), searchChars, count);
    }

    /**
     * 文字シーケンス内の指定の文字と一致する文字シーケンスを指定の開始位置と終了位置の間を検索してその開始位置のインデックスを返します。<br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param cs
     *            検索対象の文字シーケンス
     * @param fromIndex
     *            検索の開始位置を示すインデックス
     * @param toIndex
     *            検索の終了位置を示すインデックス
     * @param searchChars
     *            検索する文字シーケンス
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 発見出来た場合その開始位置のインデックス、以外は -1
     */
    public static int indexOf(CharSequence cs, int fromIndex, int toIndex, CharSequence searchChars, int count) {
        if (!Eval.isInIndex(cs, fromIndex, toIndex)) {
            throw new IllegalArgumentException(HLog.range("index err. ", cs.length(), fromIndex, toIndex));
        }

        // target Empty As Not Found.
        if (searchChars.length() == 0) {
            return -1;
        }

        int _count = 0;
        final int limit = Math.abs(count);
        if (0 < count) {
            for (int i = fromIndex; i <= toIndex; i++) {
                int plen = searchChars.length();
                if (i + plen > toIndex) {
                    break;
                }
                int to = i;
                int po = 0;
                boolean match = true;
                while (--plen >= 0) {
                    if (cs.charAt(to++) != searchChars.charAt(po++)) {
                        match = false;
                        break;
                    }
                }
                if (match) {
                    if (++_count == limit) {
                        return i;
                    }
                }
            }
        } else if (0 > count) {
            for (int i = toIndex; i >= fromIndex; i--) {
                int plen = searchChars.length();
                if (i - plen < fromIndex) {
                    break;
                }
                int to = i - plen;
                int po = 0;
                boolean match = true;
                while (--plen >= 0) {
                    if (cs.charAt(to++) != searchChars.charAt(po++)) {
                        match = false;
                        break;
                    }
                }
                if (match) {
                    if (++_count == limit) {
                        return i;
                    }
                }
            }
        }
        // Not Found.
        return -1;
    }

    /**
     * 指定された値と同値の要素を開始位置から終了位置まで線形検索しそのインデックスを返却します。 <br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param list
     *            検索元のリスト
     * @param fromIndex
     *            検索の開始インデックス
     * @param toIndex
     *            検索の終了インデックス
     * @param search
     *            検索する値
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     */
    public static int indexOf(List list, int fromIndex, int toIndex, Object search, int count) {
        if (!Eval.isInIndex(list, fromIndex, toIndex)) {
            throw new IllegalArgumentException(HLog.range("index err. ", list.size(), fromIndex, toIndex));
        }

        int _count = 0;
        final int limit = Math.abs(count);
        if (0 < count) {
            if (search == null) {
                for (int i = fromIndex; i <= toIndex; i++) {
                    if (list.get(i) == null) {
                        if (++_count == limit) {
                            return i;
                        }
                    }
                }
            } else {
                for (int i = fromIndex; i <= toIndex; i++) {
                    if (search.equals(list.get(i))) {
                        if (++_count == limit) {
                            return i;
                        }
                    }
                }
            }
        } else if (0 > count) {
            if (search == null) {
                for (int i = toIndex; i >= fromIndex; i--) {
                    if (list.get(i) == null) {
                        if (++_count == limit) {
                            return i;
                        }
                    }
                }
            } else {
                for (int i = toIndex; i >= fromIndex; i--) {
                    if (search.equals(list.get(i))) {
                        if (++_count == limit) {
                            return i;
                        }
                    }
                }
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を開始位置から終了位置まで線形検索しそのインデックスを返却します。 <br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param a
     *            検索元の配列
     * @param fromIndex
     *            検索の開始インデックス
     * @param toIndex
     *            検索の終了インデックス
     * @param search
     *            検索する値
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     */
    public static int indexOf(Object[] a, int fromIndex, int toIndex, Object search, int count) {
        if (!Eval.isInIndex(a, fromIndex, toIndex)) {
            throw new IllegalArgumentException(HLog.range("index err. ", a.length, fromIndex, toIndex));
        }

        int _count = 0;
        final int limit = Math.abs(count);
        if (0 < count) {
            if (search == null) {
                for (int i = fromIndex; i <= toIndex; i++) {
                    if (a[i] == null) {
                        if (++_count == limit) {
                            return i;
                        }
                    }
                }
            } else {
                for (int i = fromIndex; i <= toIndex; i++) {
                    if (search.equals(a[i])) {
                        if (++_count == limit) {
                            return i;
                        }
                    }
                }
            }
        } else if (0 > count) {
            if (search == null) {
                for (int i = toIndex; i >= fromIndex; i--) {
                    if (a[i] == null) {
                        if (++_count == limit) {
                            return i;
                        }
                    }
                }
            } else {
                for (int i = toIndex; i >= fromIndex; i--) {
                    if (search.equals(a[i])) {
                        if (++_count == limit) {
                            return i;
                        }
                    }
                }
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を開始位置から終了位置まで線形検索しそのインデックスを返却します。<br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param a
     *            検索元の配列
     * @param fromIndex
     *            検索の開始インデックス
     * @param toIndex
     *            検索の終了インデックス
     * @param search
     *            検索する値
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     */
    public static int indexOf(boolean[] a, int fromIndex, int toIndex, boolean search, int count) {
        if (!Eval.isInIndex(a, fromIndex, toIndex)) {
            throw new IllegalArgumentException(HLog.range("index err. ", a.length, fromIndex, toIndex));
        }

        int _count = 0;
        final int limit = Math.abs(count);
        if (0 < count) {
            for (int i = fromIndex; i <= toIndex; i++) {
                if (search == a[i]) {
                    if (++_count == limit) {
                        return i;
                    }
                }
            }
        } else if (0 > count) {
            for (int i = toIndex; i >= fromIndex; i--) {
                if (search == a[i]) {
                    if (++_count == limit) {
                        return i;
                    }
                }
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を開始位置から終了位置まで線形検索しそのインデックスを返却します。<br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param a
     *            検索元の配列
     * @param fromIndex
     *            検索の開始インデックス
     * @param toIndex
     *            検索の終了インデックス
     * @param search
     *            検索する値
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     */
    public static int indexOf(char[] a, int fromIndex, int toIndex, char search, int count) {
        if (!Eval.isInIndex(a, fromIndex, toIndex)) {
            throw new IllegalArgumentException(HLog.range("index err. ", a.length, fromIndex, toIndex));
        }

        int _count = 0;
        final int limit = Math.abs(count);
        if (0 < count) {
            for (int i = fromIndex; i <= toIndex; i++) {
                if (search == a[i]) {
                    if (++_count == limit) {
                        return i;
                    }
                }
            }
        } else if (0 > count) {
            for (int i = toIndex; i >= fromIndex; i--) {
                if (search == a[i]) {
                    if (++_count == limit) {
                        return i;
                    }
                }
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を開始位置から終了位置まで線形検索しそのインデックスを返却します。<br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param a
     *            検索元の配列
     * @param fromIndex
     *            検索の開始インデックス
     * @param toIndex
     *            検索の終了インデックス
     * @param search
     *            検索する値
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     */
    public static int indexOf(byte[] a, int fromIndex, int toIndex, byte search, int count) {
        if (!Eval.isInIndex(a, fromIndex, toIndex)) {
            throw new IllegalArgumentException(HLog.range("index err. ", a.length, fromIndex, toIndex));
        }

        int _count = 0;
        final int limit = Math.abs(count);
        if (0 < count) {
            for (int i = fromIndex; i <= toIndex; i++) {
                if (search == a[i]) {
                    if (++_count == limit) {
                        return i;
                    }
                }
            }
        } else if (0 > count) {
            for (int i = toIndex; i >= fromIndex; i--) {
                if (search == a[i]) {
                    if (++_count == limit) {
                        return i;
                    }
                }
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を開始位置から終了位置まで線形検索しそのインデックスを返却します。<br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param a
     *            検索元の配列
     * @param fromIndex
     *            検索の開始インデックス
     * @param toIndex
     *            検索の終了インデックス
     * @param search
     *            検索する値
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     */
    public static int indexOf(short[] a, int fromIndex, int toIndex, short search, int count) {
        if (!Eval.isInIndex(a, fromIndex, toIndex)) {
            throw new IllegalArgumentException(HLog.range("index err. ", a.length, fromIndex, toIndex));
        }

        int _count = 0;
        final int limit = Math.abs(count);
        if (0 < count) {
            for (int i = fromIndex; i <= toIndex; i++) {
                if (search == a[i]) {
                    if (++_count == limit) {
                        return i;
                    }
                }
            }
        } else if (0 > count) {
            for (int i = toIndex; i >= fromIndex; i--) {
                if (search == a[i]) {
                    if (++_count == limit) {
                        return i;
                    }
                }
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を開始位置から終了位置まで線形検索しそのインデックスを返却します。<br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param a
     *            検索元の配列
     * @param fromIndex
     *            検索の開始インデックス
     * @param toIndex
     *            検索の終了インデックス
     * @param search
     *            検索する値
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     */
    public static int indexOf(int[] a, int fromIndex, int toIndex, int search, int count) {
        if (!Eval.isInIndex(a, fromIndex, toIndex)) {
            throw new IllegalArgumentException(HLog.range("index err. ", a.length, fromIndex, toIndex));
        }

        int _count = 0;
        final int limit = Math.abs(count);
        if (0 < count) {
            for (int i = fromIndex; i <= toIndex; i++) {
                if (search == a[i]) {
                    if (++_count == limit) {
                        return i;
                    }
                }
            }
        } else if (0 > count) {
            for (int i = toIndex; i >= fromIndex; i--) {
                if (search == a[i]) {
                    if (++_count == limit) {
                        return i;
                    }
                }
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を開始位置から終了位置まで線形検索しそのインデックスを返却します。 <br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param a
     *            検索元の配列
     * @param fromIndex
     *            検索の開始インデックス
     * @param toIndex
     *            検索の終了インデックス
     * @param search
     *            検索する値
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     */
    public static int indexOf(long[] a, int fromIndex, int toIndex, long search, int count) {
        if (!Eval.isInIndex(a, fromIndex, toIndex)) {
            throw new IllegalArgumentException(HLog.range("index err. ", a.length, fromIndex, toIndex));
        }

        int _count = 0;
        final int limit = Math.abs(count);
        if (0 < count) {
            for (int i = fromIndex; i <= toIndex; i++) {
                if (search == a[i]) {
                    if (++_count == limit) {
                        return i;
                    }
                }
            }
        } else if (0 > count) {
            for (int i = toIndex; i >= fromIndex; i--) {
                if (search == a[i]) {
                    if (++_count == limit) {
                        return i;
                    }
                }
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を開始位置から終了位置まで線形検索しそのインデックスを返却します。<br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param a
     *            検索元の配列
     * @param fromIndex
     *            検索の開始インデックス
     * @param toIndex
     *            検索の終了インデックス
     * @param search
     *            検索する値
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     */
    public static int indexOf(float[] a, int fromIndex, int toIndex, float search, int count) {
        if (!Eval.isInIndex(a, fromIndex, toIndex)) {
            throw new IllegalArgumentException(HLog.range("index err. ", a.length, fromIndex, toIndex));
        }

        int _count = 0;
        final int limit = Math.abs(count);
        if (0 < count) {
            for (int i = fromIndex; i <= toIndex; i++) {
                if (search == a[i]) {
                    if (++_count == limit) {
                        return i;
                    }
                }
            }
        } else if (0 > count) {
            for (int i = toIndex; i >= fromIndex; i--) {
                if (search == a[i]) {
                    if (++_count == limit) {
                        return i;
                    }
                }
            }
        }
        return -1;
    }

    /**
     * 指定された値と同値の要素を開始位置から終了位置まで線形検索しそのインデックスを返却します。<br>
     * 発見出来なかった場合は -1 を返します。<br>
     * また引数 count の回数分に一致を検索します、つまり count=3 の場合は２回発見されても -1 が返却されます。<br>
     * また引数 count に負数を指定した場合は、その絶対値の回数で後方検索を実行します、つまり count=-3 は後方から 3 回目に一致したインデックスを返却します。<br>
     * また引数 count が 0 の場合は常に -1 が返却されます。
     * 
     * @param a
     *            検索元の配列
     * @param fromIndex
     *            検索の開始インデックス
     * @param toIndex
     *            検索の終了インデックス
     * @param search
     *            検索する値
     * @param count
     *            一致する回数、負数の場合後方検索
     * @return 同値の要素が最初に検出されたインデックス、同値が無い場合は -1
     */
    public static int indexOf(double[] a, int fromIndex, int toIndex, double search, int count) {
        if (!Eval.isInIndex(a, fromIndex, toIndex)) {
            throw new IllegalArgumentException(HLog.range("index err. ", a.length, fromIndex, toIndex));
        }

        int _count = 0;
        final int limit = Math.abs(count);
        if (0 < count) {
            for (int i = fromIndex; i <= toIndex; i++) {
                if (search == a[i]) {
                    if (++_count == limit) {
                        return i;
                    }
                }
            }
        } else if (0 > count) {
            for (int i = toIndex; i >= fromIndex; i--) {
                if (search == a[i]) {
                    if (++_count == limit) {
                        return i;
                    }
                }
            }
        }
        return -1;
    }
}
