/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.helpers;

import java.text.DateFormat;
import java.text.DateFormatSymbols;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;

import shohaku.core.lang.EvalSet;

/**
 * フォーマット変換を行うヘルパーメソッド群を提供します。
 */
public class HFmt {

    /*
     * Decimal
     */

    /**
     * 数値を引数で指定される書式とデフォルトロケールで文字列に変換します。
     * 
     * @param number
     *            数値
     * @param pattern
     *            書式
     * @return 変換後の文字列
     */
    public static String formatDecimal(Number number, String pattern) {
        return formatDecimal(number, Locale.getDefault(), pattern);
    }

    /**
     * 数値を引数で指定される書式と指定のロケールで文字列に変換します。
     * 
     * @param number
     *            数値
     * @param locale
     *            ロケール
     * @param pattern
     *            書式
     * @return 変換後の文字列
     */
    public static String formatDecimal(Number number, Locale locale, String pattern) {
        return formatDecimal(number, new DecimalFormatSymbols(locale), pattern);
    }

    /**
     * 数値を引数で指定される書式と指定の数値変換の記号セットで文字列に変換します。
     * 
     * @param number
     *            数値
     * @param symbols
     *            数値変換の記号セット
     * @param pattern
     *            書式
     * @return 変換後の文字列
     */
    public static String formatDecimal(Number number, DecimalFormatSymbols symbols, String pattern) {
        final DecimalFormat formatter = new DecimalFormat(pattern, symbols);
        return formatter.format(number);
    }

    /*
     * DateTime
     */

    /**
     * 日付型またはカレンダー、数値型を指定の書式で日付文字列に変換します。<br>
     * つまり有効な型は、java.util.Date、java.util.Calendar、java.lang.Number のいずれかです。
     * 
     * @param date
     *            日付型またはカレンダー、数値型
     * @param pattern
     *            書式パターン
     * @return 日付文字列
     */
    public static String formatDate(Object date, String pattern) {
        return formatDate(date, Locale.getDefault(), pattern);
    }

    /**
     * 日付型またはカレンダー、数値型を指定の書式で日付文字列に変換します。<br>
     * つまり有効な型は、java.util.Date、java.util.Calendar、java.lang.Number のいずれかです。
     * 
     * @param date
     *            日付型またはカレンダー、数値型
     * @param locale
     *            ロケール
     * @param pattern
     *            書式パターン
     * @return 日付文字列
     */
    public static String formatDate(Object date, Locale locale, String pattern) {
        return formatDate(date, new DateFormatSymbols(locale), pattern);
    }

    /**
     * 日付型またはカレンダー、数値型を指定の書式で日付文字列に変換します。<br>
     * つまり有効な型は、java.util.Date、java.util.Calendar、java.lang.Number のいずれかです。
     * 
     * @param date
     *            日付型またはカレンダー、数値型
     * @param symbols
     *            日付変換の記号セット
     * @param pattern
     *            書式パターン
     * @return 日付文字列
     */
    public static String formatDate(Object date, DateFormatSymbols symbols, String pattern) {
        if (!EvalSet.isOrAssignable(date, Date.class, Calendar.class, Number.class)) {
            throw new IllegalArgumentException("argument o is not Date or Calendar :" + date);
        }
        final DateFormat formatter = new SimpleDateFormat(pattern, symbols);
        return formatter.format(HCnv.asDataObject(date));
    }

    /**
     * 日付型またはカレンダー、数値型を指定のスタイルで日付文字列に変換します。<br>
     * つまり有効な型は、java.util.Date、java.util.Calendar、java.lang.Number のいずれかです。
     * 
     * @param date
     *            日付型またはカレンダー、数値型
     * @param style
     *            書式スタイル
     * @param locale
     *            ロケール
     * @return 日付文字列
     */
    public static String formatDate(Object date, int style, Locale locale) {
        if (!EvalSet.isOrAssignable(date, Date.class, Calendar.class, Number.class)) {
            throw new IllegalArgumentException("argument o is not Date or Calendar :" + date);
        }
        final DateFormat formatter = DateFormat.getDateInstance(style, locale);
        return formatter.format(HCnv.asDataObject(date));
    }

    /**
     * 日付型またはカレンダー、数値型を指定のスタイルで日付文字列に変換します。<br>
     * つまり有効な型は、java.util.Date、java.util.Calendar、java.lang.Number のいずれかです。
     * 
     * @param date
     *            日付型またはカレンダー、数値型
     * @param dateStyle
     *            日付書式スタイル
     * @param timeStyle
     *            時刻書式スタイル
     * @param locale
     *            ロケール
     * @return 日付文字列
     */
    public static String formatDateTime(Object date, int dateStyle, int timeStyle, Locale locale) {
        if (!EvalSet.isOrAssignable(date, Date.class, Calendar.class, Number.class)) {
            throw new IllegalArgumentException("argument o is not Date or Calendar :" + date);
        }
        final DateFormat formatter = DateFormat.getDateTimeInstance(dateStyle, timeStyle, locale);
        return formatter.format(HCnv.asDataObject(date));
    }

    /**
     * 日付型またはカレンダー、数値型を指定のスタイルで日付文字列に変換します。<br>
     * つまり有効な型は、java.util.Date、java.util.Calendar、java.lang.Number のいずれかです。
     * 
     * @param date
     *            日付型またはカレンダー、数値型
     * @param style
     *            書式スタイル
     * @param locale
     *            ロケール
     * @return 日付文字列
     */
    public static String formatTime(Object date, int style, Locale locale) {
        if (!EvalSet.isOrAssignable(date, Date.class, Calendar.class, Number.class)) {
            throw new IllegalArgumentException("argument o is not Date or Calendar :" + date);
        }
        final DateFormat formatter = DateFormat.getTimeInstance(style, locale);
        return formatter.format(HCnv.asDataObject(date));
    }

}
